/*
 * VC-1 and WMV3 decoder VLC table data
 * copyright (c) 2011 Mashiat Sarker Shakkhar
 * copyright (c) 2006 Konstantin Shishkov
 * (c) 2005 anonymous, Alex Beregszaszi, Michael Niedermayer
 *
 * This file is part of FFmpeg.
 *
 * FFmpeg is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * FFmpeg is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with FFmpeg; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

/**
 * @file
 * VC-1 tables.
 */

#ifndef AVCODEC_VC1_VLC_DATA_H
#define AVCODEC_VC1_VLC_DATA_H

#include <stdint.h>

#include "vc1data.h"

/* BitPlane IMODE - such a small table... */
static const uint8_t vc1_imode_codes[7] = {
    0, 2, 1, 3, 1, 2, 3
};
static const uint8_t vc1_imode_bits[7] = {
    4, 2, 3, 2, 4, 3, 3
};

/* Normal-2 imode */
static const uint8_t vc1_norm2_codes[4] = {
    0, 4, 5, 3
};
static const uint8_t vc1_norm2_bits[4] = {
    1, 3, 3, 2
};

static const uint16_t vc1_norm6_codes[64] = {
    0x001, 0x002, 0x003, 0x000, 0x004, 0x001, 0x002, 0x047, 0x005, 0x003, 0x004, 0x04B, 0x005, 0x04D, 0x04E, 0x30E,
    0x006, 0x006, 0x007, 0x053, 0x008, 0x055, 0x056, 0x30D, 0x009, 0x059, 0x05A, 0x30C, 0x05C, 0x30B, 0x30A, 0x037,
    0x007, 0x00A, 0x00B, 0x043, 0x00C, 0x045, 0x046, 0x309, 0x00D, 0x049, 0x04A, 0x308, 0x04C, 0x307, 0x306, 0x036,
    0x00E, 0x051, 0x052, 0x305, 0x054, 0x304, 0x303, 0x035, 0x058, 0x302, 0x301, 0x034, 0x300, 0x033, 0x032, 0x007,
};

static const uint8_t vc1_norm6_bits[64] = {
    1,  4,  4,  8,  4,  8,  8, 10,  4,  8,  8, 10,  8, 10, 10, 13,
    4,  8,  8, 10,  8, 10, 10, 13,  8, 10, 10, 13, 10, 13, 13,  9,
    4,  8,  8, 10,  8, 10, 10, 13,  8, 10, 10, 13, 10, 13, 13,  9,
    8, 10, 10, 13, 10, 13, 13,  9, 10, 13, 13,  9, 13,  9,  9,  6,
};

/* 4MV Block pattern VLC tables */
static const uint8_t vc1_4mv_block_pattern_codes[4][16] = {
    { 14, 58, 59, 25, 12, 26, 15, 15, 13, 24, 27,  0, 28,  1,  2,  2 },
    {  8, 18, 19,  4, 20,  5, 30, 11, 21, 31,  6, 12,  7, 13, 14,  0 },
    { 15,  6,  7,  2,  8,  3, 28,  9, 10, 29,  4, 11,  5, 12, 13,  0 },
    {  0, 11, 12,  4, 13,  5, 30, 16, 14, 31,  6, 17,  7, 18, 19, 10 }
};
static const uint8_t vc1_4mv_block_pattern_bits[4][16] = {
    { 5, 6, 6, 5, 5, 5, 5, 4, 5, 5, 5, 3, 5, 3, 3, 2 },
    { 4, 5, 5, 4, 5, 4, 5, 4, 5, 5, 4, 4, 4, 4, 4, 2 },
    { 4, 4, 4, 4, 4, 4, 5, 4, 4, 5, 4, 4, 4, 4, 4, 3 },
    { 2, 4, 4, 4, 4, 4, 5, 5, 4, 5, 4, 5, 4, 5, 5, 4 }
};

/* 2MV Block pattern VLC tables */
static const uint8_t vc1_2mv_block_pattern_codes[4][4] = {
    { 2, 1, 0, 3 }, { 1, 0, 2, 3 }, { 2, 0, 3, 1 }, { 1, 3, 2, 0 }
};

static const uint8_t vc1_2mv_block_pattern_bits[4][4] = {
    { 2, 2, 2, 2 }, { 1, 2, 3, 3 }, { 3, 2, 3, 1 }, { 1, 3, 3, 2 }
};

/* Interlaced frame picture 4MV MBMODE VLC tables (p. 246, p. 360) */
static const uint16_t vc1_intfr_4mv_mbmode_codes[4][15] = {
    { 22,  17,  0, 47,  32, 10,  1,  3, 67,  133, 132,  92,  19,  93,   18 },
    {  3,  45,  0,  7,  23,  6,  1,  2, 10,   39,  44,   8,  18,  77,   76 },
    { 15,   6, 28,  9,  41,  6,  2, 15, 14,    8,  40,  29,   0,  21,   11 },
    {  7, 198,  1,  2, 193, 13, 25,  0, 97, 1599,  98, 398, 798, 192, 1598 }
};

static const uint8_t vc1_intfr_4mv_mbmode_bits[4][15] = {
    { 5, 5, 2, 6, 6, 4, 2, 2, 7,  8, 8,  7,  5, 7,  5 },
    { 3, 6, 3, 3, 5, 3, 3, 3, 4,  6, 6,  4,  5, 7,  7 },
    { 4, 3, 5, 5, 7, 4, 2, 5, 5,  5, 7,  5,  2, 6,  5 },
    { 4, 9, 1, 3, 9, 5, 6, 2, 8, 12, 8, 10, 11, 9, 12 }
};

/* Interlaced frame picture NON-4MV MBMODE VLC tables (p. 363) */
static const uint8_t vc1_intfr_non4mv_mbmode_codes[4][9] = {
    {  9, 22,  0, 17, 16, 10,  1,  3, 23 },
    {  7,  0,  5,  2,  1,  1,  6,  3,  4 },
    {  1,  0, 10, 23, 44,  8,  3,  9, 45 },
    {  7, 97,  1,  2, 49, 13, 25,  0, 96 }
};

static const uint8_t vc1_intfr_non4mv_mbmode_bits[4][9] = {
    {  4,  5,  2,  5,  5,  4,  2,  2,  5 },
    {  3,  4,  6,  2,  3,  2,  3,  5,  6 },
    {  2,  2,  4,  5,  6,  4,  2,  4,  6 },
    {  4,  8,  1,  3,  7,  5,  6,  2,  8 }
};

/* Interlaced field picture MBMODE VLC tables (p. 356 - 11.4.1, 11.4.2) */
/* mixed-MV */
static const uint8_t vc1_if_mmv_mbmode_codes[8][8] = {
    { 16, 17,  3,  3,  0,  5,  9,  2 },
    {  8,  9,  3,  6,  7,  0,  5,  2 },
    { 16, 17,  5,  3,  0,  3,  9,  2 },
    { 56, 57, 15,  4,  5,  6, 29,  0 },
    { 52, 53, 27, 14, 15,  2, 12,  0 },
    { 56, 57, 29,  5,  6,  0, 15,  4 },
    { 16, 17,  6,  7,  0,  1,  9,  5 },
    { 56, 57,  0,  5,  6, 29,  4, 15 }
};
static const uint8_t vc1_if_mmv_mbmode_bits[8][8] = {
    { 6, 6, 2, 3, 2, 4, 5, 2 },
    { 5, 5, 3, 3, 3, 2, 4, 2 },
    { 6, 6, 4, 3, 2, 2, 5, 2 },
    { 6, 6, 4, 3, 3, 3, 5, 1 },
    { 6, 6, 5, 4, 4, 2, 4, 1 },
    { 6, 6, 5, 3, 3, 1, 4, 3 },
    { 5, 5, 3, 3, 2, 2, 4, 3 },
    { 6, 6, 1, 3, 3, 5, 3, 4 }
};

/* 1MV */
static const uint8_t vc1_if_1mv_mbmode_codes[8][6] = {
    {  0,  1,  1,  1,  1,  1 },
    {  0,  1,  1,  1,  1,  1 },
    { 16, 17,  3,  0,  9,  5 },
    { 20, 21,  3, 11,  0,  4 },
    {  4,  5,  2,  3,  3,  0 },
    {  4,  5,  3,  2,  0,  3 },
    {  0,  1,  1,  1,  1,  1 },
    { 16, 17,  9,  5,  3,  0 }
};
static const uint8_t vc1_if_1mv_mbmode_bits[8][6] = {
    { 5, 5, 1, 3, 2, 4 },
    { 5, 5, 1, 2, 3, 4 },
    { 5, 5, 2, 1, 4, 3 },
    { 5, 5, 2, 4, 1, 3 },
    { 4, 4, 2, 3, 2, 2 },
    { 4, 4, 3, 2, 2, 2 },
    { 5, 5, 3, 4, 1, 2 },
    { 5, 5, 4, 3, 2, 1 }
};

/* Interlaced frame/field picture MVDATA VLC tables */

/* 1-reference tables */
static const uint32_t vc1_1ref_mvdata_codes[4][72] = { /* uint32_t may be too big */
  {
    0x00005, 0x0000C, 0x0001E, 0x00012, 0x0000C, 0x00034, 0x00075, 0x00070,
    0x00000, 0x00008, 0x0001B, 0x00008, 0x0001D, 0x0007C, 0x000D6, 0x001DE,
    0x001AF, 0x00005, 0x0001B, 0x00026, 0x0001E, 0x00012, 0x00076, 0x0004D,
    0x001F6, 0x001F4, 0x00039, 0x0007F, 0x00027, 0x0006A, 0x00071, 0x00035,
    0x00071, 0x00068, 0x001DC, 0x00027, 0x00073, 0x000FF, 0x000E8, 0x000E9,
    0x0007E, 0x001F9, 0x001F5, 0x001FD, 0x0003E, 0x001CA, 0x003F9, 0x0004C,
    0x00069, 0x001FA, 0x001DF, 0x001F7, 0x00070, 0x001DD, 0x00E4D, 0x00727,
    0x00392, 0x001C8, 0x001CB, 0x003F8, 0x001AE, 0x001F8, 0x001FB, 0x0E4CE,
    0x0E4CF, 0x07260, 0x07261, 0x07262, 0x07263, 0x07264, 0x07265, 0x07266
  },
  {
    0x00007, 0x00001, 0x00007, 0x00016, 0x00001, 0x00045, 0x00018, 0x002B6,
    0x00006, 0x00004, 0x00017, 0x00010, 0x00029, 0x0002C, 0x0015A, 0x00066,
    0x0019E, 0x00009, 0x00028, 0x00017, 0x00000, 0x0002A, 0x00004, 0x0005B,
    0x000B5, 0x000CE, 0x00006, 0x00044, 0x0000F, 0x00046, 0x0000E, 0x000AC,
    0x00032, 0x00037, 0x011EB, 0x0000A, 0x0001A, 0x0011F, 0x00016, 0x00014,
    0x0002B, 0x00168, 0x00055, 0x023D5, 0x00057, 0x0002F, 0x00036, 0x0002E,
    0x00169, 0x00054, 0x0047B, 0x0019F, 0x02B7D, 0x0008E, 0x00ADE, 0x00479,
    0x0056E, 0x008F4, 0x015BF, 0x00478, 0x023D4, 0x0ADF1, 0x056F9, 0xADF0E,
    0xADF0F, 0x56F80, 0x56F81, 0x56F82, 0x56F83, 0x56F84, 0x56F85, 0x56F86
  },
  {
    0x00002, 0x00006, 0x00007, 0x0000D, 0x00007, 0x00030, 0x000FF, 0x001F0,
    0x00002, 0x00000, 0x00005, 0x00019, 0x0001E, 0x00007, 0x00063, 0x000FD,
    0x00023, 0x0000E, 0x0001B, 0x0001A, 0x00006, 0x00009, 0x00018, 0x000C5,
    0x00033, 0x001F1, 0x00002, 0x003FB, 0x001F3, 0x00022, 0x001FC, 0x00042,
    0x00623, 0x00083, 0x00620, 0x0007D, 0x00040, 0x00043, 0x003E4, 0x003E5,
    0x00191, 0x00FE9, 0x00105, 0x00208, 0x000FC, 0x00624, 0x00622, 0x00190,
    0x00626, 0x007F5, 0x00C4B, 0x01FD0, 0x0104D, 0x00065, 0x00C42, 0x000C9,
    0x00627, 0x00C43, 0x00C4A, 0x0104E, 0x01FD1, 0x0104F, 0x00412, 0x104CE,
    0x104CF, 0x08260, 0x08261, 0x08262, 0x08263, 0x08264, 0x08265, 0x08266
  },
  {
    0x0000D, 0x00001, 0x00004, 0x00000, 0x00017, 0x00005, 0x0007F, 0x0004D,
    0x00003, 0x00011, 0x0003E, 0x0003B, 0x00017, 0x00067, 0x0004A, 0x000C3,
    0x000F2, 0x0000A, 0x0002C, 0x00032, 0x0003D, 0x00015, 0x00028, 0x00093,
    0x000CC, 0x00096, 0x00003, 0x00075, 0x00020, 0x0002D, 0x00021, 0x00029,
    0x00090, 0x001D0, 0x001FB, 0x0001C, 0x0004C, 0x00060, 0x00009, 0x00008,
    0x0002D, 0x0009F, 0x001FA, 0x0013D, 0x00031, 0x000FC, 0x00058, 0x00092,
    0x000F0, 0x000F1, 0x000CD, 0x00185, 0x00165, 0x0004E, 0x00091, 0x000E9,
    0x00184, 0x001D1, 0x001E6, 0x00097, 0x001E7, 0x000B3, 0x0013C, 0x0164E,
    0x0164F, 0x00B20, 0x00B21, 0x00B22, 0x00B23, 0x00B24, 0x00B25, 0x00B26
  }
};

static const uint8_t vc1_1ref_mvdata_bits[4][72] = {
  {
     3,  4,  5,  5,  5,  6,  7,  7,  2,  4,  5,  5,  6,  7,  8,  9,  9,  4,
     6,  6,  6,  6,  7,  8,  9,  9,  6,  8,  7,  7,  7,  7,  8,  8,  9,  6,
     8,  8,  8,  8,  8,  9,  9,  9,  7, 10, 10,  8,  8,  9,  9,  9,  8,  9,
    13, 12, 11, 10, 10, 10,  9,  9,  9, 17, 17, 16, 16, 16, 16, 16, 16, 16
  },
  {
     3,  3,  4,  5,  5,  7,  8, 10,  3,  4,  5,  5,  6,  7,  9, 10, 12,  4,
     6,  6,  5,  6,  6,  8,  9, 11,  4,  7,  7,  7,  7,  8,  9,  9, 13,  5,
     8,  9,  8,  8,  9, 10, 10, 14,  7,  9,  9,  9, 10, 10, 11, 12, 14,  8,
    12, 11, 11, 12, 13, 11, 14, 16, 15, 20, 20, 19, 19, 19, 19, 19, 19, 19
  },
  {
     3,  4,  4,  4,  5,  6,  8,  9,  2,  4,  5,  5,  5,  6,  7,  8,  8,  4,
     7,  7,  6,  6,  7,  8,  8,  9,  5, 10,  9,  8,  9,  9, 11, 10, 11,  7,
     9,  9, 10, 10, 11, 12, 11, 12,  8, 11, 11, 11, 11, 11, 12, 13, 15,  9,
    12, 10, 11, 12, 12, 15, 13, 15, 13, 19, 19, 18, 18, 18, 18, 18, 18, 18
  },
  {
     4,  4,  4,  4,  5,  5,  7,  7,  3,  5,  6,  6,  6,  7,  7,  8,  8,  4,
     6,  6,  6,  6,  7,  8,  8,  8,  4,  7,  6,  6,  6,  7,  8,  9,  9,  5,
     7,  7,  6,  6,  7,  8,  9,  9,  6,  8,  8,  8,  8,  8,  8,  9, 10,  7,
     8,  8,  9,  9,  9,  8,  9,  9,  9, 14, 14, 13, 13, 13, 13, 13, 13, 13
  }
};

/* 2-reference tables */
static const uint32_t vc1_2ref_mvdata_codes[8][126] = { /* table 132 - table 139 */
  {
    0x0000C, 0x0001C, 0x0000B, 0x00000, 0x0000E, 0x0002A, 0x00050, 0x00368,
    0x00002, 0x0001A, 0x00004, 0x0003A, 0x0001D, 0x0006C, 0x000EF, 0x001BC,
    0x0015F, 0x0000F, 0x00003, 0x0001C, 0x0000D, 0x0000B, 0x0003E, 0x000A7,
    0x00146, 0x00199, 0x00006, 0x0001F, 0x00004, 0x0003C, 0x00007, 0x001BE,
    0x0008B, 0x0002C, 0x007B3, 0x00005, 0x000DB, 0x00056, 0x000EC, 0x00052,
    0x001BD, 0x00078, 0x000CF, 0x00573, 0x00009, 0x00023, 0x000ED, 0x00018,
    0x00006, 0x00044, 0x000F5, 0x00079, 0x006D2, 0x0006E, 0x0002B, 0x0015D,
    0x00017, 0x0037F, 0x00144, 0x000CE, 0x00028, 0x000AB, 0x00010, 0x001B5,
    0x000F7, 0x000A6, 0x0007B, 0x00028, 0x001ED, 0x001E9, 0x006FD, 0x00004,
    0x000F5, 0x00029, 0x0028A, 0x0028B, 0x0028F, 0x00DF9, 0x00335, 0x01E85,
    0x000EE, 0x002BD, 0x0002B, 0x003D8, 0x003D1, 0x00198, 0x001E9, 0x0051D,
    0x000B4, 0x0003F, 0x00455, 0x0022B, 0x00229, 0x00451, 0x00578, 0x007B2,
    0x00570, 0x00155, 0x00032, 0x003D0, 0x00054, 0x006D3, 0x00571, 0x00454,
    0x00334, 0x01BF1, 0x000B7, 0x00029, 0x01E84, 0x0016C, 0x0019B, 0x01BF0,
    0x00579, 0x00F43, 0x000B5, 0x008A1, 0x0002A, 0x0016D, 0x008A0, 0x007A0,
    0x003D1, 0x00AE5, 0x00154, 0x00AE4, 0x00A39, 0x00A38
  },
  {
    0x00003, 0x00009, 0x00016, 0x00010, 0x000D7, 0x00335, 0x00574, 0x00555,
    0x00000, 0x0001D, 0x00009, 0x00017, 0x0002C, 0x000AD, 0x00374, 0x006B3,
    0x00577, 0x0000F, 0x00018, 0x0000A, 0x0002E, 0x00022, 0x0017C, 0x00E7B,
    0x01B89, 0x015D8, 0x00008, 0x00034, 0x0006D, 0x00023, 0x001C2, 0x00376,
    0x002D3, 0x01C4A, 0x0330A, 0x00014, 0x0006A, 0x00072, 0x0006C, 0x000E3,
    0x0019B, 0x0073F, 0x01CF0, 0x00B41, 0x00032, 0x000E6, 0x000E0, 0x000CF,
    0x000AB, 0x0019C, 0x002AB, 0x00E2B, 0x015D9, 0x0006F, 0x001C3, 0x000AF,
    0x000BF, 0x000AC, 0x0017D, 0x006E3, 0x00E29, 0x01984, 0x00054, 0x000B5,
    0x0017A, 0x001AD, 0x00199, 0x00178, 0x00358, 0x002D2, 0x01C4B, 0x0005B,
    0x002A8, 0x00331, 0x00388, 0x0038B, 0x00370, 0x00713, 0x00CC3, 0x01CF1,
    0x001B9, 0x005EF, 0x00738, 0x002F2, 0x0033B, 0x002B9, 0x006EB, 0x00570,
    0x00E24, 0x0039D, 0x005A2, 0x005A3, 0x00E7D, 0x005EE, 0x00739, 0x00554,
    0x00AA5, 0x00AA4, 0x00377, 0x01CF5, 0x00BCE, 0x00E79, 0x00660, 0x00674,
    0x006EA, 0x00E7C, 0x00D65, 0x002F6, 0x015DA, 0x01B88, 0x005A1, 0x01CF4,
    0x005E6, 0x00E28, 0x00575, 0x00D64, 0x00334, 0x0330B, 0x015DB, 0x00B40,
    0x00BCF, 0x00DC5, 0x00E2A, 0x00675, 0x00571, 0x00553
  },
  {
    0x00004, 0x00002, 0x00010, 0x00003, 0x00017, 0x00045, 0x0003E, 0x0007E,
    0x00003, 0x00002, 0x00028, 0x0001E, 0x00015, 0x00047, 0x00002, 0x0014D,
    0x00060, 0x0000B, 0x00026, 0x00024, 0x00014, 0x00032, 0x0006F, 0x000C3,
    0x00531, 0x006E5, 0x00015, 0x0003F, 0x0002D, 0x00001, 0x0013E, 0x000DD,
    0x000F6, 0x00305, 0x00331, 0x0000E, 0x00003, 0x00034, 0x00033, 0x0001A,
    0x0014A, 0x000C5, 0x000F4, 0x006E4, 0x00001, 0x0003C, 0x0007D, 0x0008D,
    0x0009D, 0x00031, 0x0006E, 0x00296, 0x000CD, 0x00025, 0x00149, 0x00032,
    0x00089, 0x00036, 0x00088, 0x0006F, 0x00003, 0x0031D, 0x0000E, 0x001AA,
    0x0027E, 0x00061, 0x0014E, 0x0014F, 0x00067, 0x000FF, 0x00183, 0x00036,
    0x00357, 0x000F5, 0x000C6, 0x000C2, 0x00299, 0x00119, 0x00231, 0x00350,
    0x0002C, 0x0018F, 0x00530, 0x00297, 0x00004, 0x001B8, 0x000C0, 0x0027A,
    0x00311, 0x0009C, 0x00621, 0x00199, 0x0031C, 0x000F7, 0x003E3, 0x00356,
    0x00189, 0x00005, 0x0006B, 0x008C2, 0x00330, 0x004FF, 0x004F0, 0x00351,
    0x004F2, 0x001F2, 0x00373, 0x00000, 0x00C41, 0x008C3, 0x009EC, 0x003E2,
    0x00304, 0x004F7, 0x004F1, 0x001F0, 0x00148, 0x00C40, 0x009ED, 0x008C0,
    0x008C1, 0x004F3, 0x004FE, 0x000FE, 0x001F3, 0x001A9
  },
  {
    0x00000, 0x00004, 0x0002F, 0x00052, 0x00010, 0x000AD, 0x0050B, 0x00190,
    0x00003, 0x00016, 0x00007, 0x0000D, 0x000BB, 0x00173, 0x000C9, 0x0050F,
    0x0172C, 0x00003, 0x00011, 0x00005, 0x00043, 0x00023, 0x0004B, 0x0032E,
    0x02E5B, 0x00482, 0x00009, 0x0002A, 0x00014, 0x0002A, 0x00108, 0x005CA,
    0x0065A, 0x02136, 0x02132, 0x0000B, 0x00013, 0x00041, 0x000B8, 0x00174,
    0x00100, 0x014DA, 0x0404E, 0x01437, 0x0002B, 0x00085, 0x000A7, 0x000A0,
    0x0014C, 0x0029A, 0x0032C, 0x02133, 0x0142A, 0x00051, 0x00284, 0x000AC,
    0x00102, 0x00045, 0x00044, 0x0081B, 0x0065E, 0x00CB7, 0x00018, 0x0050C,
    0x00212, 0x002E4, 0x00203, 0x00094, 0x00122, 0x0081A, 0x00655, 0x00033,
    0x002BA, 0x00246, 0x00242, 0x00A6E, 0x0040C, 0x00808, 0x02134, 0x0404F,
    0x00175, 0x00405, 0x00247, 0x0012A, 0x00A14, 0x002BB, 0x00191, 0x0084F,
    0x01438, 0x000AF, 0x00B97, 0x00483, 0x0143B, 0x0032B, 0x00243, 0x0142B,
    0x00958, 0x029BF, 0x00049, 0x00A6C, 0x014DB, 0x004AD, 0x014DE, 0x0084E,
    0x01434, 0x00257, 0x02E5A, 0x00207, 0x01435, 0x01439, 0x00CB6, 0x0143A,
    0x00194, 0x00654, 0x02135, 0x0537C, 0x0015C, 0x00240, 0x01012, 0x0537D,
    0x00959, 0x01098, 0x01436, 0x0065F, 0x02026, 0x02137
  },
  {
    0x00005, 0x00019, 0x00016, 0x00011, 0x0003E, 0x0005E, 0x000EF, 0x000E2,
    0x00000, 0x00039, 0x0002B, 0x00026, 0x00028, 0x00012, 0x000C2, 0x000ED,
    0x0011D, 0x0000D, 0x00031, 0x0002A, 0x00025, 0x00020, 0x0005C, 0x001ED,
    0x0024D, 0x00770, 0x00006, 0x0007A, 0x00060, 0x0004F, 0x00048, 0x00039,
    0x00186, 0x00213, 0x00EC6, 0x0000F, 0x00026, 0x0005F, 0x00075, 0x00070,
    0x00027, 0x001DB, 0x003C6, 0x0078F, 0x0003F, 0x000A6, 0x000F0, 0x0003A,
    0x00052, 0x0004E, 0x000E3, 0x001D9, 0x0030F, 0x00010, 0x001DD, 0x000A7,
    0x000F7, 0x00022, 0x00092, 0x003C4, 0x002EF, 0x00762, 0x00079, 0x0008F,
    0x001DA, 0x00087, 0x000E8, 0x000BA, 0x00176, 0x000EE, 0x003B0, 0x00085,
    0x00119, 0x0030E, 0x00108, 0x001D2, 0x0010C, 0x00773, 0x00424, 0x00434,
    0x00071, 0x005DD, 0x001C1, 0x003A7, 0x00127, 0x0008D, 0x0021B, 0x007B2,
    0x001DF, 0x003D8, 0x00764, 0x00EE4, 0x003B3, 0x0074D, 0x001D8, 0x005DC,
    0x0084A, 0x00499, 0x003C5, 0x01D8E, 0x00765, 0x00435, 0x00771, 0x001C2,
    0x00118, 0x003BC, 0x00381, 0x00387, 0x07B33, 0x01097, 0x01096, 0x01ECD,
    0x00E99, 0x00F1C, 0x00F1D, 0x00EE5, 0x0011C, 0x07B32, 0x03D98, 0x01D8F,
    0x00E98, 0x00F67, 0x003BD, 0x00380, 0x00498, 0x00386
  },
  {
    0x0000D, 0x00010, 0x0002E, 0x00039, 0x0000D, 0x00074, 0x000ED, 0x000B6,
    0x00001, 0x00002, 0x00000, 0x00030, 0x00029, 0x00070, 0x000F3, 0x0008C,
    0x00166, 0x00009, 0x00033, 0x00078, 0x00006, 0x000C4, 0x0000B, 0x00163,
    0x000CC, 0x005BE, 0x0001F, 0x0002F, 0x00064, 0x00018, 0x000C6, 0x0000A,
    0x00162, 0x002C0, 0x00EF3, 0x00007, 0x0000F, 0x000E3, 0x000CA, 0x000B2,
    0x0018F, 0x003AE, 0x0075F, 0x00C51, 0x00015, 0x00047, 0x000EE, 0x000E2,
    0x000EA, 0x00009, 0x0016A, 0x002C3, 0x0059D, 0x0003D, 0x00008, 0x001D9,
    0x00032, 0x0000E, 0x0016E, 0x0032C, 0x0065B, 0x0196B, 0x00002, 0x0000F,
    0x001D8, 0x0008D, 0x000B4, 0x001E4, 0x00067, 0x00317, 0x00794, 0x00022,
    0x003BE, 0x00315, 0x00034, 0x00037, 0x002DE, 0x0006C, 0x00EFE, 0x0066C,
    0x00028, 0x003CB, 0x003AC, 0x00035, 0x0016B, 0x003BD, 0x002C1, 0x0062C,
    0x01DFE, 0x0000E, 0x0059E, 0x005BF, 0x000DA, 0x00629, 0x00584, 0x00EB7,
    0x00B0A, 0x0066D, 0x0000C, 0x0077E, 0x0059C, 0x00778, 0x0075E, 0x0075A,
    0x0062D, 0x00337, 0x00334, 0x00197, 0x01E57, 0x01DE4, 0x0196A, 0x01E56,
    0x00C50, 0x00B3F, 0x01E54, 0x00B0B, 0x0018E, 0x001B6, 0x01E55, 0x00CB4,
    0x00B3E, 0x00EB6, 0x01DE5, 0x01DFF, 0x00335, 0x001B7
  },
  {
    0x00001, 0x0000B, 0x00019, 0x0006F, 0x0002A, 0x00075, 0x007EB, 0x00163,
    0x00001, 0x0000E, 0x0001A, 0x0003E, 0x0001C, 0x0002D, 0x00164, 0x007EC,
    0x00165, 0x00004, 0x00006, 0x00036, 0x0007F, 0x000AE, 0x00158, 0x0015C,
    0x0056D, 0xFD510, 0x00000, 0x00004, 0x0007B, 0x000F3, 0x0003B, 0x007ED,
    0x002B3, 0x002CC, 0x0056E, 0x00018, 0x0003E, 0x00017, 0x0001E, 0x000AF,
    0x003F7, 0x0056F, 0x002CD, 0xFD511, 0x00014, 0x000AD, 0x000AA, 0x00014,
    0x000A8, 0x00153, 0x000E8, 0x001FE, 0x00DCF, 0x00078, 0x001B8, 0x00152,
    0x000FE, 0x002B1, 0x0015D, 0x00160, 0xFD512, 0xFD513, 0x0007A, 0x002B0,
    0x001E5, 0x000E9, 0x000FC, 0x006E6, 0x00DC8, 0x00584, 0xFD514, 0x000AB,
    0x00DDE, 0x00159, 0x003F4, 0x00DC9, 0x00DCA, 0x001FA, 0xFD515, 0xFD516,
    0x000FC, 0x001FF, 0x001E4, 0x000AF, 0x0015A, 0x00167, 0x00DCB, 0x00585,
    0xFD517, 0x003F7, 0x03F55, 0xFD518, 0x00DDC, 0x00586, 0x03F56, 0xFD519,
    0x03F57, 0xFD51A, 0x001BA, 0x00587, 0x00588, 0x00DDF, 0x002B2, 0xFD51B,
    0x00DCE, 0x003F6, 0xFD51C, 0x00FD4, 0xFD51D, 0xFD51E, 0xFD51F, 0x7EA80,
    0x7EA81, 0x0056C, 0x7EA82, 0x7EA83, 0x00376, 0x00589, 0x0058A, 0x7EA84,
    0x7EA85, 0x00DDD, 0x7EA86, 0x7EA87, 0x0058B, 0x07EA9
  },
  {
    0x00003, 0x0000E, 0x0000F, 0x0007E, 0x00062, 0x000C6, 0x00CD9, 0x0063E,
    0x00002, 0x00002, 0x00000, 0x00018, 0x0000C, 0x00069, 0x00039, 0x00707,
    0x00C7E, 0x00002, 0x0000D, 0x0001B, 0x0000F, 0x0019A, 0x00647, 0x01A37,
    0x346C4, 0x0346D, 0x00001, 0x0001E, 0x0007F, 0x0000A, 0x000E1, 0x00661,
    0x00CE4, 0x346C5, 0x346C6, 0x0001D, 0x00030, 0x0000D, 0x000CB, 0x00199,
    0x00320, 0x0008E, 0x0652E, 0x346C7, 0x0003E, 0x00039, 0x00035, 0x00033,
    0x0019F, 0x001C0, 0x00CDA, 0x346C8, 0x346C9, 0x0000B, 0x000D0, 0x0019E,
    0x00022, 0x00038, 0x0018E, 0x0031E, 0x03294, 0x0023C, 0x00032, 0x00012,
    0x00013, 0x00071, 0x0019D, 0x00020, 0x00C87, 0x00CC0, 0x346CA, 0x00338,
    0x00653, 0x001A2, 0x0032A, 0x00322, 0x00CE7, 0x00084, 0x0011F, 0x346CB,
    0x00325, 0x00649, 0x0032B, 0x00077, 0x00648, 0x00642, 0x00C86, 0x00C8C,
    0x346CC, 0x0003A, 0x019B7, 0x00043, 0x00327, 0x0008C, 0x0008D, 0x00C8D,
    0x346CD, 0x346CE, 0x00337, 0x00CE5, 0x00085, 0x00326, 0x00347, 0x00CA4,
    0x00C7F, 0x00D1A, 0x346CF, 0x00328, 0x1A360, 0x1A361, 0x00CD8, 0x0068C,
    0x03295, 0x03296, 0x0652F, 0x066D8, 0x00331, 0x00706, 0x0023D, 0x00076,
    0x00CC1, 0x00382, 0x00CE6, 0x066D9, 0x066DA, 0x066DB
  }
};

static const uint8_t vc1_2ref_mvdata_bits[8][126] = {
  {
     4,  5,  5,  5,  6,  7,  8, 10,  2,  5,  5,  6,  6,  7,  8,  9,
    10,  4,  5,  6,  6,  7,  8,  9, 10, 11,  4,  6,  6,  7,  7,  9,
     9, 10, 12,  5,  8,  8,  8,  8,  9,  9, 10, 12,  5,  7,  8,  7,
     7,  8,  9,  9, 11,  7,  9, 10,  9, 10, 10, 10, 10, 12,  6,  9,
     9,  9,  9,  9, 10, 10, 11,  7, 10, 10, 11, 11, 11, 12, 12, 14,
     8, 11, 10, 11, 11, 11, 11, 12, 12,  8, 12, 11, 11, 12, 12, 12,
    12, 13,  8, 12, 11, 11, 12, 12, 12, 13, 12,  9, 14, 13, 11, 13,
    12, 13, 12, 13,  9, 13, 13, 12, 12, 13, 13, 13, 13, 13
  },
  {
     3,  4,  5,  6,  8, 10, 11, 11,  2,  5,  5,  6,  7,  8, 10, 11,
    11,  4,  5,  5,  6,  7,  9, 12, 13, 13,  4,  6,  7,  7,  9, 10,
    11, 13, 14,  5,  7,  7,  7,  8,  9, 11, 13, 13,  6,  8,  8,  8,
     8,  9, 10, 12, 13,  7,  9,  8,  8,  8,  9, 11, 12, 13,  7,  9,
     9,  9,  9,  9, 10, 11, 13,  8, 10, 10, 10, 10, 10, 11, 12, 13,
     9, 11, 11, 10, 10, 10, 11, 11, 12, 10, 12, 12, 12, 11, 11, 11,
    12, 12, 10, 13, 12, 12, 11, 11, 11, 12, 12, 10, 13, 13, 12, 13,
    11, 12, 11, 12, 10, 14, 13, 13, 12, 12, 12, 11, 11, 11
  },
  {
     4,  4,  5,  5,  6,  7,  8,  9,  2,  5,  6,  6,  6,  7,  7,  9,
     9,  4,  6,  6,  6,  7,  8,  9, 11, 12,  5,  7,  7,  7,  9,  9,
    10, 11, 12,  5,  7,  7,  7,  7,  9,  9, 10, 12,  5,  8,  8,  8,
     8,  8,  9, 10, 10,  6,  9,  8,  8,  8,  8,  9,  9, 11,  6, 10,
    10,  9,  9,  9,  9, 10, 10,  7, 11, 10,  9,  9, 10,  9, 10, 11,
     7, 10, 11, 10, 10, 10,  9, 10, 11,  8, 12, 11, 11, 10, 11, 11,
    10, 10,  8, 12, 12, 11, 11, 11, 11, 10, 11,  8, 13, 12, 12, 11,
    11, 11, 11, 10,  9, 13, 12, 12, 12, 11, 11, 10, 10, 10
  },
  {
     3,  4,  6,  7,  7,  9, 11, 11,  2,  5,  5,  6,  8,  9, 10, 11,
    13,  3,  5,  5,  7,  8,  9, 12, 14, 13,  4,  6,  6,  7,  9, 11,
    13, 14, 14,  5,  7,  7,  8,  9,  9, 13, 15, 13,  6,  8,  8,  8,
     9, 10, 12, 14, 13,  7, 10,  9,  9,  9,  9, 12, 13, 14,  7, 11,
    10, 10, 10, 10, 11, 12, 13,  8, 11, 12, 12, 12, 11, 12, 14, 15,
     9, 11, 12, 11, 12, 11, 11, 12, 13,  9, 12, 13, 13, 12, 12, 13,
    14, 14,  9, 12, 13, 13, 13, 12, 13, 12, 14, 10, 13, 13, 14, 13,
    11, 13, 14, 15, 10, 12, 13, 15, 14, 13, 13, 13, 14, 14
  },
  {
     4,  5,  5,  5,  6,  7,  8,  8,  2,  6,  6,  6,  6,  6,  8,  9,
    10,  4,  6,  6,  6,  6,  7,  9, 10, 11,  4,  7,  7,  7,  7,  7,
     9, 10, 12,  5,  7,  7,  7,  7,  7,  9, 10, 11,  6,  8,  8,  7,
     7,  7,  8,  9, 10,  6,  9,  8,  8,  7,  8, 10, 10, 11,  7,  9,
     9,  8,  8,  8,  9,  9, 10,  8, 10, 10,  9,  9,  9, 11, 11, 11,
     8, 11, 10, 10,  9,  9, 10, 11, 10, 10, 12, 12, 11, 11, 10, 11,
    12, 11, 10, 13, 12, 11, 11, 10, 10, 11, 11, 11, 15, 13, 13, 13,
    12, 12, 12, 12, 10, 15, 14, 13, 12, 12, 11, 11, 11, 11
  },
  {
     4,  5,  6,  6,  6,  7,  8,  8,  2,  4,  5,  6,  6,  7,  8,  8,
     9,  4,  6,  7,  7,  8,  8,  9, 10, 11,  5,  6,  7,  7,  8,  8,
     9, 10, 12,  5,  7,  8,  8,  8,  9, 10, 11, 12,  5,  7,  8,  8,
     8,  8,  9, 10, 11,  6,  8,  9,  8,  8,  9, 10, 11, 13,  5,  8,
     9,  8,  8,  9,  9, 10, 11,  6, 10, 10,  9,  9, 10, 10, 12, 13,
     6, 10, 10,  9,  9, 10, 10, 11, 13,  7, 11, 11, 11, 11, 11, 12,
    12, 13,  7, 11, 11, 11, 11, 11, 11, 12, 12,  9, 13, 13, 13, 13,
    12, 12, 13, 12,  9, 12, 13, 12, 12, 12, 13, 13, 12, 12
  },
  {
     3,  5,  6,  8,  9, 10, 12, 12,  1,  5,  6,  7,  8,  9, 12, 12,
    12,  4,  6,  7,  8,  9, 12, 12, 14, 21,  4,  6,  8,  9,  9, 12,
    13, 13, 14,  6,  9,  8,  8,  9, 13, 14, 13, 21,  6,  9,  9,  8,
     9, 10, 11, 12, 13,  8, 10, 10, 11, 11, 12, 12, 21, 21,  8, 11,
    10, 11, 11, 12, 13, 14, 21,  9, 13, 10, 11, 13, 13, 12, 21, 21,
     9, 12, 10, 11, 12, 12, 13, 14, 21, 11, 15, 21, 13, 14, 15, 21,
    15, 21, 10, 14, 14, 13, 13, 21, 13, 13, 21, 13, 21, 21, 21, 20,
    20, 14, 20, 20, 11, 14, 14, 20, 20, 13, 20, 20, 14, 16
  },
  {
     2,  5,  6,  8,  9, 10, 13, 13,  2,  4,  5,  6,  8,  9, 10, 13,
    14,  3,  5,  7,  8, 10, 12, 15, 20, 16,  4,  6,  8,  8, 10, 12,
    13, 20, 20,  7,  8,  8,  9, 10, 11, 12, 16, 20,  7,  8,  8,  8,
    10, 11, 13, 20, 20,  8, 10, 10, 10, 10, 11, 12, 15, 14,  8,  9,
     9,  9, 10, 10, 13, 13, 20, 11, 12, 11, 11, 11, 13, 12, 13, 20,
    11, 12, 11, 11, 12, 12, 13, 13, 20, 10, 14, 11, 11, 12, 12, 13,
    20, 20, 11, 13, 12, 11, 12, 13, 14, 14, 20, 11, 19, 19, 13, 13,
    15, 15, 16, 16, 11, 13, 14, 11, 13, 12, 13, 16, 16, 16
  }
};

/* P-Picture CBPCY VLC tables */
// Looks like original tables are not conforming to standard at all. Are they used for old WMV?
static const uint16_t vc1_cbpcy_p_codes[4][64] = {
  {
      0,   6,  15,  13,  13,  11,   3,  13,   5,   8,  49,  10,  12, 114, 102, 119,
      1,  54,  96,   8,  10, 111,   5,  15,  12,  10,   2,  12,  13, 115,  53,  63,
      1,   7,   1,   7,  14,  12,   4,  14,   1,   9,  97,  11,   7,  58,  52,  62,
      4, 103,   1,   9,  11,  56, 101, 118,   4, 110, 100,  30,   2,   5,   4,   3
  },
  {
      0,   9,   1,  18,   5,  14, 237,  26,   3, 121,   3,  22,  13,  16,   6,  30,
      2,  10,   1,  20,  12, 241,   5,  28,  16,  12,   3,  24,  28, 124, 239, 247,
      1, 240,   1,  19,  18,  15,   4,  27,   1, 122,   2,  23,   1,  17,   7,  31,
      1,  11,   2,  21,  19, 246, 238,  29,  17,  13, 236,  25,  58,  63,   8, 125
  },
  {
      0, 201,  25, 231,   5, 221,   1,   3,   2, 414,   2, 241,  16, 225, 195, 492,
      2, 412,   1, 240,   7, 224,  98, 245,   1, 220,  96,   5,   9, 230, 101, 247,
      1, 102,   1, 415,  24,   3,   2, 244,   3,  54,   3, 484,  17, 114, 200, 493,
      3, 413,   1,   4,  13, 113,  99, 485,   4, 111, 194, 243,   5,  29,  26,  31
  },
  {
      0,  28,  12,  44,   3,  36,  20,  52,   2,  32,  16,  48,   8,  40,  24,  28,
      1,  30,  14,  46,   6,  38,  22,  54,   3,  34,  18,  50,  10,  42,  26,  30,
      1,  29,  13,  45,   5,  37,  21,  53,   2,  33,  17,  49,   9,  41,  25,  29,
      1,  31,  15,  47,   7,  39,  23,  55,   4,  35,  19,  51,  11,  43,  27,  31
   }
};

static const uint8_t vc1_cbpcy_p_bits[4][64] = {
  {
    13,  13,   7,  13,   7,  13,  13,  12,   6,  13,   7,  12,   6,   8,   8,   8,
     5,   7,   8,  12,   6,   8,  13,  12,   7,  13,  13,  12,   6,   8,   7,   7,
     6,  13,   8,  12,   7,  13,  13,  12,   7,  13,   8,  12,   5,   7,   7,   7,
     6,   8,  13,  12,   6,   7,   8,   8,   5,   8,   8,   6,   3,   3,   3,   2
  },
  {
    14,  13,   8,  13,   3,  13,   8,  13,   3,   7,   8,  13,   4,  13,  13,  13,
     3,  13,  13,  13,   4,   8,  13,  13,   5,  13,  13,  13,   5,   7,   8,   8,
     3,   8,  14,  13,   5,  13,  13,  13,   4,   7,  13,  13,   6,  13,  13,  13,
     5,  13,   8,  13,   5,   8,   8,  13,   5,  13,   8,  13,   6,   6,  13,   7
  },
  {
    13,   8,   6,   8,   4,   8,  13,  12,   4,   9,   8,   8,   5,   8,   8,   9,
     5,   9,  10,   8,   4,   8,   7,   8,   6,   8,   7,  13,   4,   8,   7,   8,
     5,   7,   8,   9,   6,  13,  13,   8,   4,   6,   8,   9,   5,   7,   8,   9,
     5,   9,   9,  13,   5,   7,   7,   9,   4,   7,   8,   8,   3,   5,   5,   5
  },
  {
     9,   9,   9,   9,   2,   9,   9,   9,   2,   9,   9,   9,   9,   9,   9,   8,
     3,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   8,
     2,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   8,
     9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   9,   8
  }
};

/* Interlaced CBPCY VLC tables (Table 124 - Table 131) */
static const uint16_t vc1_icbpcy_p_codes[8][63] = {
  {
    0x2F1A, 0x2F1B, 0x178C, 0x0090, 0x02A8, 0x02A9, 0x0BC7, 0x0091,
    0x02AA, 0x02AB, 0x05E0, 0x004A, 0x0096, 0x0097, 0x00BD, 0x0092,
    0x02AC, 0x02AD, 0x05E1, 0x0098, 0x0132, 0x0133, 0x0179, 0x0134,
    0x026A, 0x026B, 0x02FC, 0x004E, 0x0040, 0x0041, 0x002B, 0x0093,
    0x02AE, 0x02AF, 0x05E2, 0x0136, 0x026E, 0x026F, 0x02FD, 0x009E,
    0x013E, 0x013F, 0x017F, 0x0050, 0x0042, 0x0043, 0x002C, 0x0051,
    0x00A4, 0x00A5, 0x00BE, 0x0053, 0x0044, 0x0045, 0x002D, 0x0054,
    0x0046, 0x0047, 0x002E, 0x0003, 0x0000, 0x0001, 0x0001
  },
  {
    0x0041, 0x0042, 0x0100, 0x0043, 0x0088, 0x0089, 0x0101, 0x0045,
    0x008C, 0x008D, 0x0102, 0x0010, 0x0022, 0x0023, 0x0024, 0x0047,
    0x0010, 0x0011, 0x0103, 0x0025, 0x0058, 0x0059, 0x005A, 0x005B,
    0x005A, 0x005B, 0x005C, 0x000C, 0x0030, 0x0031, 0x0019, 0x0009,
    0x0014, 0x0015, 0x002C, 0x005C, 0x005D, 0x005E, 0x005F, 0x0026,
    0x005D, 0x005E, 0x005F, 0x000D, 0x0034, 0x0035, 0x001B, 0x0014,
    0x0027, 0x002A, 0x002B, 0x000E, 0x0038, 0x0039, 0x001D, 0x000F,
    0x003C, 0x003D, 0x001F, 0x0005, 0x0009, 0x0000, 0x0003
  },
  {
    0x0032, 0x0033, 0x001A, 0x0026, 0x00E4, 0x00E5, 0x01E6, 0x0027,
    0x00E6, 0x00E7, 0x01E7, 0x000E, 0x0063, 0x006C, 0x0077, 0x0028,
    0x00E8, 0x00E9, 0x01E8, 0x007B, 0x00DA, 0x00DB, 0x00EC, 0x00F5,
    0x01B8, 0x01B9, 0x01DA, 0x0021, 0x004B, 0x0054, 0x002B, 0x0029,
    0x00EA, 0x00EB, 0x01E9, 0x004A, 0x01BA, 0x01BB, 0x01DB, 0x0020,
    0x00DE, 0x00DF, 0x00F2, 0x0022, 0x0055, 0x0058, 0x002D, 0x000F,
    0x0070, 0x0071, 0x0078, 0x0023, 0x0059, 0x005C, 0x002F, 0x0024,
    0x005D, 0x0062, 0x0030, 0x0002, 0x001F, 0x0006, 0x0000
  },
  {
    0x0028, 0x0029, 0x009D, 0x0000, 0x01EA, 0x01EB, 0x01EC, 0x0001,
    0x01ED, 0x01EE, 0x01EF, 0x0005, 0x00F0, 0x00F1, 0x003B, 0x0002,
    0x01F0, 0x01F1, 0x01F2, 0x003F, 0x015C, 0x015D, 0x0099, 0x0010,
    0x03D0, 0x03D1, 0x0130, 0x000F, 0x009E, 0x009F, 0x00FB, 0x0003,
    0x01F3, 0x01F4, 0x01F5, 0x0011, 0x03D2, 0x03D3, 0x0131, 0x0009,
    0x015E, 0x015F, 0x009C, 0x0010, 0x00A8, 0x00A9, 0x0038, 0x0006,
    0x00F2, 0x00F3, 0x004D, 0x0011, 0x00AA, 0x00AB, 0x0039, 0x0012,
    0x00AC, 0x00AD, 0x003A, 0x0006, 0x0016, 0x0017, 0x000E
  },
  {
    0x003C, 0x003D, 0x001F, 0x000A, 0x0061, 0x0062, 0x0002, 0x000B,
    0x0063, 0x0064, 0x0003, 0x0007, 0x0003, 0x0004, 0x000B, 0x000C,
    0x0065, 0x0066, 0x0004, 0x0012, 0x000A, 0x000B, 0x0014, 0x001B,
    0x0018, 0x0019, 0x0034, 0x002C, 0x0067, 0x0068, 0x0035, 0x000D,
    0x0069, 0x006C, 0x0005, 0x0060, 0x001A, 0x001B, 0x0035, 0x0013,
    0x000E, 0x000F, 0x0015, 0x002D, 0x006D, 0x006E, 0x0038, 0x0008,
    0x0008, 0x0009, 0x000C, 0x002E, 0x006F, 0x0072, 0x003A, 0x002F,
    0x0073, 0x0000, 0x003B, 0x0007, 0x0014, 0x0015, 0x0004
  },
  {
    0x0038, 0x0039, 0x009D, 0x000A, 0x0091, 0x0092, 0x0093, 0x000B,
    0x0094, 0x0095, 0x0096, 0x0003, 0x00EE, 0x00EF, 0x0036, 0x000C,
    0x0097, 0x0098, 0x0099, 0x0008, 0x01E4, 0x01E5, 0x006A, 0x0018,
    0x03CC, 0x03CD, 0x00D6, 0x000E, 0x009E, 0x009F, 0x00F5, 0x000D,
    0x009A, 0x009B, 0x009C, 0x0019, 0x03CE, 0x03CF, 0x00D7, 0x0009,
    0x01E8, 0x01E9, 0x0090, 0x000F, 0x00E8, 0x00E9, 0x00F6, 0x0005,
    0x00F0, 0x00F1, 0x0037, 0x0010, 0x00EA, 0x00EB, 0x00F7, 0x0011,
    0x00EC, 0x00ED, 0x0034, 0x0000, 0x003E, 0x003F, 0x0002
  },
  {
    0x003C, 0x003D, 0x01CF, 0x0000, 0x00BF, 0x00E0, 0x01FC, 0x0001,
    0x00E1, 0x00E2, 0x01FD, 0x0009, 0x01F1, 0x01F2, 0x01F3, 0x0002,
    0x00E3, 0x00E4, 0x01FE, 0x0011, 0x03EE, 0x03EF, 0x03F0, 0x0021,
    0x07E2, 0x07E3, 0x07E4, 0x0018, 0x03F7, 0x03FE, 0x03FF, 0x0003,
    0x00E5, 0x00E6, 0x0080, 0x002E, 0x07E5, 0x07E6, 0x07E7, 0x0016,
    0x03F4, 0x03F5, 0x03F6, 0x0019, 0x0102, 0x0103, 0x0104, 0x000A,
    0x01F4, 0x01F5, 0x01F6, 0x001A, 0x0105, 0x0106, 0x0107, 0x001B,
    0x0178, 0x0179, 0x01CE, 0x001D, 0x00BD, 0x00BE, 0x01F0
  },
  {
    0x0003, 0x0004, 0x01B6, 0x0004, 0x002E, 0x002F, 0x000E, 0x0005,
    0x0030, 0x0031, 0x000F, 0x0003, 0x000A, 0x000B, 0x0014, 0x0006,
    0x0032, 0x0033, 0x0010, 0x0005, 0x0030, 0x0031, 0x0032, 0x0009,
    0x0066, 0x0067, 0x0068, 0x001D, 0x01B7, 0x01B8, 0x01B9, 0x0007,
    0x0034, 0x0035, 0x0011, 0x0016, 0x0069, 0x006A, 0x006B, 0x000A,
    0x0036, 0x0037, 0x00D8, 0x001E, 0x01BA, 0x01BB, 0x01BC, 0x0004,
    0x0015, 0x0016, 0x0017, 0x001F, 0x01BD, 0x01BE, 0x01BF, 0x0000,
    0x0010, 0x0011, 0x0012, 0x001C, 0x00D9, 0x00DA, 0x0013
  }
};

static const uint8_t vc1_icbpcy_p_bits[8][63] = {
  {
    15, 15, 14,  9, 11, 11, 13,  9, 11, 11, 12,  8,  9,  9,  9,  9,
    11, 11, 12,  9, 10, 10, 10, 10, 11, 11, 11,  8,  8,  8,  7,  9,
    11, 11, 12, 10, 11, 11, 11,  9, 10, 10, 10,  8,  8,  8,  7,  8,
     9,  9,  9,  8,  8,  8,  7,  8,  8,  8,  7,  3,  3,  3,  1
  },
  {
     7,  7,  9,  7,  8,  8,  9,  7,  8,  8,  9,  6,  7,  7,  7,  7,
     7,  7,  9,  7,  8,  8,  8,  8,  9,  9,  9,  6,  7,  7,  6,  6,
     7,  7,  8,  8,  9,  9,  9,  7,  8,  8,  8,  6,  7,  7,  6,  6,
     7,  7,  7,  6,  7,  7,  6,  6,  7,  7,  6,  3,  4,  3,  2
  },
  {
     6,  6,  5,  6,  8,  8,  9,  6,  8,  8,  9,  5,  7,  7,  7,  6,
     8,  8,  9,  7,  8,  8,  8,  8,  9,  9,  9,  6,  7,  7,  6,  6,
     8,  8,  9,  7,  9,  9,  9,  6,  8,  8,  8,  6,  7,  7,  6,  5,
     7,  7,  7,  6,  7,  7,  6,  6,  7,  7,  6,  3,  5,  4,  2
  },
  {
     6,  6,  8,  4,  9,  9,  9,  4,  9,  9,  9,  4,  8,  8,  7,  4,
     9,  9,  9,  6,  9,  9,  8,  6, 10, 10,  9,  5,  8,  8,  8,  4,
     9,  9,  9,  6, 10, 10,  9,  5,  9,  9,  8,  5,  8,  8,  7,  4,
     8,  8,  7,  5,  8,  8,  7,  5,  8,  8,  7,  3,  5,  5,  4
  },
  {
     6,  6,  5,  5,  7,  7,  7,  5,  7,  7,  7,  5,  6,  6,  6,  5,
     7,  7,  7,  6,  7,  7,  7,  7,  8,  8,  8,  6,  7,  7,  6,  5,
     7,  7,  7,  7,  8,  8,  8,  6,  7,  7,  7,  6,  7,  7,  6,  5,
     6,  6,  6,  6,  7,  7,  6,  6,  7,  6,  6,  4,  5,  5,  3
  },
  {
     6,  6,  8,  4,  8,  8,  8,  4,  8,  8,  8,  4,  8,  8,  7,  4,
     8,  8,  8,  5,  9,  9,  8,  6, 10, 10,  9,  5,  8,  8,  8,  4,
     8,  8,  8,  6, 10, 10,  9,  5,  9,  9,  8,  5,  8,  8,  8,  4,
     8,  8,  7,  5,  8,  8,  8,  5,  8,  8,  7,  3,  6,  6,  4
  },
  {
     6,  6,  9,  3,  8,  8,  9,  3,  8,  8,  9,  4,  9,  9,  9,  3,
     8,  8,  9,  5, 10, 10, 10,  6, 11, 11, 11,  5, 10, 10, 10,  3,
     8,  8,  8,  6, 11, 11, 11,  5, 10, 10, 10,  5,  9,  9,  9,  4,
     9,  9,  9,  5,  9,  9,  9,  5,  9,  9,  9,  5,  8,  8,  9
  },
  {
     6,  6, 10,  3,  7,  7,  7,  3,  7,  7,  7,  4,  8,  8,  8,  3,
     7,  7,  7,  5,  9,  9,  9,  6, 10, 10, 10,  6, 10, 10, 10,  3,
     7,  7,  7,  6, 10, 10, 10,  5,  9,  9,  9,  6, 10, 10, 10,  4,
     8,  8,  8,  6, 10, 10, 10,  5,  9,  9,  9,  6,  9,  9,  9
  }
};

/* MacroBlock Transform Type: 7.1.3.11, p89
 * 8x8:B
 * 8x4:B:btm  8x4:B:top  8x4:B:both,
 * 4x8:B:right  4x8:B:left  4x8:B:both
 * 4x4:B  8x8:MB
 * 8x4:MB:btm  8x4:MB:top  8x4,MB,both
 * 4x8,MB,right  4x8,MB,left
 * 4x4,MB                               */
static const uint16_t vc1_ttmb_codes[3][16] = {
  {
    0x0003,
    0x002E, 0x005F, 0x0000,
    0x0016, 0x0015, 0x0001,
    0x0004, 0x0014,
    0x02F1, 0x0179, 0x017B,
    0x0BC0, 0x0BC1, 0x05E1,
    0x017A
  },
  {
    0x0006,
    0x0006, 0x0003, 0x0007,
    0x000F, 0x000E, 0x0000,
    0x0002, 0x0002,
    0x0014, 0x0011, 0x000B,
    0x0009, 0x0021, 0x0015,
    0x0020
  },
  {
    0x0006,
    0x0000, 0x000E, 0x0005,
    0x0002, 0x0003, 0x0003,
    0x000F, 0x0002,
    0x0081, 0x0021, 0x0009,
    0x0101, 0x0041, 0x0011,
    0x0100
  }
};

static const uint8_t vc1_ttmb_bits[3][16] = {
  {
     2,
     6,  7,  2,
     5,  5,  2,
     3,  5,
    10,  9,  9,
    12, 12, 11,
     9
  },
  {
    3,
    4, 4, 4,
    4, 4, 3,
    3, 2,
    7, 7, 6,
    6, 8, 7,
    8
  },
  {
     3,
     3, 4, 5,
     3, 3, 4,
     4, 2,
    10, 8, 6,
    11, 9, 7,
    11
  }
};

/* TTBLK (Transform Type per Block) tables */
static const uint8_t vc1_ttblk_codes[3][8] = {
    {  0,  1,  3,  5, 16, 17, 18, 19 },
    {  3,  0,  1,  2,  3,  5,  8,  9 },
    {  1,  0,  1,  4,  6,  7, 10, 11 }
};
static const uint8_t vc1_ttblk_bits[3][8] = {
    {  2,  2,  2,  3,  5,  5,  5,  5 },
    {  2,  3,  3,  3,  3,  3,  4,  4 },
    {  2,  3,  3,  3,  3,  3,  4,  4 }
};

/* SUBBLKPAT tables, p93-94, reordered */
static const uint8_t vc1_subblkpat_codes[3][15] = {
    { 14, 12,  7, 11,  9, 26,  2, 10, 27,  8,  0,  6,  1, 15,  1 },
    { 14,  0,  8, 15, 10,  4, 23, 13,  5,  9, 25,  3, 24, 22,  1 },
    {  5,  6,  2,  2,  8,  0, 28,  3,  1,  3, 29,  1, 19, 18, 15 }
};
static const uint8_t vc1_subblkpat_bits[3][15] = {
    {  5,  5,  5,  5,  5,  6,  4,  5,  6,  5,  4,  5,  4,  5,  1},
    {  4,  3,  4,  4,  4,  5,  5,  4,  5,  4,  5,  4,  5,  5,  2},
    {  3,  3,  4,  3,  4,  5,  5,  3,  5,  4,  5,  4,  5,  5,  4}
};

/* MV differential tables, p265 */
static const uint16_t vc1_mv_diff_codes[4][73] = {
  {
       0,    2,    3,    8,  576,    3,    2,    6,
       5,  577,  578,    7,    8,    9,   40,   19,
      37,   82,   21,   22,   23,  579,  580,  166,
      96,  167,   49,  194,  195,  581,  582,  583,
     292,  293,  294,   13,    2,    7,   24,   50,
     102,  295,   13,    7,    8,   18,   50,  103,
      38,   20,   21,   22,   39,  204,  103,   23,
      24,   25,  104,  410,  105,  106,  107,  108,
     109,  220,  411,  442,  222,  443,  446,  447,
       7 /* 73 elements */
  },
  {
       0,    4,    5,    3,    4,    3,    4,    5,
      20,    6,   21,   44,   45,   46, 3008,   95,
     112,  113,   57, 3009, 3010,  116,  117, 3011,
     118, 3012, 3013, 3014, 3015, 3016, 3017, 3018,
    3019, 3020, 3021, 3022,    1,    4,   15,  160,
     161,   41,    6,   11,   42,  162,   43,  119,
      56,   57,   58,  163,  236,  237, 3023,  119,
     120,  242,  122,  486, 1512,  487,  246,  494,
    1513,  495, 1514, 1515, 1516, 1517, 1518, 1519,
      31 /* 73 elements */
  },
  {
       0,  512,  513,  514,  515,    2,    3,  258,
     259,  260,  261,  262,  263,  264,  265,  266,
     267,  268,  269,  270,  271,  272,  273,  274,
     275,  276,  277,  278,  279,  280,  281,  282,
     283,  284,  285,  286,    1,    5,  287,  288,
     289,  290,    6,    7,  291,  292,  293,  294,
     295,  296,  297,  298,  299,  300,  301,  302,
     303,  304,  305,  306,  307,  308,  309,  310,
     311,  312,  313,  314,  315,  316,  317,  318,
     319 /* 73 elements */
  },
  {
       0,    1,    1,    2,    3,    4,    1,    5,
       4,    3,    5,    8,    6,    9,   10,   11,
      12,    7,  104,   14,  105,    4,   10,   15,
      11,    6,   14,    8,  106,  107,  108,   15,
     109,    9,   55,   10,    1,    2,    1,    2,
       3,   12,    6,    2,    6,    7,   28,    7,
      15,    8,    5,   18,   29,  152,   77,   24,
      25,   26,   39,  108,   13,  109,   55,   56,
      57,  116,   11,  153,  234,  235,  118,  119,
      15 /* 73 elements */
  }
};
static const uint8_t vc1_mv_diff_bits[4][73] = {
  {
     6,  7,  7,  8, 14,  6,  5,  6,  7, 14, 14,  6,  6,  6,  8,  9,
    10,  9,  7,  7,  7, 14, 14, 10,  9, 10,  8, 10, 10, 14, 14, 14,
    13, 13, 13,  6,  3,  5,  6,  8,  9, 13,  5,  4,  4,  5,  7,  9,
     6,  5,  5,  5,  6,  9,  8,  5,  5,  5,  7, 10,  7,  7,  7,  7,
     7,  8, 10,  9,  8,  9,  9,  9,  3 /* 73 elements */
  },
  {
     5,  7,  7,  6,  6,  5,  5,  6,  7,  5,  7,  8,  8,  8, 14,  9,
     9,  9,  8, 14, 14,  9,  9, 14,  9, 14, 14, 14, 14, 14, 14, 14,
    14, 14, 14, 14,  2,  3,  6,  8,  8,  6,  3,  4,  6,  8,  6,  9,
     6,  6,  6,  8,  8,  8, 14,  7,  7,  8,  7,  9, 13,  9,  8,  9,
    13,  9, 13, 13, 13, 13, 13, 13,  5 /* 73 elements */

  },
  {
     3, 12, 12, 12, 12,  3,  4, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11,  1,  5, 11, 11, 11, 11,  4,  4, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11, 11,
    11, 11, 11, 11, 11, 11, 11, 11, 11 /* 73 elements */
  },
  {
    15, 11, 15, 15, 15, 15, 12, 15, 12, 11, 12, 12, 15, 12, 12, 12,
    12, 15, 15, 12, 15, 10, 11, 12, 11, 10, 11, 10, 15, 15, 15, 11,
    15, 10, 14, 10,  4,  4,  5,  7,  8,  9,  5,  3,  4,  5,  6,  8,
     5,  4,  3,  5,  6,  8,  7,  5,  5,  5,  6,  7,  9,  7,  6,  6,
     6,  7, 10,  8,  8,  8,  7,  7,  4 /* 73 elements */
  }
};

static const uint32_t vc1_ac_tables[AC_MODES][186][2] = {
{
{ 0x0001,  2}, { 0x0005,  3}, { 0x000D,  4}, { 0x0012,  5}, { 0x000E,  6}, { 0x0015,  7},
{ 0x0013,  8}, { 0x003F,  8}, { 0x004B,  9}, { 0x011F,  9}, { 0x00B8, 10}, { 0x03E3, 10},
{ 0x0172, 11}, { 0x024D, 12}, { 0x03DA, 12}, { 0x02DD, 13}, { 0x1F55, 13}, { 0x05B9, 14},
{ 0x3EAE, 14}, { 0x0000,  4}, { 0x0010,  5}, { 0x0008,  7}, { 0x0020,  8}, { 0x0029,  9},
{ 0x01F4,  9}, { 0x0233, 10}, { 0x01E0, 11}, { 0x012A, 12}, { 0x03DD, 12}, { 0x050A, 13},
{ 0x1F29, 13}, { 0x0A42, 14}, { 0x1272, 15}, { 0x1737, 15}, { 0x0003,  5}, { 0x0011,  7},
{ 0x00C4,  8}, { 0x004B, 10}, { 0x00B4, 11}, { 0x07D4, 11}, { 0x0345, 12}, { 0x02D7, 13},
{ 0x07BF, 13}, { 0x0938, 14}, { 0x0BBB, 14}, { 0x095E, 15}, { 0x0013,  5}, { 0x0078,  7},
{ 0x0069,  9}, { 0x0232, 10}, { 0x0461, 11}, { 0x03EC, 12}, { 0x0520, 13}, { 0x1F2A, 13},
{ 0x3E50, 14}, { 0x3E51, 14}, { 0x1486, 15}, { 0x000C,  6}, { 0x0024,  9}, { 0x0094, 11},
{ 0x08C0, 12}, { 0x0F09, 14}, { 0x1EF0, 15}, { 0x003D,  6}, { 0x0053,  9}, { 0x01A0, 11},
{ 0x02D6, 13}, { 0x0F08, 14}, { 0x0013,  7}, { 0x007C,  9}, { 0x07C1, 11}, { 0x04AC, 14},
{ 0x001B,  7}, { 0x00A0, 10}, { 0x0344, 12}, { 0x0F79, 14}, { 0x0079,  7}, { 0x03E1, 10},
{ 0x02D4, 13}, { 0x2306, 14}, { 0x0021,  8}, { 0x023C, 10}, { 0x0FAE, 12}, { 0x23DE, 14},
{ 0x0035,  8}, { 0x0175, 11}, { 0x07B3, 13}, { 0x00C5,  8}, { 0x0174, 11}, { 0x0785, 13},
{ 0x0048,  9}, { 0x01A3, 11}, { 0x049E, 13}, { 0x002C,  9}, { 0x00FA, 10}, { 0x07D6, 11},
{ 0x0092, 10}, { 0x05CC, 13}, { 0x1EF1, 15}, { 0x00A3, 10}, { 0x03ED, 12}, { 0x093E, 14},
{ 0x01E2, 11}, { 0x1273, 15}, { 0x07C4, 11}, { 0x1487, 15}, { 0x0291, 12}, { 0x0293, 12},
{ 0x0F8A, 12}, { 0x0509, 13}, { 0x0508, 13}, { 0x078D, 13}, { 0x07BE, 13}, { 0x078C, 13},
{ 0x04AE, 14}, { 0x0BBA, 14}, { 0x2307, 14}, { 0x0B9A, 14}, { 0x1736, 15}, { 0x000E,  4},
{ 0x0045,  7}, { 0x01F3,  9}, { 0x047A, 11}, { 0x05DC, 13}, { 0x23DF, 14}, { 0x0019,  5},
{ 0x0028,  9}, { 0x0176, 11}, { 0x049D, 13}, { 0x23DD, 14}, { 0x0030,  6}, { 0x00A2, 10},
{ 0x02EF, 12}, { 0x05B8, 14}, { 0x003F,  6}, { 0x00A5, 10}, { 0x03DB, 12}, { 0x093F, 14},
{ 0x0044,  7}, { 0x07CB, 11}, { 0x095F, 15}, { 0x0063,  7}, { 0x03C3, 12}, { 0x0015,  8},
{ 0x08F6, 12}, { 0x0017,  8}, { 0x0498, 13}, { 0x002C,  8}, { 0x07B2, 13}, { 0x002F,  8},
{ 0x1F54, 13}, { 0x008D,  8}, { 0x07BD, 13}, { 0x008E,  8}, { 0x1182, 13}, { 0x00FB,  8},
{ 0x050B, 13}, { 0x002D,  8}, { 0x07C0, 11}, { 0x0079,  9}, { 0x1F5F, 13}, { 0x007A,  9},
{ 0x1F56, 13}, { 0x0231, 10}, { 0x03E4, 10}, { 0x01A1, 11}, { 0x0143, 11}, { 0x01F7, 11},
{ 0x016F, 12}, { 0x0292, 12}, { 0x02E7, 12}, { 0x016C, 12}, { 0x016D, 12}, { 0x03DC, 12},
{ 0x0F8B, 12}, { 0x0499, 13}, { 0x03D8, 12}, { 0x078E, 13}, { 0x02D5, 13}, { 0x1F5E, 13},
{ 0x1F2B, 13}, { 0x078F, 13}, { 0x04AD, 14}, { 0x3EAF, 14}, { 0x23DC, 14}, { 0x004A,  9}
},
{
{ 0x0000,  3}, { 0x0003,  4}, { 0x000B,  5}, { 0x0014,  6}, { 0x003F,  6}, { 0x005D,  7},
{ 0x00A2,  8}, { 0x00AC,  9}, { 0x016E,  9}, { 0x020A, 10}, { 0x02E2, 10}, { 0x0432, 11},
{ 0x05C9, 11}, { 0x0827, 12}, { 0x0B54, 12}, { 0x04E6, 13}, { 0x105F, 13}, { 0x172A, 13},
{ 0x20B2, 14}, { 0x2D4E, 14}, { 0x39F0, 14}, { 0x4175, 15}, { 0x5A9E, 15}, { 0x0004,  4},
{ 0x001E,  5}, { 0x0042,  7}, { 0x00B6,  8}, { 0x0173,  9}, { 0x0395, 10}, { 0x072E, 11},
{ 0x0B94, 12}, { 0x16A4, 13}, { 0x20B3, 14}, { 0x2E45, 14}, { 0x0005,  5}, { 0x0040,  7},
{ 0x0049,  9}, { 0x028F, 10}, { 0x05CB, 11}, { 0x048A, 13}, { 0x09DD, 14}, { 0x73E2, 15},
{ 0x0018,  5}, { 0x0025,  8}, { 0x008A, 10}, { 0x051B, 11}, { 0x0E5F, 12}, { 0x09C9, 14},
{ 0x139C, 15}, { 0x0029,  6}, { 0x004F,  9}, { 0x0412, 11}, { 0x048D, 13}, { 0x2E41, 14},
{ 0x0038,  6}, { 0x010E,  9}, { 0x05A8, 11}, { 0x105C, 13}, { 0x39F2, 14}, { 0x0058,  7},
{ 0x021F, 10}, { 0x0E7E, 12}, { 0x39FF, 14}, { 0x0023,  8}, { 0x02E3, 10}, { 0x04E5, 13},
{ 0x2E40, 14}, { 0x00A1,  8}, { 0x05BE, 11}, { 0x09C8, 14}, { 0x0083,  8}, { 0x013A, 11},
{ 0x1721, 13}, { 0x0044,  9}, { 0x0276, 12}, { 0x39F6, 14}, { 0x008B, 10}, { 0x04EF, 13},
{ 0x5A9B, 15}, { 0x0208, 10}, { 0x1CFE, 13}, { 0x0399, 10}, { 0x1CB4, 13}, { 0x039E, 10},
{ 0x39F3, 14}, { 0x05AB, 11}, { 0x73E3, 15}, { 0x0737, 11}, { 0x5A9F, 15}, { 0x082D, 12},
{ 0x0E69, 12}, { 0x0E68, 12}, { 0x0433, 11}, { 0x0B7B, 12}, { 0x2DF8, 14}, { 0x2E56, 14},
{ 0x2E57, 14}, { 0x39F7, 14}, { 0x51A5, 15}, { 0x0003,  3}, { 0x002A,  6}, { 0x00E4,  8},
{ 0x028E, 10}, { 0x0735, 11}, { 0x1058, 13}, { 0x1CFA, 13}, { 0x2DF9, 14}, { 0x4174, 15},
{ 0x0009,  4}, { 0x0054,  8}, { 0x0398, 10}, { 0x048B, 13}, { 0x139D, 15}, { 0x000D,  4},
{ 0x00AD,  9}, { 0x0826, 12}, { 0x2D4C, 14}, { 0x0011,  5}, { 0x016B,  9}, { 0x0B7F, 12},
{ 0x51A4, 15}, { 0x0019,  5}, { 0x021B, 10}, { 0x16FD, 13}, { 0x001D,  5}, { 0x0394, 10},
{ 0x28D3, 14}, { 0x002B,  6}, { 0x05BC, 11}, { 0x5A9A, 15}, { 0x002F,  6}, { 0x0247, 12},
{ 0x0010,  7}, { 0x0A35, 12}, { 0x003E,  6}, { 0x0B7A, 12}, { 0x0059,  7}, { 0x105E, 13},
{ 0x0026,  8}, { 0x09CF, 14}, { 0x0055,  8}, { 0x1CB5, 13}, { 0x0057,  8}, { 0x0E5B, 12},
{ 0x00A0,  8}, { 0x1468, 13}, { 0x0170,  9}, { 0x0090, 10}, { 0x01CE,  9}, { 0x021A, 10},
{ 0x0218, 10}, { 0x0168,  9}, { 0x021E, 10}, { 0x0244, 12}, { 0x0736, 11}, { 0x0138, 11},
{ 0x0519, 11}, { 0x0E5E, 12}, { 0x072C, 11}, { 0x0B55, 12}, { 0x09DC, 14}, { 0x20BB, 14},
{ 0x048C, 13}, { 0x1723, 13}, { 0x2E44, 14}, { 0x16A5, 13}, { 0x0518, 11}, { 0x39FE, 14},
{ 0x0169,  9}
},
{
{ 0x0001,  2}, { 0x0006,  3}, { 0x000F,  4}, { 0x0016,  5}, { 0x0020,  6}, { 0x0018,  7},
{ 0x0008,  8}, { 0x009A,  8}, { 0x0056,  9}, { 0x013E,  9}, { 0x00F0, 10}, { 0x03A5, 10},
{ 0x0077, 11}, { 0x01EF, 11}, { 0x009A, 12}, { 0x005D, 13}, { 0x0001,  4}, { 0x0011,  5},
{ 0x0002,  7}, { 0x000B,  8}, { 0x0012,  9}, { 0x01D6,  9}, { 0x027E, 10}, { 0x0191, 11},
{ 0x00EA, 12}, { 0x03DC, 12}, { 0x013B, 13}, { 0x0004,  5}, { 0x0014,  7}, { 0x009E,  8},
{ 0x0009, 10}, { 0x01AC, 11}, { 0x01E2, 11}, { 0x03CA, 12}, { 0x005F, 13}, { 0x0017,  5},
{ 0x004E,  7}, { 0x005E,  9}, { 0x00F3, 10}, { 0x01AD, 11}, { 0x00EC, 12}, { 0x05F0, 13},
{ 0x000E,  6}, { 0x00E1,  8}, { 0x03A4, 10}, { 0x009C, 12}, { 0x013D, 13}, { 0x003B,  6},
{ 0x001C,  9}, { 0x0014, 11}, { 0x09BE, 12}, { 0x0006,  7}, { 0x007A,  9}, { 0x0190, 11},
{ 0x0137, 13}, { 0x001B,  7}, { 0x0008, 10}, { 0x075C, 11}, { 0x0071,  7}, { 0x00D7, 10},
{ 0x09BF, 12}, { 0x0007,  8}, { 0x00AF, 10}, { 0x04CC, 11}, { 0x0034,  8}, { 0x0265, 10},
{ 0x009F, 12}, { 0x00E0,  8}, { 0x0016, 11}, { 0x0327, 12}, { 0x0015,  9}, { 0x017D, 11},
{ 0x0EBB, 12}, { 0x0014,  9}, { 0x00F6, 10}, { 0x01E4, 11}, { 0x00CB, 10}, { 0x099D, 12},
{ 0x00CA, 10}, { 0x02FC, 12}, { 0x017F, 11}, { 0x04CD, 11}, { 0x02FD, 12}, { 0x04FE, 11},
{ 0x013A, 13}, { 0x000A,  4}, { 0x0042,  7}, { 0x01D3,  9}, { 0x04DD, 11}, { 0x0012,  5},
{ 0x00E8,  8}, { 0x004C, 11}, { 0x0136, 13}, { 0x0039,  6}, { 0x0264, 10}, { 0x0EBA, 12},
{ 0x0000,  7}, { 0x00AE, 10}, { 0x099C, 12}, { 0x001F,  7}, { 0x04DE, 11}, { 0x0043,  7},
{ 0x04DC, 11}, { 0x0003,  8}, { 0x03CB, 12}, { 0x0006,  8}, { 0x099E, 12}, { 0x002A,  8},
{ 0x05F1, 13}, { 0x000F,  8}, { 0x09FE, 12}, { 0x0033,  8}, { 0x09FF, 12}, { 0x0098,  8},
{ 0x099F, 12}, { 0x00EA,  8}, { 0x013C, 13}, { 0x002E,  8}, { 0x0192, 11}, { 0x0136,  9},
{ 0x006A,  9}, { 0x0015, 11}, { 0x03AF, 10}, { 0x01E3, 11}, { 0x0074, 11}, { 0x00EB, 12},
{ 0x02F9, 12}, { 0x005C, 13}, { 0x00ED, 12}, { 0x03DD, 12}, { 0x0326, 12}, { 0x005E, 13},
{ 0x0016,  7}
},
{
{ 0x0004,  3}, { 0x0014,  5}, { 0x0017,  7}, { 0x007F,  8}, { 0x0154,  9}, { 0x01F2, 10},
{ 0x00BF, 11}, { 0x0065, 12}, { 0x0AAA, 12}, { 0x0630, 13}, { 0x1597, 13}, { 0x03B7, 14},
{ 0x2B22, 14}, { 0x0BE6, 15}, { 0x000B,  4}, { 0x0037,  7}, { 0x0062,  9}, { 0x0007, 11},
{ 0x0166, 12}, { 0x00CE, 13}, { 0x1590, 13}, { 0x05F6, 14}, { 0x0BE7, 15}, { 0x0007,  5},
{ 0x006D,  8}, { 0x0003, 11}, { 0x031F, 12}, { 0x05F2, 14}, { 0x0002,  6}, { 0x0061,  9},
{ 0x0055, 12}, { 0x01DF, 14}, { 0x001A,  6}, { 0x001E, 10}, { 0x0AC9, 12}, { 0x2B23, 14},
{ 0x001E,  6}, { 0x001F, 10}, { 0x0AC3, 12}, { 0x2B2B, 14}, { 0x0006,  7}, { 0x0004, 11},
{ 0x02F8, 13}, { 0x0019,  7}, { 0x0006, 11}, { 0x063D, 13}, { 0x0057,  7}, { 0x0182, 11},
{ 0x2AA2, 14}, { 0x0004,  8}, { 0x0180, 11}, { 0x059C, 14}, { 0x007D,  8}, { 0x0164, 12},
{ 0x076D, 15}, { 0x0002,  9}, { 0x018D, 11}, { 0x1581, 13}, { 0x00AD,  8}, { 0x0060, 12},
{ 0x0C67, 14}, { 0x001C,  9}, { 0x00EE, 13}, { 0x0003,  9}, { 0x02CF, 13}, { 0x00D9,  9},
{ 0x1580, 13}, { 0x0002, 11}, { 0x0183, 11}, { 0x0057, 12}, { 0x0061, 12}, { 0x0031, 11},
{ 0x0066, 12}, { 0x0631, 13}, { 0x0632, 13}, { 0x00AC, 13}, { 0x031D, 12}, { 0x0076, 12},
{ 0x003A, 11}, { 0x0165, 12}, { 0x0C66, 14}, { 0x0003,  2}, { 0x0054,  7}, { 0x02AB, 10},
{ 0x0016, 13}, { 0x05F7, 14}, { 0x0005,  4}, { 0x00F8,  9}, { 0x0AA9, 12}, { 0x005F, 15},
{ 0x0004,  4}, { 0x001C, 10}, { 0x1550, 13}, { 0x0004,  5}, { 0x0077, 11}, { 0x076C, 15},
{ 0x000E,  5}, { 0x000A, 12}, { 0x000C,  5}, { 0x0562, 11}, { 0x0004,  6}, { 0x031C, 12},
{ 0x0006,  6}, { 0x00C8, 13}, { 0x000D,  6}, { 0x01DA, 13}, { 0x0007,  6}, { 0x00C9, 13},
{ 0x0001,  7}, { 0x002E, 14}, { 0x0014,  7}, { 0x1596, 13}, { 0x000A,  7}, { 0x0AC2, 12},
{ 0x0016,  7}, { 0x015B, 14}, { 0x0015,  7}, { 0x015A, 14}, { 0x000F,  8}, { 0x005E, 15},
{ 0x007E,  8}, { 0x00AB,  8}, { 0x002D,  9}, { 0x00D8,  9}, { 0x000B,  9}, { 0x0014, 10},
{ 0x02B3, 10}, { 0x01F3, 10}, { 0x003A, 10}, { 0x0000, 10}, { 0x0058, 10}, { 0x002E,  9},
{ 0x005E, 10}, { 0x0563, 11}, { 0x00EC, 12}, { 0x0054, 12}, { 0x0AC1, 12}, { 0x1556, 13},
{ 0x02FA, 13}, { 0x0181, 11}, { 0x1557, 13}, { 0x059D, 14}, { 0x2AA3, 14}, { 0x2B2A, 14},
{ 0x01DE, 14}, { 0x063C, 13}, { 0x00CF, 13}, { 0x1594, 13}, { 0x000D,  9}
},
{
{ 0x0002,  2}, { 0x0006,  3}, { 0x000F,  4}, { 0x000D,  5}, { 0x000C,  5}, { 0x0015,  6},
{ 0x0013,  6}, { 0x0012,  6}, { 0x0017,  7}, { 0x001F,  8}, { 0x001E,  8}, { 0x001D,  8},
{ 0x0025,  9}, { 0x0024,  9}, { 0x0023,  9}, { 0x0021,  9}, { 0x0021, 10}, { 0x0020, 10},
{ 0x000F, 10}, { 0x000E, 10}, { 0x0007, 11}, { 0x0006, 11}, { 0x0020, 11}, { 0x0021, 11},
{ 0x0050, 12}, { 0x0051, 12}, { 0x0052, 12}, { 0x000E,  4}, { 0x0014,  6}, { 0x0016,  7},
{ 0x001C,  8}, { 0x0020,  9}, { 0x001F,  9}, { 0x000D, 10}, { 0x0022, 11}, { 0x0053, 12},
{ 0x0055, 12}, { 0x000B,  5}, { 0x0015,  7}, { 0x001E,  9}, { 0x000C, 10}, { 0x0056, 12},
{ 0x0011,  6}, { 0x001B,  8}, { 0x001D,  9}, { 0x000B, 10}, { 0x0010,  6}, { 0x0022,  9},
{ 0x000A, 10}, { 0x000D,  6}, { 0x001C,  9}, { 0x0008, 10}, { 0x0012,  7}, { 0x001B,  9},
{ 0x0054, 12}, { 0x0014,  7}, { 0x001A,  9}, { 0x0057, 12}, { 0x0019,  8}, { 0x0009, 10},
{ 0x0018,  8}, { 0x0023, 11}, { 0x0017,  8}, { 0x0019,  9}, { 0x0018,  9}, { 0x0007, 10},
{ 0x0058, 12}, { 0x0007,  4}, { 0x000C,  6}, { 0x0016,  8}, { 0x0017,  9}, { 0x0006, 10},
{ 0x0005, 11}, { 0x0004, 11}, { 0x0059, 12}, { 0x000F,  6}, { 0x0016,  9}, { 0x0005, 10},
{ 0x000E,  6}, { 0x0004, 10}, { 0x0011,  7}, { 0x0024, 11}, { 0x0010,  7}, { 0x0025, 11},
{ 0x0013,  7}, { 0x005A, 12}, { 0x0015,  8}, { 0x005B, 12}, { 0x0014,  8}, { 0x0013,  8},
{ 0x001A,  8}, { 0x0015,  9}, { 0x0014,  9}, { 0x0013,  9}, { 0x0012,  9}, { 0x0011,  9},
{ 0x0026, 11}, { 0x0027, 11}, { 0x005C, 12}, { 0x005D, 12}, { 0x005E, 12}, { 0x005F, 12},
{ 0x0003,  7}
},
{
{ 0x0002,  2}, { 0x000F,  4}, { 0x0015,  6}, { 0x0017,  7}, { 0x001F,  8}, { 0x0025,  9},
{ 0x0024,  9}, { 0x0021, 10}, { 0x0020, 10}, { 0x0007, 11}, { 0x0006, 11}, { 0x0020, 11},
{ 0x0006,  3}, { 0x0014,  6}, { 0x001E,  8}, { 0x000F, 10}, { 0x0021, 11}, { 0x0050, 12},
{ 0x000E,  4}, { 0x001D,  8}, { 0x000E, 10}, { 0x0051, 12}, { 0x000D,  5}, { 0x0023,  9},
{ 0x000D, 10}, { 0x000C,  5}, { 0x0022,  9}, { 0x0052, 12}, { 0x000B,  5}, { 0x000C, 10},
{ 0x0053, 12}, { 0x0013,  6}, { 0x000B, 10}, { 0x0054, 12}, { 0x0012,  6}, { 0x000A, 10},
{ 0x0011,  6}, { 0x0009, 10}, { 0x0010,  6}, { 0x0008, 10}, { 0x0016,  7}, { 0x0055, 12},
{ 0x0015,  7}, { 0x0014,  7}, { 0x001C,  8}, { 0x001B,  8}, { 0x0021,  9}, { 0x0020,  9},
{ 0x001F,  9}, { 0x001E,  9}, { 0x001D,  9}, { 0x001C,  9}, { 0x001B,  9}, { 0x001A,  9},
{ 0x0022, 11}, { 0x0023, 11}, { 0x0056, 12}, { 0x0057, 12}, { 0x0007,  4}, { 0x0019,  9},
{ 0x0005, 11}, { 0x000F,  6}, { 0x0004, 11}, { 0x000E,  6}, { 0x000D,  6}, { 0x000C,  6},
{ 0x0013,  7}, { 0x0012,  7}, { 0x0011,  7}, { 0x0010,  7}, { 0x001A,  8}, { 0x0019,  8},
{ 0x0018,  8}, { 0x0017,  8}, { 0x0016,  8}, { 0x0015,  8}, { 0x0014,  8}, { 0x0013,  8},
{ 0x0018,  9}, { 0x0017,  9}, { 0x0016,  9}, { 0x0015,  9}, { 0x0014,  9}, { 0x0013,  9},
{ 0x0012,  9}, { 0x0011,  9}, { 0x0007, 10}, { 0x0006, 10}, { 0x0005, 10}, { 0x0004, 10},
{ 0x0024, 11}, { 0x0025, 11}, { 0x0026, 11}, { 0x0027, 11}, { 0x0058, 12}, { 0x0059, 12},
{ 0x005A, 12}, { 0x005B, 12}, { 0x005C, 12}, { 0x005D, 12}, { 0x005E, 12}, { 0x005F, 12},
{ 0x0003,  7}
},
{
{ 0x0000,  2}, { 0x0003,  3}, { 0x000D,  4}, { 0x0005,  4}, { 0x001C,  5}, { 0x0016,  5},
{ 0x003F,  6}, { 0x003A,  6}, { 0x002E,  6}, { 0x0022,  6}, { 0x007B,  7}, { 0x0067,  7},
{ 0x005F,  7}, { 0x0047,  7}, { 0x0026,  7}, { 0x00EF,  8}, { 0x00CD,  8}, { 0x00C1,  8},
{ 0x00A9,  8}, { 0x004F,  8}, { 0x01F2,  9}, { 0x01DD,  9}, { 0x0199,  9}, { 0x0185,  9},
{ 0x015D,  9}, { 0x011B,  9}, { 0x03EF, 10}, { 0x03E1, 10}, { 0x03C8, 10}, { 0x0331, 10},
{ 0x0303, 10}, { 0x02F1, 10}, { 0x02A0, 10}, { 0x0233, 10}, { 0x0126, 10}, { 0x07C0, 11},
{ 0x076F, 11}, { 0x076C, 11}, { 0x0661, 11}, { 0x0604, 11}, { 0x0572, 11}, { 0x0551, 11},
{ 0x046A, 11}, { 0x0274, 11}, { 0x0F27, 12}, { 0x0F24, 12}, { 0x0EDB, 12}, { 0x0C8E, 12},
{ 0x0C0B, 12}, { 0x0C0A, 12}, { 0x0AE3, 12}, { 0x08D6, 12}, { 0x0490, 12}, { 0x0495, 12},
{ 0x1F19, 13}, { 0x1DB5, 13}, { 0x0009,  4}, { 0x0010,  5}, { 0x0029,  6}, { 0x0062,  7},
{ 0x00F3,  8}, { 0x00AD,  8}, { 0x01E5,  9}, { 0x0179,  9}, { 0x009C,  9}, { 0x03B1, 10},
{ 0x02AE, 10}, { 0x0127, 10}, { 0x076E, 11}, { 0x0570, 11}, { 0x0275, 11}, { 0x0F25, 12},
{ 0x0EC0, 12}, { 0x0AA0, 12}, { 0x08D7, 12}, { 0x1E4C, 13}, { 0x0008,  5}, { 0x0063,  7},
{ 0x00AF,  8}, { 0x017B,  9}, { 0x03B3, 10}, { 0x07DD, 11}, { 0x0640, 11}, { 0x0F8D, 12},
{ 0x0BC1, 12}, { 0x0491, 12}, { 0x0028,  6}, { 0x00C3,  8}, { 0x0151,  9}, { 0x02A1, 10},
{ 0x0573, 11}, { 0x0EC3, 12}, { 0x1F35, 13}, { 0x0065,  7}, { 0x01DA,  9}, { 0x02AF, 10},
{ 0x0277, 11}, { 0x08C9, 12}, { 0x1781, 13}, { 0x0025,  7}, { 0x0118,  9}, { 0x0646, 11},
{ 0x0AA6, 12}, { 0x1780, 13}, { 0x00C9,  8}, { 0x0321, 10}, { 0x0F9B, 12}, { 0x191E, 13},
{ 0x0048,  8}, { 0x07CC, 11}, { 0x0AA1, 12}, { 0x0180,  9}, { 0x0465, 11}, { 0x1905, 13},
{ 0x03E2, 10}, { 0x0EC1, 12}, { 0x3C9B, 14}, { 0x02F4, 10}, { 0x08C8, 12}, { 0x07C1, 11},
{ 0x0928, 13}, { 0x05E1, 11}, { 0x320D, 14}, { 0x0EC2, 12}, { 0x6418, 15}, { 0x1F34, 13},
{ 0x0078,  7}, { 0x0155,  9}, { 0x0552, 11}, { 0x191F, 13}, { 0x00FA,  8}, { 0x07DC, 11},
{ 0x1907, 13}, { 0x00AC,  8}, { 0x0249, 11}, { 0x13B1, 14}, { 0x01F6,  9}, { 0x0AE2, 12},
{ 0x01DC,  9}, { 0x04ED, 12}, { 0x0184,  9}, { 0x1904, 13}, { 0x0156,  9}, { 0x09D9, 13},
{ 0x03E7, 10}, { 0x0929, 13}, { 0x03B2, 10}, { 0x3B68, 14}, { 0x02F5, 10}, { 0x13B0, 14},
{ 0x0322, 10}, { 0x3B69, 14}, { 0x0234, 10}, { 0x7935, 15}, { 0x07C7, 11}, { 0xC833, 16},
{ 0x0660, 11}, { 0x7934, 15}, { 0x024B, 11}, { 0xC832, 16}, { 0x0AA7, 12}, { 0x1F18, 13},
{ 0x007A,  7}
},
{
{ 0x0002,  2}, { 0x0000,  3}, { 0x001E,  5}, { 0x0004,  5}, { 0x0012,  6}, { 0x0070,  7},
{ 0x001A,  7}, { 0x005F,  8}, { 0x0047,  8}, { 0x01D3,  9}, { 0x00B5,  9}, { 0x0057,  9},
{ 0x03B5, 10}, { 0x016D, 10}, { 0x0162, 10}, { 0x07CE, 11}, { 0x0719, 11}, { 0x0691, 11},
{ 0x02C6, 11}, { 0x0156, 11}, { 0x0F92, 12}, { 0x0D2E, 12}, { 0x0D20, 12}, { 0x059E, 12},
{ 0x0468, 12}, { 0x02A6, 12}, { 0x1DA2, 13}, { 0x1C60, 13}, { 0x1A43, 13}, { 0x0B1D, 13},
{ 0x08C0, 13}, { 0x055D, 13}, { 0x0003,  3}, { 0x000A,  5}, { 0x0077,  7}, { 0x00E5,  8},
{ 0x01D9,  9}, { 0x03E5, 10}, { 0x0166, 10}, { 0x0694, 11}, { 0x0152, 11}, { 0x059F, 12},
{ 0x1F3C, 13}, { 0x1A4B, 13}, { 0x055E, 13}, { 0x000C,  4}, { 0x007D,  7}, { 0x0044,  8},
{ 0x03E0, 10}, { 0x0769, 11}, { 0x0E31, 12}, { 0x1F26, 13}, { 0x055C, 13}, { 0x001B,  5},
{ 0x00E2,  8}, { 0x03A5, 10}, { 0x02C9, 11}, { 0x1F23, 13}, { 0x3B47, 14}, { 0x0007,  5},
{ 0x01D8,  9}, { 0x02D8, 11}, { 0x1F27, 13}, { 0x3494, 14}, { 0x0035,  6}, { 0x03E1, 10},
{ 0x059C, 12}, { 0x38C3, 14}, { 0x000C,  6}, { 0x0165, 10}, { 0x1D23, 13}, { 0x1638, 14},
{ 0x0068,  7}, { 0x0693, 11}, { 0x3A45, 14}, { 0x0020,  7}, { 0x0F90, 12}, { 0x7CF6, 15},
{ 0x00E8,  8}, { 0x058F, 12}, { 0x2CEF, 15}, { 0x0045,  8}, { 0x0B3A, 13}, { 0x01F1,  9},
{ 0x3B46, 14}, { 0x01A7,  9}, { 0x1676, 14}, { 0x0056,  9}, { 0x692A, 15}, { 0x038D, 10},
{ 0xE309, 16}, { 0x00AA, 10}, { 0x1C611, 17}, { 0x02DF, 11}, { 0xB3B9, 17}, { 0x02C8, 11},
{ 0x38C20, 18}, { 0x01B0, 11}, { 0x16390, 18}, { 0x0F9F, 12}, { 0x16771, 18}, { 0x0ED0, 12},
{ 0x71843, 19}, { 0x0D2A, 12}, { 0xF9E8C, 20}, { 0x0461, 12}, { 0xF9E8E, 20}, { 0x0B67, 13},
{ 0x055F, 13}, { 0x003F,  6}, { 0x006D,  9}, { 0x0E90, 12}, { 0x054E, 13}, { 0x0013,  6},
{ 0x0119, 10}, { 0x0B66, 13}, { 0x000B,  6}, { 0x0235, 11}, { 0x7CF5, 15}, { 0x0075,  7},
{ 0x0D24, 12}, { 0xF9E9, 16}, { 0x002E,  7}, { 0x1F22, 13}, { 0x0021,  7}, { 0x054F, 13},
{ 0x0014,  7}, { 0x3A44, 14}, { 0x00E4,  8}, { 0x7CF7, 15}, { 0x005E,  8}, { 0x7185, 15},
{ 0x0037,  8}, { 0x2C73, 15}, { 0x01DB,  9}, { 0x59DD, 16}, { 0x01C7,  9}, { 0x692B, 15},
{ 0x01A6,  9}, { 0x58E5, 16}, { 0x00B4,  9}, { 0x1F3D0, 17}, { 0x00B0,  9}, { 0xB1C9, 17},
{ 0x03E6, 10}, { 0x16770, 18}, { 0x016E, 10}, { 0x3E7A2, 18}, { 0x011B, 10}, { 0xF9E8D, 20},
{ 0x00D9, 10}, { 0xF9E8F, 20}, { 0x00A8, 10}, { 0x2C723, 19}, { 0x0749, 11}, { 0xE3084, 20},
{ 0x0696, 11}, { 0x58E45, 20}, { 0x02DE, 11}, { 0xB1C88, 21}, { 0x0231, 11}, { 0x1C610A, 21},
{ 0x01B1, 11}, { 0x71842D, 23}, { 0x0D2B, 12}, { 0x38C217, 22}, { 0x0D2F, 12}, { 0x163913, 22},
{ 0x05B2, 12}, { 0x163912, 22}, { 0x0469, 12}, { 0x71842C, 23}, { 0x1A42, 13}, { 0x08C1, 13},
{ 0x0073,  7}
}
};

const int ff_vc1_ac_sizes[AC_MODES] = {
    186, 169, 133, 149, 103, 103, 163, 175
};

#endif /* AVCODEC_VC1_VLC_DATA_H */
