/*******************************************************\
* irmp3-ncurses - An ncurses frontend for irmp3 using   *
* the Network Control Module                            *
* (C) 2003 Ross Axe                                     *
*                                                       *
* scroller.c - Draw scrolling text                      *
\*******************************************************/

#if HAVE_CONFIG_H
#  include "config.h"
#endif

#if HAVE_UNISTD_H
#  include <unistd.h>
#endif
#include <signal.h>

#include "irmp3-ncurses.h"

vcid("$Id: scroller.c,v 1.16 2005/03/21 10:36:24 ross Exp $");


struct scrollerlist {
    struct scrollerlist *next;
    struct scroller *scroller;
} *scrollerlist_head = NULL;

static void dump_scrollerlist(void)
{
    int i;
    struct scrollerlist *scr;

    for(i = 0, scr = scrollerlist_head; scr; i++, scr = scr->next) {
	dbg_printf(2, "msg: %s", scr->scroller->msg);
    }
    dbg_printf(2, "%d items in scrollerlist", i);
}

/* scroller must be valid by the time this is called */
static void scrollerlist_add(struct scroller *scroller)
{
    struct scrollerlist *scr = malloc(sizeof *scr);

    if(!scr) {
	sbar_printf(_("No memory for scrolling data"));
	return;
    }
    *scr = (struct scrollerlist) {
	.next = scrollerlist_head,
	.scroller = scroller,
    };
    scrollerlist_head = scr;
    dump_scrollerlist();
}

/* scroller must still be valid when this is called */
static void scrollerlist_del(struct scroller *scroller)
{
    struct scrollerlist *scr;

    if(!scrollerlist_head) {
	sbar_printf(_("scrollerlist is empty"));
	return;
    }
    if(scrollerlist_head->scroller == scroller) {
	scr = scrollerlist_head;
	scrollerlist_head = scr->next;
	free(scr);
	dump_scrollerlist();
	return;
    }
    for(scr = scrollerlist_head; scr->next; scr = scr->next) {
	if(scr->next->scroller == scroller) {
	    struct scrollerlist *tmp_scr = scr->next;

	    scr->next = scr->next->next;
	    free(tmp_scr);
	    dump_scrollerlist();
	    return;
	}
    }
    sbar_printf(_("Tried to delete nonexistant scroller 0x%p"), scroller);
}


static RETSIGTYPE sigalarm_handler(int signal)
{
    struct scrollerlist *scr;

    assert(signal == SIGALRM);
    if(!scrollerlist_head)
	return;
    for(scr = scrollerlist_head; scr; scr = scr->next) {
	struct scroller *const scroller = scr->scroller;
	int newpos = scroller->pos + scroller->dir;
	int x, y;
	void *hookdata = scroller->entryhook ? scroller->entryhook(scroller)
		: NULL;

	assert(scroller->wnd);
	if(newpos < 0) {
	    newpos = 0;
	    scroller->dir = -scroller->dir;
	    assert(scroller->dir > 0);
	}
	if(newpos > strlen(scroller->msg) - scroller->len) {
	    newpos = strlen(scroller->msg) - scroller->len;
	    scroller->dir = -scroller->dir;
	    assert(scroller->dir < 0);
	}
	getyx(*scroller->wnd, y, x);
	mvwprintw(*scroller->wnd, scroller->y, scroller->x, "%.*s",
		  scroller->len, scroller->msg + newpos);
	scroller->pos = newpos;
	wmove(*scroller->wnd, y, x);
	if(scroller->exithook)
	    scroller->exithook(scroller, hookdata);
	wrefresh(*scroller->wnd);
    }
    alarm(1);
}

void wscroller_hook(WINDOW **wnd, struct scroller *scroller, const char *msg,
		    bool right, void *(*entryhook)(struct scroller *),
		    void (*exithook)(struct scroller *, void *))
{
    int x, y;
    int w, h;
    void *hookdata;

    getyx(*wnd, y, x);
    getmaxyx(*wnd, h, w);
    w -= x;
    *scroller = (struct scroller) {
	.wnd = wnd,
	.msg = strdup(msg),
	.x = x,
	.y = y,
	.len = w,
	.pos = right ? strlen(msg) - w : 0,
	.dir = right ? +1 : -1,
	.entryhook = entryhook,
	.exithook = exithook,
    };
    hookdata = entryhook ? entryhook(scroller) : NULL;
    wprintw(*wnd, "%-*.*s", w, w, msg + (scroller->pos > 0
					 ? scroller->pos : 0));
    if(exithook)
	exithook(scroller, hookdata);
    if(w >= strlen(msg)) {
	free(scroller->msg);
	*scroller = (struct scroller)SCROLLER_INITIALISER;
    } else {
	scrollerlist_add(scroller);
	signal(SIGALRM, sigalarm_handler);
	alarm(1);
    }
}

void scroller_end(struct scroller *scroller)
{
    if(scroller->msg) {
	scrollerlist_del(scroller);
	free(scroller->msg);
    }
    *scroller = (struct scroller)SCROLLER_INITIALISER;
}

void scroller_endall(void)
{
    signal(SIGALRM, SIG_IGN);
}
