/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		UDPSocket.cpp

	Contains:	Implementation of object defined in UDPSocket.h.

	$Log: UDPSocket.cpp,v $
	Revision 1.2  1999/02/19 23:14:16  ds
	Created
	
	
*/

#ifndef __MW_
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/errno.h>
#endif

#if NEED_SOCKETBITS
#if __GLIBC__ >= 2
#include <bits/socket.h>
#else
#include <socketbits.h>
#endif
#endif

#include "UDPSocket.h"
#include "OS.h"


UDPSocket::UDPSocket(UInt32 inSocketType, Task* inTask)
: Socket(inSocketType, inTask), fDemuxer(NULL)
{
	if (inSocketType & kWantsDemuxer)
		fDemuxer = new ('udpd') UDPDemuxer();
		
	//setup msghdr
	::memset(&fMsgHdr, 0, sizeof(fMsgHdr));
	::memset(&fCmsgData, 0, sizeof(fCmsgData));
	::memset(&fMsgAddr, 0, sizeof(fMsgAddr));
	
	fMsgHdr.msg_name = (char*)&fMsgAddr;
	fMsgHdr.msg_namelen = sizeof(fMsgAddr);
	fMsgHdr.msg_iov = &fMsgIovec;
	fMsgHdr.msg_iovlen = 1;
	fMsgHdr.msg_control = (char*)&fCmsgData;
	fMsgHdr.msg_flags = 0;
}


QTSS_ErrorCode
UDPSocket::SendTo(UInt32 inRemoteAddr, UInt16 inRemotePort, char* inBuffer, UInt32 inLength)
{
	Assert(inBuffer != NULL);
	
	if (fSocket == kInvalidSocket)
		return QTSS_FileNotOpen;
		
	struct sockaddr_in 	theRemoteAddr;
	theRemoteAddr.sin_family = PF_INET;
	theRemoteAddr.sin_port = htons((ushort) inRemotePort);
	theRemoteAddr.sin_addr.s_addr = htonl(inRemoteAddr);

	int theErr = ::sendto(fSocket, inBuffer, inLength, 0,
							(sockaddr*)&theRemoteAddr, sizeof(theRemoteAddr));
	if (theErr != 0)
		return (QTSS_ErrorCode)OSThread::GetErrno();
	return QTSS_NoErr;
}

QTSS_ErrorCode
UDPSocket::RecvPacket(UInt32* outDestAddr, char* ioBuffer, UInt32 inBufLen, UInt32* outRecvLen)
{
	if (fSocket == kInvalidSocket)
		return QTSS_FileNotOpen;
		
	Assert(outRecvLen != NULL);
	Assert(outDestAddr != NULL);
	Assert(ioBuffer != NULL);
	*outDestAddr = 0;
	
	fMsgHdr.msg_controllen = sizeof(fCmsgData.cmsgHdr);
	fMsgHdr.msg_iov->iov_base = ioBuffer;
	fMsgHdr.msg_iov->iov_len = inBufLen;
	int theRecvLen = ::recvmsg(fSocket, &fMsgHdr, 0);
	if (theRecvLen == -1)
		return this->CheckError(outRecvLen);

	*outRecvLen = (UInt32)theRecvLen;

	//Check to make sure the OS supports these control things
	if (fMsgHdr.msg_controllen == 0)
		return QTSS_NoErr;

#if __MacOSX__
	//Attempt to retrieve the destination IP address out of the cmsghdr
	struct cmsghdr* theHeader;
	for (theHeader = CMSG_FIRSTHDR(&fMsgHdr); theHeader != NULL;
			/*theHeader = CMSG_NXTHDR(&fMsgHdr, theHeader)*/)
	{
		Assert(0);//this loop currently doesn't really work
		if ((theHeader->cmsg_level == IPPROTO_IP) && (theHeader->cmsg_type == IP_RECVDSTADDR))
		{
			UInt32* theIPAddr = (UInt32*)CMSG_DATA(theHeader);
			*outDestAddr = *theIPAddr;
			return QTSS_NoErr;
		}
	}
#endif
	return QTSS_NoErr;		
}

QTSS_ErrorCode
UDPSocket::RecvFrom(UInt32* outRemoteAddr, UInt16* outRemotePort,
							char* ioBuffer, UInt32 inBufLen, UInt32* outRecvLen)
{
	if (fSocket == kInvalidSocket)
		return QTSS_FileNotOpen;
		
	Assert(outRecvLen != NULL);
	Assert(outRemoteAddr != NULL);
	Assert(outRemotePort != NULL);
	
	int addrLen = sizeof(fMsgAddr);
	
	SInt32 theRecvLen = ::recvfrom(fSocket, ioBuffer, inBufLen, 0, (sockaddr*)&fMsgAddr, &addrLen);
	if (theRecvLen == -1)
		return this-> CheckError(outRecvLen);

	*outRemoteAddr = ntohl(fMsgAddr.sin_addr.s_addr);
	*outRemotePort = ntohs(fMsgAddr.sin_port);
	Assert(theRecvLen >= 0);
	*outRecvLen = (UInt32)theRecvLen;
	return QTSS_NoErr;		
}

QTSS_ErrorCode UDPSocket::CheckError(UInt32* outRecvLen)
{
	//process an error from receive
	if (OSThread::GetErrno() == EAGAIN)
	{
		//we didn't get any data back from this receive, so reinstall the
		//socket on the event queue so next time there is data we can get it
		Modwatch();
		*outRecvLen = 0;
		return QTSS_NoErr;
	}
	else
	{
		//are there any errors that can happen?
		*outRecvLen = 0;
		return (QTSS_ErrorCode)OSThread::GetErrno();
	}
}

QTSS_ErrorCode UDPSocket::JoinMulticast(UInt32 inRemoteAddr, UInt16 timeToLive)
{
	struct ip_mreq	theMulti;
	theMulti.imr_multiaddr.s_addr = htonl(inRemoteAddr);
	theMulti.imr_interface.s_addr = fLocalAddr.sin_addr.s_addr;//already in network byte order
	int err = setsockopt(fSocket, IPPROTO_IP, IP_ADD_MEMBERSHIP, (char*)&theMulti, sizeof(theMulti));
	AssertV(err == 0, OSThread::GetErrno());
	if (err == -1)
		return (QTSS_ErrorCode)OSThread::GetErrno();

	// set the ttl
	u_char	nOptVal = (u_char)timeToLive;//dms - stevens pp. 496. bsd implementations barf
											//unless this is a u_char
	err = setsockopt(fSocket, IPPROTO_IP, IP_MULTICAST_TTL, (char*)&nOptVal, sizeof(nOptVal));
	AssertV(err == 0, OSThread::GetErrno());
	if (err == -1)
		return (QTSS_ErrorCode)OSThread::GetErrno();
	else
		return QTSS_NoErr;	
}
QTSS_ErrorCode UDPSocket::LeaveMulticast(UInt32 inRemoteAddr)
{
	struct ip_mreq	theMulti;
	theMulti.imr_multiaddr.s_addr = htonl(inRemoteAddr);
	theMulti.imr_interface.s_addr = htonl(fLocalAddr.sin_addr.s_addr);
	int err = setsockopt(fSocket, IPPROTO_IP, IP_DROP_MEMBERSHIP, (char*)&theMulti, sizeof(theMulti));
	AssertV(err == 0, OSThread::GetErrno());
	if (err == -1)
		return (QTSS_ErrorCode)OSThread::GetErrno();
	else
		return QTSS_NoErr;	
}
