/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		Socket.cpp

	Contains:	implements Socket class
					
	$Log: Socket.cpp,v $
	Revision 1.2  1999/02/19 23:13:18  ds
	Created
	
	
*/

#include <string.h>

#ifndef __MW_
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/errno.h>
#include <netinet/in.h>
#include <fcntl.h>
#include <sys/uio.h>
#include <unistd.h>
#include "ev.h"
#if __MacOSX__
#include "tempcalls.h" //includes MacOS X prototypes of event queue functions
#endif
#endif

#include "Socket.h"
#include "SocketUtils.h"
#include "OS.h"
#include "atomic.h"

SocketEventQueueThread* Socket::sThread = NULL;
unsigned int			Socket::sUniqueID = 1;
OSRefTable*				Socket::sRefTable = NULL;

void Socket::Initialize()
{
	if (sRefTable == NULL)
		sRefTable = new OSRefTable();
}

void Socket::StartThread()
{
	if (sThread == NULL)
	{
		sThread = new ('scEv') SocketEventQueueThread();
		Assert(sThread != NULL);
		sThread->Start();
	}
}

Socket::Socket(UInt32 inSocketType, Task *notifytask)
: 	fSocket(kInvalidSocket),
	fTask(notifytask),
	fUniqueID(0),
	fUniqueIDStr((char*)&fUniqueID, sizeof(fUniqueID)),
	fState(inSocketType),
	fLocalAddrStrPtr(fLocalAddrStr, 0),
	fLocalDNSStrPtr(NULL),
	fPortStr(fPortBuffer, kPortBufSizeInBytes)
{
	fLocalAddr.sin_addr.s_addr = 0;
	fLocalAddr.sin_port = 0;
	
	fDestAddr.sin_addr.s_addr = 0;
	fDestAddr.sin_port = 0;
}

Socket::~Socket()
{
	int err = 0;
	
	if (fSocket != kInvalidSocket)
	{
		//if this object is registered in the table, unregister it now
		if (fUniqueID > 0)
		{
			sRefTable->UnRegister(&fRef);

#if !__MacOSX__
			removeevent(fSocket);//The eventqueue / select shim requires this
#else
			//On Linux (possibly other UNIX implementations) you MUST NOT close the fd before
			//removing the fd from the select mask, and having the select function wake up
			//to register this fact. If you close the fd first, bad things may happen, like
			//the socket not getting unbound from the port & IP addr.
			//
			//So, what we do is have the select thread itself call close. This is triggered
			//by calling removeevent.
			err = ::close(fSocket);
#endif		
		}
		else
			err = ::close(fSocket);
	}	
	AssertV(err == 0, OSThread::GetErrno());//we don't really care if there was an error, but it's nice to know
}

QTSS_ErrorCode Socket::Open(int theType)
{
	Assert(fSocket == kInvalidSocket);
	fSocket = ::socket(PF_INET, theType, 0);
	if (fSocket == kInvalidSocket)
		return (QTSS_ErrorCode)OSThread::GetErrno();
			
	InitNonblocking();
	return QTSS_NoErr;
}

void Socket::ReuseAddr()
{
	int one = 1;
	int err = ::setsockopt(fSocket, SOL_SOCKET, SO_REUSEADDR, (char*)&one, sizeof(int));
	Assert(err == 0);	
}

void Socket::InitNonblocking()
{
	Assert(fSocket != kInvalidSocket);
	

	//If this isn't a nonblocking socket, just return
	if (!(fState & kNonBlocking))
		return;
	
	int flag; 
	flag = ::fcntl(fSocket, F_GETFL, 0);
	int err = ::fcntl(fSocket, F_SETFL, flag | O_NONBLOCK);
	AssertV(err == 0, OSThread::GetErrno());
	
	//If this socket doesn't want events, don't setup the event queue
	if (!(fState & kWantsEvents))
		return;
				
	//ok, we want to get events on this socket. Register it
	//and watch events

	//allocate a Unique ID for this socket, and add it to the ref table
	fUniqueID = atomic_add(&sUniqueID, 1);

	fRef.Set(fUniqueIDStr, this);
	sRefTable->Register(&fRef);
		
	//fill out the eventreq data structure
	::memset( &fEventReq, '\0', sizeof(fEventReq));
	fEventReq.er_type = EV_FD;
	fEventReq.er_handle = fSocket;
	fEventReq.er_eventbits = EV_RE;
	fEventReq.er_data = (void*)fUniqueID;

#if DEBUG
	fModwatched = true;
#endif
	if (watchevent(&fEventReq, EV_RE) != 0)
		//this should never fail, but if it does, cleanup.
		AssertV(false, OSThread::GetErrno());
}

void Socket::Modwatch(int theMask)
{
#if DEBUG
	//Assert(fModwatched == false);
	fModwatched = true;
#endif
	//Only modwatch the socket if the socket is in the event queue!
	if (!(fState & kWantsEvents))
		return;

	fEventReq.er_eventbits = theMask;
	if (modwatch(&fEventReq, theMask) != 0)
		AssertV(false, OSThread::GetErrno());//FIXME! what to do if this barfs? probably close the connection?
}

QTSS_ErrorCode Socket::Bind(UInt32 addr, UInt16 port)
{
	if (fSocket == kInvalidSocket)
		return QTSS_FileNotOpen;
	
	::memset(&fLocalAddr, 0, sizeof(fLocalAddr));
	fLocalAddr.sin_family = PF_INET;
	fLocalAddr.sin_port = htons(port);
	fLocalAddr.sin_addr.s_addr = htonl(addr);
	
	int err = ::bind(fSocket, (sockaddr *)&fLocalAddr, sizeof(fLocalAddr));
	
	if (err == -1)
	{
		fLocalAddr.sin_port = 0;
		fLocalAddr.sin_addr.s_addr = 0;
		return (QTSS_ErrorCode)OSThread::GetErrno();
	}
	fState |= kBound;
	return QTSS_NoErr;
}

StrPtrLen*	Socket::GetLocalAddrStr()
{
	//Use the array of IP addr strings to locate the string formatted version
	//of this IP address.
	Assert(fLocalAddr.sin_addr.s_addr != INADDR_ANY);
	if (fLocalAddrStrPtr.Len == 0)
	{
		fLocalAddrStrPtr.Ptr = fLocalAddrStr;
		SocketUtils::ConvertAddrToString(fLocalAddr.sin_addr, &fLocalAddrStrPtr);
	}
	return &fLocalAddrStrPtr;
#if 0
	if (fLocalAddrStrPtr == NULL)
	{
		for (UInt32 x = 0; x < SocketUtils::GetNumIPAddrs(); x++)
		{
			if (SocketUtils::GetIPAddr(x) == fLocalAddr.sin_addr.s_addr)
			{
				fLocalAddrStrPtr = SocketUtils::GetIPAddrStr(x);
				break;
			}
		}
	}
	Assert(fLocalAddrStrPtr != NULL);
	return fLocalAddrStrPtr;
#endif
}

StrPtrLen*	Socket::GetLocalDNSStr()
{
	//Do the same thing as the above function, but for DNS names
	Assert(fLocalAddr.sin_addr.s_addr != INADDR_ANY);
	if (fLocalDNSStrPtr == NULL)
	{
		for (UInt32 x = 0; x < SocketUtils::GetNumIPAddrs(); x++)
		{
			if (SocketUtils::GetIPAddr(x) == fLocalAddr.sin_addr.s_addr)
			{
				fLocalDNSStrPtr = SocketUtils::GetDNSNameStr(x);
				break;
			}
		}
	}

	//if we weren't able to get this DNS name, make the DNS name the same as the IP addr str.
	if (fLocalDNSStrPtr == NULL)
		fLocalDNSStrPtr = this->GetLocalAddrStr();

	Assert(fLocalDNSStrPtr != NULL);
	return fLocalDNSStrPtr;
}

StrPtrLen*	Socket::GetLocalPortStr()
{
	if (fPortStr.Len == kPortBufSizeInBytes)
	{
		int temp = ntohs(fLocalAddr.sin_port);
		::sprintf(fPortBuffer, "%d", temp);
		fPortStr.Len = ::strlen(fPortBuffer);
	}
	return &fPortStr;
}

QTSS_ErrorCode Socket::Send(const char* inData, const UInt32 inLength, UInt32* outLengthSent)
{
	if (fSocket == kInvalidSocket)
		return QTSS_FileNotOpen;
		
	Assert(inData != NULL);
	
	int err = ::send(fSocket, inData, inLength, 0);//flags??
	if (err == -1)
	{
		//WarnV(false, OSThread::GetErrno());//just to see what happened
		//are there any errors that can happen if the client is connected?
		//Yes... EAGAIN. Means the socket is now flow-controleld
		int theErr = OSThread::GetErrno();
		if ((theErr != EAGAIN) && (this->IsConnected()))
			fState ^= kConnected;//turn off connected state flag
		if (theErr == EAGAIN)
			this->Modwatch(EV_RE | EV_WR);
			 
		return (QTSS_ErrorCode)OSThread::GetErrno();
	}
	
	if (outLengthSent != NULL)
		*outLengthSent = err;
	return QTSS_NoErr;
}

QTSS_ErrorCode Socket::WriteV(const struct iovec* iov, const UInt32 numIOvecs)
{
	if (fSocket == kInvalidSocket)
		return QTSS_FileNotOpen;

	Assert(iov != NULL);

	int err = ::writev(fSocket, iov, numIOvecs);
	if (err == -1)
	{
		//WarnV(false, OSThread::GetErrno());//just to see what happened
		//are there any errors that can happen if the client is connected?
		if (IsConnected())
			fState ^= kConnected;//turn off connected state flag
		return (QTSS_ErrorCode)OSThread::GetErrno();
	}
	return QTSS_NoErr;
}

QTSS_ErrorCode Socket::Read(void *buffer, const UInt32 length, UInt32 *outRecvLenP)
{
	if (fSocket == kInvalidSocket)
		return QTSS_FileNotOpen;

	Assert(outRecvLenP != NULL);
	Assert(buffer != NULL);
	SInt32 theRecvLen = ::recv(fSocket, buffer, length, 0);//flags??

	QTSS_ErrorCode theErr = QTSS_NoErr;

	//we've gotten some error from recv
	if (theRecvLen == -1)
		theErr = (QTSS_ErrorCode)OSThread::GetErrno();
	//if we get 0 bytes back from read, that means the client has disconnected.
	//Note that and return the proper error to the caller
	else if (theRecvLen == 0)
		theErr = (QTSS_ErrorCode)ENOTCONN;
	
	if (theErr != QTSS_NoErr)
	{
		//process an error from receive
		if (theErr == EAGAIN)
		{
			//we didn't get any data back from this receive, so reinstall the
			//socket on the event queue so next time there is data we can get it
			Modwatch();
			*outRecvLenP = 0;
			return QTSS_NoErr;
		}
		else
		{
			//are there any errors that can happen if the client is connected?
			if (IsConnected())
				fState ^= kConnected;//turn off connected state flag
			*outRecvLenP = 0;
			return theErr;
		}
	}
	Assert(theRecvLen > 0);
	*outRecvLenP = (UInt32)theRecvLen;
	return QTSS_NoErr;
}



void SocketEventQueueThread::Entry()
{
	struct eventreq theCurrentEvent;
	::memset( &theCurrentEvent, '\0', sizeof(theCurrentEvent) );
	
	while (true)
	{
		int theErrno = EINTR;
		while (theErrno == EINTR)
		{
			int theReturnValue = waitevent(&theCurrentEvent, NULL);
			
			//Sort of a hack. In the POSIX version of the server, waitevent can return
			//an actual POSIX errorcode.
			if (theReturnValue >= 0)
				theErrno = theReturnValue;
			else
				theErrno = OSThread::GetErrno();
		}
		
		AssertV(theErrno == 0, theErrno);
		
		//ok, there's data waiting on this socket. Send a wakeup.
		Assert(theCurrentEvent.er_data != NULL);
		if (theCurrentEvent.er_data != NULL)
		{
			//The cookie in this event is an ObjectID. Resolve that objectID into
			//a pointer.
			StrPtrLen idStr((char*)&theCurrentEvent.er_data, sizeof(theCurrentEvent.er_data));
			OSRef* ref = Socket::sRefTable->Resolve(&idStr);
			if (ref != NULL)
			{
				Socket* theSocket = (Socket*)ref->GetObject();
#if DEBUG
				theSocket->fModwatched = false;
#endif
				Task::EventFlags theEvents = 0;
				if (theCurrentEvent.er_eventbits & EV_RE)
					theEvents |= Task::kReadEvent;
				if (theCurrentEvent.er_eventbits & EV_WR)
					theEvents |= Task::kWriteEvent;
				theSocket->ProcessEvent(theEvents);
				Socket::sRefTable->Release(ref);
			}
		}
	}
}

