/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		SDPParser.cpp

	Contains:	Implementation of object defined in .h file

	Copyright:	 1998 by Apple Computer, Inc., all rights reserved.

	$Log: SDPParser.cpp,v $
	Revision 1.2  1999/02/19 23:13:18  ds
	Created
	

*/

#ifndef __MW_
#include <sys/types.h>
#include <arpa/inet.h>
#endif

#include "SDPParser.h"

#include "StringParser.h"
#include "StringFormatter.h"
#include "OS.h"
#include "SocketUtils.h"
#include "StrPtrLen.h"

StrPtrLen	SDPParser::sCLine("c=IN IP4 0.0.0.0\r\na=control:*\r\n");
StrPtrLen	SDPParser::sVideoStr("video");
StrPtrLen	SDPParser::sAudioStr("audio");
StrPtrLen	SDPParser::sRtpMapStr("rtpmap");
StrPtrLen	SDPParser::sControlStr("control");

char* SDPParser::ConvertToLocalSDP(StrPtrLen* inCompleteSDPFile, UInt32* newSDPLen)
{
	bool appendCLine = true;
	UInt32 trackIndex = 0;
	
	char *localSDP = new ('sdp2') char[inCompleteSDPFile->Len + 400];
	StringFormatter localSDPFormatter(localSDP, inCompleteSDPFile->Len + 400);

	StrPtrLen sdpLine;
	StringParser sdpParser(inCompleteSDPFile);
	char trackIndexBuffer[50];

	while (true)
	{
		//stop when we reach an empty line.
		sdpParser.ConsumeUntil(&sdpLine, StringParser::sEOLMask);
		if (sdpLine.Len == 0)
			break;
			
		switch (*sdpLine.Ptr)
		{
			case 'c':
				break;//ignore connection information
			case 'm':
			{
				//append new connection information right before the first 'm'
				if (appendCLine)
				{
					localSDPFormatter.Put(sCLine);
					appendCLine = false;
				}
				//the last "a=" for each m should be the control a=
				if (trackIndex > 0)
				{
					::sprintf(trackIndexBuffer, "a=control:trackID=%ld\r\n",trackIndex);
					localSDPFormatter.Put(trackIndexBuffer, ::strlen(trackIndexBuffer));
				}
				//now write the 'm' line, but strip off the port information
				StringParser mParser(&sdpLine);
				StrPtrLen mPrefix;
				mParser.ConsumeUntil(&mPrefix, StringParser::sDigitMask);
				localSDPFormatter.Put(mPrefix);
				localSDPFormatter.Put("0", 1);
				(void)mParser.ConsumeInteger(NULL);
				localSDPFormatter.Put(mParser.GetCurrentPosition(), mParser.GetDataRemaining());
				localSDPFormatter.PutEOL();
				trackIndex++;
				break;
			}
			default:
				localSDPFormatter.Put(sdpLine);
				localSDPFormatter.PutEOL();
		}
		//make sure to go past the EOL
		if (!sdpParser.ExpectEOL())
			break;
	}
	
	if (trackIndex > 0)
	{
		::sprintf(trackIndexBuffer, "a=control:trackID=%ld\r\n",trackIndex);
		localSDPFormatter.Put(trackIndexBuffer, ::strlen(trackIndexBuffer));
	}
	*newSDPLen = (UInt32)localSDPFormatter.GetCurrentOffset();
	return localSDP;
}

SDPParser::~SDPParser()
{
	if (fDeleteSDP)
		delete [] fSDPData.Ptr;
	if (fStreamArray != NULL)
	{
		char* theCharArray = (char*)fStreamArray;
		delete [] theCharArray;
	}
}


void SDPParser::Parse(char* completeSDPFile, UInt32 completeSDPLen)
{
	Assert(fSDPData.Ptr == NULL);
	Assert(fStreamArray == NULL);
	
	fSDPData.Set(completeSDPFile, completeSDPLen);

	bool hasGlobalStreamInfo = false;
	StreamInfo theGlobalStreamInfo; //needed if there is one c= header independent of
									//individual streams

	StrPtrLen sdpLine;
	StringParser trackCounter(&fSDPData);
	StringParser sdpParser(&fSDPData);
	UInt32 theStreamIndex = 0;

	//walk through the SDP, counting up the number of tracks
	while (true)
	{
		//stop when we reach an empty line.
		trackCounter.ConsumeUntil(&sdpLine, StringParser::sEOLMask);
		if (sdpLine.Len == 0)
			break;
		//each 'm' line in the SDP file corresponds to another track.
		if (sdpLine.Ptr[0] == 'm')
			fNumStreams++;	

		if (!trackCounter.ExpectEOL())
			break;
	}

	//We should scale the # of StreamInfos to the # of trax, but we can't because
	//of an annoying compiler bug...
	
	fStreamArray = (StreamInfo*)new ('siar') char[(fNumStreams + 1) * sizeof(StreamInfo)];
	::memset(fStreamArray, 0, (fNumStreams + 1) * sizeof(StreamInfo));
	
	//Now actually get all the data on all the streams
	while (true)
	{
		//stop when we reach an empty line.
		sdpParser.ConsumeUntil(&sdpLine, StringParser::sEOLMask);
		if (sdpLine.Len == 0)
			break;

		switch (*sdpLine.Ptr)
		{
			case 'm':
			{
				if (hasGlobalStreamInfo)
				{
					fStreamArray[theStreamIndex].fIPAddr = theGlobalStreamInfo.fIPAddr;
					fStreamArray[theStreamIndex].fTimeToLive = theGlobalStreamInfo.fTimeToLive;
				}
				
				StringParser mParser(&sdpLine);
				
				//find out what type of track this is
				mParser.ConsumeLength(NULL, 2);//go past 'm='
				StrPtrLen theStreamType;
				mParser.ConsumeWord(&theStreamType);
				if (theStreamType.Equal(sVideoStr))
					fStreamArray[theStreamIndex].fCodecType = RTPStream::kVideoCodecType;
				else if (theStreamType.Equal(sAudioStr))
					fStreamArray[theStreamIndex].fCodecType = RTPStream::kAudioCodecType;
					
				//find the port for this stream
				mParser.ConsumeUntil(NULL, StringParser::sDigitMask);
				SInt32 tempPort = mParser.ConsumeInteger(NULL);
				if ((tempPort > 0) && (tempPort < 65536))
					fStreamArray[theStreamIndex].fPort = tempPort;
				theStreamIndex++;
			}
			break;
			case 'a':
			{
				//if we haven't even hit an 'm' line yet, just ignore all 'a' lines
				if (theStreamIndex == 0)
					break;
					
				StringParser aParser(&sdpLine);
				aParser.ConsumeLength(NULL, 2);//go past 'a='
				StrPtrLen aLineType;
				aParser.ConsumeWord(&aLineType);
				
				if (aLineType.Equal(sRtpMapStr))
				{
					//mark the codec type if this line has a codec name on it. If we already
					//have a codec type for this track, just ignore this line
					if ((fStreamArray[theStreamIndex - 1].fCodecName.Len == 0) &&
						(aParser.GetThru(NULL, ' ')))
						(void)aParser.GetThruEOL(&fStreamArray[theStreamIndex - 1].fCodecName);
				}
				else if (aLineType.Equal(sControlStr))
				{			
					//mark the trackID if that's what this line has
					aParser.ConsumeUntil(NULL, StringParser::sDigitMask);
					fStreamArray[theStreamIndex - 1].fTrackID = aParser.ConsumeInteger(NULL);
				}
			}
			break;
			case 'c':
			{
				//get the IP address off this header
				UInt32 tempIPAddr = 0;
				StringParser cParser(&sdpLine);
				cParser.ConsumeLength(NULL, 9);//strip off "c=in ip4 "
				StrPtrLen ipAddrStr;
				cParser.ConsumeUntil(&ipAddrStr, '/');
				char endChar = ipAddrStr.Ptr[ipAddrStr.Len];
				ipAddrStr.Ptr[ipAddrStr.Len] = '\0';
				//inet_addr returns numeric IP addr in network byte order, make
				//sure to convert to host order.
				tempIPAddr = (UInt32)ntohl(::inet_addr(ipAddrStr.Ptr));
				ipAddrStr.Ptr[ipAddrStr.Len] = endChar;
				
				//grab the ttl
				SInt32 tempTtl = kDefaultTTL;
				if (cParser.GetThru(NULL, '/'))
				{
					tempTtl = cParser.ConsumeInteger(NULL);
					Assert(tempTtl >= 0);
					Assert(tempTtl < 65536);
				}

				if (theStreamIndex > 0)
				{
					//if this c= line is part of a stream, it overrides the
					//global stream information
					fStreamArray[theStreamIndex - 1].fIPAddr = tempIPAddr;
					fStreamArray[theStreamIndex - 1].fTimeToLive = tempTtl;
				} else {
					theGlobalStreamInfo.fIPAddr = tempIPAddr;
					theGlobalStreamInfo.fTimeToLive = tempTtl;
					hasGlobalStreamInfo = true;
				}
			}
		}
		//make sure to go past the EOL
		if (!sdpParser.ExpectEOL())
			break;
	}
}

SDPParser::StreamInfo*	SDPParser::GetStreamInfo(UInt32 inStreamIndex)
{
	Assert(inStreamIndex < fNumStreams);
	if (fStreamArray == NULL)
		return NULL;
	if (inStreamIndex < fNumStreams)
		return &fStreamArray[inStreamIndex];
	else
		return NULL;
}

bool	SDPParser::IsReflectable()
{
	if (fStreamArray == NULL)
		return false;
	//each stream's info must meet certain criteria
	for (UInt32 x = 0; x < fNumStreams; x++)
	{
		if ((!this->IsReflectableIPAddr(fStreamArray[x].fIPAddr)) ||
			(fStreamArray[x].fPort == 0) ||
			(fStreamArray[x].fTimeToLive == 0))
			return false;
	}
	return true;
}

bool	SDPParser::IsReflectableIPAddr(UInt32 inIPAddr)
{
	if (SocketUtils::IsMulticastIPAddr(inIPAddr) || SocketUtils::IsLocalIPAddr(inIPAddr))
		return true;
	return false;
}

