/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPServer.cpp

	Contains:	Implements object defined in RTSPServer.h
	
	$Log: RTSPServer.cpp,v $
	Revision 1.3  1999/02/26 05:14:57  cl
	Added HTTP stats (RTSPWebStatsModule)
	
	Revision 1.2  1999/02/19 23:08:35  ds
	Created
	

*/

#include "RTSPServer.h"


#include "RTPInterfaceModule.h"
#include "RTSPWebDebugModule.h"
#include "RTSPWebStatsModule.h"
#include "RTSPLoggingModule.h"
#include "RTSPPlayListModule.h"
#if __MacOSX__
#include "RTSPSvrControlModule.h" //Module only applicable on MacOS X
#endif
#include "OS.h"
#include "SocketUtils.h"

char* RTSPServer::sPortPrefString = "rtsp_port";

RTSPServer::~RTSPServer()
{
	for (UInt32 moduleIter = 0; moduleIter < sNumRTSPModules; moduleIter++)
		if (NULL != sRTSPModuleList[moduleIter])
			sRTSPModuleList[moduleIter]->Shutdown();
}


bool RTSPServer::Initialize(RTSPPrefs* inPrefs, UInt16 inPortOverride)
{
	sSrvrPrefs = inPrefs;
	
	//Init static utils, resources
	RTSPRequestInterface::Initialize();
	
	//load modules: these get loaded pretty much before anything in the server has
	//been initialized. This is mainly for logging: if any errors happen in the
	//startup process, we want to make sure they get caught. However, we need to
	//make sure that the APIs the modules might call at startup are protected.
#if __MacOSX__
	sNumRTSPModules = 5;
#else
	sNumRTSPModules = 4;
#endif
	sRTSPModuleList = new('sMod')  RTSPModule*[sNumRTSPModules];
	//**************  sRTSPModuleList[0] = new ('rand') RTSPPlayListModule();
	sRTSPModuleList[0] = new ('rtpI') RTPInterfaceModule();
	sRTSPModuleList[1] = new ('rlog') RTSPLoggingModule();
	sRTSPModuleList[2] = new ('webd') RTSPWebDebugModule();
	sRTSPModuleList[3] = new ('webs') RTSPWebStatsModule();
#if __MacOSX__
	sRTSPModuleList[4] = new ('svrc') RTSPSvrControlModule();
#endif

	for (UInt32 moduleIter = 0; moduleIter < sNumRTSPModules; moduleIter++)
	{
		if (!sRTSPModuleList[moduleIter]->Initialize())
		{
			RTSPModule* badModule = sRTSPModuleList[moduleIter];
			sRTSPModuleList[moduleIter] = NULL;
			delete badModule;
		}
	}
	
	//check to make sure there is an available ip interface
	if (SocketUtils::GetNumIPAddrs() == 0)
	{
		RTSPModuleInterface::LogError(RTSPModule::kFatal, RTSPMessages::kNotConfiguredForIP, 0);
		return false;
	}

	//find out what our default IP addr is & dns name
	if (sSrvrPrefs->GetRTSPIPAddress() == INADDR_ANY)
		sDefaultIPAddr = SocketUtils::GetIPAddr(0);
	else
		sDefaultIPAddr = sSrvrPrefs->GetRTSPIPAddress();
		
	for (UInt32 ipAddrIter = 0; ipAddrIter < SocketUtils::GetNumIPAddrs(); ipAddrIter++)
	{
		if (SocketUtils::GetIPAddr(ipAddrIter) == sDefaultIPAddr)
		{
			sDefaultDNSName = SocketUtils::GetDNSNameStr(ipAddrIter);
			Assert(sDefaultDNSName != NULL);
			break;
		}
	}
	if (sDefaultDNSName == NULL)
	{
		//If we've gotten here, what has probably happened is the IP address (explicitly
		//entered as a preference) doesn't exist
		RTSPModuleInterface::LogError(RTSPModule::kFatal, RTSPMessages::kDefaultRTSPAddrUnavail, 0);
		return false;	
	}
				
	//at this point, we are ready to start listening
	bool openedListener = this->CreateListeners(inPortOverride);

	//Even if there is an error, these objects should still exist
	Assert(fListeners != NULL);
	Assert(fListeners[0] != NULL);
	RTSPServerInterface::sFirstListener = fListeners[0];
	
	if (!openedListener)
		return false;
	
	return true;
}

bool RTSPServer::CreateListeners(UInt16 inPortOverride)
{
	QTSS_ErrorCode err = QTSS_NoErr;
	
	//If there is a port override in effect, ignore any settings, and use that port
	if (inPortOverride != 0)
	{
		//Because of a bug in the GCC MacOS X compiler, we can't create an actual array of objects
		fListeners = new ('lsar') RTSPListenerSocket*[2];
		fListeners[0] = new ('lsck') RTSPListenerSocket();

		err = fListeners[0]->Initialize(sSrvrPrefs->GetRTSPIPAddress(), inPortOverride);
		return (err == QTSS_NoErr);
	}

	static const UInt32 kMaxPrefValueSize = 1024;
	char thePrefValue[kMaxPrefValueSize];

	//Figure out how many listeners we need (based on the number of ports we are listening on)
	UInt32 theNumPorts = 0;

	//loop through all the ports. The first time, just to find out how many,
	//the second time, to actually read the values into the array
	bool ok = true;
	for (theNumPorts = 0; ok; theNumPorts++)
	{
		thePrefValue[0] = '\0';
		// function return 0 if it couldn't find the preference
		ok = (bool)sSrvrPrefs->GetPrefsSource()->
				GetValueByIndex(sPortPrefString, theNumPorts, &thePrefValue[0]);
	}
	theNumPorts--;

	//if there aren't any ports entered, default to the compile-time default port
	if (theNumPorts == 0)
	{
		//Because of a bug in the GCC MacOS X compiler, we can't create an actual array of objects
		fListeners = new ('lsar') RTSPListenerSocket*[2];
		fListeners[0] = new ('lsck') RTSPListenerSocket();
		
		err = fListeners[0]->Initialize(sSrvrPrefs->GetRTSPIPAddress(), kDefaultRTSPPort);
		return (err == QTSS_NoErr);
	}
	
	bool allListenersOpen = true;
	
	//Now that we know how many listeners we need to create, go ahead and create them
	//Because of a bug in the GCC MacOS X compiler, we can't create an actual array of objects
	fListeners = new ('lsar') RTSPListenerSocket*[theNumPorts + 2];
	for (UInt32 theNewIter = 0; theNewIter < theNumPorts; theNewIter++)
		fListeners[theNewIter] = new ('lsck') RTSPListenerSocket();
	
	//Open all the listeners on all the proper ports
	for (UInt32 theIndex = 0; theIndex < theNumPorts; theIndex++)
	{
		thePrefValue[0] = '\0';
		ok = (bool)sSrvrPrefs->GetPrefsSource()->
				GetValueByIndex(sPortPrefString, theIndex, &thePrefValue[0]);
		Assert(ok);
		
		if (thePrefValue[0] != '\0')
		{
			UInt16 thePort = (UInt16)::strtol(thePrefValue, NULL, 10);
			err = fListeners[theIndex]->Initialize(sSrvrPrefs->GetRTSPIPAddress(), thePort);
		}
		if (err != QTSS_NoErr)
		{
			allListenersOpen = false;
			RTSPModuleInterface::LogError(RTSPModule::kFatal, RTSPMessages::kCouldntListenWarning, err);
		}
	}
	return allListenersOpen;
}

Task*	RTSPListenerSocket::GetSessionTask(TCPSocket** outSocket)
{
	Assert(outSocket != NULL);
	
	RTSPSession* theTask = new ('sSes') RTSPSession();
	*outSocket = theTask->GetSocket();
	return theTask;
}


