/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		RTSPRequest.cpp

	Contains:	Implementation of RTSPRequest class.

	$Log: RTSPRequest.cpp,v $
	Revision 1.2  1999/02/19 23:08:31  ds
	Created
	
	
*/


#include "RTSPRequest.h"
#include "RTSPProtocol.h"

#include "RTSPSessionInterface.h"
#include "StringParser.h"
#include "QTSS.h"
#include "RTSPModule.h"
#include "StringTranslator.h"
#include "OS.h"

UInt8
RTSPRequest::sURLStopConditions[] =
{
	0, 0, 0, 0, 0, 0, 0, 0, 0, 1, //0-9      //'\t' is a stop condition
	1, 0, 0, 1, 0, 0, 0, 0, 0, 0, //10-19    //'\r' & '\n' are stop conditions
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //20-29
	0, 0, 1, 0, 0, 0, 1, 0, 0, 0, //30-39    //' ' & '$' are stop conditions
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //40-49
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //50-59
	0, 0, 0, 1, 0, 0, 0, 0, 0, 0, //60-69    //'?' is a stop condition
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //70-79
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //80-89
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //90-99
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //100-109
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //110-119
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //120-129
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //130-139
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //140-149
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //150-159
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //160-169
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //170-179
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //180-189
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //190-199
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //200-209
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //210-219
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //220-229
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //230-239
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, //240-249
	0, 0, 0, 0, 0, 0 			 //250-255
};


//Parses the request
RTSPProtocol::RTSPStatusCode RTSPRequest::Parse(StrPtrLen *stream)
{
	Assert(stream != NULL);
	StringParser parser(stream);

	//parse status line.
	RTSPProtocol::RTSPStatusCode error = ParseFirstLine(parser);

	//handle any errors that come up	
	if (error != RTSPProtocol::kSuccessOK)
		return error;
		
	error = ParseHeaders(parser);
	if (error != RTSPProtocol::kSuccessOK)
		return error;
	
	//Response headers should set themselves up to reflect what's in the request headers
	fResponseKeepAlive = fRequestKeepAlive;
	
	//Make sure that there was some path that was extracted from this request. If not, there is no way
	//we can process the request, so generate an error
	if (fQTSSConnParams[qtssFilePathParam].Len == 0)
		return this->SendErrorResponse(	RTSPProtocol::kClientBadRequest,
											RTSPMessages::kNoURLInRequest);
	
	return RTSPProtocol::kSuccessOK;
}

//returns: StatusLineTooLong, SyntaxError, BadMethod
RTSPProtocol::RTSPStatusCode RTSPRequest::ParseFirstLine(StringParser &parser)
{	
	//first get the method
	parser.ConsumeWord(&fQTSSConnParams[qtssMethodParam]);
	
	//THIS WORKS UNDER THE ASSUMPTION THAT:
	//valid HTTP/1.1 headers are: GET, HEAD, POST, PUT, OPTIONS, DELETE, TRACE
	fMethod = RTSPProtocol::GetMethod(fQTSSConnParams[qtssMethodParam]);
	if (fMethod == RTSPProtocol::kIllegalMethod)
		return this->SendErrorResponse(	RTSPProtocol::kClientBadRequest,
										RTSPMessages::kBadRTSPMethod,
										&fQTSSConnParams[qtssMethodParam]);
	
	//no longer assume this is a space... instead, just consume whitespace
	parser.ConsumeWhitespace();

	//now parse the uri
	RTSPProtocol::RTSPStatusCode err = ParseURI(parser);
	if (err != RTSPProtocol::kSuccessOK)
		return err;

	//no longer assume this is a space... instead, just consume whitespace
	parser.ConsumeWhitespace();

	//if there is a version, consume the version string
	StrPtrLen versionStr;
	parser.ConsumeUntil(&versionStr, StringParser::sEOLMask);
	
	//setup the first line parameter
	fQTSSConnParams[qtssFirstLineParam].Ptr = fQTSSConnParams[qtssFullRequestParam].Ptr;
	fQTSSConnParams[qtssFirstLineParam].Len = parser.GetCurrentPosition() -
												fQTSSConnParams[qtssFullRequestParam].Ptr;
	//check the version
	if (versionStr.Len > 0)
		fVersion = RTSPProtocol::GetVersion(versionStr);

	//go past the end of line
	if (!parser.ExpectEOL())
		return this->SendErrorResponse(RTSPProtocol::kClientBadRequest,
										RTSPMessages::kNoRTSPVersion);
	return RTSPProtocol::kSuccessOK;
}

//returns: SyntaxError if there was an error in the uri. Or InternalServerError
RTSPProtocol::RTSPStatusCode RTSPRequest::ParseURI(StringParser &parser)
{
	//read in the complete URL, set it to be the qtssAbsoluteURLParam
	StrPtrLen* theAbsURLParam = &fQTSSConnParams[qtssAbsoluteURLParam];
	Assert(theAbsURLParam != NULL);
	parser.ConsumeUntil(theAbsURLParam, sURLStopConditions);
	
	StringParser urlParser(theAbsURLParam);
		
	//we always should have a slash before the uri.
	//If not, that indicates this is a full URI
	if (*theAbsURLParam->Ptr != '/')
	{
		//if it is a full URL, store the host name off in a separate parameter
		StrPtrLen theRTSPString;
		urlParser.ConsumeLength(&theRTSPString, 7);
		if (!theRTSPString.EqualIgnoreCase("rtsp://", 7))
			return this->SendErrorResponse(RTSPProtocol::kClientBadRequest,
											RTSPMessages::kNoRTSPInURL,
											&theRTSPString);

		//assign the host field here to the proper QTSS param
		urlParser.ConsumeUntil(&fHeaders[RTSPProtocol::kHostHeader], '/');
	}
	
	//whatever is in this position in the URl must be the relative URL. Store that
	//in the qtssURLParam.
	StrPtrLen* theURLParam = &fQTSSConnParams[qtssURLParam];
	theURLParam->Ptr = urlParser.GetCurrentPosition();
	theURLParam->Len = urlParser.GetDataReceivedLen() - urlParser.GetDataParsedLen();
	
	//path strings are statically allocated. Therefore, if they are longer than
	//this length we won't be able to handle the request.
	if (theURLParam->Len > RTSPRequestInterface::kMaxFilePathSizeInBytes)
		return this->SendErrorResponse(RTSPProtocol::kClientBadRequest,
										RTSPMessages::kURLTooLong,
										theURLParam);

	//decode the URL, put the result in the separate buffer for the file path,
	//set the file path StrPtrLen to the proper value
	SInt32 theBytesWritten = StringTranslator::DecodeURL(theURLParam->Ptr, theURLParam->Len,
												fFilePath, RTSPRequestInterface::kMaxFilePathSizeInBytes);
	//if negative, an error occurred, reported as an QTSS_ErrorCode
	//we also need to leave room for a terminator.
	if ((theBytesWritten < 0) || (theBytesWritten == RTSPRequestInterface::kMaxFilePathSizeInBytes))
	{
		Assert(theBytesWritten != QTSS_NotEnoughSpace);
		if (theBytesWritten == QTSS_BadURLFormat)
			return this->SendErrorResponse(RTSPProtocol::kClientBadRequest,
													RTSPMessages::kURLInBadFormat,
													theURLParam);
	}
	
	//setup the proper QTSS param
	fFilePath[theBytesWritten] = '\0';
	fQTSSConnParams[qtssFilePathParam].Ptr = fFilePath;
	fQTSSConnParams[qtssFilePathParam].Len = theBytesWritten;
	return RTSPProtocol::kSuccessOK;
}


//throws eHTTPNoMoreData and eHTTPOutOfBuffer
RTSPProtocol::RTSPStatusCode RTSPRequest::ParseHeaders(StringParser& parser)
{
	StrPtrLen theKeyWord;
	bool isStreamOK;
	
	while ((parser.PeekFast() != '\r') && (parser.PeekFast() != '\n')) 
	{
		isStreamOK = parser.GetThru(&theKeyWord, ':');
		if (!isStreamOK)
			return this->SendErrorResponse(RTSPProtocol::kClientBadRequest,
											RTSPMessages::kNoColonAfterHeader);
							
		if (parser.PeekFast() == ' ') {	// handle space, if any
			isStreamOK = parser.Expect(' ');
			Assert(isStreamOK);
		}
		
		RTSPProtocol::RTSPHeader theHeader = RTSPProtocol::GetRequestHeader(theKeyWord);
		isStreamOK = parser.GetThruEOL(&fHeaders[theHeader]);
		if (!isStreamOK)
			return this->SendErrorResponse(RTSPProtocol::kClientBadRequest,
											RTSPMessages::kNoEOLAfterHeader);
		
		//some headers require some special processing. If this code begins
		//to get out of control, we made need to come up with a function pointer table
		switch (theHeader)
		{
			case RTSPProtocol::kTransportHeader: 	ParseTransportHeader(); break;
			case RTSPProtocol::kRangeHeader:		ParseRangeHeader(); 	break;
			default:	break;
		}
	}
	isStreamOK = parser.ExpectEOL();
	Assert(isStreamOK);
	return RTSPProtocol::kSuccessOK;
}

void RTSPRequest::ParseTransportHeader()
{
	StringParser theTransportParser(&fHeaders[RTSPProtocol::kTransportHeader]);
	
	//transport header from client: Transport: RTP/AVP;unicast;client_port=5000-5001\r\n
	//throw everything away until the first client port. This is not the most
	//kosher parsing practice, so change this eventually
	theTransportParser.ConsumeUntil(NULL, StringParser::sDigitMask);
	
	//store the two client ports both as integers and strings
	fClientPortA = (UInt16)theTransportParser.ConsumeInteger(this->GetQTSSParameter(qtssClientPortA));
	theTransportParser.ConsumeLength(NULL, 1);
	fClientPortB = (UInt16)theTransportParser.ConsumeInteger(this->GetQTSSParameter(qtssClientPortB));
}

void  RTSPRequest::ParseRangeHeader()
{
	StringParser theRangeParser(&fHeaders[RTSPProtocol::kRangeHeader]);
	theRangeParser.ConsumeLength(NULL, 4);//consume "npt="
	fStartTime = (Float64)theRangeParser.ConsumeFloat();
	//see if there is a stop time as well.
	if (theRangeParser.GetDataRemaining() > 1)
	{
		theRangeParser.ConsumeLength(NULL, 1);
		fStopTime = (Float64)theRangeParser.ConsumeFloat();
	}
}
