/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		OS.h

	Contains:  	OS utility functions. Memory allocation, time, etc.

	$Log: OS.h,v $
	Revision 1.4  1999/02/26 04:55:29  cl
	Fixed ObjectDeleter reversed assert conditional
	
	Revision 1.3  1999/02/25 06:41:06  cl
	Removed "assert" call and replaced with inline code due to include files recursion
	
	Revision 1.2  1999/02/19 23:05:31  ds
	Created
		

*/

#ifndef _OS_H_
#define _OS_H_

#include "MyAssert.h"
#include "StrPtrLen.h"
#include "QTSS.h"

#define MEMORY_DEBUGGING 0 //enable this to turn on really fancy debugging of memory leaks, etc...

#if MEMORY_DEBUGGING	//+ 6.18.99 rt enable VALIDATE_MEMORY_POOL to turn on debugging of pool corruption
	#define VALIDATE_MEMORY_POOL 0
#else
	#define VALIDATE_MEMORY_POOL 0
#endif

#if MEMORY_DEBUGGING
#include "OSQueue.h"
#include "OSMutex.h"
#endif

class OS
{
	public:
	
		//call this before calling anything else
		static void	Initialize();

		static SInt32 Min(SInt32 a, SInt32 b) 	{ if (a < b) return a; return b; }
		
		static SInt64 	Milliseconds();
		static SInt64	Microseconds();

		//because the OS doesn't seem to have this function
		static SInt64	HostToNetworkSInt64(SInt64 hostOrdered);
		
		static SInt64	ConvertToSecondsSince1900(SInt64 inMilliseconds)
							{ return ConvertMsecToFixed64Sec(sMsecSince1900 + inMilliseconds); }
		static SInt64	ConvertMsecToFixed64Sec(SInt64 inMilliseconds)
							{ return (SInt64)(.5 + inMilliseconds * 4294967.296L); }
		
		//Both these functions return QTSS_NoErr, QTSS_FileExists, or POSIX errorcode
		//Makes whatever directories in this path that don't exist yet 
		static QTSS_ErrorCode RecursiveMakeDir(char *inPath, mode_t mode);
		//Makes the directory at the end of this path
		static QTSS_ErrorCode MakeDir(char *inPath, mode_t mode);
		
#if MEMORY_DEBUGGING
		//If memory debugging is on, clients can get access to data structures that give
		//memory status.
		static OSQueue*	GetTagQueue() { return &sTagQueue; }
		static OSMutex* GetTagQueueMutex() { return &sMutex;	}
		static UInt32	GetAllocatedMemory() { return sAllocatedBytes; }

		static void*	DebugNew(size_t size, UInt32 tag, bool sizeCheck);
		static void		DebugDelete(void *mem);
		static bool		MemoryDebuggingTest();
		static void		ValidateMemoryQueue();

		struct TagElem
		{
			OSQueueElem elem;
			char tag[5];
			UInt32* tagPtr;
			UInt32 tagSize; //how big are objects of this type?
			UInt32 totMemory; //how much do they currently occupy
			UInt32 numObjects;//how many are there currently?
		};
#endif
		
		//When memory allocation fails, the server just exits. This sets the code
		//the server exits with
		static void SetMemoryError(SInt32 inErr) { sMemoryErr = inErr; }

		//Used by the new operator
		static void* New(size_t s, UInt32 tag, bool sizeCheck);
	
	private:
	
#if MEMORY_DEBUGGING
		struct MemoryDebugging
		{
			OSQueueElem elem;
			UInt32 tag;
			UInt32 size;
		};
		static OSQueue sMemoryQueue;
		static OSQueue sTagQueue;
		static UInt32  sAllocatedBytes;
		static OSMutex sMutex;
		
#endif
	
		static double sDivisor;
		static double sMicroDivisor;
		static SInt64 sMsecSince1900;
		static SInt64 sInitialMsec;
		static SInt32 sMemoryErr;
		static void SetDivisor();
};

template <class T>
class OSArrayObjectDeleter
{
	public:
		OSArrayObjectDeleter(T* victim) : fT(victim)  {}
		~OSArrayObjectDeleter() { delete [] fT; }
		
		//+ 7.22.99 was T* SetObject... but has no return or reason to return
		void SetObject(T* victim) 
		{ 
			//can't use a normal assert here because "Assert.h" eventually includes this file....
			#ifdef ASSERT
				//char s[65]; 
				if (fT != NULL) printf ("_Assert: OSObjectDeleter::SetObject() %s, %d\n", __FILE__, __LINE__); 
			#endif 

			fT = victim; 
		}
		T* GetObject() { return fT; }
		
		operator T*() { return fT; }
	
	private:
	
		T* fT;
};

typedef OSArrayObjectDeleter<char> OSCharArrayDeleter;


//because $%@#%#%%** GCC doesn't implement the placement new operator
inline void* operator new(size_t, void* ptr) { return ptr;}

//The server assumes that memory allocation
//succeeds. If it doesn't succeed, we should exit the process immediately

//Memory allocation. We define a special new here simply for debugging purposes.
//This New routine allows the server to easily track memory leaks.
//Pass in a tag so that this memory allocation can later be identified
void* operator new(size_t s, UInt32 tag);
void* operator new[](size_t s, UInt32 tag);
void operator delete(void* mem);
void operator delete[](void* mem);
//make sure that the normal new operator is also overridden in case some code still uses it.
//these versions also do the debugging stuff if debugging is turned on
void* operator new (size_t s);
void* operator new[](size_t s);
#endif
