/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */


#include <stdlib.h>
#include <string.h>
#include <stdio.h>

	
#include "plassert.h"

#ifdef __MACOS__
	#include <Events.h>
#else
	#include <sys/time.h>
#endif


#include "PlaylistPicker.h"

/*
	PlaylistPicker has 3 modes
		- sequential looping through the items in the play list(s)
		in the order they are entered.
		
		- above w/ looping
		
		- weighted picking "randomly" from weighted buckets
		

*/




PlaylistPicker::PlaylistPicker( UInt32 numBuckets, UInt32 numNoRepeats )
{
	// weighted random ctor
	
	mNumToPickFrom = 0;
	mBuckets = numBuckets;
	mIsSequentialPicker = false;
	mRecentMoviesListSize = numNoRepeats;
	
	
	mPickCounts = new long[numBuckets];
	UInt32 x;
	
	for ( x = 0; x < mBuckets; x++ )
	{	mPickCounts[x] = 0;
		mElementLists[x] = new PLDoubleLinkedList<SimplePlayListElement>;
		ASSERT( mElementLists[x] );
		
	}
	
	

	mUsedElements = new NoRepeat( numNoRepeats );

	ASSERT( mUsedElements );
}

PlaylistPicker::PlaylistPicker(bool doLoop)
{
	// sequential ctor
	
	mIsSequentialLooping = doLoop;
	
	mIsSequentialPicker = true;
	mWhichSequentialBucket = 0;
	mRecentMoviesListSize = 0;
	
	mNumToPickFrom = 0;
	mBuckets = 2;	// alternating used/remaining pick buckets

	
	mPickCounts = new long[mBuckets];
	
	
	UInt32 	bucketIndex;
	
	for ( bucketIndex = 0; bucketIndex < mBuckets; bucketIndex++ )
	{	
		mPickCounts[bucketIndex] = 0;
		mElementLists[bucketIndex] = new PLDoubleLinkedList<SimplePlayListElement>;
		ASSERT( mElementLists[bucketIndex] );
		
	}


	mUsedElements = NULL;

}

PlaylistPicker::~PlaylistPicker()
{
	UInt32		bucketIndex;
	
	delete mUsedElements;
	
	for ( bucketIndex = 0; bucketIndex < mBuckets; bucketIndex++ )
	{	
		delete mElementLists[bucketIndex] ;
		
	}
	
	delete [] mPickCounts;
}


UInt32 PlaylistPicker::Random()
{
	mLastSeed = 0;
	
	#if __MACOS__
	unsigned long t = TickCount();
	#else
	struct timeval tv;
	struct timezone tz;
	::gettimeofday(&tv, &tz);

	UInt64 t;
	t = tv.tv_sec * 1000;		// sec -> msec
	t += tv.tv_usec / 1000;	// usec -> msec
	
	#endif
	
	//GetDateTime( &t );
	
	if ( mLastSeed != t )
	{	::srand((unsigned int)t);
		mLastSeed = t;
	}
	
	//while ( t == TickCount() );
	
	return ::rand();
}


char* PlaylistPicker::PickOne()
{
	
	char*	foundName = NULL;	// pointer to name of pick we find, caller deletes.
	
	
	if ( mIsSequentialPicker )
	{	
		
		if ( mElementLists[mWhichSequentialBucket]->GetNumElements() == 0 && mIsSequentialLooping )
		{	// ran out of items switch to other list.
			if ( mWhichSequentialBucket == 0 )
				mWhichSequentialBucket = 1;
			else
				mWhichSequentialBucket = 0;
	
		}
			
		if ( mElementLists[mWhichSequentialBucket]->GetNumElements() > 0 )
		{
			PLDoubleLinkedListNode<SimplePlayListElement>*	node;
			
			
			node = mElementLists[mWhichSequentialBucket]->GetFirst();
			
			ASSERT( node );
			
			int	nameLen = ::strlen( node->mElement->mElementName );
			
			foundName = new char[ nameLen +1 ];
			
			ASSERT( foundName );
			
			if ( foundName )
			{	
				int	usedBucketIndex;
				
				::strcpy( foundName, node->mElement->mElementName );
			
				// take him out of the bucket since he's now in play
				mElementLists[mWhichSequentialBucket]->RemoveElement( node );
				
				
				if ( mWhichSequentialBucket == 0 )
					usedBucketIndex = 1;
				else
					usedBucketIndex = 0;
					
				mElementLists[usedBucketIndex]->AddElementToTail( node );
			
			}
		
		}


	}
	else
	{
		SInt32		bucketIndex;
		UInt32		minimumBucket;
		UInt32		avaiableToPick;
		UInt32		theOneToPick;
		SInt32		topBucket;
		
		
		// find the highest bucket with some elements.
		bucketIndex = this->GetNumBuckets() - 1;
		
		while ( bucketIndex >= 0 &&  mElementLists[bucketIndex]->GetNumElements() == 0 )
		{
			bucketIndex--;
		}
		
		
		// adjust to 1 based  so we can use MOD
		topBucket = bucketIndex + 1;
		
		//printf( "topBucket %li \n", topBucket );
		 
		
		minimumBucket =  this->Random() % topBucket; // find our minimum bucket
		
		
		//printf( "minimumBucket %li \n", minimumBucket );
		
		// pick randomly from the movies in this and higher buckets
		// sum the available elements, then pick randomly from them.
		
		avaiableToPick = 0;
		
		bucketIndex = minimumBucket;
		
		while ( bucketIndex < topBucket )
		{
			avaiableToPick += mElementLists[bucketIndex]->GetNumElements();
				
			bucketIndex++;
		}
		
		//printf( "avaiableToPick %li \n", avaiableToPick );
		
		// was anyone left??
		
		if ( avaiableToPick )
		{

			theOneToPick = this->Random() % avaiableToPick;
			
			//printf( "theOneToPick %li \n", theOneToPick );

			// now walk through the lists unitl we get to the list
			// that contains our pick, then pick that one.
			bucketIndex = minimumBucket;
			
			while ( bucketIndex < topBucket && foundName == NULL )
			{
			
				//printf( "theOneToPick %li, index %li numelements %li\n", theOneToPick , bucketIndex, mElementLists[bucketIndex]->GetNumElements());
				
				if ( theOneToPick >= mElementLists[bucketIndex]->GetNumElements() )
					theOneToPick -= mElementLists[bucketIndex]->GetNumElements();
				else
				{	//printf( "will pick theOneToPick %li, index %li \n", theOneToPick, bucketIndex);
					foundName = this->PickFromList( mElementLists[bucketIndex], theOneToPick );
					if ( foundName )
						mPickCounts[bucketIndex]++;
				}

				bucketIndex++;
			}
		
			// we messed up if we don't have a name at this point.
			ASSERT( foundName );
		}
	}
	
	
	return foundName;

}


char*	PlaylistPicker::PickFromList( PLDoubleLinkedList<SimplePlayListElement>* list, UInt32 elementIndex )
{
	PLDoubleLinkedListNode<SimplePlayListElement>*	plNode;
	char*		foundName = NULL;
	
	
	plNode = list->GetNthNode( elementIndex );

	if ( plNode )
	{
		int	nameLen = ::strlen(plNode->mElement->mElementName );
		
		foundName = new char[ nameLen +1 ];
		
		ASSERT( foundName );
		
		if ( foundName )
		{	
			::strcpy( foundName, plNode->mElement->mElementName );
		
			// take him out of the bucket since he's now in play
			list->RemoveElement( plNode );
			
			mNumToPickFrom--;
		
			// add him to our used list, and possibly
			// get an older one to put back into play
			PLDoubleLinkedListNode<SimplePlayListElement>* recycleNode = mUsedElements->AddToList( plNode );
			
			// if we got an old one to recycle, do so.
			if ( recycleNode )
				this->AddNode( recycleNode );
		}
	}

	return foundName;

}

bool PlaylistPicker::AddToList( const char* name, int weight )
{
	bool											addedSuccesfully;
	PLDoubleLinkedListNode<SimplePlayListElement>*	node;
	SimplePlayListElement*							element;
	

	node = NULL;
	addedSuccesfully = false;
	element = new SimplePlayListElement(name);
	
	ASSERT( element );
	
	
	if ( element )
	{	element->mElementWeight = weight;
		node = new PLDoubleLinkedListNode<SimplePlayListElement>(element);

		ASSERT( node );
	}
	
	if ( node )
		addedSuccesfully = AddNode(node);
	
		
	return addedSuccesfully;
}

bool PlaylistPicker::AddNode(  PLDoubleLinkedListNode<SimplePlayListElement>* node )
{	
	bool	addSucceeded = false;

	
	ASSERT( node );
	ASSERT( node->mElement );
	
	
	if ( mIsSequentialPicker )	// make picks in sequential order, not weighted random
	{
		// add all to bucket 0
		mElementLists[0]->AddElementToTail( node );
		
		addSucceeded = true;
		mNumToPickFrom++;
	
	}
	else
	{	
		int		weight;
		
		weight = node->mElement->mElementWeight;
	
		// weights are 1 based, correct to zero based for use as array myIndex
		weight--;
		
		ASSERT( weight >= 0 );
		
		ASSERT(  (UInt32)weight < mBuckets );
	
		if ( (UInt32)weight < mBuckets )
		{
			// the elements weighting defines the list it is in.
			mElementLists[weight]->AddElement( node );
			
			addSucceeded = true;
			mNumToPickFrom++;
		
		}
	}
	
	return addSucceeded;

}