/*
 * @(#)xoct.c
 *
 * Copyright 1993 - 2010  David A. Bagley, bagleyd@tux.org
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/* Driver file for Oct */

#ifndef WINVER
#include "version.h"
static const char aboutHelp[] = {
"Oct Version " VERSION "\n"
"Send bugs (reports or fixes) to the author: "
"David Bagley <bagleyd@tux.org>\n"
"The latest version is at: "
"http://www.tux.org/~bagleyd/puzzles.html"
};

static const char optionsHelp[] = {
"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n"
"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
"[-{foreground|fg} {color}] [-{background|bg} {color}]\n"
"[-face{0|1|2|3|4|5|6|7} {color}] [-{border|bd} {color}]\n"
"[-delay msecs] [-[no]sound] [-moveSound {filename}]\n"
"[-{font|fn} {fontname}] [-{size {int} | sticky}]\n"
"[-{mode {int} | both}] [-[no]orient] [-[no]practice]\n"
"[-userName {string}] [-scoreFile {filename}] [-scores]\n"
"[-version]"
};
#endif

#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
"The original puzzle has 9 triangles per face (size = 3) "
"and has period 4 turning (i.e. the face or points turn\n"
"in 90 degree intervals).  The puzzle was designed by Uwe "
"Meffert and called the Magic Octahedron (or Star\n"
"Puzzler).  The puzzle was not widely distributed but not "
"exactly rare.  This puzzle has some analogies to the\n"
"Rubik's Cube and the solving techniques are the same to "
"that of the Pyraminx.  Christoph's Magic Jewel is\n"
"similar except there are no trivial corners to solve.  "
"This has 2^22*12! or 2,009,078,326,886,400 different\n"
"combinations.\n"
"Uwe Meffert also noticed that there could be an "
"alternate twisting for the octahedron where it has\n"
"period 3 turning (i.e. faces turn with 120 degree "
"intervals).\n"
"One is able to simulate a Trajber's Octahedron (period "
"3 turning and sticky mode).  Also one is able to\n"
"simulate one with variant turning (period 4 turning and "
"sticky mode)."
};

static const char featuresHelp[] = {
"Press \"mouse-left\" button to move a piece.  Release "
"\"mouse-left\" button on a piece on the same face and\n"
"in the same row (but not an adjacent piece or the move "
"is ambiguous).  The pieces will then turn towards\n"
"where the mouse button was released.\n"
"\n"
"Click \"mouse-center\" button, or press \"P\" or \"p\" "
"keys, to toggle the practice mode (in practice mode the\n"
"record should say \"practice\").  This is good for learning "
"moves and experimenting.\n"
"\n"
"Click \"mouse-right\" button, or press \"Z\" or \"z\" "
"keys, to randomize the puzzle (this must be done first\n"
"to set a new record).\n"
"\n"
"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
"\n"
"Press \"W\" or \"w\" keys to save (write) a puzzle.\n"
"\n"
"Press \"U\" or \"u\" keys to undo a move.\n"
"\n"
"Press \"R\" or \"r\" keys to redo a move.\n"
"\n"
"Press \"C\" or \"c\" keys to clear the puzzle.\n"
"\n"
"Press \"S\" or \"s\" keys reserved for the auto-solver "
"(not implemented).\n"
"\n"
"Press \"O\" or \"o\" keys to toggle the orient mode.  One "
"has to orient the faces in orient mode, besides\n"
"getting all the faces to be the same color.  To do this "
"one has to get the lines to be oriented in\n"
"the same direction, this only matters with center "
"pieces, if at all (i.e. those that are not on a\n"
"corner or edge).  This does add complexity so there are "
"2 sets of records.\n"
"\n"
"Press \"3\", \"4\", \"B\", or \"b\" keys (not the keypad 3, 4) "
"to change modes to Period 3, Period 4, or Both.\n"
"\n"
"Press \"Y\" or \"y\" keys to toggle sticky mode (increase/decrease "
"is disabled here if sticky mode is on).\n"
"\"Sticky\" and \"Period 3\" turning allows only the faces to "
"turn.  It is as if the middle cut of the three cuts\n"
"did not exist.\n"
"\"Sticky\" and \"Period 4\" turning allows only the points to "
"turn, it is as if the middle cut of the three cuts\n"
"did not exist.\n"
"Beware, the \"Sticky\" mode is a hack and much could be done "
"to improve its look.\n"
"\n"
"Press \"I\" or \"i\" keys to increase the number of pieces.\n"
"\n"
"Press \"D\" or \"d\" keys to decrease the number of pieces.\n"
"\n"
"Press \">\" or \".\" keys to speed up the movement of pieces.\n"
"\n"
"Press \"<\" or \",\" keys to slow down the movement of pieces.\n"
"\n"
"Press \"@\" key to toggle the sound.\n"
"\n"
"Press \"Esc\" key to hide program.\n"
"\n"
"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
"\n"
"Use the key pad or arrow keys to move without the mouse.\n"
"Key pad is defined for Oct as:\n"
"  /    Counterclockwise\n"
"7 8 9  Up, Upper Right\n"
"  ^\n"
"4<5>6  Left, Clockwise, Right\n"
"  v\n"
"1 2 3  Lower Left, Down\n"
"\n"
"Use the shift keys to access \"Period 4\" turns from "
"\"Both\" mode, otherwise it assumes \"Period 3\" turning.\n"
"Points turn in \"Period 4\" and faces turn in \"Period 3\".\n"
"Use the control key and the left mouse button, keypad, or "
"arrow keys to move the whole octahedron.  This is not\n"
"recorded as a turn."
};

static const char referencesHelp[] = {
"Beyond Rubik's Cube: spheres, pyramids, dodecahedrons and "
"God knows what else by Douglas R. Hofstadter,\n"
"Scientific American, July 1982, pp 16-31.\n"
"Magic Cubes 1996 Catalog of Dr. Christoph Bandelow."
};
#endif

static const char solveHelp[] = {
"Auto-solver: sorry, only implemented for size < 4 where "
"period = 4."
};

#include "file.h"
#ifdef WINVER
#include "OctP.h"
#define TITLE "woct"

static OctRec widget;

#ifndef SCOREPATH
#ifdef UNIXDELIM
#define SCOREPATH "c:/WINDOWS"
#else
#define SCOREPATH "c:\\WINDOWS"
#endif
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(widget.core.hWnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->oct.started = b
#else
#include "xwin.h"
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#ifdef MOUSEBITMAPS
#include "pixmaps/mouse-l.xbm"
#include "pixmaps/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) printState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtVaSetValues(w, XtNstart, b, NULL)
#include "Oct.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#ifdef CONSTPIXMAPS
#include "oct.t.xpm"
#include "oct.p.xpm"
#include "oct.s.xpm"
#include "oct.m.xpm"
#include "oct.l.xpm"
#include "oct.xpm"
#else
#include "pixmaps/oct.t.xpm"
#include "pixmaps/oct.p.xpm"
#include "pixmaps/oct.s.xpm"
#include "pixmaps/oct.m.xpm"
#include "pixmaps/oct.l.xpm"
#include "pixmaps/oct.xpm"
#endif
#define RESIZE_XPM(s) ((char **) (((s)<=32)?\
(((s)<=22)?(((s)<=16)?oct_t_xpm:oct_p_xpm):\
(((s)<=24)?oct_s_xpm:oct_m_xpm)):\
(((s)<=48)?oct_l_xpm:oct_xpm)))
#endif
#include "pixmaps/oct.xbm"
#define DEFINE_XBM (char *) oct_bits, oct_width, oct_height
#ifndef SCOREPATH
#ifdef VMS
#define SCOREPATH "SYS$LOGIN:"
#else
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif
#endif

#ifndef SCOREFILE
#define SCOREFILE "oct.scores"
#endif

#define MAX_FACETS 6
#define NEVER (-1)
#define FILE_NAME_LENGTH 1024
#define USER_NAME_LENGTH 120
#define MESSAGE_LENGTH (USER_NAME_LENGTH+64)
#define TITLE_LENGTH 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USER_NAME_LENGTH];
} PuzzleRecord;

static PuzzleRecord puzzleRecord[MAX_MODES][2][MAX_FACETS - MIN_FACETS + 2];
static int movesDsp = 0;
static char messageDsp[MESSAGE_LENGTH] = "Welcome";
static char recordDsp[MESSAGE_LENGTH] = "NOT RECORDED";
static int oldSize;

#ifdef HAVE_MOTIF
#define MIN_SPEED 1
#define MAX_SPEED 50
#define MULTSPEED 20
static Widget movesText, recordText, message;
static Widget modes[MAX_MODES], orientizeSwitch;
static Widget stickySwitch, practiceSwitch, facetSlider, speedSlider;
static char buff[21];
static const char *modeString[] =
{
	"Period 3", "Period 4", "Both"
};
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
static Arg arg[3];
#else
static char titleDsp[TITLE_LENGTH] = "";
#endif
static char scoreFileName[FILE_NAME_LENGTH] = SCOREFILE;
static char fileName[FILE_NAME_LENGTH];
static Boolean randomized = False;
#ifdef WINVER
#define PROGRAM_NAME_LENGTH 80
static char progDsp[PROGRAM_NAME_LENGTH] = TITLE;
static char userNameDsp[USER_NAME_LENGTH] = "Guest";
#else
static Pixmap pixmap = None;
static Widget topLevel, puzzle;
static char *progDsp;
static char userNameDsp[USER_NAME_LENGTH] = "";

#ifdef HAVE_MOTIF
static void
printState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("printState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtVaSetValues(w, XmNlabelString, xmstr, NULL);
}
#endif

static void
printRecords(void)
{
	int i, mode, orient;

	(void) printf("                 OCT  HALL OF FAME\n\n");
	(void) printf("MODE ORIENT FACET USER            MOVES\n");
	for (mode = 0; mode < MAX_MODES; mode++)
		for (orient = 0; orient < 2; orient++)
			for (i = 0; i <= MAX_FACETS - MIN_FACETS + 1; i++) {
				if (puzzleRecord[mode][orient][i].score > 0)
					(void) printf("%4d%7d%6d %-12s%9d\n",
						mode + 3, orient, i + 1,
						puzzleRecord[mode][orient][i].name,
						puzzleRecord[mode][orient][i].score);
			}
}
#endif

static void
initRecords(void)
{
	int i, mode, orient;

	for (mode = 0; mode < MAX_MODES; mode++)
		for (orient = 0; orient < 2; orient++)
			for (i = 0; i <= MAX_FACETS - MIN_FACETS + 1; i++) {
				puzzleRecord[mode][orient][i].score = NEVER;
				(void) strncpy(puzzleRecord[mode][orient][i].name,
					NOACCESS, USER_NAME_LENGTH);
			}
}

static void
readRecords(void)
{
	FILE *fp;
	int i, n, mode, orient;
	char userName[USER_NAME_LENGTH];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname;

	stringCat(&buf1, CURRENTDELIM, scoreFileName);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	(void) strncpy(fileName, lname, USER_NAME_LENGTH);
	if ((fp = fopen(fileName, "r")) == NULL) {
		(void) strncpy(fileName, fname, USER_NAME_LENGTH);
		/* Try installed directory. */
		if ((fp = fopen(fileName, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (mode = 0; mode < MAX_MODES; mode++)
		for (orient = 0; orient < 2; orient++)
			for (i = 0; i <= MAX_FACETS - MIN_FACETS + 1; i++) {
				(void) fscanf(fp, "%d %s\n", &n, userName);
				if (n <= puzzleRecord[mode][orient][i].score ||
						puzzleRecord[mode][orient][i].score <= NEVER) {
					puzzleRecord[mode][orient][i].score = n;
					(void) strncpy(puzzleRecord[mode][orient][i].name,
						userName, USER_NAME_LENGTH);
				}
			}
	(void) fclose(fp);
}

static void
writeRecords(void)
{
	FILE *fp;
	int i, mode, orient;
	char *buf1 = NULL;

	if ((fp = fopen(fileName, "w")) == NULL) {
		stringCat(&buf1, "Can not write to ", fileName);
		PRINT_MESSAGE(buf1);
		free(buf1);
		return;
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockFile[FILE_NAME_LENGTH];

		(void) strncpy(lockFile, fileName, FILE_NAME_LENGTH - 6);
		lockFile[FILE_NAME_LENGTH - 6] = '\0';
		(void) strcat(lockFile, ".lock");
		while (((lfd = open(lockFile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
				"Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
				"Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (mode = 0; mode < MAX_MODES; mode++) {
			for (orient = 0; orient < 2; orient++) {
				for (i = 0; i <= MAX_FACETS - MIN_FACETS + 1; i++)
					(void) fprintf(fp, "%d %s\n",
						puzzleRecord[mode][orient][i].score, puzzleRecord[mode][orient][i].name);
				(void) fprintf(fp, "\n");
			}
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockFile);
#endif
		(void) fclose(fp);
	}
}

static void
printRecord(int size, int mode,
		Boolean orient, Boolean sticky, Boolean practice)
{
	int i = mode - PERIOD3;
	int j = (orient) ? 1 : 0;
	int k = (sticky) ? MAX_FACETS - MIN_FACETS + 1 : size - MIN_FACETS;

	if (practice) {
		(void) strncpy(recordDsp, "practice", MESSAGE_LENGTH);
	} else if (!sticky && size > MAX_FACETS) {
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGE_LENGTH);
	} else if (puzzleRecord[i][j][k].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			puzzleRecord[i][j][k].score, puzzleRecord[i][j][k].name);
	}
#ifdef HAVE_MOTIF
	printState(recordText, recordDsp);
#endif
}

static void
printStatus(char *msg, int nMoves
#ifndef HAVE_MOTIF
		, int mode, int size, Boolean sticky
#endif
		)
{
#ifdef HAVE_MOTIF
	printState(message, msg);
	(void) sprintf(buff, "%d", nMoves);
	printState(movesText, buff);
#else
	char ss[10], mb[10];

	if (sticky)
		(void) strcpy(ss, "sticky");
	else
		(void) sprintf(ss, "%d", size);
	if (mode == BOTH)
		(void) strcpy(mb, "both");
	else
		(void) sprintf(mb, "%d", mode);
	(void) sprintf(titleDsp, "%s.%s: %s@ (%d/%s) - %s",
		progDsp, mb, ss, nMoves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(widget.core.hWnd, (LPSTR) titleDsp);
#else
	XtVaSetValues(XtParent(puzzle), XtNtitle, titleDsp, NULL);
#endif
#endif
}

static Boolean
handleSolved(int counter, int size, int mode, Boolean orient, Boolean sticky)
{
	int i = mode - PERIOD3;
	int j = (orient) ? 1 : 0;
	int k = (sticky) ? MAX_FACETS - MIN_FACETS + 1 : size - MIN_FACETS;

	if ((sticky || size <= MAX_FACETS) &&
			(counter < puzzleRecord[i][j][k].score ||
			puzzleRecord[i][j][k].score <= NEVER)) {
		readRecords();	/* Maybe its been updated by another */
		puzzleRecord[i][j][k].score = counter;
		(void) strncpy(puzzleRecord[i][j][k].name, userNameDsp,
			USER_NAME_LENGTH);
		if ((size < 2 && mode != PERIOD4) ||
				(size < 4 && mode == PERIOD4) || (orient &&
				(counter < puzzleRecord[i][!j][k].score ||
				puzzleRecord[i][!j][k].score <= NEVER))) {
			puzzleRecord[i][!j][k].score = counter;
			(void) strncpy(puzzleRecord[i][!j][k].name, userNameDsp,
				USER_NAME_LENGTH);
		}
		writeRecords();
		printRecord(size, mode, orient, sticky, False);
		return True;
	}
	return False;
}

static void
initialize(
#ifdef WINVER
OctWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int size, mode;
	Boolean orient, sticky, practice;
	char *userName, *scoreFile;

#ifdef WINVER
	initializePuzzle(w, brush);

	size = w->oct.size;
	mode = w->oct.mode;
	orient = w->oct.orient;
	sticky = w->oct.sticky;
	practice = w->oct.practice;
	userName = w->oct.userName;
	scoreFile = w->oct.scoreFile;
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILE_NAME_LENGTH);
#else
	Boolean scoreOnly, versionOnly;

	XtVaGetValues(w,
		XtNuserName, &userName,
		XtNscoreFile, &scoreFile,
		XtNsize, &size,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNsticky, &sticky,
		XtNpractice, &practice,
		XtNscoreOnly, &scoreOnly,
		XtNversionOnly, &versionOnly, NULL);
	if (versionOnly) {
		(void) printf("%s\n", aboutHelp);
		exit(0);
	}
	if (strcmp(scoreFile, ""))
		(void) strncpy(scoreFileName, scoreFile, FILE_NAME_LENGTH);
	if (scoreOnly) {
		initRecords();
		readRecords();
		printRecords();
		exit(0);
	}
#ifdef HAVE_MOTIF
	if (size > MAX_FACETS)
		XtVaSetValues(facetSlider, XmNmaximum, size, NULL);
	XmScaleSetValue(facetSlider, size);
	XmToggleButtonSetState(modes[mode - PERIOD3], True, False);
	XmToggleButtonSetState(orientizeSwitch, orient, True);
	XmToggleButtonSetState(stickySwitch, sticky, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
	{
		int delay;

		XtVaGetValues(w, XtNdelay, &delay, NULL);
		XmScaleSetValue(speedSlider,
			MAX_SPEED + MIN_SPEED - delay / MULTSPEED - 1);
	}
#endif
#endif
	SET_STARTED(w, False);
	initRecords();
	readRecords();
#ifndef WINVER
	(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH);
#endif
	if (!strcmp(userName, "") || !strcmp(userName, "(null)") ||
			!strcmp(userName, NOACCESS) ||
			!strcmp(userName, NOBODY)) {
#ifdef WINVER
		(void) strncpy(userNameDsp, userName, USER_NAME_LENGTH);
#else
		char *login = getlogin();

		if (login == NULL) {
			(void) strcpy(userNameDsp, "");
		} else {
			(void) sprintf(userNameDsp, "%s", login);
		}
		if (!strcmp(userNameDsp, "") ||
				!strcmp(userNameDsp, "(null)") ||
				!strcmp(userNameDsp, NOACCESS) ||
				!strcmp(userNameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(userNameDsp, "%s", "guest");
#endif
	}
	printRecord(size, mode, orient, sticky, practice);
	oldSize = size;
	printStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, mode, size, sticky
#endif
		);
}

#ifdef WINVER
void
setPuzzle(OctWidget w, int reason)
#else
static void
puzzleListener(Widget w, caddr_t clientData,
		octCallbackStruct *callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int size, mode;
	Boolean orient, sticky, practice, start, cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	size = w->oct.size;
	mode = w->oct.mode;
	orient = w->oct.orient;
	sticky = w->oct.sticky;
	practice = w->oct.practice;
	cheat = w->oct.cheat;
	start = w->oct.started;
#else
	XtVaGetValues(w,
		XtNsize, &size,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNsticky, &sticky,
		XtNpractice, &practice,
		XtNstart, &start,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
	case ACTION_HIDE:
#ifdef WINVER
		ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
		(void) XIconifyWindow(XtDisplay(topLevel),
			XtWindow(topLevel),
			XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
		break;
#ifndef WINVER
	case ACTION_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(practiceDialog);
#else
		XtVaSetValues(puzzle, XtNmenu, ACTION_PRACTICE, NULL);
#endif
		break;
	case ACTION_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
		XtManageChild(randomizeDialog);
#else
		XtVaSetValues(puzzle, XtNmenu, ACTION_RANDOMIZE, NULL);
#endif
		break;
#endif
	case ACTION_SOLVE_MESSAGE:
#ifdef WINVER
		(void) MessageBox(w->core.hWnd, solveHelp,
			"Auto-solve", MB_OK);
#else
#ifdef HAVE_MOTIF
		XtManageChild(solveDialog);
#else
		(void) strncpy(messageDsp, solveHelp, MESSAGE_LENGTH);
#endif
#endif
		break;
	case ACTION_RESTORE:
		if (practice) {
			(void) strncpy(recordDsp, "practice",
				MESSAGE_LENGTH);
#ifdef HAVE_MOTIF
			printState(recordText, recordDsp);
#endif
		}
		movesDsp = 0;
		randomized = False;
		break;
	case ACTION_RESET:
		movesDsp = 0;
		randomized = False;
		break;
	case ACTION_AMBIGUOUS:
		(void) strncpy(messageDsp, "Ambiguous move",
			MESSAGE_LENGTH);
		break;
	case ACTION_ILLEGAL:
		if (practice || randomized)
			(void) strncpy(messageDsp, "Illegal move",
				MESSAGE_LENGTH);
		else
			(void) strncpy(messageDsp,
				"Randomize to start", MESSAGE_LENGTH);
		break;
	case ACTION_MOVED:
		movesDsp++;
		SET_STARTED(w, True);
		break;
	case ACTION_CONTROL:
		return;
	case ACTION_SOLVED:
		if (practice)
			movesDsp = 0;
		else if (cheat)
			(void) sprintf(messageDsp,
				"No cheating %s!!", userNameDsp);
		else if (handleSolved(movesDsp, size, mode, orient, sticky))
			(void) sprintf(messageDsp,
				"Congratulations %s!!", userNameDsp);
		else
			(void) strncpy(messageDsp, "Solved!",
				MESSAGE_LENGTH);
		SET_STARTED(w, False);
		randomized = False;
		break;
	case ACTION_COMPUTED:
		SET_STARTED(w, False);
		break;
	case ACTION_PRACTICE:
		movesDsp = 0;
		randomized = False;
		practice = !practice;
		if (!practice)
			(void) strncpy(messageDsp, "Randomize to start",
				MESSAGE_LENGTH);
		printRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
		w->oct.practice = practice;
		w->oct.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, practice,
			XtNstart, False, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
		break;
	case ACTION_RANDOMIZE:
		movesDsp = 0;
		randomized = True;
#ifdef WINVER
		w->oct.practice = False;
		w->oct.started = False;
#else
		XtVaSetValues(w,
			XtNpractice, False,
			XtNstart, False, NULL);
#endif
		break;
	case ACTION_INCREMENT:
		if (!sticky) {
			movesDsp = 0;
			size++;
			oldSize = size;
			printRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
			w->oct.size = size;
#else
			XtVaSetValues(w, XtNsize, size, NULL);
#ifdef HAVE_MOTIF
			if (size > MAX_FACETS)
				XtVaSetValues(facetSlider, XmNmaximum, size, NULL);
			XmScaleSetValue(facetSlider, size);
#endif
#endif
		}
		break;
	case ACTION_DECREMENT:
		if (!sticky) {
			movesDsp = 0;
			size--;
			oldSize = size;
			printRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
			w->oct.size = size;
#else
			XtVaSetValues(w, XtNsize, size, NULL);
#ifdef HAVE_MOTIF
			XmScaleSetValue(facetSlider, size);
			if (size >= MAX_FACETS)
				XtVaSetValues(facetSlider, XmNmaximum, size, NULL);
#endif
#endif
		}
		break;
	case ACTION_ORIENTIZE:
		movesDsp = 0;
		orient = !orient;
		printRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
		w->oct.orient = orient;
#else
		XtVaSetValues(w, XtNorient, orient, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(orientizeSwitch, orient, True);
#endif
#endif
		break;
	case ACTION_PERIOD3:
	case ACTION_PERIOD4:
	case ACTION_BOTH:
		movesDsp = 0;
		mode = reason - ACTION_PERIOD3 + PERIOD3;
		printRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
		w->oct.mode = mode;
#else
		XtVaSetValues(w, XtNmode, mode, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(modes[mode - PERIOD3], True, True);
#endif
#endif
		break;
	case ACTION_STICKY:
		movesDsp = 0;
		sticky = !sticky;
		if (sticky)
			size = 4;
		else
			size = oldSize;
		printRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
		w->oct.sticky = sticky;
		w->oct.size = size;
#else
		XtVaSetValues(w,
			XtNsticky, sticky,
			XtNsize, size, NULL);
#ifdef HAVE_MOTIF
		XmToggleButtonSetState(stickySwitch, sticky, True);
		XmScaleSetValue(facetSlider, size);
#endif
#endif
		break;
	case ACTION_UNDO:
		movesDsp--;
		SET_STARTED(w, True);
		break;
	case ACTION_REDO:
		movesDsp++;
		SET_STARTED(w, True);
		break;
#ifdef HAVE_MOTIF
	case ACTION_SPEED:
		{
			int oldDelay, delay;

			XtVaGetValues(puzzle, XtNdelay, &delay, NULL);
			oldDelay = delay;
			if (delay > MAX_SPEED - MIN_SPEED)
				delay = MAX_SPEED - MIN_SPEED;
			if (delay < 0)
				delay = 0;
			if (delay != oldDelay) {
				XtVaSetValues(w,
					XtNdelay, delay, NULL);
			}
			XmScaleSetValue(speedSlider,
				MAX_SPEED + MIN_SPEED - delay - 1);
		}
		return;
#endif
	}
	printStatus(messageDsp, movesDsp
#ifndef HAVE_MOTIF
		, mode, size, sticky
#endif
		);
}

#ifdef WINVER
static LRESULT CALLBACK
about(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	if (message == WM_COMMAND && LOWORD(wParam) == IDOK) {
		(void) EndDialog(hDlg, TRUE);
		return TRUE;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;
	int shift = 0;

	widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.oct.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			enterPuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.oct.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_BRUSH));
			leavePuzzle(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
	case WM_CREATE:
		initialize(&widget, brush);
		break;
	case WM_DESTROY:
		destroyPuzzle(brush);
		break;
	case WM_SIZE:
		resizePuzzle(&widget);
		(void) InvalidateRect(hWnd, NULL, TRUE);
		break;
	case WM_PAINT:
		widget.core.hDC = BeginPaint(hWnd, &paint);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		exposePuzzle(&widget);
		(void) EndPaint(hWnd, &paint);
		break;
	case WM_RBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		randomizePuzzle(&widget);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONDOWN:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		selectPuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
	case WM_LBUTTONUP:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		releasePuzzle(&widget, LOWORD(lParam), HIWORD(lParam),
			((GetKeyState(VK_SHIFT) >> 1) ||
			(GetKeyState(VK_CAPITAL) & 1)),
			(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
	case WM_MOUSEWHEEL:
		widget.core.hDC = GetDC(hWnd);
		(void) SelectObject(widget.core.hDC,
			GetStockObject(NULL_PEN));
		{
			int zDelta = ((short) HIWORD(wParam));
			POINT cursor, origin;

			origin.x = 0, origin.y = 0;
			ClientToScreen(hWnd, &origin);
			(void) GetCursorPos(&cursor);
			if (zDelta > (WHEEL_DELTA >> 1)) {
				movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					TOP,
					((GetKeyState(VK_SHIFT) >> 1) ||
					(GetKeyState(VK_CAPITAL) & 1)),
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			} else if (zDelta < -(WHEEL_DELTA >> 1)) {
				movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					BOTTOM,
					((GetKeyState(VK_SHIFT) >> 1) ||
					(GetKeyState(VK_CAPITAL) & 1)),
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			}
		}
		(void) ReleaseDC(hWnd, widget.core.hDC);
		break;
#endif
	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case ACTION_GET:
			getPuzzle(&widget);
			resizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_WRITE:
			writePuzzle(&widget);
			break;
		case ACTION_EXIT:
			destroyPuzzle(brush);
			break;
		case ACTION_HIDE:
			hidePuzzle(&widget);
			break;
		case ACTION_UNDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			undoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_REDO:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			redoPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_CLEAR:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			clearPuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_RANDOMIZE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			randomizePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_PRACTICE:
			practicePuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_SOLVE:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC,
				GetStockObject(NULL_PEN));
			solvePuzzle(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case ACTION_ORIENTIZE:
			orientizePuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_STICKY:
			(void) stickyModePuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_SHIFT_TOP:
		case ACTION_SHIFT_TR:
		case ACTION_SHIFT_RIGHT:
		case ACTION_SHIFT_BR:
		case ACTION_SHIFT_BOTTOM:
		case ACTION_SHIFT_BL:
		case ACTION_SHIFT_LEFT:
		case ACTION_SHIFT_TL:
		case ACTION_SHIFT_CW:
		case ACTION_SHIFT_CCW:
			shift = 1;
		case ACTION_TOP:
		case ACTION_TR:
		case ACTION_RIGHT:
		case ACTION_BR:
		case ACTION_BOTTOM:
		case ACTION_BL:
		case ACTION_LEFT:
		case ACTION_TL:
		case ACTION_CW:
		case ACTION_CCW:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				(void) movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					(int) LOWORD(wParam) -
					((shift) ? ACTION_SHIFT_TOP : ACTION_TOP),
					shift, FALSE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_SHIFT_CONTROL_TOP:
		case ACTION_SHIFT_CONTROL_TR:
		case ACTION_SHIFT_CONTROL_RIGHT:
		case ACTION_SHIFT_CONTROL_BR:
		case ACTION_SHIFT_CONTROL_BOTTOM:
		case ACTION_SHIFT_CONTROL_BL:
		case ACTION_SHIFT_CONTROL_LEFT:
		case ACTION_SHIFT_CONTROL_TL:
		case ACTION_SHIFT_CONTROL_CW:
		case ACTION_SHIFT_CONTROL_CCW:
			shift = 1;
		case ACTION_CONTROL_TOP:
		case ACTION_CONTROL_TR:
		case ACTION_CONTROL_RIGHT:
		case ACTION_CONTROL_BR:
		case ACTION_CONTROL_BOTTOM:
		case ACTION_CONTROL_BL:
		case ACTION_CONTROL_LEFT:
		case ACTION_CONTROL_TL:
		case ACTION_CONTROL_CW:
		case ACTION_CONTROL_CCW:
			{
				POINT cursor, origin;

				widget.core.hDC = GetDC(hWnd);
				(void) SelectObject(widget.core.hDC,
					GetStockObject(NULL_PEN));
				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				(void) movePuzzleInput(&widget,
					cursor.x - origin.x,
					cursor.y - origin.y,
					(int) LOWORD(wParam) -
					((shift) ? ACTION_SHIFT_CONTROL_TOP :
					ACTION_CONTROL_TOP),
					shift, TRUE);
				(void) ReleaseDC(hWnd, widget.core.hDC);
			}
			break;
		case ACTION_INCREMENT:
			incrementPuzzle(&widget);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_DECREMENT:
			if (decrementPuzzle(&widget)) {
				sizePuzzle(&widget);
				(void) InvalidateRect(hWnd, NULL, TRUE);
			}
			break;
		case ACTION_PERIOD3:
		case ACTION_PERIOD4:
		case ACTION_BOTH:
			periodModePuzzle(&widget,
				(int) LOWORD(wParam) - ACTION_PERIOD3);
			sizePuzzle(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case ACTION_SPEED:
			speedUpPuzzle(&widget);
			break;
		case ACTION_SLOW:
			slowDownPuzzle(&widget);
			break;
		case ACTION_SOUND:
			toggleSoundPuzzle(&widget);
			break;
		case ACTION_DESCRIPTION:
			(void) MessageBox(hWnd, descriptionHelp,
				"Description", MB_OK);
			break;
		case ACTION_FEATURES:
			(void) MessageBox(hWnd, featuresHelp,
				"Features", MB_OK);
			break;
		case ACTION_REFERENCES:
			(void) MessageBox(hWnd, referencesHelp,
				"References", MB_OK);
			break;
		case ACTION_ABOUT:
			(void) DialogBox(widget.core.hInstance,
				"About", hWnd, (DLGPROC) about);
			break;
		}
		break;
	default:
		return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
usage(char *programName)
{
	unsigned int i;

	(void) fprintf(stderr, "usage: %s\n", programName);
	for (i = 0; i < strlen(optionsHelp); i++) {
		if (i == 0 || optionsHelp[i - 1] == '\n')
			(void) fprintf(stderr, "\t");
		(void) fprintf(stderr, "%c", (optionsHelp[i]));
	}
	(void) fprintf(stderr, "\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*oct.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*oct.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*oct.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*oct.reverseVideo", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*oct.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*oct.reverseVideo", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*oct.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*oct.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*oct.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*oct.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*oct.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*oct.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*oct.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*oct.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*oct.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*oct.faceColor5", XrmoptionSepArg, NULL},
	{(char *) "-face6", (char *) "*oct.faceColor6", XrmoptionSepArg, NULL},
	{(char *) "-face7", (char *) "*oct.faceColor7", XrmoptionSepArg, NULL},
	{(char *) "-delay", (char *) "*oct.delay", XrmoptionSepArg, NULL},
	{(char *) "-sound", (char *) "*oct.sound", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosound", (char *) "*oct.sound", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fn", (char *) "*oct.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*oct.font", XrmoptionSepArg, NULL},
	{(char *) "-size", (char *) "*oct.size", XrmoptionSepArg, NULL},
	{(char *) "-sticky", (char *) "*oct.sticky", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-mode", (char *) "*oct.mode", XrmoptionSepArg, NULL},
	{(char *) "-both", (char *) "*oct.mode", XrmoptionNoArg, (char *) "4"},
	{(char *) "-orient", (char *) "*oct.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*oct.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*oct.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*oct.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-userName", (char *) "*oct.userName", XrmoptionSepArg, NULL},
	{(char *) "-scoreFile", (char *) "*oct.scoreFile", XrmoptionSepArg, NULL},
	{(char *) "-scores", (char *) "*oct.scoreOnly", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-version", (char *) "*oct.versionOnly", XrmoptionNoArg, (char *) "TRUE"}
};

#ifdef HAVE_MOTIF
static void
puzzlePracticeListener(Widget w, XtPointer clientData,
		XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(puzzle, XtNmenu, ACTION_PRACTICE, NULL);
	}
}

static void
puzzleRandomizeListener(Widget w, XtPointer clientData,
		XmAnyCallbackStruct *cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtVaSetValues(puzzle, XtNmenu, ACTION_RANDOMIZE, NULL);
	}
}
static void
pieceChangeListener(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int size = cbs->value, mode;
	Boolean orient, sticky, practice;

	XtVaGetValues(puzzle,
		XtNsize, &oldSize,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNsticky, &sticky,
		XtNpractice, &practice, NULL);
	if (sticky)
		XmScaleSetValue(w, oldSize);
	else if (oldSize != size) {
		XtVaSetValues(puzzle,
			XtNsize, size, NULL);
		oldSize = size;
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		printState(movesText, buff);
		printRecord(size, mode, orient, sticky, practice);
		(void) strcpy(messageDsp, "");
		printState(message, messageDsp);
	}
}

static void
modeToggle(Widget w, int mode, XmToggleButtonCallbackStruct *cbs)
{
	int size;
	Boolean orient, sticky, practice;

	if (cbs->set) {
		XtVaGetValues(puzzle,
			XtNsize, &size,
			XtNorient, &orient,
			XtNsticky, &sticky,
			XtNpractice, &practice, NULL);
		XtVaSetValues(puzzle,
			XtNmode, mode + PERIOD3, NULL);
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		printState(movesText, buff);
		printRecord(size, mode + PERIOD3, orient, sticky, practice);
		(void) strcpy(messageDsp, "");
		printState(message, messageDsp);
	}
}

static void
orientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct *cbs)
{
	int size, mode;
	Boolean orient = cbs->set, sticky, practice;

	XtVaGetValues(puzzle,
		XtNsize, &size,
		XtNmode, &mode,
		XtNsticky, &sticky,
		XtNpractice, &practice, NULL);
	XtVaSetValues(puzzle,
		XtNorient, orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	printState(movesText, buff);
	printRecord(size, mode, orient, sticky, practice);
	(void) strcpy(messageDsp, "");
	printState(message, messageDsp);
}

static void
stickyToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct *cbs)
{
	int size, mode;
	Boolean orient, sticky = cbs->set, practice;

	XtVaGetValues(puzzle,
		XtNsize, &size,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	XtVaSetValues(puzzle,
		XtNsticky, sticky, NULL);
	if ((sticky && size != 4) || (!sticky && oldSize != 4)) {
		if (sticky)
			size = 4;
		else
			size = oldSize;
		if (size <= MAX_FACETS)
			XmScaleSetValue(facetSlider, size);
		XtVaSetValues(puzzle,
			XtNsize, size, NULL);
	}
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	printState(movesText, buff);
	printRecord(size, mode, orient, sticky, practice);
	(void) strcpy(messageDsp, "");
	printState(message, messageDsp);
}

static void
practiceToggle(Widget w, XtPointer clientData,
		XmToggleButtonCallbackStruct *cbs)
{
	int size, mode;
	Boolean orient, sticky, practice = cbs->set;

	XtVaSetValues(puzzle,
		XtNpractice, practice,
		XtNstart, False, NULL);
	XtVaGetValues(puzzle,
		XtNsize, &size,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNsticky, &sticky, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	printState(movesText, buff);
	printRecord(size, mode, orient, sticky, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGE_LENGTH);
	printState(message, messageDsp);
}

static void
speedChangeListener(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int delay = MULTSPEED * (MAX_SPEED + MIN_SPEED - cbs->value - 1), oldDelay; 
	XtVaGetValues(puzzle,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(puzzle, XtNdelay, delay, NULL);
	}
}

static void
fileMenuListener(Widget w, void *value, void *clientData)
{
	int val = (int) value + ACTION_GET;

	if (val == ACTION_EXIT)
		exit(0);
	XtVaSetValues(puzzle, XtNmenu, val, NULL);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value + ACTION_UNDO;

	XtVaSetValues(puzzle, XtNmenu, val, NULL);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILE_NAME_LENGTH + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILE_NAME_LENGTH + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpMenuListener(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpMenuListener: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
	int pixmapSize;
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, facetRowCol, labelRowCol, radioRowCol;
	Widget switchRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString undoString, redoString;
	XmString clearString, randomizeString, practiceString, solveString;
	XmString incrementString, decrementString;
	XmString orientizeString, stickyString;
	XmString speedString, slowString, soundString;
	unsigned int i;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Oct",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		usage(argv[0]);

#ifdef HAVE_MOTIF
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Exit");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu",
		0, fileMenuListener,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'x', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	undoString = XmStringCreateSimple((char *) "Undo");
	redoString = XmStringCreateSimple((char *) "Redo");
	clearString = XmStringCreateSimple((char *) "Clear");
	practiceString = XmStringCreateSimple((char *) "Practice");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "Auto-solve");
	incrementString = XmStringCreateSimple((char *) "Increment");
	decrementString = XmStringCreateSimple((char *) "Decrement");
	orientizeString = XmStringCreateSimple((char *) "Orientize");
	stickyString = XmStringCreateSimple((char *) "Sticky");
	speedString = XmStringCreateSimple((char *) "Speed >");
	slowString = XmStringCreateSimple((char *) "Slow <");
	soundString = XmStringCreateSimple((char *) "Sound @");
	(void) XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu",
		1, playCB,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, redoString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'z', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 's', NULL, NULL,
		XmVaPUSHBUTTON, incrementString, 'I', NULL, NULL,
		XmVaPUSHBUTTON, decrementString, 'D', NULL, NULL,
		XmVaPUSHBUTTON, orientizeString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, stickyString, 'y', NULL, NULL,
		XmVaPUSHBUTTON, speedString, '>', NULL, NULL,
		XmVaPUSHBUTTON, slowString, '<', NULL, NULL,
		XmVaPUSHBUTTON, soundString, '@', NULL, NULL,
		NULL);
	XmStringFree(undoString);
	XmStringFree(redoString);
	XmStringFree(clearString);
	XmStringFree(practiceString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(incrementString);
	XmStringFree(decrementString);
	XmStringFree(orientizeString);
	XmStringFree(stickyString);
	XmStringFree(speedString);
	XmStringFree(slowString);
	XmStringFree(soundString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpMenuListener, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure you want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) puzzlePracticeListener);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) puzzleRandomizeListener);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move", 5, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	movesText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	recordText = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	facetRowCol = XtVaCreateManagedWidget("facetRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	facetSlider = XtVaCreateManagedWidget("facetSlider",
		xmScaleWidgetClass, facetRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size", 5,
		XmNminimum, MIN_FACETS,
		XmNmaximum, MAX_FACETS,
		XmNvalue, MIN_FACETS,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(facetSlider, XmNvalueChangedCallback,
		(XtCallbackProc) pieceChangeListener, (XtPointer) NULL);
	speedSlider = XtVaCreateManagedWidget("speedSlider",
		xmScaleWidgetClass, facetRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MIN_SPEED,
		XmNmaximum, MAX_SPEED,
		XmNvalue, MAX_SPEED - 10,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speedSlider, XmNvalueChangedCallback,
		(XtCallbackProc) speedChangeListener, (XtPointer) NULL);
	labelRowCol = XtVaCreateManagedWidget("labelRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	(void) XtVaCreateManagedWidget("PeriodText",
		xmLabelWidgetClass, labelRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Periods:", 9, NULL);
	radioRowCol = XtVaCreateManagedWidget("radioRowCol",
		xmRowColumnWidgetClass, labelRowCol,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN,
		XmNradioBehavior, True, NULL);
	for (i = 0; i < XtNumber(modeString); i++) {
		modes[i] = XtVaCreateManagedWidget(modeString[i],
			xmToggleButtonGadgetClass, radioRowCol,
			XmNradioBehavior, True, NULL);
		XtAddCallback(modes[i], XmNvalueChangedCallback,
			(XtCallbackProc) modeToggle, (XtPointer) i);
	}
	switchRowCol = XtVaCreateManagedWidget("switchRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	orientizeSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULT_ORIENT, NULL);
	XtAddCallback(orientizeSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) orientToggle, (XtPointer) NULL);
	stickySwitch = XtVaCreateManagedWidget("Sticky",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULTSTICKY, NULL);
	XtAddCallback(stickySwitch, XmNvalueChangedCallback,
		(XtCallbackProc) stickyToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, switchRowCol,
		XmNset, DEFAULT_PRACTICE, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) practiceToggle, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	message = XtVaCreateManagedWidget("Play Oct! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);
	puzzle = XtCreateManagedWidget("oct",
		octWidgetClass, mainPanel, NULL, 0);
#else
	puzzle = XtCreateManagedWidget("oct",
		octWidgetClass, topLevel, NULL, 0);
#endif
	XtVaGetValues(puzzle,
		XtNpixmapSize, &pixmapSize, NULL);
#ifdef HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			RESIZE_XPM(pixmapSize), &pixmap, NULL,
			&xpmAttributes);
	}
	if (pixmap == (Pixmap) NULL)
#endif
		pixmap = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			DEFINE_XBM);
	XtVaSetValues(topLevel,
#ifdef HAVE_MOTIF
		XmNkeyboardFocusPolicy, XmPOINTER, /* not XmEXPLICIT */
#else
		XtNinput, True,
#endif
		XtNiconPixmap, pixmap, NULL);
	XtAddCallback(puzzle, XtNselectCallback,
		(XtCallbackProc) puzzleListener, (XtPointer) NULL);
	initialize(puzzle);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(puzzle), (unsigned int) AnyButton, AnyModifier,
		XtWindow(puzzle), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(puzzle),
		XCreateFontCursor(XtDisplay(puzzle), XC_hand2));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
