/* partspace.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: partspace.cc,v 1.4 2002/02/22 21:04:28 ralf Exp $ */

#include "partspace.h"

#ifdef WANT_THREADS
void *PS_slavestart( void *arg )
{
  PartSpace *ps = (PartSpace*)arg;
  ps->slavehandler();
  return NULL;
}
#endif

PartSpace::PartSpace()
{
#ifdef WANT_THREADS
  spaceinfo = new List();
  spacerequest = new SpaceRequestList();
  spaceanswer = new SpaceRequestList();

  slave.running = 0;
  slave.stop = false;
  if ( pthread_create( &slave.th, NULL, PS_slavestart, this ) == 0 ) {
    thread_usage = true;
  } else {
    fprintf( stderr, "Worker: Can't create thread, disabling thread-support!\n" );
    thread_usage = false;
  }
#else
  thread_usage = false;
#endif
}

PartSpace::~PartSpace()
{
#ifdef WANT_THREADS
  SpaceRequestList::space_request_list_t *te;

  slave.stop = true;
  // wake up slave
  spacerequest->signal();
  if ( thread_usage == true ) {
    pthread_join( slave.th, NULL );
  }
  while ( spaceanswer->isEmpty_locked() == false ) {
    te = spaceanswer->remove_locked();
    if ( te != NULL ) {
      _freesafe( te->name );
      delete te;
    }
  }
  while ( spacerequest->isEmpty_locked() == false ) {
    te = spacerequest->remove_locked();
    if ( te != NULL ) {
      _freesafe( te->name );
      delete te;
    }
  }

  delete spacerequest;
  delete spaceanswer;

  te = (SpaceRequestList::space_request_list_t*)spaceinfo->getFirstElement();
  while ( te != NULL ) {
    spaceinfo->removeFirstElement();
    _freesafe( te->name );
    delete te;
    te = (SpaceRequestList::space_request_list_t*)spaceinfo->getFirstElement();
  }
  delete spaceinfo;
#endif
}

// So arbeitet es im Thread-Fall
// 1.Nachschauen, ob vom Slave Resultate da sind
// 1.1.Wenn ja, alle entnehmen und falls Zeit noch gltig
//     in spaceinfo einfgen
//     falls error == 0
// 2.Prfen, ob letzter Flush lange genug weg ist
// 2.1.Wenn ja, dann ber Liste gehen
//     Fr alle zu alten:
//       Entferne Eintrag aus liste
// 3.Name suchen
// 3.1.Wenn gefunden, dann stats in lokalen buffer kopieren und 0 zurckgeben
//     Damit wird bis zum nchsten readSpace auf diese Werte zugegriffen
//     Gltig ist dieser Wert wegen 2
// 3.2.sonst: neuen space_..._t besorgen, mit dupstring(name) belegen und 
//     in slaverequest einfgen und slave aufwecken
//     dann wird EAGAIN zurckgeliefert

int PartSpace::readSpace( const char* name )
{
#ifdef WANT_THREADS
  SpaceRequestList::space_request_list_t *te;
  time_t ct;
  int id;

  if ( thread_usage == true ) {
    spaceanswer->lock();
    ct = time( NULL );
    while ( spaceanswer->isEmpty_locked() == false ) {
      te = spaceanswer->remove_locked();
      if ( te != NULL ) {
        if ( ( difftime( ct, te->readtime ) <= PARTSPACE_INFO_LIFETIME ) && ( te->error == 0 ) ) {
          // valid
          spaceinfo->addElement( te );
        } else {
          // invalid
          _freesafe( te->name );
          delete te;
        }
      }
    }
    spaceanswer->unlock();
    
    id = spaceinfo->initEnum();
    te = (SpaceRequestList::space_request_list_t*)spaceinfo->getFirstElement( id );
    while ( te != NULL ) {
      if ( difftime( ct, te->readtime ) > PARTSPACE_INFO_LIFETIME ) {
        // too old
        spaceinfo->removeFirstElement();
       _freesafe( te->name );
        delete te;
      } else break;
      te = (SpaceRequestList::space_request_list_t*)spaceinfo->getFirstElement( id );
    }
  
    // find name
    te = (SpaceRequestList::space_request_list_t*)spaceinfo->getFirstElement( id );
    while ( te != NULL ) {
      if ( strcmp( name, te->name ) == 0 ) {
        // entry found
        break;
      }
      te = (SpaceRequestList::space_request_list_t*)spaceinfo->getNextElement( id );
    }
    spaceinfo->closeEnum( id );
  } else te = NULL;
  
  if ( te != NULL ) {
    // found
#ifdef HAVE_STATVFS
    blocksize = te->fsstatistic.f_frsize;
    space = te->fsstatistic.f_blocks;
    freespace = te->fsstatistic.f_bavail;
#else
    blocksize = te->fsstatistic.f_bsize;
    space = te->fsstatistic.f_blocks;
    freespace = te->fsstatistic.f_bavail;
#endif
    return 0;
  } else {
    // not found
    if ( thread_usage == true ) {
      // when using thread put a request in the list
      te = new SpaceRequestList::space_request_list_t;
      te->name = dupstring( name );
      spacerequest->lock();
      spacerequest->put_locked( te );
      spacerequest->signal();
      spacerequest->unlock();
      return EAGAIN;
    } else {
      // not using threads so just call getSpace
      SpaceRequestList::space_request_list_t te2;
      int erg;
      
      erg = getSpace( name, &( te2.fsstatistic ) );
      if ( erg == 0 ) {
#ifdef HAVE_STATVFS
        blocksize = te2.fsstatistic.f_frsize;
        space = te2.fsstatistic.f_blocks;
        freespace = te2.fsstatistic.f_bavail;
#else
        blocksize = te2.fsstatistic.f_bsize;
        space = te2.fsstatistic.f_blocks;
        freespace = te2.fsstatistic.f_bavail;
#endif
        return 0;
      } else return erg;
    }
  }
#else

#ifdef HAVE_SYS_STATVFS_H
  struct statvfs fsstatistic;
#else
  struct statfs fsstatistic;
#endif

  if ( getSpace( name, &fsstatistic ) == 0 ) {
#ifdef HAVE_STATVFS
    blocksize = fsstatistic.f_frsize;
    space = fsstatistic.f_blocks;
    freespace = fsstatistic.f_bavail;
#else
    blocksize = fsstatistic.f_bsize;
    space = fsstatistic.f_blocks;
    freespace = fsstatistic.f_bavail;
#endif
    return 0;
  } else return EFAULT;
#endif
}

long PartSpace::getBlocksize()
{
  return blocksize;
}

long PartSpace::getFreeSpace()
{
  return freespace;
}

long PartSpace::getSpace()
{
  return space;
}

// H stands for human readable, means in B/KB/MB/GB when great enough

long PartSpace::getFreeSpaceH(char **unit_return)
{
  double dfree;
  char buffer[3];

  dfree=(double)freespace*(double)blocksize;
  if(dfree>10.0*1024.0*1024.0*1024.0) {
    dfree+=512.0*1024.0*1024.0;
    dfree/=1024.0*1024.0*1024.0;
    strcpy(buffer,"GB");
  } else if(dfree>10485760) {
    dfree+=524288;
    dfree/=1048576;
    strcpy(buffer,"MB");
  } else if(dfree>10240) {
    dfree+=512;
    dfree/=1024;
    strcpy(buffer,"KB");
  } else {
    strcpy(buffer,"B");
  }
  if(unit_return!=NULL) {
    strcpy(*unit_return,buffer);
  }
  return (long)dfree;
}

long PartSpace::getSpaceH(char **unit_return)
{
  double dsize;
  char buffer[3];

  dsize=((double)space)*((double)blocksize);
  if(dsize>10.0*1024.0*1024.0*1024.0) {
    dsize+=512.0*1024.0*1024.0;
    dsize/=1024.0*1024.0*1024.0;
    strcpy(buffer,"GB");
  } else if(dsize>10485760) {
    dsize+=524288.0;
    dsize/=1048576.0;
    strcpy(buffer,"MB");
  } else if(dsize>10240) {
    dsize+=512;
    dsize/=1024;
    strcpy(buffer,"KB");
  } else {
    strcpy(buffer,"B");
  }
  if(unit_return!=NULL) {
    strcpy(*unit_return,buffer);
  }
  return (long)dsize;
}

#ifdef WANT_THREADS
bool PartSpace::SpaceRequestList::isEmpty_locked()
{
  if ( elements < 1 ) return true;
  return false;
}

int PartSpace::SpaceRequestList::put_locked( space_request_list_t *elem )
{
  int pos;

  if ( elem == NULL ) return -1;

  // don't accept any pointer in next
  elem->next = NULL;
  if ( tail != NULL ) {
    // add behind last
    tail->next = elem;
    tail = elem;
  } else {
    head = tail = elem;
  }

  pos = elements++;
  return pos;
}

PartSpace::SpaceRequestList::space_request_list_t *PartSpace::SpaceRequestList::remove_locked()
{
  space_request_list_t *te;

  if ( elements == 0 ) return NULL;
  
  te = head;
  head = head->next;
  if ( head == NULL )
    tail = NULL;
  elements--;

  // leave no pointer in out list
  te->next = NULL;  
  return te;
}

PartSpace::SpaceRequestList::SpaceRequestList()
{
  head = NULL;
  tail = NULL;
  elements = 0;
}

PartSpace::SpaceRequestList::~SpaceRequestList()
{
  space_request_list_t *te;

  lock();
  while ( isEmpty_locked() == false ) {
    te = remove_locked();
    delete te;
  }
  unlock();
}

void PartSpace::SpaceRequestList::lock()
{
  ex.lock();
}

void PartSpace::SpaceRequestList::unlock()
{
  ex.unlock();
}

void PartSpace::SpaceRequestList::wait()
{
  ex.wait();
}

void PartSpace::SpaceRequestList::signal()
{
  ex.signal();
}

void PartSpace::slavehandler()
{
  SpaceRequestList::space_request_list_t *te;
  bool ende;
  
  if ( slave.running != 0 ) {
    fprintf( stderr, "Worker: another thread already running!\n");
    return;
  }
  slave.running = 1;
#ifdef DEBUG
  printf("entering slave handler\n");
#endif

  spacerequest->lock();
  for( ende = false; ende == false; ) {
    // wait for next element or stop
    while ( ( spacerequest->isEmpty_locked() == true ) && ( slave.stop == false ) )
      spacerequest->wait();

#ifdef DEBUG
    printf("wait finished\n");
#endif
    if ( slave.stop == false ) {
      // an element to check
      te = spacerequest->remove_locked();
      spacerequest->unlock();
      if ( te != NULL ) {
        te->error = getSpace( te->name, &(te->fsstatistic) );
        te->readtime = time(NULL);
        spaceanswer->lock();
        spaceanswer->put_locked( te );
        spaceanswer->unlock();
      }
      spacerequest->lock();
    } else {
      ende = true;
    }
  }
  spacerequest->unlock();
#ifdef DEBUG
  printf("leaving slave handler\n");
#endif
}
#endif

#ifdef HAVE_SYS_STATVFS_H
int PartSpace::getSpace( const char *name, struct statvfs *statbuf )
#else
int PartSpace::getSpace( const char *name, struct statfs *statbuf )
#endif
{
  if ( name == NULL ) return EFAULT;
  if ( statbuf == NULL ) return EFAULT;

#ifdef HAVE_STATVFS
  return statvfs( name, statbuf );
#else
  return statfs( name, statbuf );
#endif
}

