/* datei.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2001 Ralf Hoffmann.
 * You can contact me at: ralf.hoffmann@epost.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: datei.cc,v 1.50 2002/03/17 01:09:25 ralf Exp $ */

#include "datei.h"
#include <aguix/lowlevelfunc.h>
#include "wconfig.h"
#include "grouphash.h"
#include "locale.h"
#include "wdefines.h"

#if 0
#include <string>
#endif

#ifndef MAXPATHLEN
#define MAXPATHLEN 2048
#endif

int datelen=-1;

Datei::Datei()
{
  fp=NULL;
  error=0;
}

Datei::~Datei()
{
  if(fp!=NULL) close();
}

int Datei::open(const char *name,char *mode)
{
  fp=fopen(name,mode);
  if(fp==NULL) return 1;
  return 0;
}

void Datei::close()
{
  if(fp==NULL) return;
  fclose(fp);
  fp=NULL;
}

int Datei::putUChar(unsigned char value)
{
  if(fp==NULL) return -1;
  int x;
  x=fputc(value,fp);
  if(x!=EOF) return 1;
  printf("fehler bei FPutChar\n");
  return 0;
}

int Datei::putUShort(unsigned short int value)
{
  if(fp==NULL) return -1;
  int x;
  x=fputc(value>>8,fp);
  if(x==EOF) return 0;
  x=fputc(value&0xff,fp);
  if(x==EOF) return 1;
  return 2;
}

int Datei::putInt(int value)
{
  if(fp==NULL) return -1;
  int x,tv;
  if(value<0) {
    x=fputc(1,fp);
    if(x==EOF) return 0;
  } else {
    x=fputc(0,fp);
    if(x==EOF) return 0;
  }
  tv=abs(value);
  int i;
  for(i=0;i<4;i++) {
    x=fputc(tv&0xff,fp);
    if(x==EOF) return i+1;
    tv>>=8;
  }
  return i+1;
}

int Datei::getInt()
{
  if(fp==NULL) return -1;
  int vz,x,value;
  vz=fgetc(fp);
  if(vz<0) {
    error++;
    return 0;
  }
  value=0;
  for(int i=0;i<4;i++) {
    x=fgetc(fp);
    if(x==EOF) {
      error++;
      return 0;
    }
    value+=(x<<(8*i));
  }
  if(vz==1) value*=-1;
  return value;
}

int Datei::putULong(unsigned long value)
{
  if(fp==NULL) return -1;
  int x;
  x=fputc((value&0xff000000)>>24,fp);
  if(x==EOF) return 0;
  x=fputc((value&0xff0000)>>16,fp);
  if(x==EOF) return 1;
  x=fputc((value&0xff00)>>8,fp);
  if(x==EOF) return 2;
  x=fputc(value&0xff,fp);
  if(x==EOF) return 3;
  return 4;
}

int Datei::putString(char *string)
{
  if(fp==NULL) return -1;
  int x;
  x=fputs(string,fp);
  if(x==EOF) return -2;
  return strlen(string);
}

unsigned char Datei::getUChar()
{
  if(fp==NULL) {
    error++;
    return 0;
  }
  int x;
  x=fgetc(fp);
  if(x==EOF) {
    error++;
    return 0;
  }
  return (unsigned char) x;
}

int Datei::getCharAsInt()
{
  if(fp==NULL) {
    error++;
    return 0;
  }
  return fgetc(fp);
}

unsigned short Datei::getUShort()
{
  if(fp==NULL) {
    error++;
    return 0;
  }
  int x;
  unsigned short int a;
  x=fgetc(fp);
  if(x==EOF) {
    error++;
    return 0;
  }
  a=((unsigned int)x)*0x100;
  x=fgetc(fp);
  if(x==EOF) {
    error++;
    return 1;
  }
  a+=(unsigned int)x;
  return a;
}

unsigned long Datei::getULong()
{
  if(fp==NULL) {
    error++;
    return 0;
  }
  int x;
  unsigned long a;
  x=fgetc(fp);
  if(x==EOF) {
    error++;
    return 0;
  }
  a=((unsigned int)x)*0x1000000;
  x=fgetc(fp);
  if(x==EOF) {
    error++;
    return 1;
  }
  a+=((unsigned int)x)*0x10000;
  x=fgetc(fp);
  if(x==EOF) {
    error++;
    return 2;
  }
  a+=((unsigned int)x)*0x100;
  x=fgetc(fp);
  if(x==EOF) {
    error++;
    return 3;
  }
  a+=(unsigned int)x;
  return a;
}

char *Datei::getString(int count)
{
  if(fp==NULL) return NULL;
  unsigned char *tstr,*tstr2;
  int ch;
  long x;
  long size;
  if(count==-1) {
    /* lesen bis NULL-Byte */
    size=1024;
    tstr=(unsigned char*)_allocsafe(size);
    x=0;
    do {
      ch=fgetc(fp);
      if(ch!=EOF) {
        tstr[x++]=(unsigned char)ch;
      } else {
        tstr[x++]=0;
        break;
      }
      if(x==size) {
        size*=2;
        tstr2=(unsigned char*)_allocsafe(size);
        if(tstr2==NULL) break;
        strcpy((char*)tstr2,(char*)tstr);
        _freesafe(tstr);
        tstr=tstr2;
      }
    } while(ch!=0);
  } else {
    tstr=(unsigned char*)_allocsafe(count+1);
    if(tstr!=NULL) {
      for(x=0;x<count;x++) {
        ch=fgetc(fp);
        tstr[x]=(unsigned char)ch;
      }
      tstr[count]=0;
    }
  }
  return (char*)tstr;
}

int Datei::getString(char *buf,int count)
{
  buf[0]=0;
  char *str=getString(count);
  strncpy(buf,str,count);
  buf[count]=0;
  _freesafe(str);
  return strlen(buf);
}

FileEntry::FileEntry()
{
  name=NULL;
  fullname=NULL;
  filetype=NULL;
  use=true;
  isHidden=false;
}

FileEntry::~FileEntry()
{
  if(name!=NULL) _freesafe(name);
  if(fullname!=NULL) _freesafe(fullname);
}

Verzeichnis::Verzeichnis()
{
  files=NULL;
  actdir=NULL;
  actsortmode=-1;
  invalid_entries=NULL;
}

Verzeichnis::~Verzeichnis()
{
  closeDir();
}

ArrayList *Verzeichnis::getFiles()
{
  return files;
}

int Verzeichnis::readDir(const char *dirstr)
{
  if(dirstr==NULL) return -1;
  if(strlen(dirstr)<1) return -1;
  if(files!=NULL) closeDir();
  files=new ArrayList();
  invalid_entries=new ArrayList();
  // nahezu 1 zu 1 vom Original bernehmen

  int returnvalue=0;
  struct dirent *namelist;
  DIR *dir;
  FileEntry *fe1=NULL,*fe2;
  char *tstr;
  long x,ldnr;
  FileEntry *fpp;
  long havepointpoint;
  int id;
  bool corruptDir=false;
  
  fe1=NULL;
  dir=opendir(dirstr);
  if(dir!=NULL) {
//    readdir(dir);  /* berspringt den Eintrag "." */
    havepointpoint=-1;
    fpp=NULL;
    ldnr=0;
    while((namelist=readdir(dir))!=NULL) {
      // ignore "." entries (no need)
      if(strcmp(namelist->d_name,".")!=0) {
        fe1=new FileEntry();
        if(strcmp(namelist->d_name,"..")==0) {
          // this is because of problems with NTFS
          havepointpoint=ldnr;
          fpp=fe1;
        }
        fe1->name=dupstring(namelist->d_name);
        x=strlen(fe1->name);
        tstr=(char*)_allocsafe(strlen(dirstr)+1+x+1);
        strcpy(tstr,dirstr);
        if(strlen(dirstr)>1) strcat(tstr,"/");
        strcat(tstr,namelist->d_name);
        fe1->fullname=tstr;
        fe1->nr=ldnr++;
        if(fe1->readInfos()==0) {
          if(fe1->name[0]=='.') fe1->isHidden=true;
          else fe1->isHidden=false;
          files->addElement(fe1);
        } else {
          ldnr--;
#if 0
          corruptDir=true;
          delete fe1;
#else
          invalid_entries->addElement(fe1);
#endif
        }
      }
      if(corruptDir==true) break;
    }
    closedir(dir);
    if(corruptDir==false) {
      if(havepointpoint==-1) {
        // there is no parent-dir-entry
        // create one

        // first raise the nr of all FEs
        id=files->initEnum();
        fe1=(FileEntry*)files->getFirstElement(id);
        while(fe1!=NULL) {
          fe1->nr++;
          fe1=(FileEntry*)files->getNextElement(id);
        }
        fe2=(FileEntry*)files->getFirstElement(id);
        files->closeEnum(id);
        fe1=new FileEntry();
        fe1->name=dupstring("..");
        x=strlen(fe1->name);
        tstr=(char*)_allocsafe(strlen(dirstr)+1+x+1);
        strcpy(tstr,dirstr);
        if(strlen(dirstr)>1) strcat(tstr,"/");
        strcat(tstr,fe1->name);
        fe1->fullname=tstr;
        fe1->nr=0;
        if(fe1->name[0]=='.') fe1->isHidden=true;
        else fe1->isHidden=false;
        fe1->size=0;
        fe1->dirsize=-1;
        if(fe2!=NULL) {
          fe1->lastaccess=fe2->lastaccess;
          fe1->lastmod=fe2->lastmod;
          fe1->lastchange=fe2->lastchange;
          fe1->mode=fe2->mode;
          fe1->userid=fe2->userid;
          fe1->groupid=fe2->groupid;
        }
        fe1->isCorrupt=false;
        fe1->isLink=false;
        fe1->select=false;
        files->addElementAt(0,fe1);
      } else if(havepointpoint>0) {
        if(fpp!=NULL) {
          id=files->initEnum();
          fe1=(FileEntry*)files->getFirstElement(id);
          while((fe1!=NULL)&&(fe1!=fpp)) {
            fe1->nr++;
          }
          havepointpoint=files->getIndex(fpp);
          files->removeElementAt(havepointpoint);
          files->addElementAt(0,fpp);
          fpp->nr=0;
          files->closeEnum(id);
        }
      }
      actdir=dupstring(dirstr);
    }
  } else {
    returnvalue=-1;
    delete files;
    files=NULL;
    delete invalid_entries;
    invalid_entries=NULL;
  }
  if(corruptDir==true) {
    id=files->initEnum();
    fe1=(FileEntry*)files->getFirstElement(id);
    while(fe1!=NULL) {
      delete fe1;
      fe1=(FileEntry*)files->getNextElement(id);
    }
    files->closeEnum(id);

    id=invalid_entries->initEnum();
    fe1=(FileEntry*)invalid_entries->getFirstElement(id);
    while(fe1!=NULL) {
      delete fe1;
      fe1=(FileEntry*)invalid_entries->getNextElement(id);
    }
    invalid_entries->closeEnum(id);

    returnvalue=-1;
    delete files;
    files=NULL;
    delete invalid_entries;
    invalid_entries=NULL;
  }
  return returnvalue;
}

int Verzeichnis::sortfunction(void *p1,void *p2,int mode)
{
  FileEntry *fe1,*fe2;
  int comp=0;
  fe1=(FileEntry*)p1;
  fe2=(FileEntry*)p2;
  loff_t s1,s2;
  int dirm=mode&0x600;
  WCFiletype *ft1, *ft2;
  
/*  if(strcmp(fe1->name,"..")==0) return true;  // Sonderfall
  if(strcmp(fe2->name,"..")==0) return false;*/
  if((strcmp(fe1->name,"..")==0)&&(strcmp(fe2->name,"..")==0)) return 0;
  else if(strcmp(fe1->name,"..")==0) return -1;
  else if(strcmp(fe2->name,"..")==0) return 1;

  switch(mode&0xff) {
    case SORT_NAME:
      comp=strcmp(fe1->name,fe2->name);
      break;
    case SORT_SIZE:
      s1=-1;
      s2=-1;
      if(fe1->isDir()==true) s1=fe1->dirsize;
      if(s1<0) s1=fe1->size;
      if(fe2->isDir()==true) s2=fe2->dirsize;
      if(s2<0) s2=fe2->size;
/*      if(s1<s2) smaller=true;
      else if(s1==s2) {
        if(strcmp(fe1->name,fe2->name)<0) smaller=true;
      }*/
      if(s1<s2) comp=-1;
      else if(s1>s2) comp=1;
      else comp=strcmp(fe1->name,fe2->name);
      break;
    case SORT_ACCTIME:
      if(fe1->lastaccess<fe2->lastaccess) comp=-1;
      else if(fe1->lastaccess>fe2->lastaccess) comp=1;
      else comp=strcmp(fe1->name,fe2->name);
      break;
    case SORT_MODTIME:
      if(fe1->lastmod<fe2->lastmod) comp=-1;
      else if(fe1->lastmod>fe2->lastmod) comp=1;
      else comp=strcmp(fe1->name,fe2->name);
      break;
    case SORT_CHGTIME:
      if(fe1->lastchange<fe2->lastchange) comp=-1;
      else if(fe1->lastchange>fe2->lastchange) comp=1;
      else comp=strcmp(fe1->name,fe2->name);
      break;
    case SORT_TYPE:
      if ( fe1->filetype != NULL ) {
        ft1 = fe1->filetype;
      } else {
        if(fe1->isDir()==true) {
          ft1 = wconfig->getdirtype();
        } else {
          ft1 = wconfig->getnotyettype();
        }
      }
      if ( fe2->filetype != NULL ) {
        ft2 = fe2->filetype;
      } else {
        if(fe2->isDir()==true) {
          ft2 = wconfig->getdirtype();
        } else {
          ft2 = wconfig->getnotyettype();
        }
      }
      comp = strcmp( ft1->getName(), ft2->getName() );
      if ( comp == 0 )
        comp = strcmp( fe1->name, fe2->name );
      break;
    case SORT_OWNER:
      // first cmp usernames
      // if equal cmp groupnames
      // if equal cmp names
      
      // getUserNameS only look for existing entry in hashtable and returns
      // static pointer
      // so it's fast
      //TODO: perhaps it makes more sense to first compare groups
      comp = strcmp( ugdb->getUserNameS( fe1->userid ),
                     ugdb->getUserNameS( fe2->userid ) );
      if ( comp == 0 ) {
        comp = strcmp( ugdb->getGroupNameS( fe1->groupid ),
                       ugdb->getGroupNameS( fe2->groupid ) );
        if ( comp == 0 )
          comp = strcmp( fe1->name, fe2->name );
      }
      break;
  }
  if((mode&SORT_REVERSE)==SORT_REVERSE) comp*=-1;
  if(dirm!=SORT_DIRMIXED) {
    if(fe1->isDir()==true) {
      if(fe2->isDir()==false) comp=(dirm==SORT_DIRLAST)?1:-1;
    } else {
      if(fe2->isDir()==true) comp=(dirm==SORT_DIRLAST)?-1:1;
    }
  }
  return comp;
}

void Verzeichnis::closeDir()
{
  if(files!=NULL) {
    FileEntry *fe=(FileEntry*)files->getFirstElement();
    while(fe!=NULL) {
      delete fe;
      fe=(FileEntry*)files->getNextElement();
    }
    delete files;
    files=NULL;
    _freesafe(actdir);
    actdir=NULL;
    actsortmode=-1;
  }
  if(invalid_entries!=NULL) {
    FileEntry *fe=(FileEntry*)invalid_entries->getFirstElement();
    while(fe!=NULL) {
      delete fe;
      fe=(FileEntry*)invalid_entries->getNextElement();
    }
    delete invalid_entries;
    invalid_entries=NULL;
  }
}

int Verzeichnis::getSize()
{
  if(files!=NULL) return files->size();
  else return -1;
}

char *Verzeichnis::getDir()
{
  return actdir;
}

char *ParentDir(const char *pathstr,char *dirnamereturn)
{
  char *newpath;
  int x,y;
  x=strlen(pathstr);
  if(x<2) {
    newpath=(char*)_allocsafe(2);
    strcpy(newpath,"/");
    if(dirnamereturn!=NULL) strcpy(dirnamereturn,"");
  } else {
    if(pathstr[x-1]=='/') y=x-2; else y=x-1;
    while((pathstr[y]!='/')&&(y>0)) {
      y--;
    }
    if(y==0) {
      newpath=(char*)_allocsafe(2);
      strcpy(newpath,"/");
      if(dirnamereturn!=NULL) strcpy(dirnamereturn,pathstr+1);
    } else {
      newpath=(char*)_allocsafe(y+1);
      strncpy(newpath,pathstr,y);
      if(dirnamereturn!=NULL) strcpy(dirnamereturn,pathstr+y+1);
      newpath[y]=0;
    }
  }
  return newpath;
}

char *HandlePath(const char *pathstr)
{
  char *newpath,*newpath2,buffer[1024];
  int x,y,parentlevel;
  int start,end;
  int found,ende;
  x=strlen(pathstr);
  y=0;
  found=0;
  parentlevel=0;
  ende=0;
  newpath=(char*)_allocsafe(1);
  newpath[0]=0;
  do {
    start=y;
    while((pathstr[y]!='/')&&((y+1)<x)) {
      y++;
    }
    if(((y+1)>=x)&&(pathstr[y]!='/')) end=y+1; else end=y;
    if(end>start) {
      strncpy(buffer,pathstr+start,end-start);
      buffer[end-start]=0;
      if(strcmp(buffer,".")==0) {
        y++;
      } else if(strcmp(buffer,"..")==0) {
        y++;
        newpath2=ParentDir(newpath,NULL);
        _freesafe(newpath);
        newpath=newpath2;
      } else {
        newpath2=(char*)_allocsafe(strlen(newpath)+1+strlen(buffer)+1);
        strcpy(newpath2,newpath);
        if(strlen(newpath)>0) {
          if(newpath[strlen(newpath)-1]!='/') strcat(newpath2,"/");
        }
        strcat(newpath2,buffer);
        _freesafe(newpath);
        newpath=newpath2;
        y++;
      }
    } else {
      if(start==0) {
        newpath2=(char*)_allocsafe(2);
        strcpy(newpath2,"/");
        _freesafe(newpath);
        newpath=newpath2;
      }
      y++;
    }
    if(y>=x) ende=1;
  } while(ende==0);
  return newpath;
}

int Verzeichnis::sort(int mode)
{
  return sort(mode,false);
}

int Verzeichnis::sort(int mode,bool force)
{
 if((mode!=actsortmode)||(force==true)) {
   files->sort(sortfunction,mode);
   // FEs erhalten Nummer anhand der Position, damit von LVC auf FE geschlossen werden kann
   int id=files->initEnum(),pos;
   FileEntry *fe=(FileEntry*)files->getFirstElement(id);
   pos=0;
   while(fe!=NULL) {
     fe->nr=pos++;
     fe=(FileEntry*)files->getNextElement(id);
   }
   files->closeEnum(id);
   actsortmode=mode;
 }
 return 0;
}

bool FileEntry::isDir()
{
  bool dir=false;
  if(isLink==true) {
    if(S_ISDIR(dmode)) dir=true;
  } else if(S_ISDIR(mode)) dir=true;
  return dir;
}

FileEntry *FileEntry::duplicate()
{
  FileEntry *tfe=new FileEntry();
  tfe->fullname=dupstring(fullname);
  tfe->name=dupstring(name);
  tfe->size=size;
  tfe->dsize=dsize;
  tfe->dirsize=dirsize;
  tfe->select=select;
  tfe->lastaccess=lastaccess;
  tfe->dlastaccess=dlastaccess;
  tfe->lastmod=lastmod;
  tfe->dlastmod=dlastmod;
  tfe->lastchange=lastchange;
  tfe->dlastchange=dlastchange;
  tfe->mode=mode;
  tfe->dmode=dmode;
  tfe->isLink=isLink;
  tfe->isCorrupt=isCorrupt;
  tfe->userid=userid;
  tfe->duserid=duserid;
  tfe->groupid=groupid;
  tfe->dgroupid=dgroupid;
  tfe->nr=nr;
  tfe->inFilter=inFilter;
  tfe->filetype=filetype;
  return tfe;
}

bool Datei::fileExists(const char *name)
{
  struct stat stbuf;
  if(lstat(name,&stbuf)!=0) return false;
  else {
    /* kann auch verzeichnis sein */
//    if(S_ISDIR(stbuf.st_mode)) return false;
  }
  return true;
}

Datei::d_fe_t Datei::fileExistsExt(const char*name)
{
  struct stat stbuf;
  d_fe_t return_value;
  if(stat(name,&stbuf)!=0) return_value=D_FE_NOFILE;
  else {
    if(S_ISDIR(stbuf.st_mode)) return_value=D_FE_DIR;
    else return_value=D_FE_FILE;
    if(S_ISLNK(stbuf.st_mode)) return_value=D_FE_LINK;
  }
  return return_value;
}

Datei::d_fe_t Datei::lfileExistsExt(const char*name)
{
  struct stat stbuf;
  d_fe_t return_value;
  if(lstat(name,&stbuf)!=0) return_value=D_FE_NOFILE;
  else {
    if(S_ISDIR(stbuf.st_mode)) return_value=D_FE_DIR;
    else return_value=D_FE_FILE;
    if(S_ISLNK(stbuf.st_mode)) return_value=D_FE_LINK;
  }
  return return_value;
}

void Datei::seek(long offset,int mode)
{
  if(fp!=NULL) fseek(fp,offset,mode);
}

int getOwnerStringLen(uid_t user,gid_t gr)
{
  char *tstr;
  #ifdef DEBUG
//  printf("entering getOwnerStringLen\n");
  #endif
  int len=0;
  int ost=wconfig->getOwnerstringtype();
  #ifdef DEBUG
//  printf("  getUserNameS\n");
  #endif
  tstr=ugdb->getUserNameS(user);
  if(tstr!=NULL) len+=strlen(tstr);
  len+=(ost==1)?1:3;
  #ifdef DEBUG
//  printf("  getGroupNameS\n");
  #endif
  tstr=ugdb->getGroupNameS(gr);
  if(tstr!=NULL) len+=strlen(tstr);
  #ifdef DEBUG
//  printf("leaving getOwnerStringLen\n");
  #endif
  return len;
}

char* getOwnerString(uid_t user,gid_t mygroup)
{
  int len=0;
  char *tstr1,*tstr2;
  int ost=wconfig->getOwnerstringtype();
  
  tstr1=ugdb->getUserNameS(user);
  tstr2=ugdb->getGroupNameS(mygroup);
  if(tstr1!=NULL) len+=strlen(tstr1);
  len+=(ost==1)?1:3;
  if(tstr2!=NULL) len+=strlen(tstr2);
  char *str=(char*)_allocsafe((len+2)*sizeof(char));
  str[0]=0;
  if(tstr1!=NULL) strcat(str,tstr1);
  strcat(str,(ost==1)?".":" @ ");
  if(tstr2!=NULL) strcat(str,tstr2);
  return str;
}

int FileEntry::readInfos()
{
  return readInfos(false);
}

int FileEntry::readInfos(bool update)
{
  //TODO: auf update achten
  struct stat stbuf;

  select=false;
  dirsize=-1;
  filetype=NULL;
  inFilter=true;

  if(fullname==NULL) return 1;
  if(lstat(fullname,&stbuf)==0) {
    size=stbuf.st_size;
    lastaccess=stbuf.st_atime;
    lastmod=stbuf.st_mtime;
    lastchange=stbuf.st_ctime;
    mode=stbuf.st_mode;
    userid=stbuf.st_uid;
    groupid=stbuf.st_gid;
  } else return 2;
  isCorrupt=false;
  if(S_ISLNK(mode)) {
    isLink=true;
    if(stat(fullname,&stbuf)==0) {
      dsize=stbuf.st_size;
      dlastaccess=stbuf.st_atime;
      dlastmod=stbuf.st_mtime;
      dlastchange=stbuf.st_ctime;
      dmode=stbuf.st_mode;
      duserid=stbuf.st_uid;
      dgroupid=stbuf.st_gid;
    } else {
      /* korrupt */
      isCorrupt=true;
      dmode=0;  // to prevent corrupt links treated as dirs
      if(errno==ENOENT) {
        // Eventl. fuer spaetere Betrachtungen
      }
    }
  } else {
    isLink=false;
  }
  return 0;
}

char *FileEntry::getDestination()
{
  char *str=NULL,*tstr;
  if((isLink==true)&&(fullname!=NULL)) {
    tstr=(char*)_allocsafe((MAXPATHLEN+1)*sizeof(char));
    int x=readlink(fullname,tstr,MAXPATHLEN);
    if(x>=0) {
      tstr[x]=0;
      str=dupstring(tstr);
    }
    _freesafe(tstr);
  }
  return str;
}

int FileEntry::writeDestination2Buf(char *buf,int size2,bool terminated)
{
  int x=-1;
  if((isLink==true)&&(fullname!=NULL)) {
    x=readlink(fullname,buf,size2);
    if((terminated==true)&&(x>=0)) {
      buf[x]=0;
    }
  }
  return x;
}

extern double d1,d2,d3;

void Verzeichnis::setLVC4FE( FieldListView *lv, int row,FileEntry *fe,int *sets)
{
  int mlen=0;
  char *str=NULL;
  struct tm *timeptr;
  loff_t size;
  bool sfds=false;
  std::string s1;
  char pbuf[10];
  int ost, x;
  
  for(int i=0;i<9;i++) {
    if(sets[i]<0) break;
    switch(sets[i]) {
      case 0://Name
        s1 = "";
        if(fe->isLink==true) {
          if(fe->isCorrupt==true) s1 += '!';
          else {
            if(S_ISDIR(fe->dmode)) s1 += '~';
            else s1 += '@';
          }
        } else {
          if(S_ISDIR(fe->mode)) s1 += '/';
          else if(fe->isExe()==true) s1 += '*';
          else s1 += ' ';
        }
        s1 += fe->name;
        lv->setText( row, 2 * i, s1 );
        break;
      case 1://Size
        if((fe->isDir()==true)&&(fe->isLink==false)&&(fe->dirsize>=0)) size=fe->dirsize;
        else size=fe->size;

        if(fe->isDir()==true) {
          if(!((fe->isLink==false)&&(fe->dirsize>=0))) {
            if(wconfig->getShowStringForDirSize()==true)
              sfds=true;
          }
        }
        if(sfds==true) {
          s1 = wconfig->getStringForDirSize();
          lv->setText( row, 2 * i, s1 );
        } else {
          mlen = LongSize( size );
          str = (char*)_allocsafe( mlen + 1 );
          MakeLong2NiceStr( str, size );
          str[ mlen ] = '\0';
          s1 = str;
          _freesafe( str );
          lv->setText( row, 2 * i, s1 );
        }
        break;
      case 2://Type
        if(fe->filetype!=NULL) {
          s1 = fe->filetype->getName();
        } else {
          if(fe->isDir()==true) {
            // Dir
            s1 = wconfig->getdirtype()->getName();
          } else {
            s1 = wconfig->getnotyettype()->getName();
          }
        }
        lv->setText( row, 2 * i, s1 );
        break;
      case 3://Perm
        fe->writePerm2Buf( pbuf, false );
        pbuf[9] = '\0';
        s1 = pbuf;
        lv->setText( row, 2 * i, s1 );
        break;
      case 4://Owner
        // will be 1 for Owner.group, 0 else
        ost=wconfig->getOwnerstringtype();

        s1 = ugdb->getUserNameS(fe->userid);
        if(ost==1) {
          s1 += '.';
        } else {
          s1 += " @ ";
        }
        s1 += ugdb->getGroupNameS(fe->groupid);
        lv->setText( row, 2 * i, s1 );
        break;
      case 5://Dest
        str = fe->getDestination();
        if ( str != NULL ) {
          // only if there will be something to print
          lv->setText( row, 2 * i, str );
          _freesafe( str );
        } else {
          lv->setText( row, 2 * i, "" );
        }
        break;
      case 6://Mod
        timeptr = localtime( &( fe->lastmod ) );
        mlen = wconfig->getDatelen( timeptr );
        str = (char*)_allocsafe( mlen + 2 );
        memset( str, ' ', mlen + 1 );
        x = wconfig->writeDateToString( str , mlen + 1, timeptr );
        str[ x ] = '\0';
        s1 = str;
        _freesafe( str );
        lv->setText( row, 2 * i, s1 );
        break;
      case 7://Acc
        timeptr=localtime(&(fe->lastaccess));
        mlen = wconfig->getDatelen( timeptr );
        str = (char*)_allocsafe( mlen + 2 );
        memset( str, ' ', mlen + 1 );
        x = wconfig->writeDateToString( str , mlen + 1, timeptr );
        str[ x ] = '\0';
        s1 = str;
        _freesafe( str );
        lv->setText( row, 2 * i, s1 );
        break;
      case 8://Change
        timeptr=localtime(&(fe->lastchange));
        mlen = wconfig->getDatelen( timeptr );
        str = (char*)_allocsafe( mlen + 2 );
        memset( str, ' ', mlen + 1 );
        x = wconfig->writeDateToString( str , mlen + 1, timeptr );
        str[ x ] = '\0';
        s1 = str;
        _freesafe( str );
        lv->setText( row, 2 * i, s1 );
        break;
    }
  }
}

char *FileEntry::getPermStr()
{
  return dupstring(getSPermStr());
}

char *FileEntry::getSPermStr()
{
  static char str[10];
  writePerm2Buf(str,true);
  return str;
}

void FileEntry::writePerm2Buf(char*str,bool terminated)
{
  if((mode&S_IRUSR)==S_IRUSR) str[0]='r'; else str[0]='-';
  if((mode&S_IWUSR)==S_IWUSR) str[1]='w'; else str[1]='-';
  if((mode&S_ISUID)==S_ISUID) {
    if((mode&S_IXUSR)==S_IXUSR) str[2]='s'; else str[2]='S';
  } else {
    if((mode&S_IXUSR)==S_IXUSR) str[2]='x'; else str[2]='-';
  }
  if((mode&S_IRGRP)==S_IRGRP) str[3]='r'; else str[3]='-';
  if((mode&S_IWGRP)==S_IWGRP) str[4]='w'; else str[4]='-';
  if((mode&S_ISGID)==S_ISGID) {
    if((mode&S_IXGRP)==S_IXGRP) str[5]='s'; else str[5]='S';
  } else {
    if((mode&S_IXGRP)==S_IXGRP) str[5]='x'; else str[5]='-';
  }
  if((mode&S_IROTH)==S_IROTH) str[6]='r'; else str[6]='-';
  if((mode&S_IWOTH)==S_IWOTH) str[7]='w'; else str[7]='-';
  if((mode&S_ISVTX)==S_ISVTX) {
    if((mode&S_IXOTH)==S_IXOTH) str[8]='t'; else str[8]='T';
  } else {
    if((mode&S_IXOTH)==S_IXOTH) str[8]='x'; else str[8]='-';
  }
  if(terminated==true) str[9]=0;
}

bool FileEntry::isExe()
{
  if((mode&S_IXUSR)==S_IXUSR) return true;
  if((mode&S_IXGRP)==S_IXGRP) return true;
  if((mode&S_IXOTH)==S_IXOTH) return true;
  return false;
}

int getDatelen()
{
  char str[128];
  if(datelen<0) {
    /* Erst berechnen */
    time_t t1;
    struct tm *mytm;
    int x;
    time(&t1);
    // Alle 7 Tage messen
    for(int i=0;i<7;i++) {
      mytm=localtime(&t1);
      x=strftime(str,127,"%H:%M:%S %d %b %Y",mytm);
      if(x>datelen) datelen=x;
      t1+=60*60*24;  // Ein Tag drauf
    }
  }
  return datelen;
}

#define FILEOUTPUTLENGTH 4096

WCFiletype* FileEntry::checkFiletype( List *ftlist, bool skipcontent )
{
//printf("start check %s\n",name);
  unsigned char fbuffer[64];
  int reads,x,entry,found,y,minsize=0;
  int id;
  WCFiletype *ft,*unknownft;

#if 0
  char *filestr, *outname, *fileoutput;
  std::string exestr;
#endif

#ifdef HAVE_LIBZ
  gzFile fp;
#else
  FILE *fp;
#endif
  
  if ( ftlist == NULL ) return NULL;
  
  // TODO: Nicht superideal, weil position nicht zwingend richtig ist (im Moment
  //        klappt das, aber wer weiss, was mir noch alles fuer dummkramm einfaellt :-)
  //       Besser nach UNKNOWNTYPE suchen, aber zeitintensiv
  //       Viel besser: bei initList die Position garantieren
  id=ftlist->initEnum();
  unknownft=(WCFiletype*)ftlist->getElementAt(id,UNKNOWNTYPE-1);
  ftlist->closeEnum(id);
  // Vorbesetzen, damit immer unknown typ kommt, selbst wenn Datei nicht den Anforderungen
  //  fuer Dateityperkennung genuegt
  filetype=unknownft;
  if(ftlist->size()<4) {
    return filetype;
  }
  if((S_ISDIR(mode))&&(isCorrupt==false)) {
    // Wirklich ein Verzeichnis
    return filetype;
  }
  if( ((isLink==true)&&(!S_ISREG(dmode))) || ((isLink==false)&&(!S_ISREG(mode)))) {
    // Keine regulaere Datei
    return filetype;
  }
  if(size<1) {
    return filetype;
  }
#ifdef HAVE_LIBZ
  #define fopen gzopen
  #define fclose gzclose
  #define fread(B,N,L,F) gzread(F,B,(L)*(N))
#endif
  if(skipcontent==false) {
    fp=fopen(fullname,"r");
    if(fp!=NULL) {
      reads=fread(fbuffer,1,64,fp);
      fclose(fp);
      id=ftlist->initEnum();
      ft=(WCFiletype*)ftlist->getFirstElement(id);
      entry=0;
      found=-1;
      WCFiletype *foundft=NULL;
      short *filedesc;
      int fileusage = 0;
      
      while(ft!=NULL) {
        if(ft->getinternID()==NORMALTYPE) {
          if(ft->getUseFiledesc()==true) {
            filedesc=ft->getFiledesc();
            for(x=0;x<64;x++) {
              if(filedesc[x]>-1) minsize=x+1;
            }
            if(reads>=minsize) {
              y=0;
              for(x=0;x<reads;x++) {
                if(filedesc[x]>=0) {
                  if(filedesc[x]!=(short)fbuffer[x]) {
                    y=1;
                    break;
                  }
                }
              }
              if(y==0) {
                foundft=ft;
                break;
              }
            }
          }
          if(ft->getUsePattern()==true) {
            if(match(ft->getPattern())==true) {
              foundft=ft;
              break;
            }
          }
          if ( ft->getUseFileExp() == true ) {
            fileusage++;
          }
        }
        ft=(WCFiletype*)ftlist->getNextElement(id);
      }

#if 0
      // this code is functional and does exactly what it should
      // but it's just too slow and cpu-eating
      if ( ( foundft == NULL ) && ( fileusage > 0 ) ) {
        // not found

        outname = Datei::createTMPName();
        filestr = (char*)_allocsafe( sizeof(char) * ( strlen( FILE_COMMAND ) + strlen( fullname ) + 1 ) );
        sprintf( filestr, FILE_COMMAND, fullname );

        exestr = filestr;
        exestr += " > ";
        exestr += outname;
        _freesafe( filestr );
    
        // get the output of file
        system( exestr.c_str() );
        fileoutput = NULL;
        fp = fopen( outname, "r" );
        if ( fp != NULL ) {
          fileoutput = (char*)_allocsafe( sizeof(char) * FILEOUTPUTLENGTH );
          reads = fread( fileoutput, 1, FILEOUTPUTLENGTH - 1, fp );
          fileoutput[reads] = '\0';
          fclose( fp );
        }
    
        ft = (WCFiletype*)ftlist->getFirstElement( id );
        while ( ft != NULL ) {
          if ( ft->getinternID() == NORMALTYPE ) {
            if ( ft->getUseFileExp() == true ) {
              if ( ft->fileExpMatchString( fileoutput ) == true ) {
                foundft = ft;
                break;
              }
            }
          }
          ft = (WCFiletype*)ftlist->getNextElement( id );
        }
        remove( outname );
        _freesafe( outname );
        if ( fileoutput != NULL ) _freesafe( fileoutput );
      }
#endif

      if(foundft==NULL) {
        filetype=unknownft;
      } else {
        filetype=foundft;
      }
      ftlist->closeEnum(id);
    }
  } else {
    id=ftlist->initEnum();
    ft=(WCFiletype*)ftlist->getFirstElement(id);
    entry=0;
    found=-1;
    WCFiletype *foundft=NULL;
    while(ft!=NULL) {
      if(ft->getinternID()==NORMALTYPE) {
        if(ft->getUsePattern()==true) {
          if(match(ft->getPattern())==true) {
            foundft=ft;
            break;
          }
        }
      }
      ft=(WCFiletype*)ftlist->getNextElement(id);
    }
    if(foundft==NULL) {
      filetype=unknownft;
    } else {
      filetype=foundft;
    }
    ftlist->closeEnum(id);
  }
//printf("end check\n");
  return filetype;
#ifdef HAVE_LIBZ
  #undef fopen
  #undef fclose
  #undef fread
#endif
}

#undef FILEOUTPUTLENGTH

bool FileEntry::match(char* pattern)
{
  if(fnmatch(pattern,name,0)==0) return true;
  return false;
}

int Datei::overreadChunk()
{
  int chunksize=getInt();
  while(chunksize>0) {
    getUChar();
    chunksize--;
  }
  return 0;
}

int Datei::getIntSize()
{
  return 5;
}

int Datei::getUCharSize()
{
  return 1;
}

int Datei::getUShortSize()
{
  return 2;
}

int Datei::getULongSize()
{
  return 4;
}

long Datei::errors()
{
  long e=error;
  error=0;
  return e;
}

char *ParentDir(const char *pathstr,char *dirnamereturn,int returnsize)
{
  char *newpath;
  int x,y;
  x=strlen(pathstr);
  if(x<2) {
    newpath=(char*)_allocsafe(2);
    strcpy(newpath,"/");
    if(dirnamereturn!=NULL) strcpy(dirnamereturn,"");
  } else {
    if(pathstr[x-1]=='/') y=x-2; else y=x-1;
    while((pathstr[y]!='/')&&(y>0)) {
      y--;
    }
    if(y==0) {
      newpath=(char*)_allocsafe(2);
      strcpy(newpath,"/");
      if(dirnamereturn!=NULL) {
        strncpy(dirnamereturn,pathstr+1,returnsize);
        dirnamereturn[returnsize-1]=0;
      }
    } else {
      newpath=(char*)_allocsafe(y+1);
      strncpy(newpath,pathstr,y);
      if(dirnamereturn!=NULL) {
        strncpy(dirnamereturn,pathstr+y+1,returnsize);
        dirnamereturn[returnsize-1]=0;
      }
      newpath[y]=0;
    }
  }
  return newpath;
}

int Verzeichnis::getUseSize()
{
  int size=0;
  if(files!=NULL) {
    int id=files->initEnum();
    FileEntry *fe=(FileEntry*)files->getFirstElement(id);
    while(fe!=NULL) {
      if(fe->use==true) size++;
      fe=(FileEntry*)files->getNextElement(id);
    }
    files->closeEnum(id);
  }
  return size;
}

char *Datei::getFilenameFromPath(const char*path)
{
  char *tstr;
  int pos;
  if(path==NULL) return NULL;
  pos=strlen(path)-2;  // -2 because a name contains min. 1 character
                       // but it can be a slash so skiping it
  while(pos>=0) {
    if(path[pos]=='/') break;
    pos--;
  }
  if(pos>=0) tstr=dupstring(&path[pos+1]);
  else tstr=dupstring(path);
  return tstr;
}

char *Datei::getNameWithoutExt(const char *name)
{
  char *tstr;
  int pos,len;
  bool found=false;
  if(name==NULL) return NULL;
  len=strlen(name);
  pos=len-1;
  while(pos>=0) {
    if((name[pos]=='/')&&(pos<(len-1))) break;
    if(name[pos]=='.') {
      found=true;
      break;
    }
    pos--;
  }
  if(found==false) tstr=dupstring(name);
  else {
    tstr=(char*)_allocsafe(pos+1);
    if(pos>0) strncpy(tstr,name,pos);
    tstr[pos]=0;
  }
  return tstr;
}

char *Datei::createTMPName()
{
  char *returnstr,buffer[128];
  int rannr;
  
  for(;;) {
    rannr=rand();
    sprintf(buffer,"%d",rannr);
    returnstr=(char*)_allocsafe(strlen("/tmp/worker")+strlen(buffer)+1);
    sprintf(returnstr,"/tmp/worker%s",buffer);
    if(Datei::lfileExistsExt(returnstr)==Datei::D_FE_NOFILE) break;
    _freesafe(returnstr);
  }
  return returnstr;
}

int Datei::putStringExt(const char *format,const char *str)
{
  if(fp==NULL) return -1;
  fprintf(fp,format,str);
  return 0;
}

char *HandlePathExt(const char *src)
{
  char *tstr,*tstr2;
  char *newpath;
  char *buf;
  int size=4096,i,j;

  newpath=dupstring("");
  int pos=0,count;

  // resolve "."
  if(src[0]=='.') {
    if((src[1]==0)||(src[1]=='/')) {
      for(;;) {
        buf=(char*)_allocsafe(size);
        if(getcwd(buf,size)!=NULL) break;
        else if(errno!=ERANGE) {
          strcpy(buf,"/");  // use the root if getcwd doesn't report the actual
          break;
        }
        _freesafe(buf);
        size*=2;
      }
      _freesafe(newpath);
      if(src[1]=='/') newpath=catstring(buf,"/");
      else newpath=dupstring(buf);
      _freesafe(buf);
      pos=2;
    }
  }
  for(i=pos;i<(int)strlen(src);i++) {
    if(src[i]=='$') {
      // there is an env-variable to resolve
      if(i>pos) {
        // copy the chars before
        buf=(char*)_allocsafe(i-pos+1);
        strncpy(buf,src+pos,i-pos);
        buf[i-pos]=0;
        tstr=catstring(newpath,buf);
        _freesafe(newpath);
        _freesafe(buf);
        newpath=tstr;
      }
      // now find the end of the env-variable
      if(src[i+1]=='{') {
        count=1;
        // search for closing bracket
        for(j=i+2;;j++) {
          if(src[j]=='{') count++;
          else if(src[j]==0) break;
          else if(src[j]=='}') count--;
          if(count==0) {
            j++; // to point at the first char after this
            break;
          }
        }
        // j-1 is the closing bracket
        if((j-2)>=(i+2)) {
          tstr=dupstring(src+i+2);
          tstr[(j-2)-(i+2)+1]=0;
          tstr2=resolveEnv(tstr);
          _freesafe(tstr);
          tstr=catstring(newpath,tstr2);
          _freesafe(newpath);
          _freesafe(tstr2);
          newpath=tstr;
        }
      } else {
        // find the "/" or the end
        for(j=i+1;;j++) {
          if(src[j]==0) break;
          else if(src[j]=='/') break;
        }
        // j-1 is last char for the env
        if((j-1)>=(i+1)) {
          tstr=dupstring(src+i+1);
          tstr[(j-1)-(i+1)+1]=0;
          tstr2=resolveEnv(tstr);
          _freesafe(tstr);
          tstr=catstring(newpath,tstr2);
          _freesafe(newpath);
          _freesafe(tstr2);
          newpath=tstr;
        }
      }
      pos=j;
      i=pos;
    }
  }
  if(i>pos) {
    // copy the chars before
    buf=(char*)_allocsafe(i-pos+1);
    strncpy(buf,src+pos,i-pos);
    buf[i-pos]=0;
    tstr=catstring(newpath,buf);
    _freesafe(newpath);
    _freesafe(buf);
    newpath=tstr;
  }
  // now resolve the normal things like ".." "." and so on
  tstr=HandlePath(newpath);
  _freesafe(newpath);
  return tstr;
}

char *resolveEnv(const char* str)
{
  char *resstr,*tstr,*tstr2;
  char *buf;
  int i,j;

  resstr=dupstring("");
  int pos=0,count;

  for(i=pos;i<(int)strlen(str);i++) {
    if(str[i]=='$') {
      // there is an env-variable to resolve
      if(i>pos) {
        // copy the chars before
        buf=(char*)_allocsafe(i-pos+1);
        strncpy(buf,str+pos,i-pos);
        buf[i-pos]=0;
        tstr=catstring(resstr,buf);
        _freesafe(resstr);
        _freesafe(buf);
        resstr=tstr;
      }
      // now find the end of the env-variable
      if(str[i+1]=='{') {
        count=1;
        // search for closing bracket
        for(j=i+2;;j++) {
          if(str[j]=='{') count++;
          else if(str[j]==0) break;
          else if(str[j]=='}') count--;
          if(count==0) {
            j++; // to point at the first char after this
            break;
          }
        }
        // j-1 is the closing bracket
        if((j-2)>=(i+2)) {
          tstr=dupstring(str+i+2);
          tstr[(j-2)-(i+2)+1]=0;
          tstr2=resolveEnv(tstr);
          _freesafe(tstr);
          tstr=catstring(resstr,tstr2);
          _freesafe(resstr);
          _freesafe(tstr2);
          resstr=tstr;
        }
      } else {
        // find the "/" or the end
        for(j=i+1;;j++) {
          if(str[j]==0) break;
          else if(str[j]=='/') break;
        }
        // j-1 is last char for the env
        if((j-1)>=(i+1)) {
          tstr=dupstring(str+i+1);
          tstr[(j-1)-(i+1)+1]=0;
          tstr2=resolveEnv(tstr);
          _freesafe(tstr);
          tstr=catstring(resstr,tstr2);
          _freesafe(resstr);
          _freesafe(tstr2);
          resstr=tstr;
        }
      }
      pos=j;
    }
  }
  if(i>pos) {
    // copy the chars before
    buf=(char*)_allocsafe(i-pos+1);
    strncpy(buf,str+pos,i-pos);
    buf[i-pos]=0;
    tstr=catstring(resstr,buf);
    _freesafe(resstr);
    _freesafe(buf);
    resstr=tstr;
  }
  tstr=getenv(resstr);
  _freesafe(resstr);
  if(tstr==NULL) resstr=dupstring("");
  else resstr=dupstring(tstr);
  return resstr;
}

void Verzeichnis::setLVC4FE_DND( FieldListViewDND *lv, int row, FileEntry *fe, int *sets )
{
  setLVC4FE( lv, row, fe, sets );
  lv->replacedndtext(row,fe->name);
}

ArrayList *Verzeichnis::getInvalidEntries()
{
  return invalid_entries;
}

char *Datei::shrinkFilename( const char *str, int maxlength )
{
  char *tstr = NULL;
  int len, len1, len2, leftuse;
  bool stopnow;
  const char *lastl, *lastr, *startl, *startr, *str1;
  const char *filename;
  
  // first test for null pointer
  if ( str == NULL )
    return NULL;
  
  // now test if maxlength if big enough
  // let 5 chars be the minimum (1 at the beginning, 3 for "...", 1 at the end)
  if ( maxlength < 5 )
    return NULL;
  
  // from now we will always return a shorten string
  
  // now test if the str will fit in maxlength
  len = strlen( str );
  if( len <= maxlength )
    return dupstring( str );
  
  // now we really have to do something
  
  // first find the most important part, the filename
  // we will also take the beginning slash (if exists) to show the user
  // the filename is not shorten
  // first test: is there any slash
  // if not then shorten the str with lowlevelfunc shrinkstring
  // if:
  //   filename starts at the last slash or the next to last 
  //     if the str ends with a slash
  //   if this is longer then maxlength-5 (atleast 2 chars at the beginning and "...")
  //     then just do a shringstring because the filename will be shorten in any case
  //   otherwise:
  //     find the first dirname from left
  //     if it will fit into the string, add it
  //     otherwise take the previous string and stop
  //     do the same with the first dirname from right
  //   specialcase: filename is shorter then maxlength-5
  //     but the first dir doesnt fit, then use as much chars from it
  //     to fill maxlength
  
  str1 = str +
         len - // this is the 0 byte
         2;              // to skip a final / (if exists)
                         // since a filename is atleast 1 char this would
                         // point in the worst case at the beginning slash
  // remember that we require maxlength atleast 5 chars and so the str is
  // atleast 6 chars so this will always point into the string
  
  // now search for the slash backwards
  
  while ( *str1 != '/' ) {
    str1--;
    if ( str1 < str ) break;
  }
  
  if ( str1 < str ) {
    // no slash (or only a final one)
    return shrinkstring( str, maxlength );
  } else {
    // found filename
    // we will also take the slash into it
    filename = str1;
    
    // longer then maxlength-5 ?
    // then short with shrinkstring
    if ( ( (int)strlen( filename ) ) > ( maxlength - 5 ) )
      return shrinkstring( str, maxlength );
    
    // enough room for some additional chars
    startl = str;
    startr = filename;
    lastl = str;      // this are the last positions that will fit into maxlength
    lastr = filename;
    
    if ( startl[0] == '/' )  // starting slash => skip
      startl++;
    
    // startr is at the start filename and so points to a slash
    // but because we want to prefer the first dir from left, leave it so
    
    for ( stopnow = false; ; ) {
      // remember: there is a slash before the filename
      // so this will stop there (or before)
      while ( *startl != '/' ) {
        startl++;
      }
      // this could run over the left
      while ( *startr != '/' ) {
        startr--;
        if ( startr < str ) break;
      }
      
      if ( startl > startr ) {
        // since startr is before startl both found the same mid dir
        // this dir will not fit in maxlength because then everything would fit
        // but this cannot be because of previous checks
        // just take lastl und lastr to build new string
        stopnow = true;
      } else {
        // test if startl and/or startr will fit
        // otherwise stop
        
        len1 = startl - str + 1;
        len2 = len - ( lastr - str );  // ( lastr - str ) is the number of chars before lastr!
                                       // but because we sub it from len, don't add 1 like before
        if ( ( len1 + 3 + len2 ) <= maxlength ) {
          // new dir at the left will fit
          lastl = startl;
        }
        
        len1 = lastl - str + 1;
        len2 = len - ( startr - str );
        if ( ( len1 + 3 + len2 ) <= maxlength ) {
          // new dir at the right will fit
          lastr = startr;
        }
        
        if ( ( lastl != startl ) && ( lastr != startr ) ) {
          // nothing has changed so stop
          stopnow = true;
        }
      }
      
      if ( stopnow == true ) {
        // time to stop
        tstr = (char*)_allocsafe( sizeof(char) * ( maxlength + 1 ) );
        if ( lastl == str ) {
          // no fitting dirs found
          // so take as much as possibly
          // use maxlength - 3 - strlen( filename) chars from left
          leftuse = maxlength - 3 - strlen( filename );
          strncpy( tstr, str, leftuse );
          strncpy( tstr + leftuse, "...", 3 );
          strncpy( tstr + leftuse + 3, filename, maxlength - 3 - leftuse );
          tstr[ maxlength ] = '\0';
        } else {
          len1 = ( lastl - str + 1);
          strncpy( tstr, str, len1 );
          strncpy( tstr + len1, "...", 3 );
          len1 += 3;
          len2 = maxlength - len1;  // chars left in tstr
          strncpy( tstr + len1, lastr, len2 );
          // if there were less then len2 chars in lastr
          // then tstr will be null-terminated
          // otherwise we have to set the end
          tstr[ maxlength ] = '\0';
        }
        break;
      }
      
      startl++;
      startr--;
    }
  }
  
  return tstr;
}

