/*
 * beat.c
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * History:
 *
 * v 0.0.1 2000.05.03 by Reed Lai
 *   replace alarm() with setitimer(), beat resolution up to 1 microsecond
 *
 * v 0.0.0 2000.04.28 by Reed Lai
 *   create, use alarm() and SIGALRM, 1 second time beat resolution
 *
 */
#define BEAT_DEBUG		/* unmark to show beat debug message */

#include "beat.h"

struct itimerval itv, oitv;
unsigned int beat;
unsigned int beat2;

/*
 * beat_next
 *
 * set the next time beat
 *
 * Arguments:
 *   t        time in microseconds
 */
int beat_next(unsigned long long int t)
{
    itv.it_value.tv_sec = t / 1000000;
    itv.it_value.tv_usec = t % 1000000;

    itv.it_interval.tv_sec = itv.it_value.tv_sec;
    itv.it_interval.tv_usec = itv.it_value.tv_usec;

    return (setitimer(ITIMER_REAL, &itv, 0));
}

/*
 * beat_close
 *
 * restore original itimerval
 */
void beat_close()
{
    setitimer(ITIMER_REAL, &oitv, &itv);
}

/* beat_roger */
void beat_roger(int which)
{
    switch (which) {
    case 0:
	beat = 0;
	break;
    case 1:
	beat2 = 0;
	break;
    }
}

/*
 * beat_handler
 *
 * routine to handle time beat
 */
void beat_handler(int sig)
{
    ++beat;
    ++beat2;
}

/*
 * beat_init
 *
 * routine to handle time beat
 *
 * Arguments:
 *  usecs     time in microseconds
 */
int beat_init(unsigned long long int usecs)
{
    static struct sigaction act;

    /* clear old beat */
    itv.it_interval.tv_sec = 0;
    itv.it_interval.tv_usec = 0;

    itv.it_value.tv_sec = 0;
    itv.it_value.tv_usec = 0;

    setitimer(ITIMER_REAL, &itv, &oitv);
    beat = 0;

    /* set action for alarm */
    act.sa_handler = beat_handler;
    sigemptyset(&act.sa_mask);
    sigaddset(&act.sa_mask, SIGALRM);
    sigaction(SIGALRM, &act, NULL);

    /* rocking roll */
    return (beat_next(usecs));
}
