// wmmixer - A mixer designed for WindowMaker
// 05/09/98  Release 1.0 Beta1
// Copyright (C) 1998  Sam Hawker <shawkie@geocities.com>
// This software comes with ABSOLUTELY NO WARRANTY
// This software is free software, and you are welcome to redistribute it
// under certain conditions
// See the README file for a more complete notice.

// 02/04/02 Gordon Fraser <gordon@debian.org>
//   * GNU getopt
//   * Mousewheel support
//   * X handling rewrite


#include "wmmixer.h"

// Implementation
// --------------


int main(int argc, char **argv)
{
  scanArgs(argc, argv);
  initXWin(argc, argv);
  
  mixctl=new MixCtl(mixdev);
  
  if(!mixctl->openOK())
    fprintf(stderr,"%s : Unable to open mixer device '%s'.\n", NAME, mixdev);
  else{
    icon = new int[mixctl->getNrDevices()];
    channel = new int[mixctl->getNrDevices()];
    
    initialize_icon(mixctl->getNrDevices());
    
    for(int i=0;i<mixctl->getNrDevices();i++){
      if(mixctl->getSupport(i)){
	channel[channels]=i;
	channels++;
      }
    }
  }
  
  readFile();

  if(channels==0)
    fprintf(stderr,"%s : Sorry, no supported channels found.\n", NAME);
  else{
    checkVol(true);
    
    XEvent xev;
    XSelectInput(d_display, w_main, ButtonPressMask | ExposureMask | ButtonReleaseMask | PointerMotionMask | StructureNotifyMask);
    XSelectInput(d_display, w_icon, ButtonPressMask | ExposureMask | ButtonReleaseMask | PointerMotionMask | StructureNotifyMask);
    XMapWindow(d_display, w_main);

    bool done=false;
    while(!done){
      while(XPending(d_display)){
	XNextEvent(d_display, &xev);
	switch(xev.type){
	case Expose:
	  repaint();
	  break;
	case ButtonPress:
	  pressEvent(&xev.xbutton);
	  break;
	case ButtonRelease:
	  releaseEvent(&xev.xbutton);
	  break;
	case MotionNotify:
	  motionEvent(&xev.xmotion);
	  break;
	case ClientMessage:
	  if(xev.xclient.data.l[0]==deleteWin)
	    done=true;
	  break;
	}
      }
      
      if(btnstate & (BTNPREV | BTNNEXT)){
	rpttimer++;
	if(rpttimer>=RPTINTERVAL){
	  if(btnstate & BTNNEXT)
	    curchannel++;
	  else
	    curchannel--;
	  if(curchannel<0)
	    curchannel=channels-1;
	  if(curchannel>=channels)
	    curchannel=0;
	  checkVol(true);
	  rpttimer=0;
	}
      }
      else
	checkVol(false);
      XFlush(d_display);
      usleep(50000);
    }
  }
  freeXWin();
  delete mixctl;
  return 0;
}

void initialize_icon(int num) {

   int i;

   icon[0] = 0;
   icon[1] = 7;
   icon[2] = 8;
   icon[3] = 2;
   icon[4] = 1;
   icon[5] = 6;
   icon[6] = 4;
   icon[7] = 5;
   icon[8] = 3;
   for(i=9;i<num;i++)
       icon[i] = 9;
}

void initXWin(int argc, char **argv){
  int d_depth;
  int screen;
  int x_fd;
  int dummy = 0;

  XWMHints wmhints;
  XSizeHints shints;
  XClassHint classHint;
  XTextProperty	name;
  char *wname = argv[0];
  XGCValues gcv;
  unsigned long gcm;
  
  winsize=astep ? ASTEPSIZE : NORMSIZE;
  
  if((d_display=XOpenDisplay(display))==NULL) {
    fprintf(stderr,"%s : Unable to open X display '%s'.\n", NAME, XDisplayName(display));
    exit(1);
  }

  screen  = DefaultScreen(d_display);
  w_root  = RootWindow(d_display, screen);
  d_depth = DefaultDepth(d_display, screen);
  x_fd    = XConnectionNumber(d_display);
  
  _XA_GNUSTEP_WM_FUNC=XInternAtom(d_display, "_GNUSTEP_WM_FUNCTION", false);
  deleteWin=XInternAtom(d_display, "WM_DELETE_WINDOW", false);

  w_root = RootWindow(d_display, screen);

  shints.x = 0;
  shints.y = 0;
  //  shints.flags  = USSize;
  shints.flags  = 0; // Gordon
  
  back_pix = getColor("white");
  fore_pix = getColor("black");
   
  bool pos=(XWMGeometry(d_display, DefaultScreen(d_display), position, NULL, 0, &shints, &shints.x, &shints.y,
			&shints.width, &shints.height, &dummy) & (XValue | YValue));
  shints.min_width   = winsize;
  shints.min_height  = winsize;
  shints.max_width   = winsize;
  shints.max_height  = winsize;
  shints.base_width  = winsize;
  shints.base_height = winsize;
  shints.width       = winsize;
  shints.height      = winsize;
  shints.flags=PMinSize | PMaxSize | PBaseSize; // Gordon


  w_main = XCreateSimpleWindow(d_display, w_root, shints.x, shints.y,
			    shints.width, shints.height, 0, fore_pix, back_pix);
  
  w_icon = XCreateSimpleWindow(d_display, w_root, shints.x, shints.y,
				shints.width, shints.height, 0, fore_pix, back_pix);
  
  XSetWMNormalHints(d_display, w_main, &shints);
  
  
  wmhints.icon_x = shints.x;
  wmhints.icon_y = shints.y;
  
  if(wmaker || astep || pos)
    shints.flags |= USPosition;

  if(wmaker){
    wmhints.initial_state = WithdrawnState;
    wmhints.flags = StateHint | IconWindowHint | IconPositionHint | WindowGroupHint;
    wmhints.icon_window = w_icon;
    
    wmhints.icon_x = shints.x;
    wmhints.icon_y = shints.y;
    wmhints.window_group = w_main;
  } else {
    wmhints.initial_state = NormalState;
    wmhints.flags = WindowGroupHint | StateHint;
  }
  
  classHint.res_name=NAME;
  classHint.res_class=CLASS;
  
  XSetClassHint(d_display, w_main, &classHint);
  XSetClassHint(d_display, w_icon, &classHint);
   

  if (XStringListToTextProperty(&wname, 1, &name) == 0) {
    fprintf(stderr, "%s: can't allocate window name\n", wname);
    exit(1);
   }
  
  XSetWMName(d_display, w_main, &name);
  
  gcm = GCForeground | GCBackground | GCGraphicsExposures;
  gcv.graphics_exposures = 0;
  gcv.foreground = fore_pix;
  gcv.background = back_pix;
  gc_gc=XCreateGC(d_display, w_root, gcm, &gcv);
  
  XSetWMHints(d_display, w_main, &wmhints);
  
  XSetCommand(d_display, w_main, argv, argc);
  
  XSetWMProtocols(d_display, w_main, &deleteWin, 1); // Close
  
  color[0]=mixColor(ledcolor, 0, backcolor, 100);
  color[1]=mixColor(ledcolor, 100, backcolor, 0);
  color[2]=mixColor(ledcolor, 60, backcolor, 40);
  color[3]=mixColor(ledcolor, 25, backcolor, 75);
  
  XpmAttributes xpmattr;
  XpmColorSymbol xpmcsym[4]={{"back_color",     NULL, color[0]},
			     {"led_color_high", NULL, color[1]},
			     {"led_color_med",  NULL, color[2]},
			     {"led_color_low",  NULL, color[3]}};
  xpmattr.numsymbols = 4;
  xpmattr.colorsymbols=xpmcsym;
  xpmattr.exactColors=false;
  xpmattr.closeness=40000;
  xpmattr.valuemask=XpmColorSymbols | XpmExactColors | XpmCloseness;
  
  
  XpmCreatePixmapFromData(d_display, w_root, wmmixer_xpm, &pm_main, &pm_mask, &xpmattr);
  XpmCreatePixmapFromData(d_display, w_root, tile_xpm, &pm_tile, NULL, &xpmattr);
  XpmCreatePixmapFromData(d_display, w_root, icons_xpm, &pm_icon, NULL, &xpmattr);
  XpmCreatePixmapFromData(d_display, w_root, norec_xpm, &pm_nrec, NULL, &xpmattr);
  pm_disp = XCreatePixmap(d_display, w_root, 64, 64, d_depth);
  
  if(wmaker || ushape || astep) {
    XShapeCombineMask(d_display, w_icon, ShapeBounding, winsize/2-32, winsize/2-32, pm_mask, ShapeSet);
    XShapeCombineMask(d_display, w_main, ShapeBounding, winsize/2-32, winsize/2-32, pm_mask, ShapeSet);
  } else {
    XCopyArea(d_display, pm_tile, pm_disp, gc_gc, 0, 0, 64, 64, 0, 0);
  }
  
  
  XSetClipMask(d_display, gc_gc, pm_mask);
  XCopyArea(d_display, pm_main, pm_disp, gc_gc, 0, 0, 64, 64, 0, 0);
  XSetClipMask(d_display, gc_gc, None);
  
  XStoreName(d_display, w_main, NAME);
  XSetIconName(d_display, w_main, NAME); 
}

void freeXWin(){
  XFreeGC(d_display, gc_gc);
  XFreePixmap(d_display, pm_main);
  XFreePixmap(d_display, pm_tile);
  XFreePixmap(d_display, pm_disp);
  XFreePixmap(d_display, pm_mask);
  XFreePixmap(d_display, pm_icon);
  XFreePixmap(d_display, pm_nrec);

   XDestroyWindow(d_display, w_main);
   if(wmaker)
      XDestroyWindow(d_display, w_icon);
   XCloseDisplay(d_display);
}

void createWin(Window *win, int x, int y){
   XClassHint classHint;
   if((*win=XCreateSimpleWindow(d_display, w_root, x, y, winsize, winsize, 0, back_pix, fore_pix)) == 0) {
     fprintf(stderr,"Fail: XCreateSimpleWindow\n");	
     exit(-1);
   }
   

   classHint.res_name=NAME;
   classHint.res_class=CLASS;
   XSetClassHint(d_display, *win, &classHint);
}

unsigned long getColor(char *colorname){
   XColor color;
   XWindowAttributes winattr;
   XGetWindowAttributes(d_display, w_root, &winattr);
   color.pixel=0;
   XParseColor(d_display, winattr.colormap, colorname, &color);
   color.flags=DoRed | DoGreen | DoBlue;
   XAllocColor(d_display, winattr.colormap, &color);
   return color.pixel;
}

unsigned long mixColor(char *colorname1, int prop1, char *colorname2, int prop2){
   XColor color, color1, color2;
   XWindowAttributes winattr;
   XGetWindowAttributes(d_display, w_root, &winattr);
   XParseColor(d_display, winattr.colormap, colorname1, &color1);
   XParseColor(d_display, winattr.colormap, colorname2, &color2);
   color.pixel=0;
   color.red=(color1.red*prop1+color2.red*prop2)/(prop1+prop2);
   color.green=(color1.green*prop1+color2.green*prop2)/(prop1+prop2);
   color.blue=(color1.blue*prop1+color2.blue*prop2)/(prop1+prop2);
   color.flags=DoRed | DoGreen | DoBlue;
   XAllocColor(d_display, winattr.colormap, &color);
   return color.pixel;
}

void scanArgs(int argc, char **argv){
  static struct option long_opts[] = {
    {"help",       0, NULL, 'h'},
    {"version",    0, NULL, 'v'},
    {"display",    1, NULL, 'd'},
    {"geometry",   1, NULL, 'g'},
    {"withdrawn",  0, NULL, 'w'},
    {"afterstep",  0, NULL, 'a'},
    {"shaped",     0, NULL, 's'},
    {"led-color",  1, NULL, 'l'},
    {"back-color", 1, NULL, 'b'},
    {"mix-device", 1, NULL, 'm'},
    {"scrollwheel",1, NULL, 'r'},
    {NULL,         0, NULL, 0  }};
  int i, opt_index = 0;
  

  // For backward compatibility
  for(i=1; i<argc; i++) {
    if(strcmp("-position", argv[i]) == 0) {
      sprintf(argv[i], "%s", "-g");
    } else if(strcmp("-help", argv[i]) == 0) {
      sprintf(argv[i], "%s", "-h");
    } else if(strcmp("-display", argv[i]) == 0) {
      sprintf(argv[i], "%s", "-d");
    }
  }

  while ((i = getopt_long(argc, argv, "hvd:g:wasl:b:m:r:", long_opts, &opt_index)) != -1) {
    switch (i) {
    case 'h':
    case ':':
    case '?':
      usage(argv[0]);
      break;
    case 'v':
      version();
      break;
    case 'd':
      sprintf(display, "%s", optarg);
      break;
    case 'g':
      sprintf(position, "%s", optarg);
      break;
    case 'w':
      wmaker = 1;
      break;
    case 'a':
      astep = 1;
      break;
    case 's':
      ushape = 1;
      break;
    case 'l':
      sprintf(ledcolor, "%s", optarg);
      break;
    case 'b':
      sprintf(backcolor, "%s", optarg);
      break;
    case 'm':
      sprintf(mixdev, "%s", optarg);
      break;
    case 'r':
      if(atoi(optarg)>0)
	wheel_scroll = atoi(optarg);
      break;
    }
  }
  
}

void readFile(){
   FILE *rcfile;
   char rcfilen[256];
   char buf[256];
   int done;
   int current=-1;
   sprintf(rcfilen, "%s/.wmmixer", getenv("HOME"));
   if((rcfile=fopen(rcfilen, "r"))!=NULL){
      channels=0;
      do{
         fgets(buf, 250, rcfile);
         if((done=feof(rcfile))==0){
            buf[strlen(buf)-1]=0;
            if(strncmp(buf, "addchannel ", strlen("addchannel "))==0){
               sscanf(buf, "addchannel %i", &current);
               if(current>=mixctl->getNrDevices() || mixctl->getSupport(current)==false){
                  fprintf(stderr,"%s : Sorry, this channel (%i) is not supported.\n", NAME, current);
                  current=-1;
               }
               else{
                  channel[channels]=current;
                  channels++;
	       }
            }
            if(strncmp(buf, "setchannel ", strlen("setchannel "))==0){
               sscanf(buf, "setchannel %i", &current);
               if(current>=mixctl->getNrDevices() || mixctl->getSupport(current)==false){
                  fprintf(stderr,"%s : Sorry, this channel (%i) is not supported.\n", NAME, current);
                  current=-1;
               }
            }
            if(strncmp(buf, "setmono ", strlen("setmono "))==0){
               if(current==-1)
                  fprintf(stderr,"%s : Sorry, no current channel.\n", NAME);
	       else{
                  int value;
                  sscanf(buf, "setmono %i", &value);
                  mixctl->setLeft(current, value);
                  mixctl->setRight(current, value);
                  mixctl->writeVol(current);
               }
            }
            if(strncmp(buf, "setleft ", strlen("setleft "))==0){
               if(current==-1)
                  fprintf(stderr, "%s : Sorry, no current channel.\n", NAME);
	       else{
                  int value;
                  sscanf(buf, "setleft %i", &value);
                  mixctl->setLeft(current, value);
                  mixctl->writeVol(current);
	       }
            }
            if(strncmp(buf, "setright ", strlen("setright "))==0){
               if(current==-1)
                  fprintf(stderr, "%s : Sorry, no current channel.\n", NAME);
	       else{
                  int value;
                  sscanf(buf, "setleft %i", &value);
                  mixctl->setRight(current, value);
                  mixctl->writeVol(current);
	       }
            }
            if(strncmp(buf, "setrecsrc ", strlen("setrecsrc "))==0){
               if(current==-1)
                  fprintf(stderr, "%s : Sorry, no current channel.\n", NAME);
	       else
                  mixctl->setRec(current, (strncmp(buf+strlen("setrecsrc "), "true", strlen("true"))==0));
            }
         }
      }  while(done==0);
      fclose(rcfile);
      mixctl->writeRec();
   }
}

void checkVol(bool forced=true){
   mixctl->readVol(channel[curchannel], true);
   int nl=mixctl->readLeft(channel[curchannel]);
   int nr=mixctl->readRight(channel[curchannel]);
   bool nrec=mixctl->readRec(channel[curchannel], true);
   if(forced){
      curleft=nl;
      curright=nr;
      currec=nrec;
      if(nrec)
         btnstate |= BTNREC;
      else
         btnstate &= ~BTNREC;
      curshowrec=mixctl->getRecords(channel[curchannel]);
      update();
      repaint();
   }
   else{
      if(nl!=curleft || nr!=curright || nrec!=currec){
         if(nl!=curleft){
            curleft=nl;
            drawLeft();
         }
         if(nr!=curright){
            curright=nr;
            drawRight();
         }
         if(nrec!=currec){
            currec=nrec;
            if(nrec)
               btnstate |= BTNREC;
            else
               btnstate &= ~BTNREC;
            drawBtns(BTNREC);
         }
         repaint();
      }
   }
}

void pressEvent(XButtonEvent *xev) {
   int x=xev->x-(winsize/2-32);
   int y=xev->y-(winsize/2-32);
   if(x>=5 && y>=33 && x<=16 && y<=43){
      curchannel--;
      if(curchannel<0)
         curchannel=channels-1;
      btnstate |= BTNPREV;
      rpttimer=0;
      drawBtns(BTNPREV);
      checkVol(true);
      return;
   }
   if(x>=17 && y>=33 && x<=28 && y<=43){
      curchannel++;
      if(curchannel>=channels)
         curchannel=0;
      btnstate|=BTNNEXT;
      rpttimer=0;
      drawBtns(BTNNEXT);
      checkVol(true);
      return;
   }
   if(x>=37 && x<=56 && y>=8 && y<=56) {
     int v = 0;
     if(xev->button < 4) {
       v = ((60-y)*100)/(2*25);
       dragging=true;

     } else if(xev->button == 4) {
       if(x>50)
	 v = mixctl->readRight(channel[curchannel]) + wheel_scroll;
       else if(x<45)
	 v = mixctl->readLeft(channel[curchannel]) + wheel_scroll;
       else
	 v = (mixctl->readLeft(channel[curchannel]) + mixctl->readRight(channel[curchannel]))/2 + wheel_scroll;

     } else if(xev->button == 5) {
       if(x>50)
	 v = mixctl->readRight(channel[curchannel]) - wheel_scroll;
       else if(x<45)
	 v = mixctl->readLeft(channel[curchannel]) - wheel_scroll;
       else
	 v = (mixctl->readLeft(channel[curchannel]) + mixctl->readRight(channel[curchannel]))/2 - wheel_scroll;
       
     }
     if(x<=50)
       mixctl->setLeft(channel[curchannel], v);
     if(x>=45)
       mixctl->setRight(channel[curchannel], v);
     mixctl->writeVol(channel[curchannel]);
     checkVol(false);
     return;
   }
   if(x>=5 && y>=47 && x<=28 && y<=57){
      mixctl->setRec(channel[curchannel], !mixctl->readRec(channel[curchannel], false));
      mixctl->writeRec();
      checkVol(false);
   }
}

void releaseEvent(XButtonEvent *xev){
   dragging=false;
   btnstate &= ~(BTNPREV | BTNNEXT);
   drawBtns(BTNPREV | BTNNEXT);
   repaint();
}

void motionEvent(XMotionEvent *xev){
   int x=xev->x-(winsize/2-32);
   int y=xev->y-(winsize/2-32);
   if(x>=37 && x<=56 && y>=8 && dragging){
      int v=((60-y)*100)/(2*25);
      if(v<0)
         v=0;
      if(x<=50)
         mixctl->setLeft(channel[curchannel], v);
      if(x>=45)
         mixctl->setRight(channel[curchannel], v);
      mixctl->writeVol(channel[curchannel]);
      checkVol(false);
   }
}

void repaint() {
  flush_expose(w_icon);
  XCopyArea(d_display, pm_disp, w_icon, gc_gc, 0, 0, 64, 64, winsize/2-32, winsize/2-32);
  flush_expose(w_main);
  XCopyArea(d_display, pm_disp, w_main, gc_gc, 0, 0, 64, 64, winsize/2-32, winsize/2-32);

  XEvent xev;
  while(XCheckTypedEvent(d_display, Expose, &xev));
}

void update() {
  if(wmaker || ushape || astep) {
    XShapeCombineMask(d_display, w_icon, ShapeBounding, winsize/2-32, winsize/2-32, pm_mask, ShapeSet);
    XShapeCombineMask(d_display, w_main, ShapeBounding, winsize/2-32, winsize/2-32, pm_mask, ShapeSet);
  } else {
    XCopyArea(d_display, pm_tile, pm_disp, gc_gc, 0, 0, 64, 64, 0, 0);
  }

  XSetClipMask(d_display, gc_gc, pm_mask);
  XCopyArea(d_display, pm_main, pm_disp, gc_gc, 0, 0, 64, 64, 0, 0);
  XSetClipMask(d_display, gc_gc, None);
  
  XCopyArea(d_display, pm_icon, pm_disp, gc_gc, icon[channel[curchannel]]*22, 0, 22, 22, 6, 5);
  drawLeft();
  drawRight();
  drawBtns(BTNREC | BTNNEXT | BTNPREV);
}

void drawLeft() {
  XSetForeground(d_display, gc_gc, color[1]);
  for(int i=0;i<25;i++){
    if(i==(curleft*25)/100)
      XSetForeground(d_display, gc_gc, color[3]);
    XFillRectangle(d_display, pm_disp, gc_gc, 37, 55-2*i, 9, 1);
  }
}

void drawRight() {
  XSetForeground(d_display, gc_gc, color[1]);
  for(int i=0;i<25;i++){
    if(i==(curright*25)/100)
      XSetForeground(d_display, gc_gc, color[3]);
    XFillRectangle(d_display, pm_disp, gc_gc, 48, 55-2*i, 9, 1);
  }
}

void drawBtns(int btns) {
  if(btns & BTNPREV)
    drawBtn(5, 33, 12, 11, (btnstate & BTNPREV));
  if(btns & BTNNEXT)
    drawBtn(17, 33, 12, 11, (btnstate & BTNNEXT));
  if(btns & BTNREC){
    drawBtn(5, 47, 24, 11, (btnstate & BTNREC));
    if(!curshowrec)
      XCopyArea(d_display, pm_nrec, pm_disp, gc_gc, 0, 0, 6, 7, 14, 49);
    else
      XCopyArea(d_display, pm_main, pm_disp, gc_gc, 14, 49, 6, 7, 14, 49);
  }
}

void drawBtn(int x, int y, int w, int h, bool down) {
  if(!down)
    XCopyArea(d_display, pm_main, pm_disp, gc_gc, x, y, w, h, x, y);
  else {
    XCopyArea(d_display, pm_main, pm_disp, gc_gc, x, y, 1, h-1, x+w-1, y+1);
    XCopyArea(d_display, pm_main, pm_disp, gc_gc, x+w-1, y+1, 1, h-1, x, y);
    XCopyArea(d_display, pm_main, pm_disp, gc_gc, x, y, w-1, 1, x+1, y+h-1);
    XCopyArea(d_display, pm_main, pm_disp, gc_gc, x+1, y+h-1, w-1, 1, x, y);
  }
}

void usage(const char *name) {
  printf("Usage: %s [options]\n", name);
  printf("  -h,  --help                    display this help screen\n");
  printf("  -v,  --version                 display program version\n");
  printf("  -d,  --display <string>        display to use (see X manual pages)\n");
  printf("  -g,  --geometry +XPOS+YPOS     geometry to use (see X manual pages)\n");
  printf("  -w,  --withdrawn               run the application in withdrawn mode\n");
  printf("                                 (for WindowMaker, etc)\n");
  printf("  -a,  --afterstep               use smaller window (for AfterStep Wharf)\n");
  printf("  -s,  --shaped                  shaped window\n");
  printf("  -l,  --led-color <string>      use the specified color for led display\n");
  printf("  -b,  --back-color <string>     use the specified color for backgrounds\n");
  printf("  -m,  --mix-device              use specified device (rather than /dev/mixer)\n");
  printf("  -r,  --scrollwheel <number>    volume increase/decrease with mouse wheel (default: 2)\n");
  printf("\nFor backward compatibility the following obsolete options are still supported:\n");
  printf("  -help                          display this help screen\n");
  printf("  -position                      geometry to use (see X manual pages)\n");
  printf("  -display                       display to use (see X manual pages)\n");
  exit(0);
}

void version() {
  printf("wmmixer version 1.0\n");
  exit(0);
}

static int flush_expose(Window w) {  
  XEvent dummy;
  int i=0;
  
  while (XCheckTypedWindowEvent(d_display, w, Expose, &dummy))
    i++;
  
  return i;
}

