/*
 *
 *	 wmMand-1.3 (C) 1999 Mike Henderson (mghenderson@lanl.gov)
 * 
 *	- Mandelbrot explorer
 * 
 * 
 * 
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2, or (at your option)
 *	any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.	See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program (see the file COPYING); if not, write to the
 *	Free Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *		Boston, MA	02111-1307, USA
 *
 *
 * ToDo:
 *		- Activate Julia-set map button. Currently it bugs.
 *		- Colors on 8-bit displays are not yet working properly.
 *		- Tooltip giving information about the current view
 *		- Open view in XaoS
 *
 *
 * Version 1.0	- initial release, Feb. 15, 1999.
 * Version 1.1	- update by Stonehead <pspiertz@sci.kun.nl>
 *		See Debian changelog, Apr. 13, 2002.
 * Version 1.2	- update by ciotog <chris@ciotog.net>
 *		Feb 2005.
 * Version 1.2.1 - update by ciotog <chris@ciotog.net>
 *		Mar 2006.
 * Version 1.3 - update by ciotog <chris@ciotog.net>
 *		Apr 2006.
 *
 */

/*	
 *	 Includes	
 */
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <X11/X.h>
#include <err.h>

#define XK_LATIN1 1
#define XK_MISCELLANY 1
#include <X11/keysymdef.h>

#include <X11/xpm.h>
#include <sys/types.h>
#include <sys/wait.h>

#include "../wmgeneral/wmgeneral.h"
#include "Rainbow1.h"
#include "Rainbow2.h"
#include "PurpleWhite.h"
#include "BlueYellowRed.h"
#include "wmMand_master.xpm"
#include "wmMand_mask.xbm"

typedef struct {
	Display *display;
	int screen;
	Visual *visual;
	int depth;
	Colormap cmap;
	int format;
	int bitmap_pad;
	int Color[256];
	unsigned char RRR[256];
	unsigned char GGG[256];
	unsigned char BBB[256];
} DisplayInfo;

/* Delay between refreshes (in microseconds) */
#define DELAY 10000L
#define WMMAND_VERSION "1.3"
#define FALSE 0
#define TRUE	1

void ParseCMDLine(int argc, char *argv[]);
void ButtonPressEvent(XButtonEvent *xev, DisplayInfo *Info);
void KeyPressEvent(XKeyEvent *xev, DisplayInfo *Info);
void AutoZoomEvent(XButtonEvent *lastEvent, DisplayInfo *Info);
void ComputeImage(double X, double Y, int Width, int Height, double Range, unsigned char *Image);
void ComputeJulia(double X, double Y, int Width, int Height, double Range, unsigned char *Image);
void ViewLargeImage(double X, double Y, int Width, int Height, double Range, DisplayInfo *Info);
void SetColorTable(DisplayInfo *Info, int TableNumber);

const int numPalettes = 4;
const int nIterList = 6;
const int IterList[] = {64, 128, 256, 512, 1024, 2048};
int iIter;
const int smallImageSize = 56;
int largeImageSize = 540;
int autoZoomDelay = 10;

int ButtonsUp = FALSE;
int ButtonBarEvent = FALSE;
int FractalType = 1;
double Range = 2.0;
double ZOOM = 1.2;
double Center_x = 0.0;
double Center_y = 0.0;
int ColorTable;
int MaxIterations;

int WriteGIF();



/*	
 *	 main	
 */
int main(int argc, char *argv[]) {

	XImage *xim;
	XEvent event, lastEvent;
	int i, j;
	int timer = 0;
	int nextUpdate = autoZoomDelay;
	int cursorHidden = FALSE;
	int ForceUpdate = TRUE;
	unsigned char *Image;
	DisplayInfo Info;
	Cursor blankCursor;
	Pixmap blankPixmap;
	const char blankPixmapData = 0x00;
	XColor blackColor;

	MaxIterations = IterList[2];

	/* Parse any command line arguments. */
	ParseCMDLine(argc, argv);

	/* Open window */
	openXwindow(argc, argv, wmMand_master_xpm, wmMand_mask_bits, wmMand_mask_width, wmMand_mask_height);

	/* Get Display parameters */
	Info.display = display;
	Info.screen = DefaultScreen(display);
	Info.visual = DefaultVisual(display, Info.screen);
	Info.depth = DefaultDepth(display, Info.screen);
	Info.cmap = DefaultColormap(display, 0);

	/* Initialize Color Table */
	ColorTable = 3;
	SetColorTable(&Info, ColorTable);

	xim = XCreateImage(Info.display, Info.visual, Info.depth, Info.format, 0, (char *)0, smallImageSize, smallImageSize, Info.bitmap_pad, 0);
	xim->data = (char *)malloc(xim->bytes_per_line * smallImageSize);

	/* create blank cursor for hiding when using keyboard */
	blackColor.pixel = 0;	blackColor.red = 0; blackColor.blue = 0; blackColor.green = 0;
	blankPixmap = XCreateBitmapFromData(display, iconwin, &blankPixmapData, 1, 1);
	blankCursor = XCreatePixmapCursor(display, blankPixmap, blankPixmap, &blackColor, &blackColor, 0, 0);


	/* Loop until we die */

	MaxIterations = IterList[iIter];
	
	while (TRUE) {
	
		/* keep track of time */
		++timer;
	
		/* autozoom! */
		if (lastEvent.type == ButtonPress && (timer + autoZoomDelay > nextUpdate) && ButtonBarEvent == FALSE) {
			nextUpdate += autoZoomDelay;
			AutoZoomEvent(&lastEvent.xbutton, &Info);
			ForceUpdate = TRUE;
		}

		/* Process any pending X events */
		while(XPending(display)) {
			XNextEvent(display, &event);
			lastEvent = event;

			switch(event.type) {
				case Expose:
						RedrawWindow();
						break;
				case ButtonPress:
						if (cursorHidden) { /* unhide cursor */
							XUndefineCursor(display, iconwin);
							cursorHidden = FALSE;
						}
						XGrabPointer(display, iconwin, TRUE, 0, GrabModeAsync, GrabModeAsync, iconwin, None, CurrentTime);
						ButtonPressEvent(&event.xbutton, &Info);
						timer = 0;
						nextUpdate = timer + 2 * autoZoomDelay;
						ForceUpdate = TRUE;
						break;
				case ButtonRelease:
						if (cursorHidden) { /* unhide cursor */
							XUndefineCursor(display, iconwin);
							cursorHidden = FALSE;
						}
						XUngrabPointer(display, CurrentTime);
						break;
				case KeyPress:
						if (!cursorHidden) { /* hide cursor */
							XUndefineCursor(display, iconwin);
							XDefineCursor(display, iconwin, blankCursor);
							cursorHidden = TRUE;
						}
						KeyPressEvent(&event.xkey, &Info);
						ForceUpdate = TRUE;
						break;
				case EnterNotify:
						XGrabKeyboard(display, iconwin, TRUE, GrabModeAsync, GrabModeAsync, CurrentTime);
						if (Info.depth == 8) XInstallColormap(display, Info.cmap);
						break;
				case LeaveNotify:
						if (cursorHidden) { /* unhide cursor */
							XUndefineCursor(display, iconwin);
							cursorHidden = FALSE;
						}
						XUngrabKeyboard(display, CurrentTime);
						if (Info.depth == 8) XUninstallColormap(display, Info.cmap);
						break;
			}
		}
		
		if (ForceUpdate) {
			/* allocate temp memory for Image */
			Image = (unsigned char *) malloc(sizeof(unsigned char) * smallImageSize * smallImageSize);

			/* create image */
			if (FractalType == 1)
				ComputeImage(Center_x, Center_y, smallImageSize, smallImageSize, Range, Image);
			else
				ComputeJulia(Center_x, Center_y, smallImageSize, smallImageSize, Range, Image);
		
			/* Clear window. */
			copyXPMArea(70, 70, smallImageSize, smallImageSize, 4, 4);
			
			/* Paste up image. */
			for ( i=0; i<smallImageSize; ++i ) {
				for ( j=0; j<smallImageSize; ++j ) {
					XPutPixel(xim, i, j,	Info.Color[*(Image + j*smallImageSize + i)]);
					XFlush(display);
				}
			}

			XPutImage(display, wmgen.pixmap, NormalGC, xim, 0, 0, 4, 4, smallImageSize, smallImageSize);
	
			/* Paste up buttons if ButtonsUp is toggled. */
			if (ButtonsUp) {
		
				/* paste button bar */
				copyXPMArea(4, 69, smallImageSize, 10, 4, 50);
		
				/* paste maxiterations button */
				switch(MaxIterations) {
					case 64:
						copyXPMArea(75, 69, 9, 6, 19, 52);
						break;
					case 128:
						copyXPMArea(71, 77, 13, 6, 17, 52);
						break;
					case 256:
						copyXPMArea(70, 85, 14, 6, 17, 52);
						break;
					case 512:
						copyXPMArea(71, 93, 13, 6, 17, 52);
						break;
					case 1024:
						copyXPMArea(67, 101, 17, 6, 15, 52);
						break;
					case 2048:
						copyXPMArea(66, 109, 18, 6, 14, 52);
						break;
				}
			}

			RedrawWindow(); /* Make changes visible */
			free(Image); /* Free Image Memory */
		}

		ForceUpdate = 0; /* Clear ForceUpdate Flag */
		
		waitpid(-1, NULL, WNOHANG); /* clean up any children */
		
		usleep(DELAY); /* Wait for next update */
	}

	/* Quit program	(this code is never reached..) */
	free(xim->data);
	XDestroyImage(xim);
}



/*
 *	 ParseCMDLine()
 */
void ParseCMDLine(int argc, char *argv[]) {

	int	i, nIter;
 
	for (i=1; i<argc; i++) {

		if (!strcmp(argv[i], "--zoom") || !strcmp(argv[i], "-z")) {
			/* set zoom level (no checks for reasonableness) */
			ZOOM = atof(argv[++i]);
		}
		else if (!strcmp(argv[i], "--iterations") || !strcmp(argv[i], "-i")) {
			/* set initial max iterations */
			nIter = atof(argv[++i]);
			switch (nIter) {
				case 64:
						iIter = 0; 
							break;
				case 128:
						iIter = 1;
						break;
				case 256:
						iIter = 2;
						break;
				case 512:
						iIter = 3;
						break;
				case 1024:
						iIter = 4;
						break;
				case 2048:
						iIter = 5;
						break;
			}
		}
		else if (!strcmp(argv[i], "--largesize") || !strcmp(argv[i], "-l")) {
			/* set size of large image */
			largeImageSize = atof(argv[++i]);
		}
		else if (!strcmp(argv[i], "--delayzoom") || !strcmp(argv[i], "-d")) {
			/* set autozoom delay */
			autoZoomDelay = atof(argv[++i]);
		}
		else {
			printf("\nwmMand version %s\n", WMMAND_VERSION);
			printf("\nUsage: wmMand [OPTION] ...\n\n");
			printf("Options:\n");
			printf("\t-z, --zoom <factor>\t\tset zoom factor (default is 1.2)\n");
			printf("\t-i, --iterations <number>\tset initial max iterations (64 (default), 128, 256, 512, 1024, 2048)\n");
			printf("\t-l, --largesize <number>\tset size of large image (square) (default is 540)\n");
			printf("\t-d, --delayzoom <number>\tset delay for autozooming with mouse (default 10, larger values give longer delay)\n");
			printf("\t-h, --help\t\t\tdisplay help screen\n");
			exit(1);
		}
	}
}



/*
 *	This routine handles key presses. 
 *
 */
void KeyPressEvent(XKeyEvent *xev, DisplayInfo *Info) {

	if ((xev->keycode == XKeysymToKeycode(display, XK_z))
			|| (xev->keycode == XKeysymToKeycode(display, XK_Z))
			|| (xev->keycode == XKeysymToKeycode(display, XK_plus))
			|| (xev->keycode == XKeysymToKeycode(display, XK_KP_Add))
			|| (xev->keycode == XKeysymToKeycode(display, XK_KP_5))) {
		/* z, + or 5 key on numberpad: zoom in */
		Range *= 1.0/ZOOM;
	}
	else if ((xev->keycode == XKeysymToKeycode(display, XK_o))
			|| (xev->keycode == XKeysymToKeycode(display, XK_O))
			|| (xev->keycode == XKeysymToKeycode(display, XK_minus))
			|| (xev->keycode == XKeysymToKeycode(display, XK_KP_Subtract))
			|| (xev->keycode == XKeysymToKeycode(display, XK_KP_0))) {
		/* o or - key: zoom out */
		Range *= ZOOM;
	/* 
	 * Arrow keys shift the center point with the zoom level as the shift factor
	 */
 	}
	else if ((xev-> keycode == XKeysymToKeycode(display, XK_Up))
			|| (xev->keycode == XKeysymToKeycode(display, XK_KP_Up))) {
 		/* u or up arrow: shift up */
		Center_y -= (ZOOM - 1.0) * Range;
 	}
	else if ((xev-> keycode == XKeysymToKeycode(display, XK_Down))
			|| (xev->keycode == XKeysymToKeycode(display, XK_KP_Down))) {
 		/* d or down arrow: shift down */
		Center_y += (ZOOM - 1.0) * Range;
 	}
	else if ((xev-> keycode == XKeysymToKeycode(display, XK_Left))
			|| (xev->keycode == XKeysymToKeycode(display, XK_KP_Left))) {
 		/* l or left arrow: shift left */
		Center_x -= (ZOOM - 1.0) * Range;
 	}
	else if ((xev-> keycode == XKeysymToKeycode(display, XK_Right))
			|| (xev->keycode == XKeysymToKeycode(display, XK_KP_Right))) {
 		/* r or right arrow: shift right */
		Center_x += (ZOOM - 1.0) * Range;
 	}
	else if (xev->keycode == XKeysymToKeycode(display, XK_KP_Home)) {
 		/* Home or 7 on numberpad: shift up and left */
		Center_y -= (ZOOM - 1.0) * Range;
		Center_x -= (ZOOM - 1.0) * Range;
 	}
	else if (xev->keycode == XKeysymToKeycode(display, XK_KP_Page_Up)) {
 		/* PgUp or 9 on numberpad: shift up and right */
		Center_y -= (ZOOM - 1.0) * Range;
		Center_x += (ZOOM - 1.0) * Range;
 	}
	else if (xev->keycode == XKeysymToKeycode(display, XK_KP_Page_Down)) {
 		/* PgDn or 3 on numberpad: shift down and right */
		Center_y += (ZOOM - 1.0) * Range;
		Center_x += (ZOOM - 1.0) * Range;
 	}
	else if (xev->keycode == XKeysymToKeycode(display, XK_KP_End)) {
 		/* End or 1 on numberpad: shift down and left */
		Center_y += (ZOOM - 1.0) * Range;
		Center_x -= (ZOOM - 1.0) * Range;
	}
	else if ((xev-> keycode == XKeysymToKeycode(display, XK_c))
			|| (xev->keycode == XKeysymToKeycode(display, XK_C))) {
		/* c key: change color table */
		ColorTable = (ColorTable + 1) % numPalettes;
		SetColorTable(Info, ColorTable);
	}
	else if ((xev-> keycode == XKeysymToKeycode(display, XK_i))
			|| (xev->keycode == XKeysymToKeycode(display, XK_I))) {
		/* i key: change iterations level */
		iIter = (iIter + 1) % nIterList;
		MaxIterations = IterList[iIter];
	}
	else if ((xev-> keycode == XKeysymToKeycode(display, XK_r))
			|| (xev->keycode == XKeysymToKeycode(display, XK_R))) {
		/* r key: reset view to default */
		Center_x =	0.0;
		Center_y =	0.0;
		Range = 2.0;
	}
	else if ((xev-> keycode == XKeysymToKeycode(display, XK_b))
			|| (xev->keycode == XKeysymToKeycode(display, XK_B))) {
		/* b key: toggle button bar */
		ButtonsUp = !ButtonsUp;
	}
	else if ((xev-> keycode == XKeysymToKeycode(display, XK_v))
			|| (xev->keycode == XKeysymToKeycode(display, XK_V))) {
		/* v key: view big image with ImageMagic */
		ViewLargeImage(Center_x, Center_y, largeImageSize, largeImageSize, Range, Info);
	}
}



/*
 *	This routine handles button presses.
 *
 * - Left Mouse single click: make cursor position the new image center an zooms in
 * - Right Mouse single click: make cursor position the new image center an zooms out
 * - Middle Mouse single click: toggle the button bar
 *
 */
void ButtonPressEvent(XButtonEvent *xev, DisplayInfo *Info) {

	int	x, y;
	double X, Y;

	if (ButtonsUp && ((xev->x > 3)&&(xev->x < 60)&&(xev->y > 49)&&(xev->y < 60)) ) {
		/* button bar event */
		
		ButtonBarEvent = TRUE;
		
		if (xev->x < 13) {
			/* C button: change color table */
			if (xev->button == Button1)
				ColorTable = (ColorTable + 1) % numPalettes;
			else if (xev->button == Button3)
				ColorTable = (ColorTable + numPalettes - 1) % numPalettes;
			SetColorTable(Info, ColorTable);
		}
		else if (xev->x < 34) {
			/* number button: change number of iterations */
			if (xev->button == Button1)
				iIter = (iIter + 1) % nIterList;
			else if (xev->button == Button3)
				iIter = (iIter + nIterList - 1) % nIterList;
			MaxIterations = IterList[iIter];
		}
		else if (xev->x < 42) {
			/* M button: not yet defined */
		}
		else if (xev->x < 50) {
			/* R button: reset */
			Center_x =	0.0;
			Center_y =	0.0;
			Range = 2.0;
		}
		else {
			/* V button: view large image */
			ViewLargeImage(Center_x, Center_y, largeImageSize, largeImageSize, Range, Info);
		}
		return;
	}
	else if ((xev->x > 3)&&(xev->x < 60)&&(xev->y > 3)&&(xev->y < 60)) {
	
		ButtonBarEvent = FALSE;

		/* Button bar was not up, Click was on image */
		x = xev->x - 4;
		y = xev->y - 4;
	}
	else {
		/* Click was on border */
		return;
	}

	if (xev->button == Button1) {
		/*
		 * left mouse button pressed: zoom in
		 *	 compute the physical (X, Y) values of the point defined by the image (x, y)
		 *	 recenter on the point half way from center to click
		 */
		X = Center_x + Range * ((double) x - 28.0) / 28.0 ;
		Y = Center_y + Range * ((double) y - 28.0) / 28.0 ;
		Range *= 1.0/ZOOM;
		X -= Range * ((double)x - 28.0) / 28.0;
		Y -= Range * ((double)y - 28.0) / 28.0;

		Center_x = X;
		Center_y = Y;
	}
	else if (xev->button == Button2) { 
		/* middle mouse button pressed: show/hide buttons */
		ButtonsUp = !ButtonsUp;
	}
	else if (xev->button == Button3) {
		/* right mouse button pressed: zoom out
		 *	 compute the physical (X, Y) values of the point defined by the image (x, y)
		 *	 recenter on point a little more than half way from center to click
		 */
		X = Center_x + 1.1 * Range * ((double) x - 28.0) / 28.0 ;
		Y = Center_y + 1.1 * Range * ((double) y - 28.0) / 28.0 ;
		Range *= ZOOM;
		X -= Range * ((double) x - 28.0) / 28.0;
		Y -= Range * ((double) y - 28.0) / 28.0;

		Center_x = X;
		Center_y = Y;
	}
}




/*
 *	This routine handles autozooming with the mouse
 *
 * - Left Mouse button zooms in
 * - Right Mouse button zooms out
 * - Middle Mouse button does nothing
 *
 */
void AutoZoomEvent(XButtonEvent *lastEvent, DisplayInfo *Info) {

	int	x, y;
	double X, Y;
	
	/* dummy variables for the XQueryPointer call */
	Window rr, cr;
	int rx, ry;
	unsigned int mr;

	XQueryPointer(display, iconwin, &rr, &cr, &rx, &ry, &x, &y, &mr);
	
	x -= 4;
	y -= 4;
	
	if (lastEvent->button == Button1) {
		/*
		 * left mouse button pressed: zoom in
		 *	 compute the physical (X, Y) values of the point defined by the image (x, y)
		 *	 recenter on the point half way from center to click
		 */
		X = Center_x + Range * ((double) x - 28.0) / 28.0 ;
		Y = Center_y + Range * ((double) y - 28.0) / 28.0 ;
		Range *= 1.0/ZOOM;
		X -= Range * ((double)x-28.0)/28.0;
		Y -= Range * ((double)y-28.0)/28.0;
			Center_x = X;
		Center_y = Y;
	}
	else if (lastEvent->button == Button3) { 
		/* right mouse button pressed: zoom out
		 *	 compute the physical (X, Y) values of the point defined by the image (x, y)
		 *	 recenter on point a little more than half way from center to click
		 */
		X = Center_x + 1.1 * Range * ((double) x - 28.0) / 28.0 ;
		Y = Center_y + 1.1 * Range * ((double) y - 28.0) / 28.0 ;
		Range *= ZOOM;
		X -= Range * ((double) x - 28.0) / 28.0;
		Y -= Range * ((double) y - 28.0) / 28.0;
			Center_x = X;
		Center_y = Y;
	}
}



/*
 *	 This routine computes the Mandelbrot fractal Image center on (X, Y)
 *	 of the current Image. 
 */
void ComputeImage(double X, double Y, int Width, int Height, double Range, unsigned char *Image) {

	int *IntImage;
	int i, j, done, n, nmin, nmax;
	int ncount, count, *hist, cutoff;
	double f, re, im;
	double a, b, d, a2, b2;
	double nrange, uval, HalfWidth, HalfHeight;
	
	/* Initialize hist[] */
	hist = (int *) malloc(sizeof(int) * MaxIterations);
	for (i=0; i<MaxIterations; ++i) hist[i] = 0;
	
	/* allocate memory for IntImage */
	IntImage = (int *) malloc(sizeof(int) * Width * Height);
 
	/* compute the (integer) map first */
	nmin = 9999, nmax = -9999, ncount = 0;
	HalfWidth = Width/2.0;
	HalfHeight = Height/2.0;
	f = Range/HalfWidth;

	for (i=0; i<Width; ++i) {
		re = f * ((double)i-HalfWidth) + X;

		for (j=0; j<Height; ++j) {
			im = f * ((double)j-HalfHeight) + Y;
			n = 0; a = b = 0.0;
			a2 = 0.0; b2 = 0.0;

			while ((n < MaxIterations) && ((a2 + b2) < 4.0)) {
				d = a;
				a = a2 - b2 + re;
				b = 2.0*d*b + im;
				a2 = a*a; b2 = b*b;
				++n;
			}

			if (n > nmax) nmax = n;
			if (n < nmin) nmin = n;

			if (n >= MaxIterations) {
				*(IntImage + Height*j + i) = 0;
			} else{
				*(IntImage + Height*j + i) = n;
				++ncount;
			}

			++hist[*(IntImage + Height*j + i)];
		}
	}

	/*	 Figure out what nmax should be. The problem is that you might only get
	 *	 one or two points in the upper half of the given range. A quick and dirty
	 *	 method is to find out at what n the f(n) starts to just become outliers.
	 *	 define outliers as mostly zeros. e.g. say more than 50% zero in a stretch
	 *	 that is 10 contiguous values long. Or you could count backwards until you
	 *	 get to the 99.5% level and call that the cutoff. 
	 */
	i = nmax-1, done = FALSE, count = 0, cutoff = 0;

	while((i > 0) && (done == FALSE)) {
		count += hist[i];
		if ((double)count/(double)ncount >= 0.025) {
			done = TRUE;
			cutoff = i;
		}
		--i;
	}
 
	/* 
	 *	 Then map the integer map (IntImage) into a byte map (i.e. with 256 colours) 
	 */
	uval = 0, nrange = (double)(cutoff - nmin) + 1.0;

	for (i=0; i<Width; ++i) {

		for (j=0; j<Height; ++j) {
			n = *(IntImage + Height*j + i);

			if (n == 0) {
				uval = 0;
			}
			else if (n >= cutoff) {
				uval = 255;
			}
			else if (n != 0) {
				/* 
				 *	make sure roundoff doesnt put index to 0. Otherwise it'll
				 *	look like its in the set! 
				 */
				uval = (unsigned char)(((double)(n-nmin)+1.0)/nrange * 255.0);
			}

			*(Image + Height*j + i) =	uval;
		}
	}

	free(hist);
	free(IntImage);
}



/*
 *	 This routine computes the Julia fractal Image center on (X, Y)
 *	 of the current Image. Ripped from xmand.c
 */
void ComputeJulia(double X, double Y, int Width, int Height, double Range, unsigned char *Image) {

int	 *IntImage;
int	i, j, done, n, nmin, nmax;
int	ncount, count, *hist, cutoff;
double	f, re, im;
double	a, b, d, a2, b2;
double	nrange, uval, HalfWidth, HalfHeight;
/* CX, CY refer to the c parameter needed in the Julia set iterations. */
double	CX, CY;

	/* Initialize hist[] */
	hist = (int *) malloc(sizeof(int) * MaxIterations);
	for (i=0; i<MaxIterations; ++i) hist[i] = 0;

	/* allocate memory for IntImage */
	IntImage = (int *) malloc(sizeof(int) * Width * Height);
 
	/* compute the (integer) map first */
	nmin = 9999, nmax = -9999, ncount = 0;
	HalfWidth = Width/2.0;
	HalfHeight = Height/2.0;

	CX = Range * ((double)X-HalfWidth)/HalfWidth;
		 CY = Range * ((double)Y-HalfWidth)/HalfWidth;

	f = Range/HalfWidth;
	for (i=0; i<Width; ++i) {

		re = f * ((double)i-HalfWidth) + X;
		for (j=0; j<Height; ++j) {

			im = f * ((double)j-HalfHeight) + Y;
			a = re;
			b = im;

			n = 0;
			a2 = a * a;
			b2 = b * b;

			while ((n < MaxIterations)&&((a2 + b2) < 4.0)) {
				d = a;
				a = a2 - b2 + CX;
				b = 2.0*d*b + CY;
	
				a2 = a*a; b2 = b*b;
				++n;
			}

			if (n > nmax) nmax = n;
			if (n < nmin) nmin = n;
			if (n >= MaxIterations) {
				*(IntImage + Height*j + i) = 0;
			}
			else{
				*(IntImage + Height*j + i) = n;
				++ncount;
			}
			++hist[*(IntImage + Height*j + i)];
		}
	}

	/*	 Figure out what nmax should be. The problem is that you might only get
	 *	 one or two points in the upper half of the given range. A quick and dirty
	 *	 method is to find out at what n the f(n) starts to just become outliers.
	 *	 define outliers as mostly zeros. e.g. say more than 50% zero in a stretch
	 *	 that is 10 contiguous values long. Or you could count backwards until you
	 *	 get to the 99.5% level and call that the cutoff. 
	 */
	i = nmax-1, done = FALSE, count = 0, cutoff = 0;
	while((i > 0)&&(done == FALSE)) {
		count += hist[i];
		if ((double)count/(double)ncount >= 0.025) {
			done = TRUE;
			cutoff = i;
		}
		--i;
	}
 
	/* 
	 *	 Then map the integer map (IntImage) into a byte map (i.e. with 256 colours) 
	 */
	uval = 0;
	nrange = (double)(cutoff-nmin)+1.0;
	for (i=0; i<Width; ++i) {
		for (j=0; j<Height; ++j) {
			n = *(IntImage + Height*j + i);
			if (n == 0) {
				uval = 0;
			}
			else if (n >= cutoff) {
				uval = 255;
			}
			else if (n != 0) {

				/* 
				 *	make sure roundoff doesnt put index to 0. Otherwise it'll
				 *	look like its in the set! 
				 */
				uval = (unsigned char)(((double)(n-nmin)+1.0)/nrange * 255.0);
				*(Image + Height * j + i) = (uval == 0) ? 1 : uval;
			}
		}
	}
	free(hist);
	free(IntImage);
}



void ViewLargeImage(double X, double Y, int Width, int Height, double Range, DisplayInfo *Info) {

	unsigned char *Image;
	FILE *fp_gif;
	char *fp_command;
	pid_t fp_pid;

	/* allocate memory for image data */
	Image = (unsigned char *)malloc(sizeof(unsigned char)*Height*Width);

	/* allocate memory for the command */
	fp_command = (char *)malloc(sizeof(char)*80);

	/* create large fractal */
	if (FractalType == 1)
		ComputeImage(X, Y, Height, Width, Range, Image);
	else
		ComputeJulia(X, Y, Height, Width, Range, Image);

	/* Build command */
	sprintf(fp_command, "display -title wmMand_Re%3.3E_Im%3.3E_Ra%3.3E.gif -", X, Y, Range);
	
	if ((fp_pid = fork()) == -1)
		fprintf(stderr, "wmMand: fork error\n");
	else {
		if (fp_pid == (pid_t) 0) { // Child process, which just handles 'display'
			if ((fp_gif = popen(fp_command, "w")) < 0 ) {
				fprintf(stderr, "wmMand: error opening 'display' program\n");
				exit(EXIT_FAILURE);
			}
			else {
				WriteGIF(fp_gif, Image, 0, Width, Height, Info->RRR, Info->GGG, Info->BBB, 256, 0, "");
			}
			fclose(fp_gif);
			exit(EXIT_SUCCESS);
		}
	}

	/* free Image memory and command string */
	free(fp_command);
	free(Image);
}



void SetColorTable(DisplayInfo *Info, int TableNumber) {

	XColor xColor, xColors[256];
	int i;

	switch(TableNumber) {
		case 0:
			for (i=0; i<256; ++i) {
				Info->RRR[i] = Rainbow2_Red[i];
				Info->GGG[i] = Rainbow2_Grn[i];
				Info->BBB[i] = Rainbow2_Blu[i];
			}
			break;
		case 1:
			for (i=0; i<256; ++i) {
				Info->RRR[i] = Rainbow1_Red[i];
				Info->GGG[i] = Rainbow1_Grn[i];
				Info->BBB[i] = Rainbow1_Blu[i];
			}
			break;
		case 2:
			for (i=0; i<256; ++i) {
				Info->RRR[i] = PurpleWhite_Red[i];
				Info->GGG[i] = PurpleWhite_Grn[i];
				Info->BBB[i] = PurpleWhite_Blu[i];
			}
			break;
		case 3:
			for (i=0; i<256; ++i) {
				Info->RRR[i] = BlueYellowRed_Red[i];
				Info->GGG[i] = BlueYellowRed_Grn[i];
				Info->BBB[i] = BlueYellowRed_Blu[i];
			}
			break;
	}

	/*
	 *	 Create an XImage with null data. Then allocate space for data. 
	 */
	Info->format = ZPixmap;

	if (Info->depth == 8) {

		Info->bitmap_pad = 8;
		/*
		 *	 Set a private colormap
		 */
		Info->cmap = XCreateColormap(Info->display, RootWindow(Info->display, Info->screen), Info->visual, AllocAll);

		for (i=0; i<256; ++i) {
			Info->Color[i] = i;
			xColors[i].pixel = i;
			xColors[i].red	 = (unsigned short) Info->RRR[i] << 8;
			xColors[i].green = (unsigned short) Info->GGG[i] << 8;
			xColors[i].blue	= (unsigned short) Info->BBB[i] << 8;
			xColors[i].flags = DoRed | DoGreen | DoBlue;
		}

		XStoreColors(Info->display, Info->cmap, xColors, 256);
		XSetWindowColormap(Info->display, win, Info->cmap);
	}
	else if (Info->depth > 8) {

		/* Allocate Colors */
		for (i=0; i<256; ++i) {
			xColor.red	 = (unsigned short) Info->RRR[i] << 8;
			xColor.green = (unsigned short) Info->GGG[i] << 8;
			xColor.blue	= (unsigned short) Info->BBB[i] << 8;
			xColor.flags = DoRed | DoGreen | DoBlue;
			XAllocColor(Info->display, Info->cmap, &xColor);
			Info->Color[i] = xColor.pixel;
		}

		Info->bitmap_pad = 32;
	}
	else {
		fprintf(stderr, "wmMand: Need at least 8-bit display!\n");
		exit(-1);
	}
}
