// This is the wiplc program. Most of the intelligence of this program is
// in the wiplcexpr module. The module here is compareativly simple and is
// mostly concerned with doing simple input/output.

#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>
#include <unistd.h>

#include "common.h"
#include "shmem.h"
#include "wiplcexpr.h"  

enum exprtype { EXPR,MAC,IP1,IP2,IP3,IP4 };

//-----------------------------------------------------------------------------
// Base class used for outputing a table:

// NOTE: 
//   If you whish to extend this program with your own output format
//   see the XmlOutput class for an example on how to do it.

class Output {
  private:
  // Used for sorting:
  int* base_sorted;
  static const double* val;
  static const ip_addr* ip;
  static const mac_addr* mac;
  static int rv;
  
  static inline int rev(int a) {
    return rv ? -a : a;
  }
  
  static int sort_val(const void* a, const void* b) {
    double v1=val[*((const int*)a)];
    double v2=val[*((const int*)b)];
    if(v1<v2) return rev(-1);    
    if(v1>v2) return rev(1);

    // Put non numbers in the end of the list:
    if(isnan(v1) || isinf(v1)) {
      return 1;
    } else if(isnan(v2) || isinf(v1)) {
      return -1;
    }
    
    return 0;
  }
  
  static int sort_mac(const void* a, const void* b) {
    return rev( mac_addr::compare(mac[*((const int*)a)],mac[*((const int*)b)]) );
  }
  
  static int sort_ip(const void* a, const void* b) {
    return rev( ip_addr::compare(ip[*((const int*)a)],ip[*((const int*)b)]) );
  }

  protected:
  // Parameters - can be freely read by subclasses:
  int clientcolc;
  exprtype* et;
  int* digits1;
  int* digits2;
  char** titels;
  char** units;
  char* outfile;
  char* head;
  int sortorder;
  int sortrev;
  int clearscr;
  int period_len;
  int prnt_allzero;
  
  // When a subclass want to write a table it should call this to get a
  // filehandle. Retruns 0 and sets parameter on error:
  FILE* startPage(char*& error);
  
  // When finishing writing a table this should be called.
  void endPage(FILE* f);
  
  // Returns an array with the sequence the rows should be printed in
  // according to the sorting order selected by the user.
  int* sort(const double** colums, const ip_addr*, const mac_addr*, int cardc);
  
  // Tells if a given row has non non-zero values:
  int allZero(const double** colums, int row);

  // Can be freely used in subclasses to calculate a total row:
  double* total;
    
  public:

  // This is called to write table. It should return an error message or 0.
  virtual char* writeInfo(const double** colums, const ip_addr*, 
                          const mac_addr*, int cardc, 
			  int printfrom, Time from, Time to)=0;
  
  // Initialices this class with configuration information. 
  // See main method for a description of the variables.
  void init(int maxcardc, int clientcolc,
            exprtype* et, int* digits1,int* digits2,
      	    char** titels, char** units, char* outfile,
   	    char* head, int sortorder, int sortrev, int clearscr,
	    int period_len, int prnt_allzero) {
    this->clientcolc=clientcolc;
    this->et=et;
    this->digits1=digits1;	 
    this->digits2=digits2;	 
    this->titels=titels;
    this->units=units;
    this->outfile=outfile;
    this->sortorder=sortorder;
    this->sortrev=sortrev;
    this->head=head;
    this->clearscr=clearscr;
    this->period_len=period_len;
    this->prnt_allzero=prnt_allzero;
    base_sorted=new int[maxcardc];
    total=new double[clientcolc];    
    super_inited();
  }
  
  // Caled by init. Subclasses can do their own initialization
  // in this:
  virtual void super_inited() {}
};

// Declaration of static values:
const double*   Output::val;
const ip_addr*  Output::ip;
const mac_addr* Output::mac;
int             Output::rv;

FILE* Output::startPage(char*& error) {
  FILE* f;
  if(outfile==0)
    f=stdout;
  else
    f=fopen(outfile,"w");
  if(!f) error="Error opening output file";
  else if(clearscr) fprintf(f,"\E[H\E[J");  
  fflush(f);
  return f;
}

void Output::endPage(FILE* f) {
  if(outfile!=0 && f) 
    fclose(f);
  else
    fflush(f);
}
  
int* Output::sort(const double** colums, const ip_addr* ip,
                  const mac_addr* mac, int cardc) {
  for(int i=0; i<cardc; i++) base_sorted[i]=i;
  
  rv=sortrev;
  
  if(sortorder==-1) 
    return base_sorted;
    
  if(et[sortorder]==MAC) {
    this->mac=mac;
    qsort(base_sorted,cardc,sizeof(int),sort_mac);
  } else if(et[sortorder]==EXPR) {
    this->val=colums[sortorder];
    qsort(base_sorted,cardc,sizeof(int),sort_val);
  } else {
    this->ip=ip;
    qsort(base_sorted,cardc,sizeof(int),sort_ip);
  }
  
  return base_sorted;
}

inline int Output::allZero(const double** cols, int row) {
  int j;
  for(j=0; j<clientcolc; j++) {
    if(et[j]==EXPR && cols[j][row]!=0.0) return 0;
  }
  return 1;
}

//-----------------------------------------------------------------------------
// Simple ASCII-table output class

class AscOutput:public Output {
  // Calculated from parameters given to super class:
  int* width;       // The width of each colum without any margin
  int totdispwidth; // The total width of a row
  char* colhead;    // A line with the headlines for all the colums
  char* bar;        // A vertial line with a length of totdispwidth
  char* outbuffer;  // Temporary variable in which we build up rows
                    // before printing them.
    
  virtual void super_inited();
  
  public:
  
  virtual void writePage(FILE* f,const double** cols, const ip_addr* ip, 
                         const mac_addr* mac, int cardc, 
			 int printfrom, Time from, Time t);
};

void AscOutput::super_inited() {
  // Calulate contained values:
  totdispwidth=0;
  width=new int[clientcolc];
  for(int i=0; i<clientcolc; i++) {
    int w;
    switch(et[i]) {
      case MAC: w=mac_addr::getmaxlen_tostr(); break;
      case IP1: w=ip_addr::getmaxlen_tostr(1); break;
      case IP2: w=ip_addr::getmaxlen_tostr(2); break;
      case IP3: w=ip_addr::getmaxlen_tostr(3); break;
      case IP4: w=ip_addr::getmaxlen_tostr(4); break;
      case EXPR:
        w=digits1[i]+(digits2[i]==0 ? 0 : 1+digits2[i])+strlen(units[i]);
        break;
    }
    if(w<strlen(titels[i])) w=strlen(titels[i]);
    width[i]=w;
    totdispwidth+=w;
    if(i!=clientcolc-1) totdispwidth+=3;
  }
    
  // Create bar:
  bar=new char[totdispwidth+1];
  int j=0;
  for(int i=0; i<clientcolc; i++) {
    memset(bar+j,'-',width[i]);
    if(i!=clientcolc-1) {
      memcpy(bar+j+width[i],"-+-",3);
      j+=width[i]+3;
    }
  }
  bar[totdispwidth]=0;
    
  // Create outbuffer. We make it big because numbers might actually
  // be bigger than indicated by the parameters when written (which will 
  // destory the whole layout, of course)
  outbuffer=new char[totdispwidth*2+clientcolc*100];
    
  // Calulate colhead:
  colhead=new char[totdispwidth+1];
  colhead[0]='\x0';
  for(int i=0; i<clientcolc; i++) {
    sprintf(colhead+strlen(colhead),"%-*s",width[i],titels[i]);
    if(i!=clientcolc-1) strcat(colhead," | ");
  }
}

void AscOutput::writePage(FILE* f, const double** cols, const ip_addr* ip, 
                          const mac_addr* mac, int cardc, 
			  int printfrom, Time from, Time to) {
  // Write time:  
  if(printfrom) {
    fprintf(f,"Start time: %s\n",from.tostr());
    fprintf(f,"End time  : %s\n\n",to.tostr());
  } else {
    fprintf(f,"Time: %s\n\n",to.tostr());  
  }
  
  // Write colum headlines:
  fprintf(f,"%s\n",colhead);
  fprintf(f,"%s\n",bar);
  
  // Reset totals:
  for(int i=0; i<clientcolc; i++) {
    total[i]=0.0;
  }
  
  // Sort:
  int* sorted=sort(cols,ip,mac,cardc);
  
  // And print all the rows:
  for(int pi=0; pi<cardc; pi++) {  
    // Find card we should print now:
    int i=sorted[pi];
  
    if(!prnt_allzero && allZero(cols,i)) continue;
  
    // Build up outbuffer for this row:
    outbuffer[0]='\x0';    
    int j;
    for(j=0; j<clientcolc; j++) {
      switch(et[j]) {
        case MAC:
	  sprintf(outbuffer+strlen(outbuffer),"%-*s",width[j],mac[i].tostr());
	  break;
        case IP1:
	  sprintf(outbuffer+strlen(outbuffer),"%-*s",width[j],ip[i].tostr(1));
	  break;
        case IP2:
	  sprintf(outbuffer+strlen(outbuffer),"%-*s",width[j],ip[i].tostr(2));
	  break;
        case IP3:
	  sprintf(outbuffer+strlen(outbuffer),"%-*s",width[j],ip[i].tostr(3));
	  break;
        case IP4:
	  sprintf(outbuffer+strlen(outbuffer),"%-*s",width[j],ip[i].tostr(4));
	  break;
	case EXPR:	
	  sprintf(outbuffer+strlen(outbuffer),"%*.*f%s",
	          width[j]-strlen(units[j]),digits2[j],cols[j][i],units[j]);
	  total[j]+=cols[j][i];
	  break;
      }
      if(j!=clientcolc-1) {
        strcat(outbuffer," | ");
      }
    }
    fprintf(f,"%s\n",outbuffer);
  }
  
  // Print a bar
  fprintf(f,"%s\n",bar);
  
  // Print totals:
  for(int i=0; i<cardc; i++) {  
    outbuffer[0]='\x0';    
    for(int j=0; j<clientcolc; j++) {
      if(et[j]!=EXPR) 
        sprintf(outbuffer+strlen(outbuffer),"%-*s",width[j],"");
      else
        sprintf(outbuffer+strlen(outbuffer),"%*.*f%s",
	        width[j]-strlen(units[j]),digits2[j],total[j],units[j]);
      if(j!=clientcolc-1) {
        strcat(outbuffer," | ");
      }
    }
  }
  fprintf(f,"%s\n",outbuffer);
 }

//-----------------------------------------------------------------------------
// Output in text format:

class TxtOutput:public AscOutput {
  public:
  
  virtual char* writeInfo(const double** colums, const ip_addr*, 
                          const mac_addr*, int cardc, int, Time, Time);
};

char* TxtOutput::writeInfo(const double** colums, const ip_addr* ip, 
                           const mac_addr* mac, int cardc, int printfrom,
			   Time from, Time to) {
  // Start page:
  char* error=0;
  FILE* f=startPage(error);
  if(!f) return error;
  
  // Print a title if any
  if(head) {
    fprintf(f,"%s\n\n",head);
  }
  
  // Print the table itself:
  writePage(f,colums, ip, mac, cardc, printfrom, from,to);
  
  // And stop:  
  endPage(f);
  
  return 0;
}  
  
//-----------------------------------------------------------------------------
// Output in html format:

class HtmlOutput:public AscOutput {
  public:
  
  virtual char* writeInfo(const double** colums, const ip_addr*, 
                          const mac_addr*, int cardc, int printfrom, Time, Time);
};

char* HtmlOutput::writeInfo(const double** colums, const ip_addr* ip, 
                            const mac_addr* mac, int cardc, int printfrom,
			    Time from, Time to) {
  // Start page:
  char* error=0;
  FILE* f=startPage(error);
  if(!f) return error;

  // Print header:
  fprintf(f,"<HTML>\n");
  if(head) {
    fprintf(f,"<HEAD><TITLE>%s</TITLE></HEAD>\n",head);
  }
  fprintf(f,"<BODY bgcolor=\"white\" fgcolor=\"black\">\n");
  if(period_len!=0)
    fprintf(f,"<META http-equiv=\"refresh\" content=\"%i\">\n",period_len);
  fprintf(f,"<PRE>");
  
  // Print contents:
  writePage(f,colums, ip, mac, cardc, printfrom, from, to);
    
  // Print spam line:
  fprintf(f,"\nPage generated by <A HREF=\"http://wipl-wrr.sourceforge.net\">wipl</A>.\n");
  
  // And print footer:
  fprintf(f,"</PRE></BODY></HTML>\n");
  
  // And stop:  
  endPage(f);
  
  return 0;
}  

//-----------------------------------------------------------------------------
// XML Output:

class XmlOutput:public Output {
  public:  
  virtual char* writeInfo(const double** colums, const ip_addr*, 
                          const mac_addr*, int cardc, int printfrom, 
			  Time, Time);
};

char* XmlOutput::writeInfo(const double** cols, const ip_addr* ip, 
                           const mac_addr* mac, int cardc, int printfrom, 
			   Time from, Time to) {
  // Open output file:
  char* error=0;
  FILE* f=startPage(error);
  if(!f) return error;

  // Print header:
                    fprintf(f,"<stat>\n");
                    fprintf(f,"<info>\n");
  if(head)          fprintf(f,"  <title>%s</title>\n",head);
  if(printfrom)     fprintf(f,"  <start>%s</start>\n",from.tostr());
                    fprintf(f,"  <end>%s</end>\n",to.tostr());
  if(period_len!=0) fprintf(f,"  <periodlen>%i</periodlen>\n",period_len);
                    fprintf(f,"</info>\n");

  // Reset total row.
  for(int i=0; i<clientcolc; i++) {
    total[i]=0.0;
  }
  
  // Print information about colum names:
  fprintf(f,"<cols>\n");
  for(int i=0; i<clientcolc; i++) {
    switch(et[i]) {
      case MAC: fprintf(f,"  <colmac/>\n"); break;
      case IP1: fprintf(f,"  <colip1/>\n"); break;
      case IP2: fprintf(f,"  <colip2/>\n"); break;
      case IP3: fprintf(f,"  <colip3/>\n"); break;
      case IP4: fprintf(f,"  <colip4/>\n"); break;
      case EXPR:
        fprintf(f,
	  "  <colval digit1=\"%i\" digit2=\"%i\" unit=\"%s\" title=\"%s\"/>\n",
	  digits1[i],digits2[i],units[i],titels[i]);
	break;
    }
  }
  fprintf(f,"</cols>\n");
  
  // Print the rows:
  fprintf(f,"<rows>\n");

  // Sort the rows:
  int* sorted=sort(cols,ip,mac,cardc);
  
  // Walk through all rows:
  for(int pi=0; pi<cardc; pi++) {  
    int i=sorted[pi];

    if(!prnt_allzero && allZero(cols,i)) continue;
   
    fprintf(f,"  <row>\n");
    for(int j=0; j<clientcolc; j++) {      
      switch(et[j]) {
        case MAC: fprintf(f,"    <col>%s</col>\n",mac[i].tostr()); break;
        case IP1: fprintf(f,"    <col>%s</col>\n",ip[i].tostr(1)); break;
        case IP2: fprintf(f,"    <col>%s</col>\n",ip[i].tostr(2)); break;
        case IP3: fprintf(f,"    <col>%s</col>\n",ip[i].tostr(3)); break;
        case IP4: fprintf(f,"    <col>%s</col>\n",ip[i].tostr(4)); break;
	case EXPR:fprintf(f,"    <col>%.*f</col>\n",digits2[j],cols[j][i]);
      	          total[j]+=cols[j][i]; break;
      }
    }
    fprintf(f,"  </row>\n");
  }
  
  fprintf(f,"</rows>\n");
  
  // Output the total row:
  fprintf(f,"<rowtotal>\n");
  for(int j=0; j<clientcolc; j++) {      
    if(et[j]==EXPR) {
      fprintf(f,"  <col>%*.f</col>\n",digits2[j],total[j]);
    } else {
      fprintf(f,"  <col></col>\n");
    }
  }
  fprintf(f,"</rowtotal>\n");
  fprintf(f,"</stat>\n");

  // And stop:  
  endPage(f);
  
  return 0;
}  

//-----------------------------------------------------------------------------
// The main program

main(int argc, char* argv[]) {
  // Read options into these variables:
  // Read the parameters to this variabels:
  enum {HTML,XML,TXT } format=TXT;  
                     // -f --format    Output format
  char* ipcid=DEFAULT_DAEMONFILE;  
                     // -d --daemon    Daemon file to use  
  char* outfile=0;   // -o --outfile   The file to print to or 0 on stdout 
  char* head=0;      // -t --title     A title for the tabels or 0
  int clearscr=0;    // -c --clearscr  Should a clear-page command be sent to 
                     //                the linux console before each table?
  int sortorder=-1;  // -s --sort      The client colum to sort by. -1 on no 
                     //                sorting
  int sortrev=0;     // -r --sortrev   Reverse sort order
  int period_len=0;  // -p --periodlen The length of each period in seconds.
                     //                Is 0 if we shall just print a single
		     //                table.
  int end_time=0;    // -q --quittime  The time_t time where the client should 
                     //                exit. Is 0 if it should never exit.
  int starttime=-3;  // -a --starttime Indicates which "Start time" we shall 
                     //                write:
		     //                  -3: None
		     //                  -2: Server start
		     //                  -1: Client start, 
		     //                   n, n>=0: The time as it was n 
		     //                   iterations back.
  int prnt_allzero=1;// -z --zero      Shall we print rows which doesn't have
                     //                any non-zero values?
  int clientcolums;  // Number of colums we display
  
  // Parse the option arguments:
  for(;;) {
    static option long_options[]=
      {
       {"daemon",   1,0,'d'},
       {"outfile",  1,0,'o'},
       {"clearscr", 0,0,'c'},
       {"sort",     1,0,'s'},
       {"sortrev",  0,0,'r'},
       {"periodlen",1,0,'p'},
       {"quittime", 1,0,'q'},
       {"title",    1,0,'t'},
       {"format",   1,0,'f'},
       {"starttime",1,0,'a'},
       {"zero",     0,0,'z'}
      };
     
     int c=getopt_long(argc,argv,"d:o:t:cs:rp:q:f:a:z",long_options,0);
     if(c==EOF) break;
     
     switch(c) {
       // Simple arguments:
       case 'c':
         clearscr=1;
	 break;
       case 'r':
         sortrev=1;
	 break;
       case 'z':
         prnt_allzero=0;
	 break;
              
       // Options taking an integer argument:
       case 's':
       case 'p':
       case 'q':
       case 'a':
          { 
	    int n;
  	    if(sscanf(optarg,"%i",&n)!=1 ||
	      (c=='s' && n<0) ||
	      (c=='p' && n<=0) ||
	      (c=='q' && n<=0) ||
	      (c=='a' && n<-2)) {
	      fprintf(stderr,"Number has wrong format or wrong range\n");
	      return 1;
	    }
            switch(c) {
              case 's': sortorder=n; break;
              case 'p': period_len=n; break;
  	      case 'q': end_time=n; break;
	      case 'a': starttime=n; break;
	    }
	  }
	  break;
	  
       // Options taking other arguments:
       case 'f':
	  if(!strcmp(optarg,"txt")) format=TXT;
	  else if(!strcmp(optarg,"html")) format=HTML;
	  else if(!strcmp(optarg,"xml")) format=XML;
	  else {
            fprintf(stderr,"Unkown output format: %s\n",optarg);
	    return 1;
	  }
	  break;
	  
       case 'o':
         outfile=optarg;
	 break;	  
       case 'd':
         ipcid=optarg;
	 break;	  
       case 't':
         head=optarg;
	 break;
	 
       // Error situations:
       case ':':
         fprintf(stderr,"Missing parameter for option\n");
	 return 1;
       case '?':
         return 1; // The getopt library prints an error message
	 break;	 
       default:
         // Should not happen, I guess
	 fprintf(stderr,"Error parsing commandline arguments\n");
	 return 1;
       
     }
  }
    
  // Find out how many counters to show:
  int clientcounters=argc-optind;
  
  // Check parameters for consistensi:
  if(clientcounters<=0) {
    fprintf(stderr,"At least one colum must be displayed\n");
    return 1;
  }
  
  if(sortorder>=clientcounters && sortorder!=-1) {
    fprintf(stderr,"Sort colum does not exists\n");
    return 1;  
  }
  
  if(period_len==0 && end_time!=0) {
    fprintf(stderr,"Period length must be specified when an quit time is.\n");
    return 1;
  }

  // Connect to the server: 
  ShMemClient clnt;
  char* er=clnt.init(ipcid,1);
  if(er) { fprintf(stderr,"%s\n",er); return 1; }
  
  // Lock the server:
  ServerMem* src=(ServerMem*)clnt.Lock();
  if(!src) { fprintf(stderr,"%s\n",LockErrString); return 1; }
  
  // Read server values and create WiplcExpr class:
  int maxcardc=src->maxcardc;
  Time serverstart=src->start;
  WiplcExpr we(clientcounters,src->Counterc,src->maxcardc,src->cit,
               starttime<-1 ? -1 : starttime);
  
  // Unlock server:
  clnt.UnLock(src);
  
  // Now parse the tupels given by the user into these values:  
  exprtype* et=new exprtype[clientcounters];
  char** titels=new char*[clientcounters];
  char** units=new char*[clientcounters];  
  int* digits1=new int[clientcounters];    // These are only defined for
  int* digits2=new int[clientcounters];    // colums where et is EXPR
  char** exprs=new char*[clientcounters];  //
  
  for(int i=0; i<clientcounters; i++) {
    char* s=argv[i+argc-clientcounters];
    
    if(!strcmp(s,"mac"))      { titels[i]="MAC"; units[i]=""; et[i]=MAC; }
    else if(!strcmp(s,"ip1")) { titels[i]="IP";  units[i]=""; et[i]=IP1; }
    else if(!strcmp(s,"ip2")) { titels[i]="IP";  units[i]=""; et[i]=IP2; }
    else if(!strcmp(s,"ip3")) { titels[i]="IP";  units[i]=""; et[i]=IP3; }
    else if(!strcmp(s,"ip4")) { titels[i]="IP";  units[i]=""; et[i]=IP4; }
    else {
      et[i]=EXPR;
      
      // Copy argument to new location without paranthesis:
      if(!s[0] || s[0]!='(' || s[strlen(s)-1]!=')') {
        fprintf(stderr,"Invalid specification for colum %i\n",i);
        return 1;
      }    
      char* copy=new char[strlen(s)-1];
      memcpy(copy,s+1,strlen(s)-2);
      copy[strlen(s)-2]='\x0';
    
      // Find the components
      char* t[5];
      char* tmp=copy;
      t[0]=tmp; 
      for(int j=0; j<4; j++) {
        tmp=strchr(tmp,',');
	if(tmp==0) {
	  fprintf(stderr,"Too few components in tuple for colum %i\n",i);
	  return 1;
	}
	*tmp='\x0';
	tmp++;
        t[j+1]=tmp;
      }
      
      // And save values based on them:
      if(sscanf(t[1],"%i",&digits1[i])!=1 ||
         sscanf(t[2],"%i",&digits2[i])!=1) {
           fprintf(stderr,"Error in floating point precision numbers for colum %i\n",i);
  	   return 1;
         }
      titels[i]=t[0];
      units[i]=t[3];
      exprs[i]=t[4];
    }
  }
  
  // Parse the arithmetic expressions for each colum:
  for(int i=0; i<clientcounters; i++) {
    if(et[i]==EXPR) {
      er=we.setExpression(i,exprs[i]);
      if(er) { fprintf(stderr,"%s\n",er); return 1; }
    }
  }  
  er=we.noMoreExpressions();
  if(er) { fprintf(stderr,"%s\n",er); return 1; }
  
  // Create output file:
  Output* out;
  switch(format) {
    case TXT: out=new TxtOutput();
              break;
    case XML: out=new XmlOutput();
              break;
    case HTML:out=new HtmlOutput();
              break;
  }
  
  // Do common initialization for output file:
  out->init(maxcardc,clientcounters,et,digits1,digits2,
            titels,units,outfile,head,sortorder,sortrev,clearscr,
	    period_len,prnt_allzero);

  const double** colums=new const double*[clientcounters];    
  
  Time clientstart; // Exact time for first statistic
  
  for(;;) {
    // Read information from the server:
    Time t;
    int cardc;
    ServerMem* src=(ServerMem*)clnt.Lock();
    gettimeofday(&t.tv,0);    
    if(!src) { fprintf(stderr,"%s\n",LockErrString); return 1; }
    we.nextIteration(src,t);
    cardc=src->curcardc;
    clnt.UnLock(src);
    
    // Update clienstart:
    if(clientstart.isZero()) clientstart=t;
    
    // Update expressions:
    we.updateValues();

    // Print information:
    for(int i=0; i<clientcounters; i++) {
      if(et[i]==EXPR) {
        colums[i]=we.getColum(i);
      }
    }    
        
    Time start;
    if(starttime==-2)      start=serverstart;
    else if(starttime==-1) start=clientstart;
    else if(starttime>=0)  start=we.getTimeBack(starttime);
    
    er=out->writeInfo(colums,we.getIpColum(),we.getMacColum(),
                      cardc,starttime!=-3,start,t);
    if(er) { fprintf(stderr,"%s\n",er); return 1; }
    
    // Exit if we are over-time:
    if(t.tv.tv_sec>end_time && end_time!=0 || period_len==0) break;
    
    // Else sleep:
    sleep(period_len);
  }
    
  delete out;  
}
