/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2003 Ron Steinke <rsteinke@w-link.net>

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "screenarea.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"
#include "debug_timer.h"
#include "mouse.h"

#include <cassert>
#include <cctype>
#include <typeinfo>

namespace wftk {

ScreenArea::ScreenArea() :
  RefCountObj(true),
  rect_(0,0,0,0),
  shape_(rect_),
  parent_(0),
  hidden_(true)
{
  Debug::channel(Debug::WIDGET_CREATE) << "Creating " << name() << " " << this << Debug::endl;

  show();
}

ScreenArea::~ScreenArea()
{
  Debug::channel(Debug::WIDGET_CREATE) << name() << " " << this << " has been hit for death." << Debug::endl;

  if(parent_)
    setParent(0);

  removeChildren();

  deleted.emit();
}

bool ScreenArea::show()
{
  if(!hidden_)
    return false;

  Debug::channel(Debug::INVALIDATE) << "Showing " << name() << " " << this << Debug::endl;

  hidden_ = false;

  packingUpdate(); // recalculate child widget packing
  doExpose(shape_); // and redraw everything

  return true;
}

bool ScreenArea::hide()
{
  if(hidden_)
    return false;

  Debug::channel(Debug::INVALIDATE) << "Hiding " << name() << " " << this << Debug::endl;

  hidden_ = true;
  doExpose(exposed_);

  return true;
}

void ScreenArea::raise()
{
  if(!parent_ || this == parent_->childs_.front())
    return;

  Debug::channel(Debug::INVALIDATE) << "Raising " << name() << " to the front"
	<< Debug::endl;

  Children::iterator I = parent_->find(this);
  assert(I != childs_.end());
  parent_->childs_.erase(I);
  parent_->childs_.push_front(this);
  doExpose(shape_);
}

void ScreenArea::lower()
{
  if(!parent_ || this == parent_->childs_.back())
    return;

  Debug::channel(Debug::INVALIDATE) << "Lowering " << name() << " to the back"
	<< Debug::endl;

  Children::iterator I = parent_->find(this);
  assert(I != childs_.end());
  parent_->childs_.erase(I);
  parent_->childs_.push_back(this);
  doExpose(exposed_);
}

void ScreenArea::resize(const Rect& r)
{
  Debug out(Debug::PACKING);

  out << "Resizing " << name() << " to " << r << Debug::endl;
  out << name() << " has prefered size (" << packing_info_.x.pref
	<< "," << packing_info_.y.pref << ") and minimum size ("
	<< packing_info_.x.min << "," << packing_info_.y.min << ")"
	<< Debug::endl;

  // resize() handles screen area invalidation by calling handleResize(),
  // which in turn calls setShape() through the individual widget
  // implementations. The trick is, setShape() uses exposed_ to
  // help calculate what part of the screen to invalidate. Since exposed_
  // is defined relative to the widget's origin, changing rect_
  // invalidates its value. We don't need to worry about any of
  // the other Region members besides exposed_, as setShape()
  // recalculates all of them.

  exposed_.offset(Point(rect_.x - r.x, rect_.y - r.y));

  rect_ = r;

  handleResize(r.w, r.h);

  resized(r.w, r.h);
}

void ScreenArea::setParent(ScreenArea* parent)
{
  assert(parent != this);

  if(parent == parent_)
    return;

  bool was_shown = hide();

  if(parent_) {
    Children::iterator I = parent_->find(this);
    assert(I != parent_->childs_.end());
    parent_->childs_.erase(I);
  }

  Debug out(Debug::INVALIDATE);
  out << "Setting the parent of " << name() << " " << this << " to ";
  if(parent)
    out << parent->name() << " " << parent << Debug::endl;
  else
    out << "nothing" << Debug::endl;

  bool had_parent = parent_ != 0;

  parent_ = parent;

  if(parent)
    parent->childs_.push_back(this);

  if(was_shown)
    show();

  // effectively, the parent holds a reference to the child
  if(!parent && had_parent)
    unref();
  else if(parent && !had_parent)
    ref();
}

bool ScreenArea::contains(const ScreenArea& in) const
{
  const ScreenArea* sa = &in;

  while(sa) {
    if(sa == this)
      return true;
    sa = sa->parent_;
  }

  return false;
}


ScreenArea* ScreenArea::getContainer(const Point& p)
{
  if(hidden_ || !shape_.contains(p))
    return 0;

  std::list<ScreenArea*>::const_iterator I;
  for(I = childs_.begin(); I != childs_.end(); ++I) {
    ScreenArea* w = (*I)->getContainer(p - (*I)->rect_.origin());
    if(w)
      return w; 
  }

  return this;
}

void ScreenArea::packingUpdate()
{
  Debug::channel(Debug::PACKING) << "Updating packing info for " << name()
	<< " " << this << Debug::endl;

  setPackingInfo();
  packingUpdateParent();
}

Rect ScreenArea::globalCoord (const Rect& local) const
{
  //transform local to glocal coords
  
  Rect result = parent_ ? parent_->globalCoord(local) : local;

  result.translate(rect_.origin());

  return result;
}

Rect ScreenArea::localCoord (const Rect& global) const
{
  //transform global to local coords
  
  Rect result = parent_ ? parent_->localCoord(global) : global;

  result.translate(-rect_.origin());

  return result;
}

std::string ScreenArea::name() const
{
  // This works with g++ 2.95.4, probably need lots of messy
  // platform dependent typedefs
  std::string type_name = typeid(*this).name();
/*
  std::string::size_type loc = type_name.find("wftk") + 4;
  while(true) {
    if(type_name.size() <= loc)
      return "";
    if(!isdigit(type_name[loc]))
      return type_name.substr(loc);
    ++loc;
  }
*/

  return type_name;
}

bool ScreenArea::hasMouse() const
{
  Point p = Mouse::instance()->position();

  for(const ScreenArea* sc = this; sc != 0; sc = sc->parent_)
    p -= sc->rect_.origin();

  return shape_.contains(p);
}

void ScreenArea::removeChildren()
{
  while(!childs_.empty())
    childs_.front()->setParent(0);
}

void ScreenArea::setShape(const Region& shape, const Region& coverage)
{
  shape_ = shape & Rect(0, 0, rect_.w, rect_.h);
  covered_ = shape_ & coverage;

  // expose both the new area (shape_) and the old area (exposed_)
  doExpose(shape_ | exposed_);
}

void ScreenArea::invalidate(const Region& r)
{
  Debug::channel(Debug::INVALIDATE) << "Calling invalidate for "
	<< name() << " " << this << Debug::endl;

  if(hidden_)
    return;

  // r defaults to rect_ instead of exposed_
  // because it will make (r & foo) quicker

  Region target(r);
  // only invalidate exposed parts of the widget we haven't already invalidated
  target &= exposed_ - dirtyFull_;
  if(target.empty())
    return;

  // Call invalidateRecurse() in the top level parent
  ScreenArea *top_parent = this;
  while(top_parent->parent_) {
    target.offset(top_parent->rect_.origin());
    top_parent = top_parent->parent_;
  }

  top_parent->invalidateRecurse(target);
}

void ScreenArea::invalidateRecurse(Region& invalid)
{
  Debug out(Debug::INVALIDATE);

  out << "Calling invalidateRecurse() for " << name() << " " << this << Debug::endl;

  if(hidden_)
    return;

  dirtyFull_ |= invalid & exposed_;

  if(!dirtyFull_.empty())
    out << this << " might be dirty" << Debug::endl;

  for(Children::iterator I = childs_.begin(); I != childs_.end(); ++I) {
    Point off = (*I)->rect_.origin();
    invalid.offset(-off);
    (*I)->invalidateRecurse(invalid);
    invalid.offset(off);
  }

  dirty_ |= invalid & exposed_;

  if(!dirty_.empty())
    out << this << " is dirty" << Debug::endl;

  // subtract the area this widget covers
  invalid -= covered_;
}

void ScreenArea::blit(Surface& target, const Point& offset)
{
  if(hidden_ || dirtyFull_.empty())
    return;

  Debug out(Debug::INVALIDATE);

  out << "Drawing " << name() << " " << this << ", invalid area lies inside "
	<< dirtyFull_.getClipbox() << Debug::endl;

  DebugTimer timing(Debug::DRAW_TIMING);

  if(!dirty_.empty()) {
    out << "Calling draw() for " << this << " for dirty area lying inside "
	<< dirty_.getClipbox() << Debug::endl;
    dirty_.offset(offset); // make dirty_ relative to the target
    draw(target, offset, dirty_);
    dirty_.clear();
  }

  timing.mark("Widget " + name(), "in draw()");

  // Note that blit() iterates though child widgets in the
  // opposite order of expose()
  for(Children::reverse_iterator I = childs_.rbegin(); I != childs_.rend(); ++I) {
    out << "About to draw child " << (*I)->name() << " " << *I << Debug::endl;
    out << "It is " << ((*I)->hidden_ ? "" : "not ") << "hidden, and is "
	<< ((*I)->dirtyFull_.empty() ? "not " : "") << "dirty" << Debug::endl;
    (*I)->blit(target, offset + (*I)->rect_.origin());
  }

  timing.set();

  dirtyFull_.offset(offset); // make dirtyFull_ relative to the target
  drawAfter(target, offset, dirtyFull_);
  out << "Clearing dirtyFull_ for " << this << " in blit()" << Debug::endl;
  dirtyFull_.clear();

  timing.mark("Widget " + name(), "in drawAfter()");
}

void ScreenArea::doExpose(const Region& r)
{
  // We assume here that the value of (exposed_ & r)
  // is garbage for this widget or any of its children,
  // but that (exposed_ - r) is still valid. We also
  // assume that this widget's parent (if it has one)
  // has the correct value for exposed_.

  if(parent_) {
    Region full = r;
    full.offset(getRect().origin());
    full &= parent_->exposed_;
    Region remaining = full;
    // The widget's peers may also have
    // invalid exposed_, so we need
    // to call expose() in the parent.
    parent_->expose(full, remaining);
  }
  else {
    Region full = r & shape_;
    Region remaining = full;
    expose(full, remaining);
  }

  dirtyFull_ -= r; // dirtyFull_ gets subtraced from r in invalidate()
  invalidate(r);
}

void ScreenArea::expose(Region& full, Region& remaining)
{
  // We _never_ subtract anything from full, just offset it
  // to an appropriate position for the child widgets.
  // Remaining is the amount of full that's still exposed
  // at this level, it gets subtracted from a lot.

  Debug out(Debug::INVALIDATE);

  out << "Calling expose for " << name() << " " << this << Debug::endl;

  out << "The full invalidated area lies inside " << full.getClipbox()
	<< ", and the part that remains exposed lies inside "
	<< remaining.getClipbox() << Debug::endl;
  out << "The area that has already been handled lies inside "
	<< (full - remaining).getClipbox() << Debug::endl;

  assert((remaining - full).empty());

  if(hidden_ || (full & shape_).empty())
    return;

  // This widget and its children should only have exposed area lying inside
  // shape_
  Region new_remaining(remaining & shape_);
  // remove parts that will be handled by this widget or its children
  remaining -= new_remaining;

  // The part of the widget that's exposed
  exposed_ -= full; // cut out all points in the invalidated region
  exposed_ |= new_remaining; // add those back in which we're exposed for

  out << name() << " " << this << " is ";
  if(out && exposed_ != shape_)
    out << "not ";
  out << "fully exposed" << Debug::endl;

  // The part of the widget that needs to be redrawn
  dirtyFull_ -= full;
  dirtyFull_ |= new_remaining;
  dirtyFull_ &= shape_;

  if(!dirtyFull_.empty())
    out << this << " might be dirty" << Debug::endl;

  for(Children::iterator I = childs_.begin(); I != childs_.end(); ++I) {
    Point off = (*I)->rect_.origin();
    full.offset(-off);
    new_remaining.offset(-off);
    (*I)->expose(full, new_remaining);
    full.offset(off);
    new_remaining.offset(off);
  }

  if(out && new_remaining.empty())
    out << "The children of " << this << " covered the whole invalidated area"
	<< Debug::endl;

  // The part of the widget that needs to be redrawn and will not be
  // handled by child widgets.
  dirty_ -= full;
  dirty_ |= new_remaining;

  if(!dirty_.empty())
    out << this << " is dirty" << Debug::endl;

  // Subtract from 'remaining' the parts that
  // this region will draw opaquely.
  new_remaining -= covered_;

  // and add back the parts we didn't handle
  remaining |= new_remaining;
}

ScreenArea::Children::iterator ScreenArea::find(ScreenArea *sa)
{
  if(!sa || sa->parent_ != this)
    return childs_.end();

  Children::iterator I = childs_.begin();

  while(I != childs_.end()) {
    if(*I == sa)
      break;
    ++I;
  }

  assert(I != childs_.end());

  return I;
}

Region ScreenArea::getFullObscure()
{
  Region out = exposed_ & covered_;
  for(Children::iterator I = childs_.begin(); I != childs_.end(); ++I) {
    if((*I)->hidden_)
      continue;
    Region tmp = (*I)->getFullObscure();
    tmp.offset((*I)->rect_.origin());
    out |= tmp;
  }
  return out;
}

void
ScreenArea::PackingInfo::Expander::extend(const Expander& input)
{
  pref += input.pref;
  min += input.min;

  if(input.expand) {
    expand = true;
    if(input.filler > filler + SUBTRACT)
      filler = input.filler - SUBTRACT;
  }
}

void
ScreenArea::PackingInfo::Weights::extend(const Expander& input)
{
  if(input.pref >= input.min)
    shrink_ += input.pref - input.min;
  else
    Debug::channel(Debug::PACKING) <<
	"Error: got widget with minimum size larger than prefered size" << Debug::endl;

  if(!input.expand)
    return;

  if(input.filler > high_fill_) {
    high_fill_ = input.filler;
    expand_ = 1;
  }
  else if(input.filler == high_fill_) {
    // weight is per child for filler, pref size otherwise
    expand_ += high_fill_ ? 1 : input.pref;
  }
}

void
ScreenArea::PackingInfo::Expander::contain(const Expander& input)
{
  if(input.pref > pref)
    pref = input.pref;
  if(input.min > min)
    min = input.min;

  if(input.expand) {
    expand = true;
    if(input.filler > filler + SUBTRACT)
      filler = input.filler - SUBTRACT;
  }
}

void
ScreenArea::PackingInfo::Weights::setExpand(Uint16 pref_size, Uint16 real_size)
{
  if(pref_size > real_size)
    expand_frac_ = (pref_size < real_size + shrink_)
	? -((double) (pref_size - real_size)) / shrink_
	: -1;
  else
    expand_frac_ = expand_ ? ((double) (real_size - pref_size)) / expand_ : 0;

  Debug::channel(Debug::PACKING) << "Set expand_frac_ to " << expand_frac_
	<< " in PackingInfo::Weights" << Debug::endl;
}

double
ScreenArea::PackingInfo::Weights::padding(const Expander& info) const
{
 if(expand_frac_ < 0)
   return expand_frac_ * (info.pref - info.min);
 else if(info.expand && info.filler == high_fill_)
   return high_fill_ ? expand_frac_ : (expand_frac_ * info.pref); 
 else
   return 0;
}

} // namespace wftk
