/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "lineedit.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "mixer.h"
#include "keysym.h"
#include "debug.h"

// for isprint
#include <cctype>

#include <sigc++/object_slot.h>

namespace wftk {

LineEdit::LineEdit(const std::string& text, const Font &font):
  textFont_(font.valid() ? font : Font::textFont()),
  cursorPos_(0),
  cursor_(false),
  drawCursor_(false),
  blockInput_(false),
  cursorCount_(0),
  textSurface_(NULL)
{
  textCursor_ = Rect(0,getRect().height()-5, 12, 4);

  Focus::instance()->CursorSparkle.alarm.connect(slot(*this, &LineEdit::timer));

  setClickToFocus(true);

//  setTextColor(Font::textColor());

  getResourceBackground("lineedit");

  setText(text);

  setPackingInfo();
}

LineEdit::~LineEdit()
{
  if(textSurface_ != NULL)
    delete textSurface_;
}

void
LineEdit::updateText()
{
  delete textSurface_;
  textSurface_ = textFont_.getString(text_);
  textCursor_.warp(Point(textSurface_->width() + 1, textSurface_->height()-textCursor_.height()));
}

Point
LineEdit::drawText(Surface& surf, const Point& offset, const Region& r)
{
  Debug::channel(Debug::DRAWING) << "Drawing text in LineEdit"
	<< Debug::endl;

  Point dest = offset;
  dest.y += getRect().height() - textSurface_->height() -1;

  textSurface_->blit(surf, dest, r);

  //there is no space anymore
  if(textCursor_.origin().x + 2*textCursor_.width() >= getRect().width()) {
    blockInput_ = true;
    Debug::channel(Debug::TEXT_WIDGETS) << "blocking Input, " << textCursor_.origin().x + 2*textCursor_.width() << " >= " << getRect().width() << Debug::endl;
  }

  return Point(textSurface_->width(), height() - 1);
}

void
LineEdit::draw(Surface& surf, const Point& offset, const Region& r)
{
  Widget::draw(surf, offset, r);

  Region cursor_mask(textCursor_);
  cursor_mask.offset(offset);
  if(!(r - cursor_mask).empty()) // more than just the cursor
    textCursor_.warp(drawText(surf, offset, r) + Point(1,-textCursor_.height()));

  //draw the cursor ...
  if(hasFocus()) 
    drawCursor(surf, offset, r);
}

void
LineEdit::setPackingInfo()
{
  packing_info_.y.expand = false;
  // single character height
  packing_info_.y.min = packing_info_.y.pref = textFont_.metrics().height / 64;
  // single character width
  packing_info_.x.min = textFont_.metrics().max_advance_width / 64;
  // enough space for several characters
  packing_info_.x.pref = packing_info_.x.min * 20;

  packing_info_.x.filler = PackingInfo::Expander::FILL;
}
  
bool
LineEdit::keyEvent(const SDL_keysym& sym, bool pressed)
{
  assert(hasFocus());

  bool handled = false;

  if(!pressed)
    return false;

  char input = sym.unicode & 0x7f;

  //basic editing: there are ENTER and BS 
  //nearly emacs ... 
  if(!blockInput_ && isprint(input))
    {
      text_ += std::string(1,input);
      textUpdate();
      textChanged();
      handled = true;
    }
  if((input =='\n') || (input == '\r'))
    {
      enterPressed();
      handled = true;
    }
  
  if(((int) sym.sym == (int) KEY_DELETE) || ((int) sym.sym == (int) KEY_BACKSPACE))
    {
      if(text_.length() > 0)
        {
          text_.erase(text_.length()-1,1);
          textUpdate();
          blockInput_ = false;
          textChanged();
        }
      handled = true;
    }
    
  if(handled)
    {
      //emit signal
      keyInput();
      Mixer::instance()->playSample("click");
    }
  if(sym.sym == (int) KEY_TAB)
    {
      //TAB changes focus to next one,
      //Shift+TAB to previous one
      if(sym.mod & KEY_MOD_SHIFT)
        Focus::instance()->prevFocus();
      else
        Focus::instance()->nextFocus();
      handled = true;
    }

  return handled;
}

void LineEdit::timer()
{
  if(hasFocus())
    {
      drawCursor_ = true;
      
      cursorCount_++;
      if(cursorCount_ > 15)
	cursorCount_ = 0;
      
      invalidate(textCursor_);
    }
}

void 
LineEdit::drawCursor(Surface& surf, const Point& offset, const Region& r)
{  
  cursor_ = true;

  unsigned char c = 0;
  if (cursorCount_ % 2)
    c = 128;

  Color color = Color(c,c,c);

  Region copy(textCursor_);
  copy.offset(offset);
  surf.fill(copy & r, color);
}
 
void 
LineEdit::setTextColor(const Color& fontCol)
{
  Debug::channel(Debug::TEXT_WIDGETS) << "Setting text color in LineEdit"
	<< Debug::endl;

  textFont_.setColor(fontCol);

  invalidate();
}

void LineEdit::setText(const std::string& data)
{
  Debug::channel(Debug::TEXT_WIDGETS) << "Setting text in LineEdit:"
	<< Debug::endl << data << Debug::endl;

  text_ = data;
  blockInput_ = false;
  Debug::channel(Debug::TEXT_WIDGETS) << "unblocking input\n";
  
  //we need to be reblitted !
  textUpdate();
}

} // namespace wftk
