/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#include "color.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "debug.h"
#include <iostream>
#include "string_hash_map.h"

using namespace wftk;

ResourceRegistry<Color,Color::ResLoad,Color::ResInval> Color::registry;

std::ostream& wftk::operator<<(std::ostream& s, const Color& c)
{
  s << "(R"<<(int)c.r<<" G"<<(int)c.g<<" B"<<(int)c.b<<" A"<<(int)c.a<<")";
  return s;
}

std::pair<Color,bool>
Color::ResLoad::operator()(const std::string& colordesc)
{
  unsigned int r,g,b,a;
  int conversions = 
    sscanf(colordesc.c_str(),"%d %d %d %d",&r,&g,&b,&a);

  std::pair<Color,bool> out;
  out.second = true;
  
  if(conversions == 4)
    out.first = Color(r,g,b,a);
  else if(conversions == 3)
    out.first = Color(r,g,b);
  else
    {
      Debug::channel(Debug::GENERIC) << "Resource: error parsing Colorvalue: \""
		<< colordesc << "\"" << Debug::endl;
      out.first = "blue";
    }

  return out;
}

namespace {

class Colormap
{
  public:
   Colormap();

   const Color& find(const std::string& name) const
   {
     Map::const_iterator I = map_.find(name);
     if (I != map_.end()) 
       return I->second;

     static const Color trans(0, 0, 0, Color::WFTK_TRANSPARENT);
     Debug::channel(Debug::GENERIC) << "Couldn't find color " << name
	<< ", returning transparent" << Debug::endl;
     return trans;
   }

  private:
   typedef StringHash<Color>::Map Map;
   Map map_;
};

}

const Color&
Color::find(const std::string& name)
{
  static const Colormap colormap;
  return colormap.find(name);
}

Colormap::Colormap()
{
  map_["transparent"] = Color(0, 0, 0, Color::WFTK_TRANSPARENT);

  const struct {
    const char* name;
    Uint8 r, g, b;
  } color_info[] = {
      {"black",0,0,0},
      {"darkgrey",8,8,8},
      {"darkgreen",32,64,32},
      {"darkslateblue",32,0,64},
      {"lightblue",128,0,255},
      {"grey",64,64,64},
      {"lightgrey",128,128,128},
      {"red",128,32,32},
      {"blue",32,32,128},
      {"green",32,128,32},
      {"seagreen",16,64,64},
      {"white",254,254,254},
      {"snow",255,250,250},
      {"GhostWhite",248,248,255},
      {"WhiteSmoke",245,245,245},
      {"gainsboro",220,220,220},
      {"FloralWhite",255,250,240},
      {"OldLace",253,245,230},
      {"linen",250,240,230},
      {"AntiqueWhite",250,235,215},
      {"PapayaWhip",255,239,213},
      {"BlanchedAlmond",255,235,205},
      {"bisque",255,228,196},
      {"PeachPuff",255,218,185},
      {"NavajoWhite",255,222,173},
      {"moccasin",255,228,181},
      {"cornsilk",255,248,220},
      {"ivory",255,255,240},
      {"LemonChiffon",255,250,205},
      {"seashell",255,245,238},
      {"honeydew",240,255,240},
      {"MintCream",245,255,250},
      {"azure",240,255,255},
      {"AliceBlue",240,248,255},
      {"lavender",230,230,250},
      {"LavenderBlush",255,240,245},
      {"MistyRose",255,228,225},
      {"DarkSlateGray",47,79,79},
      {"DarkSlateGrey",47,79,79},
      {"DimGray",105,105,105},
      {"DimGrey",105,105,105},
      {"SlateGray",112,128,144},
      {"SlateGrey",112,128,144},
      {"LightSlateGray",119,136,153},
      {"LightSlateGrey",119,136,153},
      {"gray",190,190,190},
      {"LightGrey",211,211,211},
      {"LightGray",211,211,211},
      {"MidnightBlue",25,25,112},
      {"navy",0,0,128},
      {"NavyBlue",0,0,128},
      {"CornflowerBlue",100,149,237},
      {"DarkSlateBlue",72,61,139},
      {"SlateBlue",106,90,205},
      {"MediumSlateBlue",123,104,238},
      {"LightSlateBlue",132,112,255},
      {"MediumBlue",0,0,205},
      {"RoyalBlue",65,105,225},
      {"DodgerBlue",30,144,255},
      {"DeepSkyBlue",0,191,255},
      {"SkyBlue",135,206,235},
      {"LightSkyBlue",135,206,250},
      {"SteelBlue",70,130,180},
      {"LightSteelBlue",176,196,222},
      {"LightBlue",173,216,230},
      {"PowderBlue",176,224,230},
      {"PaleTurquoise",175,238,238},
      {"DarkTurquoise",0,206,209},
      {"MediumTurquoise",72,209,204},
      {"turquoise",64,224,208},
      {"cyan",0,255,255},
      {"LightCyan",224,255,255},
      {"CadetBlue",95,158,160},
      {"MediumAquamarine",102,205,170},
      {"aquamarine",127,255,212},
      {"DarkGreen",0,100,0},
      {"DarkOliveGreen",85,107,47},
      {"DarkSeaGreen",143,188,143},
      {"SeaGreen",46,139,87},
      {"MediumSeaGreen",60,179,113},
      {"LightSeaGreen",32,178,170},
      {"PaleGreen",152,251,152},
      {"SpringGreen",0,255,127},
      {"LawnGreen",124,252,0},
      {"chartreuse",127,255,0},
      {"MediumSpringGreen",0,250,154},
      {"GreenYellow",173,255,47},
      {"LimeGreen",50,205,50},
      {"YellowGreen",154,205,50},
      {"ForestGreen",34,139,34},
      {"OliveDrab",107,142,35},
      {"DarkKhaki",189,183,107},
      {"khaki",240,230,140},
      {"PaleGoldenrod",238,232,170},
      {"LightGoldenrodYellow",250,250,210},
      {"LightYellow",255,255,224},
      {"yellow",255,255,0},
      {"gold",255,215,0},
      {"LightGoldenrod",238,221,130},
      {"goldenrod",218,165,32},
      {"DarkGoldenrod",184,134,11},
      {"RosyBrown",188,143,143},
      {"IndianRed",205,92,92},
      {"SaddleBrown",139,69,19},
      {"sienna",160,82,45},
      {"peru",205,133,63},
      {"burlywood",222,184,135},
      {"beige",245,245,220},
      {"wheat",245,222,179},
      {"SandyBrown",244,164,96},
      {"tan",210,180,140},
      {"chocolate",210,105,30},
      {"firebrick",178,34,34},
      {"brown",165,42,42},
      {"DarkSalmon",233,150,122},
      {"salmon",250,128,114},
      {"LightSalmon",255,160,122},
      {"orange",255,165,0},
      {"DarkOrange",255,140,0},
      {"coral",255,127,80},
      {"LightCoral",240,128,128},
      {"tomato",255,99,71},
      {"OrangeRed",255,69,0},
      {"HotPink",255,105,180},
      {"DeepPink",255,20,147},
      {"pink",255,192,203},
      {"LightPink",255,182,193},
      {"PaleVioletRed",219,112,147},
      {"maroon",176,48,96},
      {"MediumVioletRed",199,21,133},
      {"VioletRed",208,32,144},
      {"magenta",255,0,255},
      {"violet",238,130,238},
      {"plum",221,160,221},
      {"orchid",218,112,214},
      {"MediumOrchid",186,85,211},
      {"DarkOrchid",153,50,204},
      {"DarkViolet",148,0,211},
      {"BlueViolet",138,43,226},
      {"purple",160,32,240},
      {"MediumPurple",147,112,219},
      {"thistle",216,191,216},
      {"snow1",255,250,250},
      {"snow2",238,233,233},
      {"snow3",205,201,201},
      {"snow4",139,137,137},
      {"seashell1",255,245,238},
      {"seashell2",238,229,222},
      {"seashell3",205,197,191},
      {"seashell4",139,134,130},
      {"AntiqueWhite1",255,239,219},
      {"AntiqueWhite2",238,223,204},
      {"AntiqueWhite3",205,192,176},
      {"AntiqueWhite4",139,131,120},
      {"bisque1",255,228,196},
      {"bisque2",238,213,183},
      {"bisque3",205,183,158},
      {"bisque4",139,125,107},
      {"PeachPuff1",255,218,185},
      {"PeachPuff2",238,203,173},
      {"PeachPuff3",205,175,149},
      {"PeachPuff4",139,119,101},
      {"NavajoWhite1",255,222,173},
      {"NavajoWhite2",238,207,161},
      {"NavajoWhite3",205,179,139},
      {"NavajoWhite4",139,121,94},
      {"LemonChiffon1",255,250,205},
      {"LemonChiffon2",238,233,191},
      {"LemonChiffon3",205,201,165},
      {"LemonChiffon4",139,137,112},
      {"cornsilk1",255,248,220},
      {"cornsilk2",238,232,205},
      {"cornsilk3",205,200,177},
      {"cornsilk4",139,136,120},
      {"ivory1",255,255,240},
      {"ivory2",238,238,224},
      {"ivory3",205,205,193},
      {"ivory4",139,139,131},
      {"honeydew1",240,255,240},
      {"honeydew2",224,238,224},
      {"honeydew3",193,205,193},
      {"honeydew4",131,139,131},
      {"LavenderBlush1",255,240,245},
      {"LavenderBlush2",238,224,229},
      {"LavenderBlush3",205,193,197},
      {"LavenderBlush4",139,131,134},
      {"MistyRose1",255,228,225},
      {"MistyRose2",238,213,210},
      {"MistyRose3",205,183,181},
      {"MistyRose4",139,125,123},
      {"azure1",240,255,255},
      {"azure2",224,238,238},
      {"azure3",193,205,205},
      {"azure4",131,139,139},
      {"SlateBlue1",131,111,255},
      {"SlateBlue2",122,103,238},
      {"SlateBlue3",105,89,205},
      {"SlateBlue4",71,60,139},
      {"RoyalBlue1",72,118,255},
      {"RoyalBlue2",67,110,238},
      {"RoyalBlue3",58,95,205},
      {"RoyalBlue4",39,64,139},
      {"blue1",0,0,255},
      {"blue2",0,0,238},
      {"blue3",0,0,205},
      {"blue4",0,0,139},
      {"DodgerBlue1",30,144,255},
      {"DodgerBlue2",28,134,238},
      {"DodgerBlue3",24,116,205},
      {"DodgerBlue4",16,78,139},
      {"SteelBlue1",99,184,255},
      {"SteelBlue2",92,172,238},
      {"SteelBlue3",79,148,205},
      {"SteelBlue4",54,100,139},
      {"DeepSkyBlue1",0,191,255},
      {"DeepSkyBlue2",0,178,238},
      {"DeepSkyBlue3",0,154,205},
      {"DeepSkyBlue4",0,104,139},
      {"SkyBlue1",135,206,255},
      {"SkyBlue2",126,192,238},
      {"SkyBlue3",108,166,205},
      {"SkyBlue4",74,112,139},
      {"LightSkyBlue1",176,226,255},
      {"LightSkyBlue2",164,211,238},
      {"LightSkyBlue3",141,182,205},
      {"LightSkyBlue4",96,123,139},
      {"SlateGray1",198,226,255},
      {"SlateGray2",185,211,238},
      {"SlateGray3",159,182,205},
      {"SlateGray4",108,123,139},
      {"LightSteelBlue1",202,225,255},
      {"LightSteelBlue2",188,210,238},
      {"LightSteelBlue3",162,181,205},
      {"LightSteelBlue4",110,123,139},
      {"LightBlue1",191,239,255},
      {"LightBlue2",178,223,238},
      {"LightBlue3",154,192,205},
      {"LightBlue4",104,131,139},
      {"LightCyan1",224,255,255},
      {"LightCyan2",209,238,238},
      {"LightCyan3",180,205,205},
      {"LightCyan4",122,139,139},
      {"PaleTurquoise1",187,255,255},
      {"PaleTurquoise2",174,238,238},
      {"PaleTurquoise3",150,205,205},
      {"PaleTurquoise4",102,139,139},
      {"CadetBlue1",152,245,255},
      {"CadetBlue2",142,229,238},
      {"CadetBlue3",122,197,205},
      {"CadetBlue4",83,134,139},
      {"turquoise1",0,245,255},
      {"turquoise2",0,229,238},
      {"turquoise3",0,197,205},
      {"turquoise4",0,134,139},
      {"cyan1",0,255,255},
      {"cyan2",0,238,238},
      {"cyan3",0,205,205},
      {"cyan4",0,139,139},
      {"DarkSlateGray1",151,255,255},
      {"DarkSlateGray2",141,238,238},
      {"DarkSlateGray3",121,205,205},
      {"DarkSlateGray4",82,139,139},
      {"aquamarine1",127,255,212},
      {"aquamarine2",118,238,198},
      {"aquamarine3",102,205,170},
      {"aquamarine4",69,139,116},
      {"DarkSeaGreen1",193,255,193},
      {"DarkSeaGreen2",180,238,180},
      {"DarkSeaGreen3",155,205,155},
      {"DarkSeaGreen4",105,139,105},
      {"SeaGreen1",84,255,159},
      {"SeaGreen2",78,238,148},
      {"SeaGreen3",67,205,128},
      {"SeaGreen4",46,139,87},
      {"PaleGreen1",154,255,154},
      {"PaleGreen2",144,238,144},
      {"PaleGreen3",124,205,124},
      {"PaleGreen4",84,139,84},
      {"SpringGreen1",0,255,127},
      {"SpringGreen2",0,238,118},
      {"SpringGreen3",0,205,102},
      {"SpringGreen4",0,139,69},
      {"green1",0,255,0},
      {"green2",0,238,0},
      {"green3",0,205,0},
      {"green4",0,139,0},
      {"chartreuse1",127,255,0},
      {"chartreuse2",118,238,0},
      {"chartreuse3",102,205,0},
      {"chartreuse4",69,139,0},
      {"OliveDrab1",192,255,62},
      {"OliveDrab2",179,238,58},
      {"OliveDrab3",154,205,50},
      {"OliveDrab4",105,139,34},
      {"DarkOliveGreen1",202,255,112},
      {"DarkOliveGreen2",188,238,104},
      {"DarkOliveGreen3",162,205,90},
      {"DarkOliveGreen4",110,139,61},
      {"khaki1",255,246,143},
      {"khaki2",238,230,133},
      {"khaki3",205,198,115},
      {"khaki4",139,134,78},
      {"LightGoldenrod1",255,236,139},
      {"LightGoldenrod2",238,220,130},
      {"LightGoldenrod3",205,190,112},
      {"LightGoldenrod4",139,129,76},
      {"LightYellow1",255,255,224},
      {"LightYellow2",238,238,209},
      {"LightYellow3",205,205,180},
      {"LightYellow4",139,139,122},
      {"yellow1",255,255,0},
      {"yellow2",238,238,0},
      {"yellow3",205,205,0},
      {"yellow4",139,139,0},
      {"gold1",255,215,0},
      {"gold2",238,201,0},
      {"gold3",205,173,0},
      {"gold4",139,117,0},
      {"goldenrod1",255,193,37},
      {"goldenrod2",238,180,34},
      {"goldenrod3",205,155,29},
      {"goldenrod4",139,105,20},
      {"DarkGoldenrod1",255,185,15},
      {"DarkGoldenrod2",238,173,14},
      {"DarkGoldenrod3",205,149,12},
      {"DarkGoldenrod4",139,101,8},
      {"RosyBrown1",255,193,193},
      {"RosyBrown2",238,180,180},
      {"RosyBrown3",205,155,155},
      {"RosyBrown4",139,105,105},
      {"IndianRed1",255,106,106},
      {"IndianRed2",238,99,99},
      {"IndianRed3",205,85,85},
      {"IndianRed4",139,58,58},
      {"sienna1",255,130,71},
      {"sienna2",238,121,66},
      {"sienna3",205,104,57},
      {"sienna4",139,71,38},
      {"burlywood1",255,211,155},
      {"burlywood2",238,197,145},
      {"burlywood3",205,170,125},
      {"burlywood4",139,115,85},
      {"wheat1",255,231,186},
      {"wheat2",238,216,174},
      {"wheat3",205,186,150},
      {"wheat4",139,126,102},
      {"tan1",255,165,79},
      {"tan2",238,154,73},
      {"tan3",205,133,63},
      {"tan4",139,90,43},
      {"chocolate1",255,127,36},
      {"chocolate2",238,118,33},
      {"chocolate3",205,102,29},
      {"chocolate4",139,69,19},
      {"firebrick1",255,48,48},
      {"firebrick2",238,44,44},
      {"firebrick3",205,38,38},
      {"firebrick4",139,26,26},
      {"brown1",255,64,64},
      {"brown2",238,59,59},
      {"brown3",205,51,51},
      {"brown4",139,35,35},
      {"salmon1",255,140,105},
      {"salmon2",238,130,98},
      {"salmon3",205,112,84},
      {"salmon4",139,76,57},
      {"LightSalmon1",255,160,122},
      {"LightSalmon2",238,149,114},
      {"LightSalmon3",205,129,98},
      {"LightSalmon4",139,87,66},
      {"orange1",255,165,0},
      {"orange2",238,154,0},
      {"orange3",205,133,0},
      {"orange4",139,90,0},
      {"DarkOrange1",255,127,0},
      {"DarkOrange2",238,118,0},
      {"DarkOrange3",205,102,0},
      {"DarkOrange4",139,69,0},
      {"coral1",255,114,86},
      {"coral2",238,106,80},
      {"coral3",205,91,69},
      {"coral4",139,62,47},
      {"tomato1",255,99,71},
      {"tomato2",238,92,66},
      {"tomato3",205,79,57},
      {"tomato4",139,54,38},
      {"OrangeRed1",255,69,0},
      {"OrangeRed2",238,64,0},
      {"OrangeRed3",205,55,0},
      {"OrangeRed4",139,37,0},
      {"red1",255,0,0},
      {"red2",238,0,0},
      {"red3",205,0,0},
      {"red4",139,0,0},
      {"DeepPink1",255,20,147},
      {"DeepPink2",238,18,137},
      {"DeepPink3",205,16,118},
      {"DeepPink4",139,10,80},
      {"HotPink1",255,110,180},
      {"HotPink2",238,106,167},
      {"HotPink3",205,96,144},
      {"HotPink4",139,58,98},
      {"pink1",255,181,197},
      {"pink2",238,169,184},
      {"pink3",205,145,158},
      {"pink4",139,99,108},
      {"LightPink1",255,174,185},
      {"LightPink2",238,162,173},
      {"LightPink3",205,140,149},
      {"LightPink4",139,95,101},
      {"PaleVioletRed1",255,130,171},
      {"PaleVioletRed2",238,121,159},
      {"PaleVioletRed3",205,104,137},
      {"PaleVioletRed4",139,71,93},
      {"maroon1",255,52,179},
      {"maroon2",238,48,167},
      {"maroon3",205,41,144},
      {"maroon4",139,28,98},
      {"VioletRed1",255,62,150},
      {"VioletRed2",238,58,140},
      {"VioletRed3",205,50,120},
      {"VioletRed4",139,34,82},
      {"magenta1",255,0,255},
      {"magenta2",238,0,238},
      {"magenta3",205,0,205},
      {"magenta4",139,0,139},
      {"orchid1",255,131,250},
      {"orchid2",238,122,233},
      {"orchid3",205,105,201},
      {"orchid4",139,71,137},
      {"plum1",255,187,255},
      {"plum2",238,174,238},
      {"plum3",205,150,205},
      {"plum4",139,102,139},
      {"MediumOrchid1",224,102,255},
      {"MediumOrchid2",209,95,238},
      {"MediumOrchid3",180,82,205},
      {"MediumOrchid4",122,55,139},
      {"DarkOrchid1",191,62,255},
      {"DarkOrchid2",178,58,238},
      {"DarkOrchid3",154,50,205},
      {"DarkOrchid4",104,34,139},
      {"purple1",155,48,255},
      {"purple2",145,44,238},
      {"purple3",125,38,205},
      {"purple4",85,26,139},
      {"MediumPurple1",171,130,255},
      {"MediumPurple2",159,121,238},
      {"MediumPurple3",137,104,205},
      {"MediumPurple4",93,71,139},
      {"thistle1",255,225,255},
      {"thistle2",238,210,238},
      {"thistle3",205,181,205},
      {"thistle4",139,123,139},
      {"DarkGrey",169,169,169},
      {"DarkGray",169,169,169},
      {"DarkBlue",0,0,139},
      {"DarkCyan",0,139,139},
      {"DarkMagenta",139,0,139},
      {"DarkRed",139,0,0},
      {"LightGreen",144,238,144},
    };
  const int ncolors = sizeof(color_info) / sizeof(color_info[0]);

  for(int i = 0; i < ncolors; ++i)
    map_[color_info[i].name] =
      Color(color_info[i].r, color_info[i].g, color_info[i].b);

  // add the indexed greys in a loop, instead of putting them
  // in by hand

  const char digits[10] = {'0', '1', '2', '3', '4', '5', '6', '7', '8', '9'};
  for(int i = 0; i <= 100; ++i) {
    Uint8 index = (int) ((i * 255)/100.0 + 0.5);
    Color col(index, index, index);

    std::string num;
    if(i >= 100)
      num += digits[i%100];
    if(i >= 10)
      num += digits[(i/10)%10];
    num += digits[i%10];

    map_["grey" + num] = col;
    map_["gray" + num] = col;
  }

  Debug::channel(Debug::GENERIC) << (ncolors + 101 /*greys*/ + 1 /*transparent*/)
	<< " colors available in Color::find()" << Debug::endl;
}

