/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
/* <<< guess */
#include <unistd.h> /* for close() */
#include <netinet/in.h> /* for IPPROTO_IP */
#include <sys/ioctl.h>
#include <errno.h>
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
#include <sys/socket.h>
#include <net/if_dl.h> /* for struct sockaddr_dl */
#include <net/if_types.h> /* for IFT_ETHER */
#endif
/* guess >>> */

#include "checks.h"
#include "wfiface.h"
#include "defs.h"


wf_iface::wf_iface() :
  name(),
  comment(),
  macaddr(),
  ipaddr(),
  network(),
  broadcast(),
  flags(0),
  statdyn(WF_IP_STATDYN_UNKNOWN),
  ipguess(WF_IP_GUESS_UNKNOWN)
{
}

wf_iface::~wf_iface() {}

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
bool
wf_iface::guess(struct ifaddrs* ifa, bool skip_useless) {
  name = ifa->ifa_name;
  // fprintf(stderr, _("Scanning interface `%s'\n"), ifa->ifa_name);

  if (ifa->ifa_addr->sa_family == AF_INET) {
    /* IP address */
    ipaddr.set(((struct sockaddr_in*)ifa->ifa_addr)->sin_addr.s_addr);
    
    /* IP netmask address */
    network.netmask.set(((struct sockaddr_in*)ifa->ifa_netmask)->sin_addr.s_addr);
    network.network.set(ipaddr.get() & network.netmask.get());
    if (skip_useless && network.ishost())
      return false;
    
    /* IP broadcast address */
    if (((struct sockaddr_in*)ifa->ifa_broadaddr)->sin_addr.s_addr !=
	INADDR_ANY)
      broadcast.set(((struct sockaddr_in*)ifa->ifa_broadaddr)->sin_addr.s_addr);
  }

  /* Interface flags */
  /* MAC address */
  if (ifa->ifa_flags & (IFF_LOOPBACK|IFF_POINTOPOINT)) {
    if (skip_useless)
      return false;
  }
  else {
    /* no MAC address available for loopback and ppp */
    if (ifa->ifa_addr->sa_family == AF_LINK) {
      struct sockaddr_dl *sdl = (struct sockaddr_dl*)ifa->ifa_addr;
      if (sdl->sdl_type == IFT_ETHER && sdl->sdl_alen == ETHER_ADDR_LEN) {
	/* use wf_macaddr::set(unsigned char *) */
	macaddr.set((unsigned char*)LLADDR(sdl));
      }
    }
  }
  flags = ifa->ifa_flags;

  return true;
}
#else
bool
wf_iface::guess(struct ifreq* ifr, int sock, bool skip_useless) {
  struct sockaddr_in sin;
  struct sockaddr saddr;

  name = ifr->ifr_name;
  // fprintf(stderr, _("Scanning interface `%s'\n"), ifr->ifr_name);

  /* IP address */
  if (ioctl(sock, SIOCGIFADDR, (char*)ifr) != -1) {
    memcpy(&sin, &(ifr->ifr_addr), sizeof(ifr->ifr_addr));
    ipaddr.set(sin.sin_addr.s_addr);
  }

  /* IP netmask address */
  if (ioctl(sock, SIOCGIFNETMASK, (char*)ifr) != -1) {
#ifndef ifr_netmask /* ifr_netmask probably does not exist */
#define ifr_netmask  ifr_ifru.ifru_addr   /* interface net mask   */
#endif
    memcpy(&sin, &(ifr->ifr_netmask), sizeof(ifr->ifr_netmask));
    network.netmask.set(sin.sin_addr.s_addr);
    network.network.set(ipaddr.get() & network.netmask.get());
    if (skip_useless && network.ishost())
      return false;
  }

  /* IP broadcast address */
  if (ioctl(sock, SIOCGIFBRDADDR, (char*)ifr) != -1) {
    memcpy(&sin, &(ifr->ifr_broadaddr), sizeof(ifr->ifr_broadaddr));
    if (sin.sin_addr.s_addr != INADDR_ANY)
      broadcast.set(sin.sin_addr.s_addr);
  }
  
  /* Interface flags */
  if (ioctl(sock, SIOCGIFFLAGS, (char*)ifr) != -1) {
    /* MAC address */
    if (ifr->ifr_flags & (IFF_LOOPBACK|IFF_POINTOPOINT)) {
      if (skip_useless)
	return false;
    }
    else {
      /* no MAC address available for loopback and ppp */
      if (ioctl(sock, SIOCGIFHWADDR, (char*)ifr) != -1) {
	memcpy(&saddr, &(ifr->ifr_hwaddr), sizeof(ifr->ifr_hwaddr));
	macaddr.set(&saddr);
      }
    }
    flags = ifr->ifr_flags;
  }
  return true;
}
#endif

#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
bool
wf_iface::guess(const string& ifacename, bool skip_useless) {
  struct ifaddrs *ifap, *ifa;
  if (getifaddrs(&ifap) < 0) {
    perror("getifaddrs");
    return false;
  }
  
  for (ifa = ifap; ifa; ifa = ifa->ifa_next) {
    if (ifa->ifa_name != ifacename)
      continue;
    if (guess(ifa, skip_useless) == false) {
      freeifaddrs(ifap);
      return false;
    }
  }
  freeifaddrs(ifap);
  if (name.empty()) /* interface could not be found */
    return false;

  return true;
}
#else
bool
wf_iface::guess(const string& ifacename, bool skip_useless) {
  int sock = socket(AF_INET, SOCK_DGRAM, IPPROTO_IP);
  if (sock < 0) {
    perror("socket creation");
    return false;
  }

  struct ifreq ifr;
  strncpy(ifr.ifr_name, ifacename.c_str(), IFNAMSIZ);

  bool ret = guess(&ifr, sock, skip_useless);
  close(sock);
  if (ret == false)
    return false;

  name = ifacename;
  return true;
}
#endif

wf_ipaddr
wf_iface::broadcast_get() const {
  if (broadcast.isdefined())
    return broadcast;
  if (network.isdefined() == false)
    return wf_ipaddr();
  return network.broadcast();
}

bool
wf_iface::complete(bool guess) {
  if (network.network.isdefined()) {
    if (network.netmask.isdefined() && broadcast.isdefined())
      return true; /* everything is defined: ok */

    if (guess && !network.netmask.isdefined() && !broadcast.isdefined()) {
      int netmaskaddr;  /* try to guess from network address... */
      if (wf_network_tonetmask(network.network.get(), &netmaskaddr) == false)
	return false;
      network.netmask.set(netmaskaddr);
    }

    if (network.netmask.isdefined())
      broadcast = network.broadcast();
    else if (broadcast.isdefined())
      network.netmask.set(~(network.network.get() ^ broadcast.get()));
  }
  else if (network.netmask.isdefined() && broadcast.isdefined())
    network.network.set(broadcast.get() & network.netmask.get());
  else
    return false;
  
  return true;
}

bool
wf_iface::check() const {
  if (name.empty()) {
    fprintf(stderr, _("Interface has no name.\n"));
     return false;
  }
  if (wf_interface_name_check(name) == false) {
    fprintf(stderr, _("Interface `%s': invalid name.\n"), name.c_str());
    return false;
  }
  if (name.length() > IFNAMSIZ) {
    fprintf(stderr, _("Interface `%s': name is too long (>%i).\n"),
	    name.c_str(), IFNAMSIZ);
    return false;
  }
  if (ipaddr.isdefined() && network.isdefined()) {
    if (network.belong(ipaddr) == false) {
      fprintf(stderr, _("Interface `%s': IP address %s does not belong to network `%s'.\n"),
	      name.c_str(), ipaddr.tostr().c_str(), network.tostr().c_str());
      return false;
    }
  }
  if (broadcast.isdefined() && network.network.isdefined()) {
    if (network.belong(broadcast) == false) {
      fprintf(stderr, _("Interface `%s': broadcast IP address %s does not belong to network `%s'.\n"),
	      name.c_str(), broadcast.tostr().c_str(), network.tostr().c_str());
      return false;
    }
  }

  if (ipaddr.isdefined()) {
    if (ipguess == WF_IP_GUESS_COMPILETIME || ipguess == WF_IP_GUESS_RUNTIME) {
      fprintf(stderr, _("Interface `%s': no guess can be made, as IP address is already defined to %s.\n"),
	      name.c_str(), ipaddr.tostr().c_str());
      return false;
    }
    if (statdyn == WF_IP_DYNAMIC) {
      fprintf(stderr, _("Interface `%s': IP address is defined as %s, but is declared as dynamic.\n"),
	      name.c_str(), ipaddr.tostr().c_str());
      return false;
    }
  }
  else { /* no address specified */
    if (ipguess == WF_IP_GUESS_NONE) {
      fprintf(stderr, _("Interface `%s': IP address is missing, and no guess will be performed."),
	      name.c_str());
      return false;
    }
  }
  
  return true;
}


ostream&
wf_iface::debugprint(ostream& os) const {
  os << _("Interface:") << endl;

  os << _("comment:\t") << comment << endl;

  os << _("name:\t\t");
  if (name.empty() == false)
    os << name;
  else
    os << _("(undefined)");
  os << endl;

  macaddr.debugprint(os);
  ipaddr.debugprint(os);
  network.debugprint(os);
  os << _("broadcast:\t") << broadcast << endl;
  os << _("flags:\t\t") << flags << endl;
  os << _("statdyn:\t") << statdyn << endl;
  os << _("guess:\t\t") << ipguess << endl;
  return os;
}

bool
operator==(const wf_iface& iface1, const wf_iface& iface2) {
  /* we'll ignore 'comment' members during comparison */
  return (iface1.name == iface2.name &&
	  iface1.macaddr == iface2.macaddr &&
	  iface1.ipaddr == iface2.ipaddr &&
	  iface1.network == iface2.network &&
	  iface1.broadcast == iface2.broadcast &&
	  iface1.flags == iface2.flags &&
	  iface1.statdyn == iface2.statdyn &&
	  iface1.ipguess == iface2.ipguess);
}
