/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "wflogs_filter.h"
#include "filter.h"
#include "defs.h"


exprnode* exprnode_parse(const string& expr);
static const wf_logentry* _entry; /* not very clean RV@@5 */

/* called if type is numerical */
int
exprnode_var_id_wflogs::eval() const {
  int ret;
  switch (_num) {
  case VARS_COUNT:
    ret = _entry->count; break;
  case VARS_START_TIME:
    ret = _entry->start_time; break;
  case VARS_END_TIME:
    ret = _entry->end_time; break;
  case VARS_PROTOCOL:
    ret = _entry->protocol; break;
  case VARS_DATALEN:
    ret = _entry->datalen; break;
  case VARS_SPORT:
    ret = _entry->sport; break;
  case VARS_DPORT:
    ret = _entry->dport; break;
  case VARS_TCPFLAGS:
    ret = _entry->tcpflags; break;
  default:
    ret = 0; // RV@@9 shouldn't happen
  }
  return ret;
}

string
exprnode_var_id_wflogs::str() const {
  string str;
  switch (_num) {
  case VARS_FORMAT:
    str = _entry->format; break;
  case VARS_HOSTNAME:
    str = _entry->hostname; break;
  case VARS_CHAINLABEL:
    str = _entry->chainlabel; break;
  case VARS_BRANCHNAME:
    str = _entry->branchname; break;
  case VARS_INPUT_IFACE:
    str = _entry->input_iface; break;
  case VARS_OUTPUT_IFACE:
    str = _entry->output_iface; break;
  default:
    ; // RV@@9 shouldn't happen
  }
  return str;
}

wf_ipaddr
exprnode_var_id_wflogs::ipaddr() const {
  wf_ipaddr ipaddr;
  switch (_num) {
  case VARS_SIPADDR:
    ipaddr = _entry->sipaddr; break;
  case VARS_DIPADDR:
    ipaddr = _entry->dipaddr; break;
  default:
    ; // RV@@9 shouldn't happen
  }
  return ipaddr;
}

wf_macaddr
exprnode_var_id_wflogs::macaddr() const {
  wf_macaddr macaddr;
  switch (_num) {
  case VARS_SMACADDR:
    macaddr = _entry->smacaddr; break;
  case VARS_DMACADDR:
    macaddr = _entry->dmacaddr; break;
  default:
    ; // RV@@9 shouldn't happen
  }
  return macaddr;
}

wflogs_filter::wflogs_filter() :
  _tree(NULL),
  _count(0)
{
}

wflogs_filter::~wflogs_filter() {}


bool
wflogs_filter::set(const string& expr) {
  _tree = exprnode_parse(expr);
  if (_tree == NULL)
    return false;

  return true;
}

void
wflogs_filter::count_reset() {
  _count = 0;
}

unsigned int
wflogs_filter::count_get() const {
  return _count;
}

static inline bool
filter_match_func(const exprnode* tree, const wf_logentry* entry) {
  _entry = entry; /* set "global" var _entry used in the tree */
  return tree->eval() != 0;
}

bool
wflogs_filter::operator() (const wf_logentry* entry) {
  if (filter_match_func(_tree, entry))
    return false;
  ++_count;
  return true;
}

ostream&
wflogs_filter::print(ostream& os) const {
  return _tree->print(os);
}
