/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "commentview.h"

#include <QTextEdit>
#include <QTextBlock>
#include <QAction>
#include <QMenu>
#include <QApplication>
#include <QClipboard>
#include <QMessageBox>
#include <QLocale>

#include "commands/commandmanager.h"
#include "data/datamanager.h"
#include "data/issuebatch.h"
#include "data/updateevent.h"
#include "data/connectioninfo.h"
#include "models/columnconditionsettings.h"
#include "dialogs/checkmessagebox.h"
#include "xmlui/builder.h"
#include "configdata.h"
#include "viewmanager.h"
#include "connectionmanager.h"
#include "iconloader.h"

CommentView::CommentView( QObject* parent, QWidget* parentWidget ) : View( parent ),
    m_sending( false ),
    m_folderId( 0 )
{
    QAction* action;

    action = new QAction( IconLoader::icon( "comment-send" ), tr( "&Send Comment" ), this );
    action->setShortcut( tr( "Ctrl+Return" ) );
    connect( action, SIGNAL( triggered() ), this, SLOT( sendComment() ) );
    setAction( "sendComment", action );

    m_edit = new QTextEdit( parentWidget );
    m_edit->setAcceptRichText( false );
    m_edit->setContextMenuPolicy( Qt::CustomContextMenu );

    connect( m_edit, SIGNAL( customContextMenuRequested( const QPoint& ) ),
        this, SLOT( contextMenu( const QPoint& ) ) );

    m_edit->setHtml( "<pre>X</pre>" );
    QFont fixedFont = m_edit->document()->findBlock( 0 ).charFormat().font();
    m_edit->clear();
    m_edit->setFont( fixedFont );

    action = new QAction( IconLoader::icon( "edit-undo" ), tr( "&Undo" ), this );
    action->setShortcut( QKeySequence::Undo );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( undo() ) );
    connect( m_edit, SIGNAL( undoAvailable( bool ) ), action, SLOT( setEnabled( bool ) ) );
    setAction( "editUndo", action );

    action = new QAction( IconLoader::icon( "edit-redo" ), tr( "&Redo" ), this );
    action->setShortcut( QKeySequence::Redo );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( redo() ) );
    connect( m_edit, SIGNAL( redoAvailable( bool ) ), action, SLOT( setEnabled( bool ) ) );
    setAction( "editRedo", action );

    action = new QAction( IconLoader::icon( "edit-cut" ), tr( "Cu&t" ), this );
    action->setShortcut( QKeySequence::Cut );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( cut() ) );
    connect( m_edit, SIGNAL( copyAvailable( bool ) ), action, SLOT( setEnabled( bool ) ) );
    setAction( "editCut", action );

    action = new QAction( IconLoader::icon( "edit-copy" ), tr( "&Copy" ), this );
    action->setShortcut( QKeySequence::Copy );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( copy() ) );
    connect( m_edit, SIGNAL( copyAvailable( bool ) ), action, SLOT( setEnabled( bool ) ) );
    setAction( "editCopy", action );

    action = new QAction( IconLoader::icon( "edit-paste" ), tr( "&Paste" ), this );
    action->setShortcut( QKeySequence::Paste );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( paste() ) );
    setAction( "editPaste", action );

    action = new QAction( tr( "Select &All" ), this );
    action->setShortcut( QKeySequence::SelectAll );
    connect( action, SIGNAL( triggered() ), m_edit, SLOT( selectAll() ) );
    setAction( "editSelectAll", action );

    setTitle( "menuMain", tr( "&Comment" ) );
    setTitle( "menuEdit", tr( "&Edit" ) );

    setButtonStyle( "sendComment", Qt::ToolButtonTextBesideIcon );

    loadXmlUiFile( ":/resources/commentview.xml" );

    connect( m_edit, SIGNAL( textChanged() ), this, SLOT( updateActions() ) );
    connect( m_edit, SIGNAL( textChanged() ), this, SLOT( updateSummary() ) );

    connect( QApplication::clipboard(), SIGNAL( dataChanged() ), this, SLOT( updateClipboard() ) );

    setMainWidget( m_edit );

    setViewerSizeHint( QSize( 600, 400 ) );
}

CommentView::~CommentView()
{
}

void CommentView::initialUpdate()
{
    setAccess( checkDataAccess(), true );
}

Access CommentView::checkDataAccess()
{
    const IssueRow* issue = dataManager->issues()->find( id() );
    if ( !issue )
        return UnknownAccess;

    m_folderId = issue->folderId();

    const FolderRow* folder = dataManager->folders()->find( issue->folderId() );
    if ( !folder )
        return NoAccess;

    if ( connectionManager->connectionInfo()->access() != AdminAccess ) {
        int userId = connectionManager->connectionInfo()->userId();
        const MemberRow* member = dataManager->members()->find( userId, folder->projectId() );
        if ( !member )
            return NoAccess;
    }

    const TypeRow* type = dataManager->types()->find( folder->typeId() );
    if ( !type )
        return UnknownAccess;

    return NormalAccess;
}

void CommentView::enableView()
{
    updateCaption();
    updateActions();
    updateClipboard();
    updateSummary();
}

void CommentView::disableView()
{
    updateCaption();
}

void CommentView::updateCaption()
{
    QString name = tr( "Unknown Issue" ) ;
    if ( isEnabled() ) {
        const IssueRow* row = dataManager->issues()->find( id() );
        if ( row )
            name = row->name();
    }
    setCaption( tr( "Add Comment - %1" ).arg( name ) );
}

void CommentView::updateActions()
{
    action( "sendComment" )->setEnabled( !m_edit->document()->isEmpty() );
    action( "editUndo" )->setEnabled( m_edit->document()->isUndoAvailable() );
    action( "editRedo" )->setEnabled( m_edit->document()->isRedoAvailable() );
    action( "editCut" )->setEnabled( m_edit->textCursor().hasSelection() );
    action( "editCopy" )->setEnabled( m_edit->textCursor().hasSelection() );
    action( "editSelectAll" )->setEnabled( !m_edit->document()->isEmpty() );
}

void CommentView::updateClipboard()
{
    action( "editPaste" )->setEnabled( !QApplication::clipboard()->text().isEmpty() );
}

void CommentView::updateSummary()
{
    QTextDocument* document = m_edit->document();

    int chars = -1; // last end-of-block is removed
    int lines = 0;
    for ( QTextBlock block = document->begin(); block != document->end(); block = block.next() ) {
        chars += block.length();
        lines++;
    }

    QLocale locale;
    QString status = tr( "%1 paragraphs, %2 characters" ).arg( locale.toString( lines ), locale.toString( chars ) );

    int maxSize = configData->warnCommentSize();

    if ( maxSize > 0 && chars > maxSize * 1024 )
        showSummary( IconLoader::pixmap( "status-warning" ), status );
    else
        showSummary( QPixmap(), status );
}

void CommentView::sendComment()
{
    if ( !m_sending && isEnabled() ) {
        QString text = m_edit->toPlainText();

        int size = text.length();
        int maxSize = configData->warnCommentSize();

        if ( maxSize > 0 && size > maxSize * 1024 ) {
            CheckMessageBox box( mainWidget() );

            box.setIcon( QMessageBox::Warning );
            box.setWindowTitle( tr( "Warning" ) );
            box.setText( tr( "This comment is longer than %1 kilobytes. The server may not support\n"
                "such large comments. Are you sure you want to continue?" ).arg( maxSize ) );
            box.setStandardButtons( QMessageBox::Ok | QMessageBox::Cancel );

            if ( box.exec() != QMessageBox::Ok )
                return;

            if ( box.isChecked() ) {
                configData->setWarnCommentSize( 0 );
                configData->saveSettings();
            }
        }

        showBusy( tr( "Sending comment..." ) );

        IssueBatch* batch = new IssueBatch( id() );
        batch->addComment( text );
        batch->setAutoNotify( issueAutoNotify() );

        connect( batch, SIGNAL( completed( bool ) ), this, SLOT( sendCompleted( bool ) ) );

        commandManager->execute( batch );

        m_sending = true;
    }
}

void CommentView::sendCompleted( bool successful )
{
    if ( successful )
        viewManager->closeView( this );
    else
        showWarning( commandManager->errorMessage( tr( "Sending failed" ) ) );

    m_sending = false;
}

bool CommentView::queryClose()
{
    if ( isEnabled() && !m_edit->document()->isEmpty() ) {
        if ( QMessageBox::warning( mainWidget(), tr( "Warning" ),
            tr( "This comment has not been sent. You will lose all changes\n"
            "when you close it. Are you sure you want to continue?" ),
            QMessageBox::Ok | QMessageBox::Cancel ) != QMessageBox::Ok )
            return false;
    }

    return true;
}

void CommentView::updateEvent( UpdateEvent* e )
{
    setAccess( checkDataAccess() );

    if ( isEnabled() && e->unit() == UpdateEvent::Issue && e->id() == id() )
        updateCaption();

    if ( isEnabled() && e->unit() == UpdateEvent::Folder && e->id() == m_folderId )
        updateCaption();
}

void CommentView::contextMenu( const QPoint& pos )
{
    QMenu* menu = builder()->contextMenu( "contextEdit" );
    if ( menu )
        menu->exec( m_edit->mapToGlobal( pos ) );
}

bool CommentView::issueAutoNotify()
{
    if ( !dataManager->isIssueNotify( id() ) ) {
        const IssueRow* issue = dataManager->issues()->find( id() );
        if ( issue != NULL ) {
            const FolderRow* folder = dataManager->folders()->find( issue->folderId() );
            if ( folder != NULL ) {
                ColumnConditionSettings settings;
                settings.openIssueFilters( folder->typeId() );
                return settings.autoNotify( folder->folderId() );
            }
        }
    }
    return false;
}
