/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "linklocator.h"

#include <QRegExp>
#include <QCoreApplication>

static QRegExp* linkExp = NULL;

static void postRoutine()
{
    delete linkExp;
}

LinkLocator::LinkLocator( int reserve, int flags ) :
    m_result( (QChar*)NULL, reserve ),
    m_flags( flags ),
    m_column( 0 ),
    m_space( false )
{
    // \b \w [^\s@]* @ [^\s@]* \w
    // \b ( mailto: | https?:// | ftp:// | www\. | ftp\. ) \S* [\w/]
    // # \d+ \b

    if ( !linkExp ) {
        QString pattern( "\\b\\w[^\\s@]*@[^\\s@]*\\w|\\b(mailto:|https?://|ftp://|www\\.|ftp\\.)\\S*[\\w/]|#\\d+\\b" );
        linkExp = new QRegExp( pattern );
        qAddPostRoutine( postRoutine );
    }
}

LinkLocator::~LinkLocator()
{
}

QString LinkLocator::convertToHtml( const QString& text, int flags )
{
    LinkLocator locator( text.length() * 2, flags );

    int pos = 0;
    for ( ; ; ) {
        int oldpos = pos;
        pos = linkExp->indexIn( text, pos );

        if ( pos < 0 ) {
            locator.append( text.mid( oldpos ) );
            break;
        }

        if ( pos > oldpos )
            locator.append( text.mid( oldpos, pos - oldpos ) );

        QString link = linkExp->cap( 0 );

        QString href;
        if ( link[ 0 ] == QLatin1Char( '#' ) )
            href = ( ( flags & NoInternalLinks ) ? "#id" : "id://" ) + link.mid( 1 );
        else if ( linkExp->cap( 1 ).isEmpty() )
            href = "mailto:" + link;
        else if ( link.startsWith( "www." ) )
            href = "http://" + link;
        else if ( link.startsWith( "ftp." ) )
            href = "ftp://" + link;
        else
            href = link;

        href = escape( href );

        locator.appendTag( QString( "<a href=\"%1\">" ).arg( href ) );
        locator.append( link );
        locator.appendTag( "</a>" );

        pos += linkExp->matchedLength();
    }

    return locator.result();
}

QString LinkLocator::escape( const QString& text )
{
    QString result;
    for ( int i = 0; i < text.length(); i++ ) {
        QChar ch = text[ i ];
        if ( ch == QLatin1Char( '<' ) )
            result += QLatin1String( "&lt;" );
        else if ( ch == QLatin1Char( '>' ) )
            result += QLatin1String( "&gt;" );
        else if ( ch == QLatin1Char( '&' ) )
            result += QLatin1String( "&amp;" );
        else if ( ch == QLatin1Char( '"' ) )
            result += QLatin1String( "&quot;" );
        else
            result += ch;
    }
    return result;
}

void LinkLocator::append( const QString& text )
{
    for ( int i = 0; i < text.length(); i++ ) {
        QChar ch = text[ i ];

        if ( m_flags & PreserveSpaces ) {
            if ( ch == QLatin1Char( ' ' ) ) {
                if ( m_space || m_column == 0 ) {
                    m_result += QLatin1String( "&nbsp;" );
                    m_space = false;
                } else {
                    m_result += QLatin1Char( ' ' );
                    m_space = true;
                }
                m_column++;
                continue;
            }
            if ( ch == QLatin1Char( '\t' ) ) {
                do {
                    m_result += QLatin1String( "&nbsp;" );
                    m_column++;
                } while ( ( m_column & 7 ) != 0 );
                m_space = false;
                continue;
            }
            if ( ch == QLatin1Char( '\n' ) ) {
                m_result += QLatin1String( "<br>" );
                m_space = false;
                m_column = 0;
                continue;
            }
        }

        if ( ch == QLatin1Char( '<' ) )
            m_result += QLatin1String( "&lt;" );
        else if ( ch == QLatin1Char( '>' ) )
            m_result += QLatin1String( "&gt;" );
        else if ( ch == QLatin1Char( '&' ) )
            m_result += QLatin1String( "&amp;" );
        else if ( ch == QLatin1Char( '"' ) )
            m_result += QLatin1String( "&quot;" );
        else
            m_result += ch;

        m_column++;
        m_space = false;
    }
}

void LinkLocator::appendTag( const QString& tag )
{
    m_result += tag;
    m_space = false;
}
