/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "htmlreportgenerator.h"

#include "attributes/datetimehelper.h"
#include "commands/commandmanager.h"
#include "data/datamanager.h"
#include "data/connectioninfo.h"
#include "htmlreportwriter.h"
#include "issuehistoryprovider.h"
#include "issuehistorywriter.h"
#include "tablemodelshelper.h"
#include "tablemodels.h"
#include "treeviewsettings.h"
#include "issueviewsettings.h"
#include "application.h"
#include "connectionmanager.h"

HtmlReportGenerator::HtmlReportGenerator( Source source ) :
    m_source( source ),
    m_folderId( 0 ),
    m_table( NoTable ),
    m_summary( false ),
    m_history( false ),
    m_writer( NULL )
{
}

HtmlReportGenerator::~HtmlReportGenerator()
{
}

void HtmlReportGenerator::setIssue( int issueId )
{
    m_folderId = 0;
    m_issues.clear();
    m_issues.append( issueId );
    m_filterName.clear();
}

void HtmlReportGenerator::setFolder( int folderId, const QList<int>& issues, const QString& filterName )
{
    m_folderId = folderId;
    m_issues = issues;
    m_filterName = filterName;
}

void HtmlReportGenerator::setTable( Table table )
{
    m_table = table;
}

void HtmlReportGenerator::setSummaryEnabled( bool enabled )
{
    m_summary = enabled;
}

void HtmlReportGenerator::setHistoryEnabled( bool enabled )
{
    m_history = enabled;
}

void HtmlReportGenerator::write( HtmlReportWriter* writer )
{
    m_writer = writer;

    if ( m_source == Folder ) {
        const FolderRow* folder = dataManager->folders()->find( m_folderId );
        if ( folder ) {
            writer->begin( generator(), writer->titleName( folder->name() ) );

            writeFolderInfo( folder );

            if ( m_table != NoTable ) {
                TreeViewSettings settings;
                settings.openIssuesList( folder->typeId() );

                if ( m_table == CurrentTable )
                    writeIssuesTable( settings.loadColumns() );
                else if ( m_table == FullTable )
                    writeIssuesTable( settings.availableColumns() );
            }

            if ( m_summary ) {
                IssueViewSettings settings;
                settings.openIssueView( folder->typeId() );
                QList<int> attributes = settings.loadAttributes();

                for ( int i = 0; i < m_issues.count(); i++ ) {
                    int issueId = m_issues.at( i );
                    const IssueRow* issue = dataManager->issues()->find( issueId );
                    if ( issue ) {
                        writer->writeHeader2( issueId, issueHeader( issueId, issue->name() ) );
                        writeIssueSummary( issue, attributes );
                    }
                }
            }

        } else {
            writer->begin( generator(), tr( "Unknown Folder" ) );
        }
    } else if ( m_source == Issue && !m_issues.isEmpty() ) {
        const IssueRow* issue = dataManager->issues()->find( m_issues.first() );
        if ( issue ) {
            const FolderRow* folder = dataManager->folders()->find( issue->folderId() );
            if ( folder ) {
                writer->begin( generator(), writer->titleName( issue->name() ) );

                writeIssueInfo( issue, folder );

                if ( m_summary ) {
                    writer->writeHeader3( tr( "Issue Summary" ) );

                    IssueViewSettings settings;
                    settings.openIssueView( folder->typeId() );
                    QList<int> attributes = settings.loadAttributes();

                    writeIssueSummary( issue, attributes );
                }

                if ( m_history ) {
                    writer->writeHeader3( tr( "Issue History" ) );

                    writeIssueHistory( issue );
                }
            } else {
                writer->begin( generator(), tr( "Unknown Issue" ) );
            }
        } else {
            writer->begin( generator(), tr( "Unknown Issue" ) );
        }
    } else {
        writer->begin( generator(), tr( "Unknown Source" ) );
    }

    writer->writeFooter( footer() );

    writer->end();
}

void HtmlReportGenerator::writeFolderInfo( const FolderRow* folder )
{
    const ProjectRow* project = dataManager->projects()->find( folder->projectId() );
    const TypeRow* type = dataManager->types()->find( folder->typeId() );

    QString typeName = type ? type->name() : QString();

    m_writer->beginSummary();
    m_writer->writeSummaryItem( m_writer->item( tr( "Project" ) ), m_writer->item( project->name() ) );
    m_writer->writeSummaryItem( m_writer->item( tr( "Issue Type" ) ), m_writer->item( typeName ) );
    m_writer->writeSummaryItem( m_writer->item( tr( "Filter" ) ), m_writer->item( m_filterName ) );
    m_writer->endSummary();
}

void HtmlReportGenerator::writeIssueInfo( const IssueRow* issue, const FolderRow* folder )
{
    const ProjectRow* project = dataManager->projects()->find( folder->projectId() );
    const TypeRow* type = dataManager->types()->find( folder->typeId() );

    QString id = TableModelsHelper::formatId( issue->issueId() );
    QString projectName = project ? project->name() : QString();
    QString typeName = type ? type->name() : QString();

    m_writer->beginSummary();
    m_writer->writeSummaryItem( m_writer->item( tr( "ID" ) ), m_writer->item( id ) );
    m_writer->writeSummaryItem( m_writer->item( tr( "Project" ) ), m_writer->item( projectName ) );
    m_writer->writeSummaryItem( m_writer->item( tr( "Folder" ) ), m_writer->item( folder->name() ) );
    m_writer->writeSummaryItem( m_writer->item( tr( "Issue Type" ) ), m_writer->item( typeName ) );
    m_writer->endSummary();
}

void HtmlReportGenerator::writeIssuesTable( const QList<int>& columns )
{
    IssuesTableModel model( m_folderId, NULL );

    m_writer->beginTable();

    QStringList cells;
    for ( int i = 0; i < columns.count(); i++ ) {
        int column = columns.at( i );
        QString name = model.columnName( column );
        cells.append( m_writer->item( name ) );
    }
    m_writer->writeTableHeader( cells );

    for ( int i = 0; i < m_issues.count(); i++ ) {
        int issueId = m_issues.at( i );
        cells.clear();
        for ( int j = 0; j < columns.count(); j++ ) {
            int column = columns.at( j );
            QString text = model.text( issueId, column );
            cells.append( m_writer->value( text ) );
        }
        m_writer->writeTableRow( issueId, cells );
    }

    m_writer->endTable();
}

void HtmlReportGenerator::writeIssueSummary( const IssueRow* issue, const QList<int>& attributes )
{
    m_writer->writeParagraph( issueCreated( issue->createdUser(), issue->createdDate() ) );
    m_writer->writeParagraph( issueModified( issue->modifiedUser(), issue->modifiedDate() ) );

    bool inSummary = false;

    for ( int i = 0; i < attributes.count(); i++ ) {
        int attributeId = attributes.at( i );
        const ValueRow* value = dataManager->values()->find( attributeId, issue->issueId() );
        if ( value ) {
            if ( !inSummary ) {
                m_writer->beginSummary();
                inSummary = true;
            }
            QString name = TableModelsHelper::attributeName( attributeId );
            m_writer->writeSummaryItem( m_writer->item( name ), m_writer->value( value->value() ) );
        }
    }

    if ( inSummary )
        m_writer->endSummary();
}

void HtmlReportGenerator::writeIssueHistory( const IssueRow* issue )
{
    IssueHistoryProvider provider;
    provider.extractIssueHistory( issue->issueId() );

    IssueHistoryWriter writer( IssueHistoryWriter::Report );
    provider.write( &writer );

    m_writer->writeRawHtml( writer.result() );
}

QString HtmlReportGenerator::issueHeader( int issueId, const QString& name )
{
    QString idArg = TableModelsHelper::formatId( issueId );
    QString nameArg = m_writer->titleName( name );

    return QString( "%1 %2" ).arg( idArg, nameArg );
}

QString HtmlReportGenerator::issueCreated( int userId, const QDateTime& date )
{
    QString userArg = m_writer->name( TableModelsHelper::userName( userId ) );
    QString dateArg = m_writer->date( DateTimeHelper::formatDateTime( date ) );

    return tr( "Created by %1 on %2" ).arg( userArg, dateArg );
}

QString HtmlReportGenerator::issueModified( int userId, const QDateTime& date )
{
    QString userArg = m_writer->name( TableModelsHelper::userName( userId ) );
    QString dateArg = m_writer->date( DateTimeHelper::formatDateTime( date ) );

    return tr( "Modified by %1 on %2" ).arg( userArg, dateArg );
}

QString HtmlReportGenerator::generator()
{
    QString version = ( (Application*)qApp )->version();

    return QString( "WebIssues Client %1" ).arg( version );
}

QString HtmlReportGenerator::footer()
{
    QString date = m_writer->date( DateTimeHelper::formatDateTime( QDateTime::currentDateTime() ) );
    QString serverName = m_writer->name( connectionManager->connectionInfo()->serverName() );
    QString serverUrl = m_writer->item( commandManager->serverUrl().toString() );

    return QString( "Generated on %1 for %2 at %3" ).arg( date, serverName, serverUrl );
}
