/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "folderwatchprovider.h"

#include <QTreeWidget>

#include "data/datamanager.h"
#include "data/datarows.h"
#include "rdb/utilities.h"
#include "viewmanager.h"
#include "columnconditionsettings.h"
#include "rowfilters.h"
#include "iconloader.h"

FolderWatchProvider::FolderWatchProvider( QObject* parent ) : QObject( parent )
{
    m_filter = new IssueRowFilter( this );
}

FolderWatchProvider::~FolderWatchProvider()
{
}

void FolderWatchProvider::updateAll()
{
    m_map.clear();

    RDB::IndexConstIterator<TypeRow> it( dataManager->types()->index() );
    while ( it.next() )
        updateFolders( it.key() );
}

void FolderWatchProvider::updateFolders( int typeId )
{
    ColumnConditionSettings settings;
    settings.openIssueFilters( typeId );

    RDB::IndexConstIterator<FolderRow> it( dataManager->folders()->index() );
    while ( it.next() ) {
        if ( it.get()->typeId() == typeId )
            calculateWatches( it.key(), &settings );
    }
}

void FolderWatchProvider::updateFolder( int folderId )
{
    const FolderRow* folder = dataManager->folders()->find( folderId );
    if ( folder != NULL ) {
        ColumnConditionSettings settings;
        settings.openIssueFilters( folder->typeId() );

        calculateWatches( folderId, &settings );
    }
}

void FolderWatchProvider::calculateWatches( int folderId, ColumnConditionSettings* settings )
{
    WatchState folderState;

    QStringList watches = settings->loadWatches( folderId );

    if ( watches.isEmpty() ) {
        folderState = NotWatched;

        m_map.remove( folderId );
    } else {
        folderState = WatchNormal;

        WatchList list;

        for ( int i = 0; i < watches.count(); i++ ) {
            Watch watch;
            watch.m_filter = watches[ i ];
            watch.m_total = 0;
            watch.m_new = 0;
            watch.m_updated = 0;

            if ( !watches[ i ].isEmpty() ) {
                QList<ColumnCondition> conditions = settings->loadFilter( watches[ i ] );
                m_filter->setConditions( conditions );
            } else {
                m_filter->setConditions( QList<ColumnCondition>() );
            }

            RDB::ForeignConstIterator<IssueRow> it( dataManager->issues()->parentIndex(), folderId );
            while ( it.next() ) {
                int issueId = it.get()->issueId();
                if ( m_filter->filterRow( issueId ) ) {
                    watch.m_total++;
                    WatchState issueState = dataManager->issueWatchState( issueId, true );
                    if ( issueState == WatchUpdated ) {
                        watch.m_updated++;
                        if ( folderState < WatchUpdated )
                            folderState = WatchUpdated;
                    } else if ( issueState == WatchNew ) {
                        watch.m_new++;
                        if ( folderState < WatchNew )
                            folderState = WatchNew;
                    }
                }
            }

            list.append( watch );
        }

        m_map.insert( folderId, list );
    }

    if ( dataManager->folderWatchState( folderId ) != folderState ) {
        dataManager->setFolderWatchState( folderId, folderState );

        viewManager->postViewEvent( NULL, ViewEvent::UpdateWatches, folderId );
    }
}

bool FolderWatchProvider::isEmpty()
{
    return m_map.isEmpty();
}

void FolderWatchProvider::setupColumns( QTreeWidget* tree )
{
    QTreeWidgetItem* item = new QTreeWidgetItem();

    item->setText( 0, tr( "Name" ) );
    item->setTextAlignment( 0, Qt::AlignVCenter | Qt::AlignLeft );

    item->setText( 1, tr( "Total" ) );
    item->setTextAlignment( 1, Qt::AlignVCenter | Qt::AlignRight );

    item->setText( 2, tr( "New" ) );
    item->setTextAlignment( 2, Qt::AlignVCenter | Qt::AlignRight );

    item->setText( 3, tr( "Modified" ) );
    item->setTextAlignment( 3, Qt::AlignVCenter | Qt::AlignRight );

    item->setText( 4, QString() );

    tree->setHeaderItem( item );

    tree->setColumnWidth( 0, 150 );
    tree->setColumnWidth( 1, 70 );
    tree->setColumnWidth( 2, 70 );
    tree->setColumnWidth( 3, 70 );
    tree->setColumnWidth( 4, 0 );
}

void FolderWatchProvider::populateTree( QTreeWidget* tree )
{
    tree->clear();

    RDB::IndexConstIterator<ProjectRow> itp( dataManager->projects()->index() );
    QList<const ProjectRow*> projects = localeAwareSortRows( itp, &ProjectRow::name );

    for ( int i = 0; i < projects.count(); i++ ) {
        int projectId = projects.at( i )->projectId();

        RDB::ForeignConstIterator<FolderRow> itf( dataManager->folders()->parentIndex(), projectId );
        QList<const FolderRow*> folders = localeAwareSortRows( itf, &FolderRow::name );

        QTreeWidgetItem* project = NULL;

        for ( int j = 0; j < folders.count(); j++ ) {
            int folderId = folders.at( j )->folderId();

            WatchList watches = m_map.value( folderId );
            if ( !watches.isEmpty() ) {
                if ( !project ) 
                    project = addProjectItem( tree, projects.at( i ) );

                QTreeWidgetItem* folder = addFolderItem( project, folders.at( j ) );

                qSort( watches.begin(), watches.end(), Watch::compare );

                for ( int k = 0; k < watches.count(); k++ ) {
                    QString filter = watches.at( k ).m_filter;

                    QTreeWidgetItem* item;
                    if ( filter.isEmpty() )
                        item = folder;
                    else
                        item = addFilterItem( folder, folderId, filter );

                    setCount( item, 1, watches.at( k ).m_total );
                    setCount( item, 2, watches.at( k ).m_new );
                    setCount( item, 3, watches.at( k ).m_updated );
                }
            }
        }
    }

    tree->expandAll();
}

QTreeWidgetItem* FolderWatchProvider::addProjectItem( QTreeWidget* tree, const ProjectRow* project )
{
    QTreeWidgetItem* item = new QTreeWidgetItem( tree );
    item->setText( 0, project->name() );
    item->setIcon( 0, IconLoader::icon( "project" ) );

    QFont font = item->font( 0 );
    font.setBold( true );
    item->setFont( 0, font );

    return item;
}

QTreeWidgetItem* FolderWatchProvider::addFolderItem( QTreeWidgetItem* parent, const FolderRow* folder )
{
    QTreeWidgetItem* item = new QTreeWidgetItem( parent );
    item->setText( 0, folder->name() );
    item->setIcon( 0, IconLoader::icon( "folder" ) );

    item->setData( 0, FolderIdRole, folder->folderId() );

    return item;
}

QTreeWidgetItem* FolderWatchProvider::addFilterItem( QTreeWidgetItem* parent, int folderId, const QString& filter )
{
    QTreeWidgetItem* item = new QTreeWidgetItem( parent );
    item->setText( 0, filter );
    item->setIcon( 0, IconLoader::icon( "filter" ) );

    item->setData( 0, FolderIdRole, folderId );
    item->setData( 0, FilterNameRole, filter );

    return item;
}

void FolderWatchProvider::setCount( QTreeWidgetItem* item, int column, int count )
{
    item->setText( column, QString::number( count ) );
    item->setTextAlignment( column, Qt::AlignVCenter | Qt::AlignRight );

    if ( count > 0 ) {
        switch ( column ) {
            case 2:
                item->setIcon( column, IconLoader::icon( "overlay-yellow" ) );
                break;

            case 3:
                item->setIcon( column, IconLoader::icon( "overlay-green" ) );
                break;
        }
    } else {
        QColor bg = item->treeWidget()->palette().color( QPalette::Active, QPalette::Base );
        QColor fg = item->treeWidget()->palette().color( QPalette::Active, QPalette::Text );
        item->setForeground( column, QColor( ( fg.red() + 2 * bg.red() ) / 3,
            ( fg.green() + 2 * bg.green() ) / 3, ( fg.blue() + 2 * bg.blue() ) / 3 ) );
    }
}

bool FolderWatchProvider::Watch::compare( const Watch& w1, const Watch& w2 )
{
    return QString::localeAwareCompare( w1.m_filter, w2.m_filter ) < 0;
}
