/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2008 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "attributesettingsdialog.h"

#include <QLayout>
#include <QLabel>
#include <QToolButton>
#include <QHeaderView>
#include <QPushButton>
#include <QComboBox>
#include <QListWidget>
#include <QDialogButtonBox>

#include "data/datamanager.h"
#include "models/treeviewsettings.h"
#include "models/tablemodelshelper.h"
#include "widgets/gradientwidget.h"
#include "iconloader.h"

using namespace WebIssues;

/* TRANSLATOR WebIssues::AttributeSettingsDialog */

AttributeSettingsDialog::AttributeSettingsDialog( QWidget* parent ) : QDialog( parent )
{
    setWindowTitle( tr( "Attribute Settings" ) );

    QVBoxLayout* topLayout = new QVBoxLayout( this );
    topLayout->setMargin( 0 );
    topLayout->setSpacing( 0 );

    QWidget* promptWidget = new GradientWidget( this );
    topLayout->addWidget( promptWidget );

    QHBoxLayout* promptLayout = new QHBoxLayout( promptWidget );
    promptLayout->setSpacing( 10 );

    QLabel* promptPixmap = new QLabel( promptWidget );
    promptPixmap->setPixmap( IconLoader::pixmap( "configure-columns", 22 ) );
    promptLayout->addWidget( promptPixmap, 0 );

    m_promptLabel = new QLabel( promptWidget );
    m_promptLabel->setWordWrap( true );
    m_promptLabel->setMinimumWidth( 250 );
    promptLayout->addWidget( m_promptLabel, 1 );

    QFrame* separator = new QFrame( this );
    separator->setFrameStyle( QFrame::HLine | QFrame::Sunken );
    topLayout->addWidget( separator );

    QVBoxLayout* mainLayout = new QVBoxLayout();
    mainLayout->setMargin( 9 );
    mainLayout->setSpacing( 6 );
    topLayout->addLayout( mainLayout );

    QGridLayout* gridLayout = new QGridLayout();
    mainLayout->addLayout( gridLayout );

    m_list = new QListWidget( this );
    gridLayout->addWidget( m_list, 0, 0 );

    connect( m_list, SIGNAL( currentItemChanged( QListWidgetItem*, QListWidgetItem* ) ), this, SLOT( itemSelected() ) );

    QVBoxLayout* upDownLayout = new QVBoxLayout();
    gridLayout->addLayout( upDownLayout, 0, 1 );

    upDownLayout->addStretch( 1 );

    m_upButton = new QToolButton( this );
    m_upButton->setIcon( IconLoader::icon( "move-up" ) );
    m_upButton->setAutoRepeat( true );
    upDownLayout->addWidget( m_upButton );

    m_downButton = new QToolButton( this );
    m_downButton->setIcon( IconLoader::icon( "move-down" ) );
    m_downButton->setAutoRepeat( true );
    upDownLayout->addWidget( m_downButton );

    upDownLayout->addStretch( 1 );

    connect( m_upButton, SIGNAL( clicked() ), this, SLOT( upClicked() ) );
    connect( m_downButton, SIGNAL( clicked() ), this, SLOT( downClicked() ) );

    mainLayout->addSpacing( 10 );

    m_buttonBox = new QDialogButtonBox( QDialogButtonBox::Ok | QDialogButtonBox::Apply | QDialogButtonBox::Cancel,
        Qt::Horizontal, this );
    mainLayout->addWidget( m_buttonBox );

    m_buttonBox->button( QDialogButtonBox::Ok )->setText( tr( "&OK" ) );
    m_buttonBox->button( QDialogButtonBox::Apply )->setText( tr( "&Apply" ) );
    m_buttonBox->button( QDialogButtonBox::Cancel )->setText( tr( "&Cancel" ) );

    connect( m_buttonBox, SIGNAL( accepted() ), this, SLOT( accept() ) );
    connect( m_buttonBox, SIGNAL( rejected() ), this, SLOT( reject() ) );
    connect( m_buttonBox->button( QDialogButtonBox::Apply ), SIGNAL( clicked() ), this, SIGNAL( settingsApplied() ) );

    resize( 100, 350 );
}

AttributeSettingsDialog::~AttributeSettingsDialog()
{
}

void AttributeSettingsDialog::setPrompt( const QString& text )
{
    m_promptLabel->setText( text );

    m_promptLabel->setFixedHeight( m_promptLabel->heightForWidth( m_promptLabel->minimumWidth() ) );
}

void AttributeSettingsDialog::setAttributes( const QList<int>& attributes )
{
    m_list->clear();

    for ( int i = 0; i < attributes.count(); i++ ) {
        int attributeId = attributes.at( i );
        QListWidgetItem* item = new QListWidgetItem( m_list );
        item->setText( TableModelsHelper::attributeName( attributeId ) );
        item->setData( Qt::DecorationRole, IconLoader::pixmap( "attribute" ) );
        item->setData( Qt::UserRole, QVariant( attributeId ) );
    }

    m_upButton->setEnabled( false );
    m_downButton->setEnabled( false );
}

QList<int> AttributeSettingsDialog::attributes() const
{
    QList<int> attributes;

    for ( int i = 0; i < m_list->count(); i++ ) {
        QListWidgetItem* item = m_list->item( i );
        int attributeId = item->data( Qt::UserRole ).toInt();
        attributes.append( attributeId );
    }

    return attributes;
}

void AttributeSettingsDialog::itemSelected()
{
    QListWidgetItem* item = m_list->currentItem();
    if ( item ) {
        int row = m_list->row( item );

        if ( row > 0 )
            m_upButton->setEnabled( true );
        else
            m_upButton->setEnabled( false );

        if ( row < m_list->count() - 1 )
            m_downButton->setEnabled( true );
        else
            m_downButton->setEnabled( false );
    } else {
        m_upButton->setEnabled( false );
        m_downButton->setEnabled( false );
    }
}

void AttributeSettingsDialog::upClicked()
{
    QListWidgetItem* item = m_list->currentItem();

    int row = m_list->row( item );
    if ( row <= 0 )
        return;

    m_list->takeItem( row );
    m_list->insertItem( row - 1, item );

    m_list->setCurrentItem( item );

    itemSelected();
}

void AttributeSettingsDialog::downClicked()
{
    QListWidgetItem* item = m_list->currentItem();

    int row = m_list->row( item );
    if ( row >= m_list->count() - 1 )
        return;

    m_list->takeItem( row );
    m_list->insertItem( row + 1, item );

    m_list->setCurrentItem( item );

    itemSelected();
}

void AttributeSettingsDialog::accept()
{
    emit settingsApplied();

    QDialog::accept();
}
