#ifndef GVT_MANAGER_OBJECT_RECORD_H
#define GVT_MANAGER_OBJECT_RECORD_H

// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

#include "warped.h"

// a token can be white, red, or uncolored
enum gVTTokenColor {WHITE, RED, UNCOLORED};

// are we sending or receiving
enum messageStatus {SEND, RECEIVED};

/** The GVTManagerObjectRecord class.

This class is an abstract base class for GVTManagers. Specific
GVTManager implementations will derive from this base
class. Currently it doesn't contain any virtual methods as none
are needed at the moment. When other GVTManager implementations
are constructed, I am sure we will require virtual methods in this
class.

*/
class GVTManagerObjectRecord {
public:
  /**@name Public Class Methods of GVTManagerObjectRecord. */
  //@{

  /// Constructor.
  GVTManagerObjectRecord(){};

  /// Destructor.
  ~GVTManagerObjectRecord(){};

  //@} // End of Public Class Methods of GVTManagerObjectRecord.
};

/** The MatterObjectRecord class.

The MatternObjectRecord class is the data structure that the
MatternGVTManager uses to keep track of sent/received messages of
each simulation manager in the distributed simulation. Currently
an array of these data structure is allocated by the
MatternGVTManager. The size of the array is equal to the number of
simulation managers in the simulation. Each entry in this array
contains the color of the simulation manager, the number of white
messages the simulation manager has sent, the number of iterations
the gvt token has taken (currently this is used only by the
initiator - simulation manager 0), and the minimal timestamp of
the red messages.

*/
class MatternObjectRecord : public GVTManagerObjectRecord {
public:

  /**@name Public Class Methods of MatternObjectRecord. */
  //@{

  /// Constructor.
  MatternObjectRecord():
    color( WHITE ),
    tokenIterationNumber(0),
    tMin( 0 ){}

  /// Destructor.
  ~MatternObjectRecord(){};

  /** Set the color

  @param newColor the new color type
  */
  void setColor(gVTTokenColor newColor){
    color = newColor;
  }

  /** Set the minimal timestamp of red messages

  @param newTime the new time to set it to
  */
  void setMinTimeStamp(const VTime& newTime){
    tMin = newTime.clone();
  }

  /** Set the number of white messages.

  @param newValue the new value for number of white messages
  */
  void setNumberOfWhiteMessages(int newValue){
    numberOfWhiteMessages = newValue;
  }

  /// increment the token iteration number
  void incrementTokenIterationNumber(){
    tokenIterationNumber++;
  }

  /** return the token iteration number

  @return tokenIterationNumber the token iteration number
  */
  unsigned int getTokenIterationNumber(){
    return tokenIterationNumber;
  }

  /** Set the token iteration number

  @param value the new value for the token iteration number
  */
  void setTokenIterationNumber(unsigned int value){
    tokenIterationNumber = value;
  }

  /// increment the number of white messages
  void incrementNumberOfWhiteMessages(){
    numberOfWhiteMessages++;
  }

  /// decrement the number of white messages
  void decrementNumberOfWhiteMessages(){
    numberOfWhiteMessages--;
  }
   
  /** Get the color of this simulation manager.

  @return color the color type of this simulation manager
  */
  gVTTokenColor getColor(){
    return color;
  }

  /** Get the number of white messages.

  @return numberOfWhiteMesssages the number of white messages
  */
  int getNumberOfWhiteMessages(){
    return numberOfWhiteMessages;
  }

  /** Get the minimal timestamp of red messages.

  @return tMin the minimal timestamp of red messages
  */
  const VTime &getMinTimeStamp(){
    return *tMin;
  }

  //@} // End of Public Class Methods of MatternObjectRecord.
   
private:

  /**@name Private Class Attributes of MatternObjectRecord. */
  //@{

  /// color of this simulation manager
  gVTTokenColor color;

  /// the number of white messages it has sent out
  int numberOfWhiteMessages;

  /// the number of iterations the gvttoken has taken
  unsigned int tokenIterationNumber;

  /// the minimal timestamp of red messages
  VTime *tMin;

  //@} // End of Private Class Attributes of MatternObjectRecord.
};
#endif
   
