// Copyright (c) The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Malolan Chetlur             mal@ececs.uc.edu
//          Jorgen Dahl                 dahlj@ececs.uc.edu
//          Dale E. Martin              dmartin@cliftonlabs.com
//          Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Dhananjai Madhava Rao       dmadhava@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
// 
// $Id: FactoryImplementationBase.cpp
// 
//---------------------------------------------------------------------------

#include "ObjectStub.h"
#include "FactoryImplementationBase.h"

FactoryImplementationBase::FactoryImplementationBase(FactoryImplementationBase *parent){
   parentFactory = parent;
   mySymbolTable = new FactorySymbolTable(true);
}

FactoryImplementationBase::~FactoryImplementationBase(){
   mySymbolTable->deleteAllSymbols();
   delete mySymbolTable;
}

bool
FactoryImplementationBase::add(ObjectStub *stub){
   return mySymbolTable->addSymbol((Factory *)stub);
}

bool
FactoryImplementationBase::add(FactoryImplementationBase *subFactory){
   return mySymbolTable->addSymbol((Factory *)subFactory);
}

ObjectStub *
FactoryImplementationBase::isObjectPresent( const string &objectName ){

   string *prefix = getPrefix(objectName);
   if(prefix->empty()){ 
      // empty string - return NULL
      delete prefix;
      return NULL;
   }

   Factory *searchResult = NULL;

   if((searchResult = mySymbolTable->searchSymbol(*prefix)) == NULL){
      delete prefix;
      return NULL;
   }
   
   if(searchResult->getKind() == STUB) {
      delete prefix;
      return (ObjectStub *)searchResult;
   }

   string *suffix = getSuffix(objectName);
   ObjectStub *stub =
      ((FactoryImplementationBase *)searchResult)->isObjectPresent(*suffix);
   delete suffix;
   return stub;
}

ObjectStub *
FactoryImplementationBase::isObjectPresent(const Factory *objectClass){
   return isObjectPresent(objectClass->getName());
}

FactoryImplementationBase *
FactoryImplementationBase::getParentFactory() const {
   return parentFactory;
}

FactoryImplementationBase *
FactoryImplementationBase::getMainFactory() const {
   FactoryImplementationBase *parent = parentFactory;

   while (parent->getParentFactory() != NULL) {
      parent = parent->getParentFactory();
   }

   return parent;
}

string
FactoryImplementationBase::listAll(const string &prefix){
  Factory *base;

  string newPrefix = prefix + getName();
  string list;

  base = mySymbolTable->iterativeSearch(ANY);
  while (base != NULL) {
    if (base->getKind() == STUB) {
      list += newPrefix + "." + base->getName();
      list += " [" + ((ObjectStub *)base)->getInformation() + "]\n";
    }
    else {
      list +=
	((FactoryImplementationBase *)base)->listAll(newPrefix);
    }
    
    base = mySymbolTable->successor();
  }

  return list;
}

string * 
FactoryImplementationBase::getPrefix(const string &objectName){
   string *returnString = NULL;
   string::size_type position = objectName.find(".");
   if(position != string::npos){
      // everything before the "." is the prefix
      returnString = new string(objectName.substr(0,position));
   }
   else {
      returnString = new string(objectName);
   }

   return returnString;
}

string *
FactoryImplementationBase::getSuffix(const string &objectName){
   string *returnString = NULL;
   string::size_type position = objectName.find(".");
   if(position != string::npos){
      // everything after the "." is the suffix
      returnString = new string(objectName.substr(position+1));
   }
   else {
      returnString = new string(objectName);
   }

   return returnString;
}

