
// Copyright (c) 1995-2003 The University of Cincinnati.
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF THE
// SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
// IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE,
// OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE FOR ANY DAMAGES SUFFERED BY
// LICENSEE AS A RESULT OF USING, RESULT OF USING, MODIFYING OR
// DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the U.S.,
// and the terms of this license.

// You may modify, distribute, and use the software contained in this
// package under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE"
// version 2, June 1991. A copy of this license agreement can be found in
// the file "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	philip.wilsey@ieee.org
//          Dale E. Martin	dmartin@cliftonlabs.com

//---------------------------------------------------------------------------
// 
// $Id: std_textioPkg.cc,v 1.1 2003/12/04 19:13:19 dmartin Exp $
// 
//---------------------------------------------------------------------------

#include "AccessVariable.hh"
#include "std_textioPkg.hh"
#include "VHDLKernel.hh"
#include "AccessType.hh"
#include "SharedFileType.hh"
#include "UniversalLongLongInteger.hh"
#include "UniversalReal.hh"
#include <sstream>
#include <iostream>

using std::ostringstream;
using std::ends;
using std::cout;

static AccessVariable
buildTempVar( const AccessType &line ){
  ostringstream outbuf;

  if( line.val != NULL ){
    outbuf << line;
  }

  AccessVariable tempStr;
  tempStr.setVal( outbuf.str() );

  return tempStr;
}

static void
buildLineVal( int,
	      AccessType &line,
	      AccessVariable &tempStr ){
  delete line.val;
  line.val = new ArrayType( ObjectBase::VARIABLE,
			    SavantstringType_info, 
			    -1,
			    tempStr.getVal().c_str() );
}

static void
buildLineVal( AccessType &line,
	      AccessVariable &tempStr ){
  buildLineVal( 0, line, tempStr );
}

static char *SavantsideType_info_imageMap[] = {"right", "left"};
EnumerationTypeInfo SavantsideType_info(2, SavantsideType_info_imageMap, 0, ArrayInfo::to);

int
savantread( VHDLKernel *,
	    AccessVariable &line, 
	    const PhysicalType &var ){
  int startPos = line.getCurrentPos();
  double timeval = 0.0;
  char *endptr = 0;


  if (eatwhite(line) == false) {
    abort();
  }

  const char *tempLine = line.getVal().c_str();
  unsigned int tempPos = line.getCurrentPos();
   
  // extract decimal value
  timeval = strtod( &tempLine[tempPos],
		    &endptr );

  if ( endptr == &(tempLine[tempPos]) && timeval == 0 ) {
    // strtod failed to make a number out of the string
    cout << " read(int time) failed to find a number for time" << endl;
    abort();
  }
  else {
    // now get the units
    line.setCurrentPos( tempPos + (endptr - &(tempLine[tempPos] )) );
    if (eatwhite(line) == false) {
      abort();
    }
    tempPos = line.getCurrentPos();
    // handle times with units fs, ps, ns, us, ms
    if (toupper(tempLine[tempPos + 1]) == 'S') {
      double tempTimeval = timeval;
      switch (toupper(tempLine[tempPos])) {
      case 'F':
	break;
      case 'P':
	tempTimeval *= 1000.0;
	break;   
      case 'N':
	tempTimeval *= 1000000.0;
	break;
      case 'U':
	tempTimeval *= 1000000000.0;
	break;
      case 'M':
	tempTimeval *= 1.0e12;
	break;
      default:
	cout << " read(int time) warning: bad units! ("
	     << tempLine[tempPos] << tempLine[tempPos + 1]
	     << ")" << endl;
	line.setCurrentPos( startPos );
	abort();
      }
      var.getObject()->updateVal( UniversalLongLongInteger(tempTimeval) );
      line.setCurrentPos( line.getCurrentPos() + 2 );
    }
    else {
      double tempTimeval = timeval;
      // handle times with units sec, min, hr
      if (toupper(tempLine[tempPos]) == 'S' &&
	  toupper(tempLine[tempPos + 1]) == 'E' &&
	  toupper(tempLine[tempPos + 2]) == 'C') {
	tempTimeval *= 1.0e15;
	line.setCurrentPos( line.getCurrentPos() + 3 );
      } 
      else if (toupper(tempLine[tempPos]) == 'M' &&
		 toupper(tempLine[tempPos + 1]) == 'I' &&
		 toupper(tempLine[tempPos + 2]) == 'N') {
	tempTimeval *= 6.0 * 1.0e15;
	line.setCurrentPos( line.getCurrentPos() + 3 );
      }
      else if (toupper(tempLine[tempPos]) == 'H' &&
	       toupper(tempLine[tempPos + 1]) == 'R') {
	tempTimeval *= 3600 * 1.0e15;
	line.setCurrentPos( line.getCurrentPos() + 2 );
      }
      else {
	cout<< " read(VTime) warning: time units bogus" << endl;
	abort();
      }
      var.getObject()->updateVal(UniversalLongLongInteger(tempTimeval));
    }
  }
  return NORMAL_RETURN;
}

int
savantwrite( VHDLKernel *,
	     AccessVariable &line, 
	     const PhysicalType &var,
	     const VHDLData::Side_t justified, 
	     const int field,
	     VHDLData::TimeUnit_t unit ){

  static const char *units[] =  { "FS", "PS", "NS", "US", "MS", "SEC", "MIN", "HR" };
  static const int unitmult[] = { 1000, 1000, 1000, 1000, 1000, 60, 60 };
  double num;
  LONG value;
  int originalLength;
  unsigned int length = 0;

  string numstring;
  ostringstream numstream;

  value = longlongint(dynamic_cast<const UniversalLongLongInteger &>(var.getObject()->readVal()));
  // set output format, remove trailing zeros
  num = double(value);
  numstream.setf(ios::fixed, ios::floatfield);
  numstream.unsetf(ios::showpoint);
  if( num < 100 ){
    if( value % 10 == 0 ){
      numstream.precision(5);
    }
    else {
      numstream.precision(6);
    }
  }
  for ( int i = 0; i < (int)unit; i++ ){ // convert FS time to appropriate value
    num = num / unitmult[i];
  }
  if( num == (LONG)num ){
    numstream.precision(0);
    numstream << (LONG)num << " " << units[(int)unit];
  } else {
    numstream << num << " " << units[(int)unit];
  }
  numstring = numstream.str();
  length = numstring.size();
  originalLength = length;
  int spaceLength = field - length;

  // create new string
  if( field > (int)length ){  // space for user-specified padding
    length = field;
  }

  string newLine = line.getVal();

#ifdef VHDLDBG
  cout << "::write(integer) adding |" <<  numstring<< "| to line|";
  line.print(cout);
  cout << "|" << endl;
#endif

  if (field < originalLength) {
    newLine += numstring;
  }
  else {
    if (justified == VHDLData::LEFT) { 
      newLine += numstring;
      for ( int i = 0; i < spaceLength; i++ ){
	newLine += " ";
      }
    }
    if (justified == VHDLData::RIGHT) {
      for ( int i=0; i < spaceLength; i++ ){
	newLine += " ";
      }
      newLine += numstring;
    }
  }//endofelse
 
  line.setVal( newLine );

  return NORMAL_RETURN;
}

int savantread(VHDLKernel*, AccessVariable &line, VHDLVTime &time) {
  PhysicalType val(ObjectBase::VARIABLE);
  savantread(NULL, line, val);
  time = val.getVTime();
  return NORMAL_RETURN;
}

int 
savantwrite( VHDLKernel *, 
	     AccessVariable &line, 
	     const VHDLVTime &time, 
	     const VHDLData::Side_t justified, 
	     const int field, 
	     VHDLData::TimeUnit_t unit) {

  return savantwrite( NULL, 
		      line, 
		      PhysicalType( ObjectBase::VARIABLE, 
				    UniversalLongLongInteger(time.getMajor()),
				    SavanttimeType_info), 
		      justified, 
		      field, 
		      unit );
}

int 
savantread_bit( VHDLKernel *, 
		AccessVariable &line, 
		EnumerationType &var ){
  ASSERT (var.getTypeInfo().get_imageMap() == SavantbitType_info.get_imageMap());

  if (eatwhite(line) == false) {
    abort();
  }
  
  UniversalInteger tempval((int)line.getVal().c_str()[line.getCurrentPos()] - '0');
  var.getObject()->updateVal(tempval);
  line.setCurrentPos( line.getCurrentPos() + 1 );
  return NORMAL_RETURN;
}

int
savantread_character( VHDLKernel *,
		      AccessVariable &line,
		      EnumerationType &var ){
  ASSERT (var.getTypeInfo().get_imageMap() == SavantcharacterType_info.get_imageMap());
  
  if (eatwhite(line) == false) {
    abort();
  }
  var.getObject()->updateVal(UniversalInteger((int)line.getVal().c_str()[line.getCurrentPos()]));
  line.setCurrentPos( line.getCurrentPos() + 1 );
  return NORMAL_RETURN;
}

int
savantread_severity_level( VHDLKernel*,
			   AccessVariable &line,
			   EnumerationType &var ){
  ASSERT (var.getTypeInfo().get_imageMap() == Savantseverity_levelType_info.get_imageMap());

  if (eatwhite(line) == false) {
    abort();
  }

  switch (line.getVal().c_str()[line.getCurrentPos()]) {
  case 'N':
    var.getObject()->updateVal(UniversalInteger(0));
    line.setCurrentPos( line.getCurrentPos() + 4 );
    break;

  case 'W':
    var.getObject()->updateVal(UniversalInteger(1));
    line.setCurrentPos( line.getCurrentPos() + 7 );
    break;

  case 'E':
    var.getObject()->updateVal(UniversalInteger(2));
    line.setCurrentPos( line.getCurrentPos() + 5 );
    break;

  case 'F':
    var.getObject()->updateVal(UniversalInteger(3));
    line.setCurrentPos( line.getCurrentPos() + 7 );
    break;

  default:
    cerr << "Illegal severity_level value read !!" << endl;
    abort();
  }
  
  return NORMAL_RETURN;
}

int
savantread_boolean( VHDLKernel *,
		    AccessVariable &line,
		    EnumerationType &var ){
  ASSERT (var.getTypeInfo().get_imageMap() == SavantbooleanType_info.get_imageMap());

  if (eatwhite(line) == false) {
    abort();
  }

  if (line.getVal().c_str()[line.getCurrentPos()] == 'T') {
    var.getObject()->updateVal(UniversalInteger(1));
    line.setCurrentPos( line.getCurrentPos() + 4 );
  }
  else if (line.getVal().c_str()[line.getCurrentPos()] == 'F') {
    var.getObject()->updateVal(UniversalInteger(0));
    line.setCurrentPos( line.getCurrentPos() + 5 );
  }
  else {
    cerr << "Illegal boolean value read !!" << endl;
    abort();
  }

  return NORMAL_RETURN;
}

int 
savantread( VHDLKernel *kb,
	    AccessVariable &line,
	    EnumerationType &var ){
  if (eatwhite(line) == false) {
    abort();
  }

  if( var.getTypeInfo().get_imageMap() == SavantcharacterType_info.get_imageMap() ){
    savantread_character(kb, line, var);
  }
  else if( var.getTypeInfo().get_imageMap() == SavantbitType_info.get_imageMap() ){
    savantread_bit(kb, line, var);
  }
  else if ( var.getTypeInfo().get_imageMap() == SavantbooleanType_info.get_imageMap() ){
    savantread_boolean(kb, line, var);
  }
  else if( var.getTypeInfo().get_imageMap() == Savantseverity_levelType_info.get_imageMap() ){
    savantread_severity_level(kb, line, var);
  }
  else {
    UniversalInteger tempval(line.getVal().c_str()[line.getCurrentPos()] - '0');
    var.getObject()->updateVal(tempval);
    line.setCurrentPos( line.getCurrentPos() + 1 );
  }
  
  return NORMAL_RETURN;
}

int 
savantwrite( VHDLKernel*,
	     AccessVariable &line, 
	     const EnumerationType &var,
	     const VHDLData::Side_t justified,
	     const int field ){
  string newLine;
  string intString;
  ostringstream outstream;

  if( var.getTypeInfo().get_imageMap() == SavantcharacterType_info.get_imageMap() ){
    // Okay. This is a simple character so dump it as is...
    outstream << (char)var.getIntValue();
  }
  else if( var.getTypeInfo().get_imageMap() == SavantbitType_info.get_imageMap() ){
    // Okay. This is a simple bit so dump it as is...
    outstream << (char)(var.getIntValue() + '0');
  }
  else if( var.getTypeInfo().get_imageMap() == SavantbooleanType_info.get_imageMap() ){
    int value = var.getIntValue();
    outstream << var.getTypeInfo().get_imageMap()[value];
  }
  else {
    outstream << var.getTypeInfo().get_imageMap()[var.getIntValue()];
  }
  intString = outstream.str();
  unsigned int length = intString.size();
  unsigned int originalLength = length;
  unsigned int spaceLength = field - length;

  if( field > (int)originalLength ){  // space for user-specified padding
    length = field;
  }
  newLine = line.getVal();
  
  if( field < (int)originalLength ){
    newLine += intString;
  }
  else {
    if (justified == VHDLData::LEFT) { 
      newLine += intString;
      for ( unsigned int i = 0 ;
	    i < spaceLength; 
	    i++ ){
	newLine += ' ';
      }
    }
    if (justified == VHDLData::RIGHT) {
      for ( unsigned int i = 0;
	    i < spaceLength; i++ ){
	newLine += ' ';
      }
      newLine += intString;
    }
  } //endofelse
  line.setVal( newLine );

  return NORMAL_RETURN;
}

int
savantread( VHDLKernel*,
	    AccessVariable &line, 
	    IntegerType &var ){
  int integerVal =0;
  char *endPtr;
  
  if (eatwhite(line) == false) {
    abort();
  }
  
  // extract integer value
  integerVal = strtol((const char *)&(line.getVal().c_str()[line.getCurrentPos()]), &endPtr, 10);
  if (endPtr == &(line.getVal().c_str()[line.getCurrentPos()]) && integerVal == 0) {
    cout << "SavantintegerType::read(int time) failed to find a number "
	 <<"for time" << endl;
    abort();
  }
  else {
    line.setCurrentPos( line.getCurrentPos() + 
			endPtr - &line.getVal().c_str()[line.getCurrentPos()] );
    var.getObject()->updateVal(UniversalInteger(integerVal));
  }
  return NORMAL_RETURN;
}

int 
savantwrite( VHDLKernel *,
	     AccessVariable &line, 
	     const IntegerType &var,
	     const VHDLData::Side_t justified, 
	     const int field ){
  string intString, newLine;
  ostringstream outstream;
  var.getObject()->readVal().print(outstream);
  intString = outstream.str();

  unsigned int length = intString.size();
  int originalLength;

  int spaceLength = field - length;
  originalLength = length;
#ifdef VHDLDBG
  cout << "savantwrite(SavantintegerType) adding |" <<  intString << "| to line|";
  line.print(cout);
  cout << "|" << endl;
#endif

  if (field > originalLength) {  // space for user-specified padding
    length = field;
  }
  newLine += line.getVal();

  if (field < originalLength) {
    newLine += intString;
  }
  else {
    if( justified == VHDLData::LEFT ){ 
      newLine += intString;
      for ( int i = 0; i < spaceLength; i++ ){
	newLine += " ";
      }
    }
    if( justified == VHDLData::RIGHT ){
      for ( int i = 0; i < spaceLength; i++ ){
	newLine += " ";
      }
      newLine += intString;
    }
  } //endofelse

  line.setVal( newLine );

  return NORMAL_RETURN;
}                      

int
savantread( VHDLKernel *,
	    AccessVariable &line, 
	    RealType &var) {
  double doubleVal =0;
  char *endPtr;

  if (eatwhite(line) == false) {
    abort();
  }

  // extract integer value
  doubleVal = strtod((const char *)&(line.getVal().c_str()[line.getCurrentPos()]), &endPtr);
  if( endPtr == &(line.getVal().c_str()[line.getCurrentPos()]) && doubleVal == 0 ){
    cout << "SavantrealType::read failed to find a number" << endl;
    abort();
  }
  else {
    line.setCurrentPos( line.getCurrentPos() +
			endPtr - &line.getVal().c_str()[line.getCurrentPos()] );
    var.getObject()->updateVal(UniversalReal(doubleVal));
  }
  return NORMAL_RETURN;
}

int
savantwrite( VHDLKernel *,
	     AccessVariable &line, 
	     const RealType &var,
	     const VHDLData::Side_t justified,
	     const int field,
	     const int digits ){
  string newLine;
  string intString;
  ostringstream outstream;

  if(digits == 0) {
    outstream.setf(ios::scientific);
  }
  var.getObject()->readVal().print(outstream);
  intString = outstream.str();

  unsigned int length = intString.size();
    
  unsigned int originalLength;
  unsigned int spaceLength = field - length;
  originalLength = length;

  if( field > (int)originalLength ){  // space for user-specified padding
    length = field;
  }
  newLine = line.getVal();

  if( field < (int)originalLength ) {
    newLine += intString;
  }
  else {
    if( justified == VHDLData::LEFT ){ 
      newLine += intString;
      for ( unsigned int i = 0;
	    i < spaceLength;
	    i++ ){
	newLine += ' ';
      }
    }
    else if( justified == VHDLData::RIGHT ){
      for ( unsigned int i = 0;
	    i < spaceLength;
	    i++ ){
	newLine += ' ';
      }
      newLine += intString;
    }
  } //endofelse

  line.setVal( newLine.c_str() );

  return NORMAL_RETURN;
}                      

int
savantread( VHDLKernel *kb,
	    AccessVariable &line,
	    VHDLType &var ){

  switch (var.get_kind()) {
  case INTEGER_TYPE:
    return savantread(kb, line, dynamic_cast<IntegerType &>(var));

  case ENUMERATION_TYPE:
    return savantread(kb, line, dynamic_cast<EnumerationType &>(var));

  case REAL_TYPE:
    return savantread(kb, line, dynamic_cast<RealType &>(var));

  case PHYSICAL_TYPE:
    return savantread(kb, line, dynamic_cast<PhysicalType &>(var));

  case ARRAY_TYPE:
    return savantread(kb, line, dynamic_cast<ArrayType &>(var));

  case RECORD_TYPE:
    return savantread(kb, line, dynamic_cast<RecordType &>(var));

  default:
    break;
  }

  return NORMAL_RETURN;
}

int
savantread( VHDLKernel *,
	    AccessVariable &line, 
	    ArrayType &var) {
  if( var.get_bounds(0).dirn() == ArrayInfo::to ){
    for( int i = var.get_bounds(0).left(); 
	 i <= var.get_bounds(0).right(); 
	 i++ ) {
      savantread(NULL, line, var[i]);
    }
  }
  else {
    for( int i = var.get_bounds(0).left();
	 i >= var.get_bounds(0).right();
	 i-- ){
      savantread(NULL, line, var[i]);
    }
  }
  return NORMAL_RETURN;
}

int
savantread( VHDLKernel *,
	    AccessVariable &line, 
	    RecordType &var) {
  for(int field = 1;
      field <= var.get_number_of_fields();
      field++ ){
    savantread(NULL, line, var.get_field(field));
  }
  
  return NORMAL_RETURN;
}

int
savantwrite( VHDLKernel *kb,
	     AccessVariable &line, 
	     const VHDLType &var,
	     const VHDLData::Side_t justified, 
	     const int field ){
  switch( var.get_kind() ){
  case INTEGER_TYPE:
    return savantwrite( kb, line, dynamic_cast<const IntegerType &>(var), justified, field);
  case ENUMERATION_TYPE:
    return savantwrite( kb, line, dynamic_cast<const EnumerationType &>(var), justified, field);
  case REAL_TYPE:
    return savantwrite( kb, line, dynamic_cast<const RealType &>(var), justified, field);
  case PHYSICAL_TYPE:
    return savantwrite( kb, line, dynamic_cast<const PhysicalType &>(var), justified, field);
  case ARRAY_TYPE:
    return savantwrite( kb, line, dynamic_cast<const ArrayType &>(var), justified, field);
  case RECORD_TYPE:
    return savantwrite( kb, line, dynamic_cast<const RecordType &>(var), justified, field);
  default:
    return NORMAL_RETURN;
  }
}
  
int
savantwrite( VHDLKernel *kb,
	     AccessVariable &line, 
	     const ArrayType &var,
	     const VHDLData::Side_t justified, 
	     const int field ){
  bool extraSpaceFlag = true;
  
  // Determine if element subtype of this array is character...
  if( var.getArrayTypeInfo().getElementTypeInfo() != NULL &&
      var.getArrayTypeInfo().getElementTypeInfo()->is_character_type() ){
    extraSpaceFlag = false;
  }
  
  if(var.get_bounds(0).dirn() == ArrayInfo::to) {
    for( int i = var.get_bounds(0).left(); i <= var.get_bounds(0).right(); i++ ){
      if( i > var.get_bounds(0).left() && extraSpaceFlag ){
	// Add a space to delimit fields...
	line.setVal( line.getVal() + " " );
      }
      savantwrite(kb, line, var[i], justified, field);
    }
  }
  else {
    for( int i = var.get_bounds(0).left(); i >= var.get_bounds(0).right(); i-- ){
      if( i > var.get_bounds(0).left() && extraSpaceFlag ){
	// Add a space to delimit fields...
	line.setVal( line.getVal() + " " );
      }
      savantwrite(kb, line, var[i], justified, field);
    }
  }
  return NORMAL_RETURN;
}

int
savantwrite( VHDLKernel *kb,
	     AccessVariable &line, 
	     const RecordType &var,
	     const VHDLData::Side_t justified, 
	     const int field ){
  for( int counter = 1; 
       counter <= var.get_number_of_fields(); 
       counter++ ){
    if (counter > 1) {
      line.setVal( line.getVal() + " " );
    }
    savantwrite(kb, line, var.get_field(counter), justified, field);
  }

  return NORMAL_RETURN;
}


// Modified from here

int
savantwrite( VHDLKernel *base, 
	     AccessType &line, 
	     const PhysicalType &var, 
	     const VHDLData::Side_t side, 
	     const int field, 
	     VHDLData::TimeUnit_t unit ){
  int retval = -1;

  AccessVariable tempStr = buildTempVar( line );
  retval = savantwrite( base, tempStr, var, side, field, unit );

  line.val = new ArrayType(ObjectBase::VARIABLE, SavantstringType_info, -1, tempStr.getVal().c_str() );

  return retval;
}

int
savantwrite( VHDLKernel *base, 
	     AccessType &line, 
	     const VHDLVTime &time, 
	     const VHDLData::Side_t side,
	     const int i,
	     VHDLData::TimeUnit_t t_unit ){

  int retval = -1;
  AccessVariable tempStr = buildTempVar( line );

  retval = savantwrite(base, tempStr, time, side, i, t_unit);
  
  line.val = new ArrayType( ObjectBase::VARIABLE,
			    SavantstringType_info, 
			    -1,
			    tempStr.getVal().c_str() );

  return retval;
}

int
savantwrite( VHDLKernel *base,
	     AccessType &line, 
	     const EnumerationType &var,
	     const VHDLData::Side_t justified,
	     const int field ){
  int returnValue = -1;

  AccessVariable tempStr = buildTempVar( line );

  returnValue = savantwrite( base, tempStr, var, justified, field );

  delete line.val;
  line.val = new ArrayType( ObjectBase::VARIABLE,
			    SavantstringType_info,
			    -1,
			    tempStr.getVal().c_str() );

  return returnValue;
}


int
savantwrite( VHDLKernel *kb,
	     AccessType &line, 
	     const VHDLType &var,
	     const VHDLData::Side_t justified, 
	     const int field ) {
  switch( var.get_kind() ){
  case INTEGER_TYPE:
    return savantwrite(kb, line, dynamic_cast<const IntegerType &>(var), justified, field);
    
  case ENUMERATION_TYPE:
    return savantwrite(kb, line, dynamic_cast<const EnumerationType &>(var), justified, field);
    
  case REAL_TYPE:
    return savantwrite(kb, line, dynamic_cast<const RealType &>(var), justified, field);
    
  case PHYSICAL_TYPE:
    return savantwrite(kb, line, dynamic_cast<const PhysicalType &>(var), justified, field);

  case ARRAY_TYPE:
    return savantwrite(kb, line, dynamic_cast<const ArrayType &>(var), justified, field);

  case RECORD_TYPE:
    return savantwrite(kb, line, dynamic_cast<const RecordType &>(var), justified, field);

  default:
    return NORMAL_RETURN;
  }
}

int
savantwrite( VHDLKernel *base,
	     AccessType &line, 
	     const ArrayType &var,
	     const VHDLData::Side_t justified, 
	     const int field ){
  int retval = -1;

  AccessVariable tempStr = buildTempVar( line );

  retval = savantwrite( base,
			tempStr,
			var,
			justified,
			field);
  buildLineVal( line, tempStr );

  return retval;
}

int
savantwrite( VHDLKernel *base,
	     AccessType &line, 
	     const RecordType &var,
	     const VHDLData::Side_t justified, 
	     const int field ){
  AccessVariable tempStr = buildTempVar( line );

  int retval = savantwrite(base, tempStr, var, justified, field);
  
  buildLineVal( line, tempStr );

  return retval;
}

int
savantwrite(VHDLKernel *base,
	    AccessType &line, 
	    const IntegerType &var, 
	    const VHDLData::Side_t justified, 
	    const int field ){

  AccessVariable tempStr = buildTempVar( line );
  int retval = savantwrite(base, tempStr, var, justified, field);
  buildLineVal( line, tempStr );

  return retval;
}

int
savantwrite( VHDLKernel *base,
	     AccessType &line, 
	     const RealType &var, const VHDLData::Side_t justified,
	     const int field ){
  AccessVariable tempStr = buildTempVar( line );
  int retval = savantwrite( base, tempStr, var, justified, field );
  buildLineVal( line, tempStr );
  return retval;
}

int
savantread( VHDLKernel *base,
	    AccessType &line,
	    PhysicalType &var ){

  int retval = -1;

  AccessVariable tempStr = buildTempVar( line );
  
  retval = savantread(base, tempStr, var);

  if( tempStr.getCurrentPos() < tempStr.length() ){
    buildLineVal( tempStr.getCurrentPos(), line, tempStr );
  }
  else {
    delete line.val;
    line.val = NULL;
  }
  
  return retval;
}
  
int
savantread( VHDLKernel *base,
	    AccessType &line,
	    EnumerationType &var ){
  int retval = -1;
  
  AccessVariable tempStr = buildTempVar( line );
  retval = savantread(base, tempStr, var);

  if( tempStr.getCurrentPos() < tempStr.length() ){
    buildLineVal( tempStr.getCurrentPos(), line, tempStr );
  }
  else {
    delete line.val;
    line.val = NULL;
  }
  
  return retval;
}
  
int
savantread( VHDLKernel *base,
	    AccessType &line,
	    IntegerType &var ){

  AccessVariable tempStr = buildTempVar( line );
  int retval = savantread(base, tempStr, var);

  if( tempStr.getCurrentPos() < tempStr.length() ) {
    buildLineVal( tempStr.getCurrentPos(), line, tempStr );
  }
  else {
    delete line.val;
    line.val = NULL;
  }
  
  return retval;
}
  
int
savantread( VHDLKernel *base,
	    AccessType &line,
	    RealType &var ){

  AccessVariable tempStr = buildTempVar( line );
  
  int retval = savantread( base, tempStr, var );

  if( tempStr.getCurrentPos() < tempStr.length() ){
    buildLineVal( tempStr.getCurrentPos(), line, tempStr );
  }
  else {
    delete line.val;
    line.val = NULL;
  }
  
  return retval;
}

int
savantread( VHDLKernel *kb,
	    AccessType &line, 
	    VHDLType &var ) {
  switch ( var.get_kind() ) {
  case INTEGER_TYPE:
    return savantread(kb, line, dynamic_cast<IntegerType &>(var));

  case ENUMERATION_TYPE:
    return savantread(kb, line, dynamic_cast<EnumerationType &>(var));

  case REAL_TYPE:
    return savantread(kb, line, dynamic_cast<RealType &>(var));

  case PHYSICAL_TYPE:
    return savantread(kb, line, dynamic_cast<PhysicalType &>(var));

  case ARRAY_TYPE:
    return savantread(kb, line, dynamic_cast<ArrayType &>(var));

  case RECORD_TYPE:
    return savantread(kb, line, dynamic_cast<RecordType &>(var));

  default:
    break;
  }

  return NORMAL_RETURN;
}
  

int
savantread( VHDLKernel *base,
	    AccessType &line,
	    ArrayType &var ){
  AccessVariable tempStr = buildTempVar( line );
  
  int retval = savantread(base, tempStr, var);

  if( tempStr.getCurrentPos() < tempStr.length() ){
    buildLineVal( tempStr.getCurrentPos(), line, tempStr );
  }
  else {
    delete line.val;
    line.val = NULL;
  }
  
  return retval;
}


int
savantread( VHDLKernel *base,
	    AccessType &line,
	    RecordType &var ){
  int retval = -1;

  AccessVariable tempStr = buildTempVar( line );
  retval = savantread( base, tempStr, var );
  if( tempStr.getCurrentPos() < tempStr.length() ){
    buildLineVal( tempStr.getCurrentPos(), line, tempStr );
  }
  else {
    delete line.val;
    line.val = NULL;
  }
  
  return retval;
}

// Second set of modifications start here

int
savantwrite_time( VHDLKernel *base,
		  AccessType &line, 
		  const PhysicalType &var,
		  const EnumerationType &side,
		  const IntegerType &width,
		  const PhysicalType &unit ){ 

  const UniversalLongLongInteger unit_val 
    = dynamic_cast<const UniversalLongLongInteger&>(unit.readVal());
  VHDLData::TimeUnit_t time = VHDLData::NS;
  
  if( longlongint(unit_val) == (LONG) 1 ){
    time = VHDLData::PS;
  } 
  else if(longlongint(unit_val) == (LONG) 2 ){
    time = VHDLData::NS;
  }
  else if(longlongint(unit_val) == (LONG) 3 ){
    time = VHDLData::US;
  }
  else if(longlongint(unit_val) == (LONG) 4 ){
    time = VHDLData::MS;
  }
  else if(longlongint(unit_val) == (LONG) 5 ){
    time = VHDLData::SECONDS;
  }
  else if(longlongint(unit_val) == (LONG) 6 ){
    time = VHDLData::MIN;
  }
  else if(longlongint(unit_val) == (LONG) 7 ){
    time = VHDLData::HR;
  }
  
  if( side.getIntValue() == SavantsideType_info.get_right() ){
    return savantwrite( base,
			line,
			var,
			VHDLData::RIGHT,
			width.getIntValue(),
			time );
  }
  else {
    return savantwrite( base,
			line,
			var,
			VHDLData::LEFT,
			width.getIntValue(),
			time );
  }
}

int
savantwrite( VHDLKernel *base,
	     AccessType &line,
	     const PhysicalType &var,
	     const EnumerationType &side,
	     const IntegerType &width,
	     const PhysicalType &unit ){
  if( var.range.get_unit_info() == SavanttimeType_info.get_unit_info() ){
    return savantwrite_time(base, line, var, side, width, unit);
  }

  return NORMAL_RETURN;
}

int
savantwrite( VHDLKernel *base,
	     AccessType &line, 
	     const VHDLVTime &var,
	     const EnumerationType &side,
	     const IntegerType &width,
	     VHDLData::TimeUnit_t time ){
  if( side.getIntValue() == SavantsideType_info.get_right() ){
    return savantwrite( base,
			line,
			var,
			VHDLData::RIGHT,
			width.getIntValue(),
			time);
  }
  else {
    return savantwrite( base,
			line,
			var,
			VHDLData::LEFT,
			width.getIntValue(),
			time);
  }
}

int
savantwrite( VHDLKernel *base,
	     AccessType &line, 
	     const EnumerationType &var,
	     const EnumerationType &side,
	     const IntegerType &width ){
  if( side.getIntValue() == SavantsideType_info.get_right()) {
    return savantwrite( base,
			line,
			var,
			VHDLData::RIGHT,
			width.getIntValue() );
  }
  else {
    return savantwrite( base,
			line,
			var,
			VHDLData::LEFT,
			width.getIntValue() );
  }
}

int
savantwrite( VHDLKernel *base, 
	     AccessType &line, 
	     const IntegerType &var, 
	     const EnumerationType &side,
	     const IntegerType &width ){
  VHDLData::Side_t direction;
  if( side.getIntValue() == SavantsideType_info.get_right()) {
    direction = VHDLData::RIGHT;
  }
  else {
    direction = VHDLData::LEFT; 
  }
  return savantwrite( base,
		      line,
		      var,
		      direction, 
		      width.getIntValue() );

}


int
savantwrite( VHDLKernel *base,
	     AccessType &line, 
	     const RealType &var,
	     const EnumerationType &side,
	     const IntegerType &width,
	     const IntegerType & ){
  if( dynamic_cast<const UniversalInteger &>(side.getObject()->readVal()).getValue() ==
      SavantsideType_info.get_right()) {
    return savantwrite( base,
			line,
			var,
			VHDLData::RIGHT,
			dynamic_cast<const UniversalInteger &>(width.getObject()->readVal()).getValue() );
  }
  else {
    return savantwrite( base,
			line,
			var,
			VHDLData::LEFT,
			dynamic_cast<const UniversalInteger &>(width.getObject()->readVal()).getValue() );
  }
}

int
savantwrite( VHDLKernel *kb,
	     AccessVariable &line,
	     const VHDLType &var,
	     const EnumerationType &justified,
	     const IntegerType &field ){
  switch (var.get_kind()) {
  case INTEGER_TYPE:
    return savantwrite(kb, line, dynamic_cast<const IntegerType &>(var), justified, field);
    
  case ENUMERATION_TYPE:
    return savantwrite(kb, line, dynamic_cast<const EnumerationType &>(var), justified, field);
    
  case REAL_TYPE:
    return savantwrite(kb, line, dynamic_cast<const RealType &>(var), justified, field);
    
  case PHYSICAL_TYPE:
    return savantwrite(kb, line, dynamic_cast<const PhysicalType &>(var), justified, field);
    
  case ARRAY_TYPE:
    return savantwrite(kb, line, dynamic_cast<const ArrayType &>(var), justified, field);
    
  case RECORD_TYPE:
    return savantwrite(kb, line, dynamic_cast<const RecordType &>(var), justified, field);
    
  default:
    break;
  }
  
  return NORMAL_RETURN;
}

int
savantwrite( VHDLKernel *base,
	     AccessType &line, 
	     const ArrayType &var,
	     const EnumerationType &side,
	     const IntegerType &width ){
  if ( dynamic_cast<const UniversalInteger &>(side.getObject()->readVal()).getValue()
       == SavantsideType_info.get_right() ) {
    return savantwrite( base, 
			line, 
			var, 
			VHDLData::RIGHT, 
			dynamic_cast<const UniversalInteger &>(width.getObject()->readVal()).getValue());
  }
  else {
    return savantwrite( base,
			line,
			var,
			VHDLData::LEFT,
			dynamic_cast< const UniversalInteger &>(width.getObject()->readVal()).getValue());
  }
}

int
savantwrite( VHDLKernel *base, 
	     AccessType &line, 
	     const RecordType &var, 
	     const EnumerationType &side,
	     const IntegerType &width ){
  if( side.getIntValue() == SavantsideType_info.get_right()) {
    return savantwrite( base,
			line,
			var,
			VHDLData::RIGHT,
			width.getIntValue() );
  }
  else {
    return savantwrite( base,
			line,
			var,
			VHDLData::LEFT,
			width.getIntValue() );
  }
}

// These global pointers are initialized in VHDLKernel::initializeGlobalObjectId()

SharedFileType *output = NULL;
SharedFileType *input  = NULL;
