#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <string>
#include <sstream>
#include <GL/gl.h>
#include <GL/glu.h>

#include <tulip/SelectionProxy.h>
#include <tulip/StringProxy.h>
#include <tulip/MetricProxy.h>
#include <tulip/MetaGraphProxy.h>
#include <tulip/IntProxy.h>
#include <tulip/LayoutProxy.h>
#include <tulip/SizesProxy.h>
#include "tulip/GlLines.h"
#include "tulip/GlGraph.h"
#include "tulip/Glyph.h"
#include "tulip/GlTools.h"

using namespace std;

//=====================================================
//this is used to sort node or edges from a select buffer according to their depth
template<class Element>
struct lessElementZ : public std::binary_function<Element, Element, bool> {
  GLuint (*v)[4];
  MutableContainer<GLint> *mapping;
  bool operator()(Element e1, Element e2) {
    GLuint z1, z2;
    z1 = v[mapping->get(e1.id)][1]/2 + v[mapping->get(e1.id)][2]/2;
    z2 = v[mapping->get(e2.id)][1]/2 + v[mapping->get(e2.id)][2]/2;
    return z1 < z2;
  }
};
//=====================================================
//Selection d'objet dans le monde 3D
//=====================================================
void GlGraph::initDoSelect(GLint x, GLint y, GLint w,GLint h) {
  makeCurrent();
  selectBuf = new GLuint[(_superGraph->numberOfEdges()+_superGraph->numberOfNodes())][4];
  glSelectBuffer((_superGraph->numberOfEdges()+_superGraph->numberOfNodes())*4 , (GLuint *)selectBuf);
  glRenderMode(GL_SELECT);
  glInitNames();
  glPushName(~0);

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();
  x += w/2;
  y = winH - (y + h/2);
  glGetIntegerv(GL_VIEWPORT, viewportArray);
  gluPickMatrix(x, y, w, h, viewportArray);
  initProjection(false);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  initModelView();
  glGetIntegerv(GL_VIEWPORT, viewportArray);
  tlp::multMatrix(projectionMatrix, modelviewMatrix, transformMatrix);

  glPolygonMode(GL_FRONT, GL_FILL);
  glDisable(GL_LIGHTING);
  glDisable(GL_BLEND);
  glDisable(GL_STENCIL_TEST);
}
//=====================================================
void GlGraph::endSelect() {
  glMatrixMode(GL_MODELVIEW);
  glPopMatrix();
  glGetDoublev (GL_MODELVIEW_MATRIX, modelviewMatrix);
  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glGetDoublev (GL_PROJECTION_MATRIX, projectionMatrix);
  glGetIntegerv(GL_VIEWPORT, viewportArray);
  tlp::multMatrix(projectionMatrix, modelviewMatrix, transformMatrix);
  glRenderMode(GL_RENDER);
}
//=====================================================
bool GlGraph::doNodeSelect(const int x, const int y, 
			   const int width , const int height, 
			   vector<node> &vNode , bool ordered) {
  if (_superGraph==0) return false;
  GLint hits;
  initDoSelect(x , y, width, height);
  makeNodeSelect(0);
  glFlush();
  hits = glRenderMode(GL_RENDER);
  if (hits <= 0) {
    delete [] selectBuf; 
    endSelect(); 
    return false;
  }
  MutableContainer<GLint> nodeToSelectBuffer;
  while(hits>0) {
    vNode.push_back(node(selectBuf[hits-1][3]));
    nodeToSelectBuffer.set(selectBuf[hits-1][3], hits -1);
    --hits;
  }
  if (ordered) {
    struct lessElementZ<node> comp;
    comp.v = selectBuf;
    comp.mapping = &nodeToSelectBuffer;
    sort(vNode.begin(), vNode.end(), comp);
  }
  delete [] selectBuf;
  endSelect();
  return true;
}
//=====================================================
bool GlGraph::doEdgeSelect(const int x, const int y, 
			   const int width, const int height, 
			   vector<edge> &vEdge, bool ordered) {
  if (_superGraph==0) return false;
  GLint hits;
  initDoSelect(x , y, width, height);
  makeEdgeSelect(0);
  glFlush();
  hits = glRenderMode(GL_RENDER);
  if (hits <= 0) {
    delete [] selectBuf; 
    endSelect(); 
    return false;
  }
  MutableContainer<GLint> edgeToSelectBuffer;
  while(hits>0) {
    vEdge.push_back(edge(selectBuf[hits-1][3]));
    edgeToSelectBuffer.set(selectBuf[hits-1][3], hits -1);
    --hits;
  }
  if (ordered) {
    struct lessElementZ<edge> comp;
    comp.v = selectBuf;
    comp.mapping = &edgeToSelectBuffer;
    sort(vEdge.begin(), vEdge.end(), comp);
  }
  delete [] selectBuf;
  endSelect();
  return true;
}
//=====================================================
void GlGraph::doSelect(const int x, const int y, 
		       const int width ,const int height,
                       vector<node> &sNode, vector<edge> &sEdge) {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << " x:" << x << ", y:" <<y <<", wi:"<<width<<", height:" << height << endl;
#endif
  timerStop();
  makeCurrent();
  glPushAttrib(GL_ALL_ATTRIB_BITS); 
  doNodeSelect(x, y, width, height, sNode, false);
  doEdgeSelect(x, y, width, height, sEdge, false);
  glPopAttrib();
  timerStart(0);
  //  cerr << "bool GlGraph::doSelect End" << endl;
}
//=====================================================
bool GlGraph::doSelect(GLint x, GLint y, tlp::ElementType &type ,node &n, edge &e ) {
#ifndef NDEBUG
  cerr << __PRETTY_FUNCTION__ << endl;
#endif
  timerStop();
  makeCurrent();
  glPushAttrib(GL_ALL_ATTRIB_BITS); 
  bool result;
  vector<node> tmpSetNode;
  //Check that line
  result = doNodeSelect(x-3, y-3, 6, 6, tmpSetNode);
  if (result){
    n = (*(tmpSetNode.begin()));
    type = tlp::NODE;
  }
  else { 
    type = tlp::EDGE;
    vector<edge> tmpSetEdge;
    result = doEdgeSelect(x-3, y-3, 6, 6, tmpSetEdge);
    if (result) 
      e = (*(tmpSetEdge.begin()));
  }
  glPopAttrib();
  timerStart(0);
  return result;
  //  cerr << "bool GlGraph::doSelect End" << endl;
}
//====================================lines================
//Construction des objets avec affectation de nom
//======================================================================
//Placement des sommets du graphe  l'index i=startIndex..startIndex+n-1
void GlGraph::makeNodeSelect(int startIndex) {
  glMatrixMode(GL_MODELVIEW);
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  Iterator<node> *itN=_superGraph->getNodes();
  while (itN->hasNext()) {
    node itv=itN->next();
    const Size &nSize=elementSize->getNodeValue(itv);
    if( nSize == Size(0,0,0) )	continue;
    const Coord &tmpCoord = elementLayout->getNodeValue(itv);	  
    glLoadName(itv.id);
    glPushMatrix();
    glTranslatef( tmpCoord.getX() , tmpCoord.getY() , tmpCoord.getZ() );
    glScalef(nSize.getW(), nSize.getH(), nSize.getD());
    if (elementMetaGraph->getNodeValue(itv) == 0)
      glyphs.get(elementShape->getNodeValue(itv))->draw(itv);
    else
      glyphs.get(0)->draw(itv);
    glPopMatrix();
  } delete itN;
  glPopAttrib();
}
//=================================================================
//Placement des artes du graphe  l'index i=startIndex..startIndex+n-number_of_edges
void GlGraph::makeEdgeSelect(int startIndex) {
  if (!_displayEdges) return;
  initProxies();
  glMatrixMode(GL_MODELVIEW);
  glPushAttrib(GL_ALL_ATTRIB_BITS);
  glDisable(GL_CULL_FACE);
  glDisable(GL_LINE_SMOOTH);
  glDisable(GL_POLYGON_SMOOTH);
  glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
  static const Color tmpColor(255,255,255,100);
  Iterator<edge> *itE = _superGraph->getEdges();
  while (itE->hasNext()) {
    edge ite=itE->next();
    glLoadName(ite.id);
    //    drawEdge(ite);
    const Coord &tmpCoord=elementLayout->getNodeValue(_superGraph->source(ite));
    const Coord &tmpCoord2=elementLayout->getNodeValue(_superGraph->target(ite));
    const LineType::RealType &lCoord=elementLayout->getEdgeValue(ite);
    drawEdge(tmpCoord, tmpCoord2, tmpCoord, lCoord, tmpCoord2,
             tmpColor, tmpColor, elementSize->getEdgeValue(ite), elementShape->getEdgeValue(ite), 
	     false);
  } delete itE;
  glPopAttrib();
}
//=====================================================
