/***************************************************************************
                             th-job-config-dialog.c
                             ----------------------
    begin                : Wed Nov 10 2004
    copyright            : (C) 2004 by Tim-Philipp Mller
    email                : t.i.m@orange.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "th-job.h"
#include "th-job-config-dialog.h"
#include "th-job-cropping-dialog.h"
#include "th-utils.h"

#include <gtk/gtk.h>
#include <glib/gi18n.h>

#include <string.h>

enum
{
	COL_PICSIZE_SIZE = 0,
	COL_PICSIZE_DESC,
	COL_PICSIZE_XYSTR,
	PICSIZE_NUM_COLS
};

enum
{
	COL_AUDIO_STREAM_ID = 0,
	COL_LANG_NAME
};

struct _ThJobConfigDialogPrivate
{
	ThJob        *job;
	
	GtkWidget    *top_label;
	
	GtkWidget    *fn_label;
	GtkWidget    *fn_button;

	GtkWidget    *crop_top_label;
	GtkWidget    *crop_left_label;
	GtkWidget    *crop_right_label;
	GtkWidget    *crop_bottom_label;
	GtkWidget    *crop_button;
	
	gulong        lang_combobox_sigid;
	GtkWidget    *lang_combobox;
	gulong        lang_all_sigid;
	GtkWidget    *lang_all;
	GtkListStore *lang_list;
	
	GtkWidget    *picsize_combobox;
	GtkListStore *picsize_list;

	gulong        quality_spinbutton_sigid; 
	GtkWidget    *quality_radiobutton;
	GtkWidget    *quality_spinbutton;
	GtkWidget    *quality_vbox;
	
	gulong        filesize_spinbutton_sigid;
	GtkWidget    *filesize_radiobutton;
	GtkWidget    *filesize_spinbutton;
	GtkWidget    *filesize_vbox;

	GtkWidget    *info_expander;
	GtkWidget    *info_title_entry;
	GtkWidget    *info_comment_entry;

	GtkWidget    *ok_button;
	GtkWidget    *close_button;
};

static void             job_config_dialog_class_init       (ThJobConfigDialogClass *klass);

static void             job_config_dialog_instance_init    (ThJobConfigDialog *cp);

static void             job_config_dialog_finalize         (GObject *object);

static gboolean         job_config_dialog_configure_job    (ThJobConfigDialog *jcd, ThJob *job);

static void             job_config_dialog_update_crop_labels (ThJobConfigDialog *jcd);

static void             job_config_dialog_update_file_size_range (ThJobConfigDialog *jcd);

static void             job_config_dialog_job_update_picsizes (ThJobConfigDialog *jcd);

static void             job_config_dialog_fill_language_combobox (ThJobConfigDialog *jcd);

/* variables */

static GObjectClass    *jcd_parent_class;          /* NULL */


/***************************************************************************
 *
 *   job_config_dialog_class_init
 *
 ***************************************************************************/

static void
job_config_dialog_class_init (ThJobConfigDialogClass *klass)
{
	GObjectClass  *object_class; 

	object_class = G_OBJECT_CLASS (klass);
	
	jcd_parent_class = g_type_class_peek_parent (klass);

	object_class->finalize = job_config_dialog_finalize;
}

/***************************************************************************
 *
 *   job_config_dialog_instance_init
 *
 ***************************************************************************/

static void
job_config_dialog_instance_init (ThJobConfigDialog *jcd)
{
	jcd->priv = g_new0 (ThJobConfigDialogPrivate, 1);
}

/***************************************************************************
 *
 *   job_config_dialog_finalize
 *
 ***************************************************************************/

static void
job_config_dialog_finalize (GObject *object)
{
	ThJobConfigDialog *jcd;

	jcd = (ThJobConfigDialog*) object;

	th_log ("ThJobConfigDialog: finalize\n");

	memset (jcd->priv, 0xab, sizeof (ThJobConfigDialogPrivate));
	g_free (jcd->priv);
	jcd->priv = NULL;

	/* chain up */
	jcd_parent_class->finalize (object);
}


/***************************************************************************
 *
 *   th_job_config_dialog_get_type
 *
 ***************************************************************************/

GType
th_job_config_dialog_get_type (void)
{
	static GType type; /* 0 */

	if (type == 0)
	{
		static GTypeInfo info =
		{
			sizeof (ThJobConfigDialogClass),
			(GBaseInitFunc) NULL,
			(GBaseFinalizeFunc) NULL,
			(GClassInitFunc) job_config_dialog_class_init,
			NULL, NULL,
			sizeof (ThJobConfigDialog),
			0,
			(GInstanceInitFunc) job_config_dialog_instance_init
		};

		type = g_type_register_static (TH_TYPE_FAKE_DIALOG, "ThJobConfigDialog", &info, 0);
	}

	return type;
}

/***************************************************************************
 *
 *   job_config_dialog_run_cropping_dialog
 *
 ***************************************************************************/

static void
job_config_dialog_run_cropping_dialog (ThJobConfigDialog *jcd, GtkButton *b)
{
	GtkWidget *appwin;

	appwin = gtk_widget_get_toplevel (GTK_WIDGET (jcd));

	th_job_cropping_dialog_run_for_job (jcd->priv->job, GTK_WINDOW (appwin));
	
	job_config_dialog_update_crop_labels (jcd);
	job_config_dialog_job_update_picsizes (jcd);
	th_job_save_config (jcd->priv->job);
}

/***************************************************************************
 *
 *   job_config_dialog_update_crop_labels
 *
 ***************************************************************************/

static void
job_config_dialog_update_crop_labels (ThJobConfigDialog *jcd)
{
	guint left = 0, right = 0, top = 0, bottom = 0;

	g_return_if_fail (jcd->priv->job != NULL);
	g_return_if_fail (TH_IS_JOB (jcd->priv->job));
	
	g_object_get (jcd->priv->job,
	              "crop-top", &top, "crop-left", &left,
	              "crop-right", &right, "crop-bottom", &bottom,
	              NULL);

	th_label_set_text (GTK_LABEL (jcd->priv->crop_top_label), "%u", top);
	th_label_set_text (GTK_LABEL (jcd->priv->crop_left_label), "%u", left);
	th_label_set_text (GTK_LABEL (jcd->priv->crop_right_label), "%u", right);
	th_label_set_text (GTK_LABEL (jcd->priv->crop_bottom_label), "%u", bottom);
}

/***************************************************************************
 *
 *   job_config_dialog_run_filechooser
 *
 ***************************************************************************/

static void
job_config_dialog_run_filechooser (ThJobConfigDialog *jcd, GtkButton *b)
{
	GtkWidget *fc, *appwin;
	gchar     *output_fn;
	
	appwin = gtk_widget_get_toplevel (GTK_WIDGET (jcd));
	fc = gtk_file_chooser_dialog_new (NULL, /* _("Save DVD Title As ..."),*/
	                                  GTK_WINDOW (appwin),
	                                  GTK_FILE_CHOOSER_ACTION_SAVE, 
	                                  GTK_STOCK_SAVE_AS, GTK_RESPONSE_ACCEPT,
	                                  NULL);
	
	g_object_get (jcd->priv->job, "output-fn", &output_fn, NULL);
	
	if (output_fn)
	{
		gchar *basename = g_path_get_basename (output_fn);
		gtk_file_chooser_set_filename (GTK_FILE_CHOOSER (fc), output_fn);
		if (g_utf8_validate (basename, -1, NULL))
		{
			gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (fc), basename);
		}
		else
		{
			gchar *base_loc;
		 
			if ((base_loc = g_locale_to_utf8 (basename, -1, NULL, NULL, NULL)))
				gtk_file_chooser_set_current_name (GTK_FILE_CHOOSER (fc), base_loc);
			
			g_free (base_loc);
		}
		g_free (basename);
	}

	{
		gchar *last_output_dir;

		g_object_get (appwin, "last-output-directory", &last_output_dir, NULL);

		if (last_output_dir)
			gtk_file_chooser_set_current_folder (GTK_FILE_CHOOSER (fc), last_output_dir);

		g_free (last_output_dir);
	}
	
	if (gtk_dialog_run (GTK_DIALOG (fc)) == GTK_RESPONSE_ACCEPT)
	{
		gchar *fn, *dir;
		 
		fn = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (fc));
		
		g_object_set (jcd->priv->job, "output-fn", fn, NULL);

		dir = g_path_get_dirname (fn);

		g_object_set (appwin, "last-output-directory", dir, NULL);

		g_free (dir);
		g_free (fn);
	}
	
	gtk_widget_destroy (fc);
	g_free (output_fn);
}

/***************************************************************************
 *
 *   job_config_dialog_job_update_picsizes
 *
 ***************************************************************************/

static void
job_config_dialog_job_update_picsizes (ThJobConfigDialog *jcd)
{
	GtkTreeIter  iter;
	guint        n, width, height;
	
	/* Fill in the height/width for each size into each row. This needs
	 *   to be done dynamically, because later the sizes have to be calculated
	 *   taking cropping into account (once that's implemented, that is) */
	n = 0;
	while (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (jcd->priv->picsize_list), &iter, NULL, n))
	{
		if (th_job_get_effective_picture_size (jcd->priv->job, n, &width, &height))
		{
			gchar *s = g_strdup_printf ("%3u x %3u", width, height);
			
			gtk_list_store_set (jcd->priv->picsize_list, &iter, COL_PICSIZE_XYSTR, s, -1);

			g_free (s);
		}
		
		++n;
	}
}

/***************************************************************************
 *
 *   job_config_dialog_job_fn_notify
 *
 ***************************************************************************/

static void
job_config_dialog_job_fn_notify (ThJobConfigDialog *jcd, GParamSpec *p, ThJob *job)
{
	const gchar *home;
	gchar *fn;
	
	g_object_get (jcd->priv->job, "output-fn", &fn, NULL);
	
	home = g_get_home_dir();
	if (fn && home && strncmp (fn, home, strlen(home)) == 0 && fn[strlen(home)] == G_DIR_SEPARATOR)
	{
		gchar *prettyfn;
		prettyfn = g_strdup_printf ("~%s", fn + strlen(home));
		gtk_label_set (GTK_LABEL (jcd->priv->fn_label), prettyfn);
		g_free (prettyfn);
	}
	else
	{
		gtk_label_set (GTK_LABEL (jcd->priv->fn_label), fn);
	}
	
	g_free (fn);
}

/***************************************************************************
 *
 *   job_config_dialog_picture_size_changed
 *
 ***************************************************************************/

static void
job_config_dialog_picture_size_changed (ThJobConfigDialog *jcd, GtkComboBox *cbox)
{
	ThPictureSize size;
	GtkTreeIter   iter;
	
	if (!gtk_combo_box_get_active_iter (cbox, &iter))
		g_return_if_reached ();

	gtk_tree_model_get (gtk_combo_box_get_model (cbox), &iter, COL_PICSIZE_SIZE, &size, -1);
	
	g_object_set (jcd->priv->job, "picture-size", size, NULL);
}

/***************************************************************************
 *
 *   job_config_dialog_setup_picsize_combobox
 *
 ***************************************************************************/

static void
job_config_dialog_setup_picsize_combobox (ThJobConfigDialog *jcd)
{
	GtkCellRenderer *cell;
	const gchar     *sizestr[] = { N_("Extra Small"), N_("Small"), N_("Medium Small"), 
	                               N_("Medium Large"), N_("Large"), N_("Extra Large"), 
	                               N_("Full") };
	guint i;

	g_assert (G_N_ELEMENTS (sizestr) == TH_PICTURE_SIZE_FULL+1);

	jcd->priv->picsize_list = gtk_list_store_new (PICSIZE_NUM_COLS, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_STRING);

	g_assert (PICSIZE_NUM_COLS == 3);

	for (i = 0;  i <= TH_PICTURE_SIZE_FULL;  ++i)
	{
		GtkTreeIter  iter;
	
		gtk_list_store_append (jcd->priv->picsize_list, &iter);

		gtk_list_store_set (jcd->priv->picsize_list, &iter, 
		                    COL_PICSIZE_SIZE, i, 
		                    COL_PICSIZE_DESC, _(sizestr[i]),
		                    -1);
	}

	gtk_cell_layout_clear (GTK_CELL_LAYOUT (jcd->priv->picsize_combobox));

	cell = gtk_cell_renderer_text_new ();
	g_object_set (cell, "xalign", (gdouble) 0.0, NULL);
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (jcd->priv->picsize_combobox), cell, TRUE);
	gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (jcd->priv->picsize_combobox), cell, 
	                               "text", COL_PICSIZE_DESC);

	cell = gtk_cell_renderer_text_new ();
	g_object_set (cell, "xalign", (gdouble) 1.0, "xpad", 4, NULL);
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (jcd->priv->picsize_combobox), cell, FALSE);
	gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (jcd->priv->picsize_combobox), cell, 
	                               "text", COL_PICSIZE_XYSTR);
	
	gtk_combo_box_set_model (GTK_COMBO_BOX (jcd->priv->picsize_combobox), 
	                         GTK_TREE_MODEL (jcd->priv->picsize_list));

	g_signal_connect_swapped (jcd->priv->picsize_combobox, "changed",
	                          G_CALLBACK (job_config_dialog_picture_size_changed),
	                          jcd);
}

/***************************************************************************
 *
 *   job_config_dialog_language_changed
 *
 ***************************************************************************/

static void
job_config_dialog_language_changed (ThJobConfigDialog *jcd, GtkComboBox *cbox)
{
	GtkTreeIter   iter;
	guint         aid;
	
	if (!gtk_combo_box_get_active_iter (cbox, &iter))
		g_return_if_reached ();

	gtk_tree_model_get (gtk_combo_box_get_model (cbox), &iter, COL_AUDIO_STREAM_ID, &aid, -1);
	
	g_object_set (jcd->priv->job, "audio-stream-id", aid, NULL);
}

/***************************************************************************
 *
 *   job_config_dialog_language_all_toggled
 *
 ***************************************************************************/

static void
job_config_dialog_language_all_toggled (ThJobConfigDialog *jcd, GtkCheckButton *cbox)
{
  gboolean active;

  active = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (cbox));

  /* Disable combobox, set all audio flag */
  gtk_widget_set_sensitive (jcd->priv->lang_combobox, !active);
  g_object_set (jcd->priv->job, "audio-all", active, NULL);

  job_config_dialog_update_file_size_range (jcd);

  /* force re-calculation of effective video bitrate if the number of audio
   * streams has changed */
  if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (jcd->priv->filesize_radiobutton))) {
    gdouble target_size;

    g_object_get (jcd->priv->job, "target-output-size", &target_size, NULL);
    g_object_set (jcd->priv->job, "target-output-size", target_size, NULL);
  }
}

/***************************************************************************
 *
 *   job_config_dialog_setup_language_combobox
 *
 ***************************************************************************/

static void
job_config_dialog_setup_language_combobox (ThJobConfigDialog *jcd)
{
	GtkCellRenderer *cell;

	jcd->priv->lang_list = gtk_list_store_new (3, G_TYPE_UINT, G_TYPE_STRING, G_TYPE_STRING);

	gtk_cell_layout_clear (GTK_CELL_LAYOUT (jcd->priv->lang_combobox));

	cell = gtk_cell_renderer_text_new ();
	g_object_set (cell, "xalign", (gdouble) 0.0, NULL);
	gtk_cell_layout_pack_start (GTK_CELL_LAYOUT (jcd->priv->lang_combobox), cell, TRUE);
	gtk_cell_layout_add_attribute (GTK_CELL_LAYOUT (jcd->priv->lang_combobox), cell, 
	                               "text", COL_LANG_NAME);

	gtk_combo_box_set_model (GTK_COMBO_BOX (jcd->priv->lang_combobox), 
	                         GTK_TREE_MODEL (jcd->priv->lang_list));

	jcd->priv->lang_combobox_sigid = 
		g_signal_connect_swapped (jcd->priv->lang_combobox, "changed",
		                          G_CALLBACK (job_config_dialog_language_changed),
		                          jcd);
	jcd->priv->lang_all_sigid = 
		g_signal_connect_swapped (jcd->priv->lang_all, "toggled",
		                          G_CALLBACK (job_config_dialog_language_all_toggled),
		                          jcd);
}

/***************************************************************************
 *
 *   job_config_dialog_fill_language_combobox
 *
 ***************************************************************************/

static void
job_config_dialog_fill_language_combobox (ThJobConfigDialog *jcd)
{
	guint i, default_aid, num_audio_streams = 0;

	/* Need to block 'changed' signal handler here, otherwise
	 *  the active iter might change while the list is cleared,
	 *  leading to a wrong default audio id to be selected. */
	g_signal_handler_block (jcd->priv->lang_combobox, 
	                        jcd->priv->lang_combobox_sigid);
	
	gtk_list_store_clear (jcd->priv->lang_list);
	
	g_signal_handler_unblock (jcd->priv->lang_combobox, 
	                          jcd->priv->lang_combobox_sigid);

	g_object_get (jcd->priv->job, 
	              "audio-stream-id", &default_aid,
	              NULL);

	th_log ("default_aid = %u\n", default_aid);

	num_audio_streams = jcd->priv->job->audio_streams->len;
	for (i = 0;  i < num_audio_streams;  ++i)
	{
		const gchar *lang, *desc, *lang_name;
		gchar        nolangstr[256] = { 0x00, };
		guint        aid;
	
		if (th_job_get_audio_stream (jcd->priv->job, i, &aid, &lang, &desc))
		{
			GtkTreeIter  iter;

			gtk_list_store_append (jcd->priv->lang_list, &iter);

			lang_name = th_utils_get_language_name (lang);
			
			if (lang_name == NULL)
			{
				g_snprintf (nolangstr, sizeof (nolangstr), "%s (audio_%02u)", _("Unknown Language"), aid);
				lang_name = nolangstr;
			}

			gtk_list_store_set (jcd->priv->lang_list, &iter, 
			                    COL_AUDIO_STREAM_ID, aid,
			                    COL_LANG_NAME, lang_name,
			                    -1);
		
			if (aid == default_aid)
			{
				gtk_combo_box_set_active_iter (GTK_COMBO_BOX (jcd->priv->lang_combobox), &iter);
			}
		}
	}
}

/***************************************************************************
 *
 *   job_config_dialog_radio_button_toggled
 *
 ***************************************************************************/

static void
job_config_dialog_radio_button_toggled (ThJobConfigDialog *jcd, GtkWidget *rb)
{
	/* only interested in the currently active one */
	if (!gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (rb)))
		return;

	if (rb == jcd->priv->quality_radiobutton)
	{
		gtk_widget_show (jcd->priv->quality_vbox);
		gtk_widget_hide (jcd->priv->filesize_vbox);
		gtk_widget_set_sensitive (jcd->priv->quality_spinbutton, TRUE);
		gtk_widget_set_sensitive (jcd->priv->filesize_spinbutton, FALSE);
	}
	else if (rb == jcd->priv->filesize_radiobutton)
	{
		gtk_widget_show (jcd->priv->filesize_vbox);
		gtk_widget_hide (jcd->priv->quality_vbox);
		gtk_widget_set_sensitive (jcd->priv->filesize_spinbutton, TRUE);
		gtk_widget_set_sensitive (jcd->priv->quality_spinbutton, FALSE);
	}
	else 
	{
		g_return_if_reached ();
	}
}

/***************************************************************************
 *
 *   job_config_dialog_spinbutton_changed
 *
 ***************************************************************************/

static void
job_config_dialog_spinbutton_changed (ThJobConfigDialog *jcd, GtkSpinButton *sb)
{
	gdouble val;
	
	if (jcd->priv->job == NULL)
		return;

	val = gtk_spin_button_get_value (sb);

	if (GTK_WIDGET (sb) == jcd->priv->quality_spinbutton)
	{
		g_object_set (jcd->priv->job, "target-quality", (guint) val, NULL);
	}
	else if (GTK_WIDGET (sb) == jcd->priv->filesize_spinbutton)
	{
		g_object_set (jcd->priv->job, "target-output-size", val, NULL);
	} 
	else 
	{
		g_return_if_reached ();
	}
}

/***************************************************************************
 *
 *   job_config_dialog_filesize_spinbutton_output
 *
 ***************************************************************************/

static gboolean
job_config_dialog_filesize_spinbutton_output (ThJobConfigDialog *jcd, GtkSpinButton *sb)
{
	guint64  bytes;
	gchar   *s;
	
	bytes = (guint64) (gtk_spin_button_get_value (sb) * 1024.0 * 1024.0);
	s = th_utils_get_human_size_str (bytes);
	
	if (s && !g_str_equal (s, gtk_entry_get_text (GTK_ENTRY (sb))))
		gtk_entry_set_text (GTK_ENTRY (sb), s);
	
	g_free (s);
	
	return TRUE;
}

/***************************************************************************
 *
 *   jcd_info_entry_changed
 *
 ***************************************************************************/

static void
jcd_info_entry_changed (ThJobConfigDialog *jcd, GtkEditable *editable)
{
	const gchar *txt;
	GtkWidget   *entry;

	entry = GTK_WIDGET (editable);
	txt = gtk_entry_get_text (GTK_ENTRY (entry));

	if (entry == jcd->priv->info_title_entry)
		g_object_set (jcd->priv->job, "title-tag", txt, NULL);
	else if (entry == jcd->priv->info_comment_entry)
		g_object_set (jcd->priv->job, "comment-tag", txt, NULL);
	else 
		g_return_if_reached ();

	th_job_save_config (jcd->priv->job);
}

/***************************************************************************
 *
 *   th_job_config_dialog_new
 *
 ***************************************************************************/

GtkWidget *
th_job_config_dialog_new (void)
{
	ThJobConfigDialog *jcd;
	GtkWidget         *glade_window = NULL;
	GtkWidget         *toplevel_vbox;

	jcd = (ThJobConfigDialog *) g_object_new (TH_TYPE_JOB_CONFIG_DIALOG, NULL);
	
	if (!th_utils_ui_load_interface ("th-job-config-dialog.glade", 
	                                 FALSE,
	                                 "th-job-config-dialog",   &glade_window,
	                                 "th-toplevel-vbox",       &toplevel_vbox,
	                                 "th-top-label",           &jcd->priv->top_label,
	                                 "th-filename-label",      &jcd->priv->fn_label,
	                                 "th-filename-button",     &jcd->priv->fn_button,
	                                 "th-language-combobox",   &jcd->priv->lang_combobox,
	                                 "th-language-all",        &jcd->priv->lang_all,
	                                 "th-picsize-combobox",    &jcd->priv->picsize_combobox,
	                                 "th-crop-top-label",      &jcd->priv->crop_top_label,
	                                 "th-crop-left-label",     &jcd->priv->crop_left_label,
	                                 "th-crop-right-label",    &jcd->priv->crop_right_label,
	                                 "th-crop-bottom-label",   &jcd->priv->crop_bottom_label,
	                                 "th-crop-button",         &jcd->priv->crop_button,
	                                 "th-vbox-video-size",     &jcd->priv->filesize_vbox,
	                                 "th-vbox-video-quality",  &jcd->priv->quality_vbox,
	                                 "th-filesize-spinbutton", &jcd->priv->filesize_spinbutton,
	                                 "th-quality-spinbutton",  &jcd->priv->quality_spinbutton,
	                                 "th-filesize-radiobutton", &jcd->priv->filesize_radiobutton,
	                                 "th-quality-radiobutton",  &jcd->priv->quality_radiobutton,
	                                 "th-info-expander",        &jcd->priv->info_expander,
	                                 "th-info-title-entry",     &jcd->priv->info_title_entry,
	                                 "th-info-comment-entry",   &jcd->priv->info_comment_entry,
	                                 NULL))
	{
		g_warning ("th_utils_ui_load_interface (\"th-ui-job-config-dialog.glade\") failed.\n");
		if (glade_window)
			gtk_widget_destroy (glade_window);
		gtk_widget_destroy (GTK_WIDGET (jcd));
		return NULL;
	}
	
	g_object_ref (toplevel_vbox);
	gtk_container_remove (GTK_CONTAINER (glade_window), toplevel_vbox);
	gtk_container_add (GTK_CONTAINER (TH_FAKE_DIALOG (jcd)->vbox), toplevel_vbox);
	g_object_unref (toplevel_vbox);

	g_signal_connect_swapped (jcd->priv->crop_button, "clicked",
	                          G_CALLBACK (job_config_dialog_run_cropping_dialog),
	                          jcd);
	
	g_signal_connect_swapped (jcd->priv->fn_button, "clicked",
	                          G_CALLBACK (job_config_dialog_run_filechooser),
	                          jcd);

	g_signal_connect_swapped (jcd->priv->quality_radiobutton, "toggled",
	                          G_CALLBACK (job_config_dialog_radio_button_toggled),
	                          jcd);

	g_signal_connect_swapped (jcd->priv->filesize_radiobutton, "toggled",
	                          G_CALLBACK (job_config_dialog_radio_button_toggled),
	                          jcd);

	/* so it's wide enough to show the 'MB' as well */
	gtk_spin_button_set_digits (GTK_SPIN_BUTTON (jcd->priv->filesize_spinbutton), 4);

	g_signal_connect_swapped (jcd->priv->filesize_spinbutton, "output",
	                          G_CALLBACK (job_config_dialog_filesize_spinbutton_output),
	                          jcd);

	jcd->priv->quality_spinbutton_sigid = 
		g_signal_connect_swapped (jcd->priv->quality_spinbutton, "value-changed",
		                          G_CALLBACK (job_config_dialog_spinbutton_changed),
		                          jcd);
	
	jcd->priv->filesize_spinbutton_sigid = 
		g_signal_connect_swapped (jcd->priv->filesize_spinbutton, "value-changed",
		                          G_CALLBACK (job_config_dialog_spinbutton_changed),
		                          jcd);

	job_config_dialog_setup_language_combobox (jcd);
	
	job_config_dialog_setup_picsize_combobox (jcd);

	jcd->priv->close_button = th_fake_dialog_add_button (TH_FAKE_DIALOG (jcd),
	                                                     GTK_STOCK_CANCEL,
	                                                     GTK_RESPONSE_REJECT);

	/* makes cancel button stick to the left; let's see
	 *  how long it takes until some HIGian complains ;) */
	gtk_button_box_set_child_secondary (GTK_BUTTON_BOX (TH_FAKE_DIALOG (jcd)->action_area),
	                                    jcd->priv->close_button,
	                                    TRUE);

	jcd->priv->ok_button = th_fake_dialog_add_button (TH_FAKE_DIALOG (jcd),
	                                                  GTK_STOCK_OK,
	                                                  GTK_RESPONSE_ACCEPT);

	gtk_expander_set_expanded (GTK_EXPANDER (jcd->priv->info_expander), FALSE);


	g_signal_connect_swapped (jcd->priv->info_title_entry, "changed",
	                          G_CALLBACK (jcd_info_entry_changed),
	                          jcd);

	g_signal_connect_swapped (jcd->priv->info_comment_entry, "changed",
	                          G_CALLBACK (jcd_info_entry_changed),
	                          jcd);
	
	return GTK_WIDGET (jcd);
}

/***************************************************************************
 *
 *   th_job_config_dialog_configure_jobs
 *
 ***************************************************************************/

gboolean
th_job_config_dialog_configure_jobs (ThJobConfigDialog *jcd, GList *jobs)
{
	GList *l;
	
	g_return_val_if_fail (TH_IS_JOB_CONFIG_DIALOG (jcd), FALSE);
	g_return_val_if_fail (jobs != NULL, FALSE);

	for (l = jobs;  l;  l = l->next)
	{
		ThJob *job = TH_JOB (l->data);
		
		if (!job_config_dialog_configure_job (jcd, job))
			return FALSE;
	}
	
	return TRUE;
}

static void
job_config_dialog_update_file_size_range (ThJobConfigDialog *jcd)
{
  GtkSpinButton *filesize_spin;
  gdouble  min_fsize, max_fsize;

  filesize_spin = GTK_SPIN_BUTTON (jcd->priv->filesize_spinbutton);

  th_job_get_file_size_range (jcd->priv->job, &min_fsize, &max_fsize);

  /* +/- 0.5 because of double => int conversion (don't 
   *   want to end up with a value too low or high */
  g_signal_handler_block (filesize_spin, jcd->priv->filesize_spinbutton_sigid);
  gtk_spin_button_set_range (filesize_spin, min_fsize + 0.5, max_fsize - 0.5);
  g_signal_handler_unblock (filesize_spin, jcd->priv->filesize_spinbutton_sigid);
}

/***************************************************************************
 *
 *   job_config_dialog_configure_job
 *
 ***************************************************************************/

static gboolean
job_config_dialog_configure_job (ThJobConfigDialog *jcd, ThJob *job)
{
	ThPictureSize picsize;
	GtkTreeIter   iter;
	gboolean      ret = TRUE;
	gdouble       target_output_size;
	gulong        notify_id1;
	guint         title_num, target_quality, response_id;
	gchar        *title_tag, *comment_tag;
	GString      *s;
	
	g_return_val_if_fail (TH_IS_JOB (job), TRUE);

	jcd->priv->job = job;

	job_config_dialog_update_crop_labels (jcd);
	job_config_dialog_update_file_size_range (jcd);

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (jcd->priv->quality_radiobutton), TRUE);
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (jcd->priv->filesize_radiobutton), TRUE);

	g_object_get (job, "title-tag", &title_tag, "comment-tag", &comment_tag, NULL);

	gtk_entry_set_text (GTK_ENTRY (jcd->priv->info_title_entry), (title_tag) ? title_tag : "");
	gtk_entry_set_text (GTK_ENTRY (jcd->priv->info_comment_entry), (comment_tag) ? comment_tag : "");

	g_free (title_tag);
	g_free (comment_tag);

	notify_id1 = 
		g_signal_connect_swapped (job, "notify::output-fn",
		                          G_CALLBACK (job_config_dialog_job_fn_notify),
		                          jcd);

	g_object_notify (G_OBJECT (job), "output-fn");

	g_object_get (job, 
	              "title-num", &title_num, 
	              "picture-size", &picsize, 
	              "target-output-size", &target_output_size,
	              "target-quality", &target_quality,
	              NULL);

	th_log ("target_output_size = %.1fM\n", target_output_size);

	gtk_spin_button_set_value (GTK_SPIN_BUTTON (jcd->priv->filesize_spinbutton), target_output_size);
	
	if (target_quality > 0)
	{
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (jcd->priv->quality_radiobutton), TRUE);
		gtk_spin_button_set_value (GTK_SPIN_BUTTON (jcd->priv->quality_spinbutton), target_quality);
	}

	/* no need to show the 'all languages' checkbox if there is only one */
	if (job->audio_streams->len == 1) {
	  gtk_widget_hide (jcd->priv->lang_all);
	} else {
	  gtk_widget_show (jcd->priv->lang_all);
	}
	
	s = g_string_new (NULL);
	g_string_append (s, "<b>");
	g_string_append_printf (s, _("Configure DVD Title %u"), title_num + 1);
	g_string_append (s, "</b>");
	gtk_label_set_markup (GTK_LABEL (jcd->priv->top_label), s->str);

	job_config_dialog_job_update_picsizes (jcd);
	
	if (gtk_tree_model_iter_nth_child (GTK_TREE_MODEL (jcd->priv->picsize_list), &iter, NULL, picsize))
		gtk_combo_box_set_active_iter (GTK_COMBO_BOX (jcd->priv->picsize_combobox), &iter);

	job_config_dialog_fill_language_combobox (jcd);

	response_id = th_fake_dialog_run (TH_FAKE_DIALOG (jcd));
	if (response_id != GTK_RESPONSE_ACCEPT)
	{
		ret = FALSE;
		goto done;
	}

	/* make sure that the right target is selected even if
	 *  we just switch between the two radio buttons, but do
	 *  not change the value in either of the spin buttons */
	if (GTK_WIDGET_SENSITIVE (jcd->priv->filesize_spinbutton))
		g_signal_emit_by_name (jcd->priv->filesize_spinbutton, "value-changed");
	else if (GTK_WIDGET_SENSITIVE (jcd->priv->quality_spinbutton))
		g_signal_emit_by_name (jcd->priv->quality_spinbutton, "value-changed");
	else g_warning ("%s: Neither of the two spin buttons is active?!\n", G_STRLOC);

	th_job_save_config (job);
	
done:
	
	g_signal_handler_disconnect (job, notify_id1);

	g_string_free (s, TRUE);

	if (response_id != TH_RESPONSE_DESTROYED)
		jcd->priv->job = NULL;
	
	return ret;
}

