# Ruby/GraphViz
[![All Contributors](https://img.shields.io/badge/all_contributors-31-orange.svg)](#contributors)
[![Travis CI build](https://secure.travis-ci.org/glejeune/Ruby-Graphviz.svg)](https://travis-ci.org/glejeune/Ruby-Graphviz)
[![Gem Version](https://badge.fury.io/rb/ruby-graphviz.svg)](https://rubygems.org/gems/ruby-graphviz)

Copyright (C) 2004-2018 Gregoire Lejeune

* Doc : http://rdoc.info/projects/glejeune/Ruby-Graphviz
* Sources : https://github.com/glejeune/Ruby-Graphviz
* On Rubygems: https://rubygems.org/gems/ruby-graphviz

## INSTALLATION

Add this line to your application's Gemfile:

	gem 'ruby-graphviz'

And then execute:

	$ bundle

Or install it yourself as:

	$ gem install ruby-graphviz

## DESCRIPTION

Interface to the GraphViz graphing tool

## TODO

* New FamilyTree


## SYNOPSIS

A basic example

```ruby
require 'ruby-graphviz'

# Create a new graph
g = GraphViz.new( :G, :type => :digraph )

# Create two nodes
hello = g.add_nodes( "Hello" )
world = g.add_nodes( "World" )

# Create an edge between the two nodes
g.add_edges( hello, world )

# Generate output image
g.output( :png => "hello_world.png" )
```

The same but with a block

```ruby
require 'ruby-graphviz'

GraphViz::new( :G, :type => :digraph ) { |g|
  g.world( :label => "World" ) << g.hello( :label => "Hello" )
}.output( :png => "hello_world.png" )
```

Or with the DSL

```ruby
    require 'ruby-graphviz/dsl'
    digraph :G do
      world[:label => "World"] << hello[:label => "Hello"]

      output :png => "hello_world.png"
    end
```

Create a graph from a file 

```ruby
    require 'ruby-graphviz'

    # In this example, hello.dot is :
    #   digraph G {Hello->World;}

    GraphViz.parse( "hello.dot", :path => "/usr/local/bin" ) { |g|
      g.get_node("Hello") { |n|
        n[:label] = "Bonjour"
      }
      g.get_node("World") { |n|
        n[:label] = "Le Monde"
      }
    }.output(:png => "sample.png")
```

[GraphML](http://graphml.graphdrawing.org/) support

```ruby
    require 'ruby-graphviz/graphml'

    g = GraphViz::GraphML.new( "graphml/cluster.graphml" )
    g.graph.output( :path => "/usr/local/bin", :png => "#{$0}.png" )
```

## TOOLS

Ruby/GraphViz also includes :

* `ruby2gv`, a simple tool that allows you to create a dependency graph from a ruby script. Example : http://drp.ly/dShaZ

```ruby
ruby2gv -Tpng -oruby2gv.png ruby2gv
```

* `gem2gv`, a tool that allows you to create a dependency graph between gems. Example : http://drp.ly/dSj9Y

```ruby
gem2gv -Tpng -oruby-graphviz.png ruby-graphviz
```

* `dot2ruby`, a tool that allows you to create a ruby script from a graphviz script

```ruby
$ cat hello.dot
digraph G {Hello->World;}

$ dot2ruby hello.dot
# This code was generated by dot2ruby.g

require 'rubygems'
require 'ruby-graphviz'
graph_g = GraphViz.digraph( "G" ) { |graph_g|
  graph_g[:bb] = '0,0,70,108'
  node_hello = graph_g.add_nodes( "Hello", :height => '0.5', :label => '\N', :pos => '35,90', :width => '0.88889' )
  graph_g.add_edges( "Hello", "World", :pos => 'e,35,36.413 35,71.831 35,64.131 35,54.974 35,46.417' )
  node_world = graph_g.add_nodes( "World", :height => '0.5', :label => '\N', :pos => '35,18', :width => '0.97222' )
}
puts graph_g.output( :canon => String )
```

* `git2gv`, a tool that allows you to show your git commits

* `xml2gv`, a tool that allows you to show a xml file as graph.


## GRAPH THEORY

```ruby
require 'ruby-graphviz'
require 'ruby-graphviz/theory'

g = GraphViz.new( :G ) { ... }

t = GraphViz::Theory.new( g )

puts "Adjancy matrix : "
puts t.adjancy_matrix

puts "Symmetric ? #{t.symmetric?}"

puts "Incidence matrix :"
puts t.incidence_matrix

g.each_node do |name, node|
  puts "Degree of node `#{name}' = #{t.degree(node)}"
end

puts "Laplacian matrix :"
puts t.laplacian_matrix

puts "Dijkstra between a and f"
r = t.moore_dijkstra(g.a, g.f)
if r.nil?
  puts "No way !"
else
  print "\tPath : "; p r[:path]
  puts "\tDistance : #{r[:distance]}"
end

print "Ranges : "
rr = t.range
p rr
puts "Your graph contains circuits" if rr.include?(nil)

puts "Critical path : "
rrr = t.critical_path
print "\tPath "; p rrr[:path]
puts "\tDistance : #{rrr[:distance]}"
```

## INSTALLATION

```
sudo gem install ruby-graphviz
```

You also need to install [GraphViz](http://www.graphviz.org) 

On **Windows** you also need to install win32-open3. This is not an absolute
requirement.

## LICENCES

### Ruby/GraphViz 

Ruby/GraphViz is freely distributable according to the terms of the GNU
General Public License (see the file 'COPYING').

This program is distributed without any warranty. See the file 'COPYING' for
details.

GNU General Public Licence: https://en.wikipedia.org/wiki/Gpl

### nothugly.xsl

By Vidar Hokstad and Ryan Shea; Contributions by Jonas Tingborn, Earl
Cummings, Michael Kennedy (Graphviz 2.20.2 compatibility, bug fixes, testing,
lots of gradients)

Copyright (c) 2009 Vidar Hokstad

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in all
copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
SOFTWARE.

MIT license: https://en.wikipedia.org/wiki/MIT_License

## Contributors

Thanks goes to these wonderful people ([emoji key](https://github.com/kentcdodds/all-contributors#emoji-key)):

<!-- ALL-CONTRIBUTORS-LIST:START - Do not remove or modify this section -->
<!-- prettier-ignore-start -->
<!-- markdownlint-disable -->
<table>
  <tr>
    <td align="center"><a href="https://github.com/dburt"><img src="https://avatars0.githubusercontent.com/u/139988?v=4" width="100px;" alt=""/><br /><sub><b>Dave Burt</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=dburt" title="Code">💻</a></td>
    <td align="center"><a href="https://github.com/obruening"><img src="https://avatars2.githubusercontent.com/u/146361?v=4" width="100px;" alt=""/><br /><sub><b>obruening</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=obruening" title="Code">💻</a></td>
    <td align="center"><a href="https://github.com/hipe"><img src="https://avatars1.githubusercontent.com/u/22006?v=4" width="100px;" alt=""/><br /><sub><b>Chip Malice</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=hipe" title="Code">💻</a></td>
    <td align="center"><a href="https://github.com/MSNexploder"><img src="https://avatars3.githubusercontent.com/u/101456?v=4" width="100px;" alt=""/><br /><sub><b>Stefan Stüben</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=MSNexploder" title="Code">💻</a></td>
    <td align="center"><a href="https://github.com/oupo"><img src="https://avatars1.githubusercontent.com/u/143470?v=4" width="100px;" alt=""/><br /><sub><b>oupo</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=oupo" title="Code">💻</a></td>
    <td align="center"><a href="http://lejeun.es"><img src="https://avatars1.githubusercontent.com/u/15168?v=4" width="100px;" alt=""/><br /><sub><b>Gregoire Lejeune</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=glejeune" title="Code">💻</a> <a href="#design-glejeune" title="Design">🎨</a> <a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=glejeune" title="Documentation">📖</a></td>
    <td align="center"><a href="https://github.com/markus1189"><img src="https://avatars0.githubusercontent.com/u/591567?v=4" width="100px;" alt=""/><br /><sub><b>Markus Hauck</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=markus1189" title="Code">💻</a></td>
  </tr>
  <tr>
    <td align="center"><a href="https://github.com/khalilfazal"><img src="https://avatars2.githubusercontent.com/u/308027?v=4" width="100px;" alt=""/><br /><sub><b>Khalil Fazal</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=khalilfazal" title="Code">💻</a></td>
    <td align="center"><a href="https://github.com/kachick"><img src="https://avatars2.githubusercontent.com/u/1180335?v=4" width="100px;" alt=""/><br /><sub><b>Kenichi Kamiya</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=kachick" title="Code">💻</a></td>
    <td align="center"><a href="https://github.com/nevenh"><img src="https://avatars0.githubusercontent.com/u/1448453?v=4" width="100px;" alt=""/><br /><sub><b>Neven Has</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=nevenh" title="Code">💻</a></td>
    <td align="center"><a href="https://Andrew.Kvalhe.im"><img src="https://avatars1.githubusercontent.com/u/1844746?v=4" width="100px;" alt=""/><br /><sub><b>Andrew</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=AndrewKvalheim" title="Code">💻</a></td>
    <td align="center"><a href="https://github.com/dznz"><img src="https://avatars3.githubusercontent.com/u/245206?v=4" width="100px;" alt=""/><br /><sub><b>Daniel Zollinger</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=dznz" title="Code">💻</a></td>
    <td align="center"><a href="https://guilhermesimoes.github.io"><img src="https://avatars0.githubusercontent.com/u/531168?v=4" width="100px;" alt=""/><br /><sub><b>Guilherme Simoes</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=guilhermesimoes" title="Code">💻</a></td>
    <td align="center"><a href="https://github.com/OrelSokolov"><img src="https://avatars0.githubusercontent.com/u/2205407?v=4" width="100px;" alt=""/><br /><sub><b>Oleg Orlov</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=OrelSokolov" title="Code">💻</a></td>
  </tr>
  <tr>
    <td align="center"><a href="https://www.instagram.com/stewiecutedog/"><img src="https://avatars3.githubusercontent.com/u/222582?v=4" width="100px;" alt=""/><br /><sub><b>Gabe Kopley</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=gkop" title="Code">💻</a></td>
    <td align="center"><a href="http://jakegoulding.com"><img src="https://avatars0.githubusercontent.com/u/174509?v=4" width="100px;" alt=""/><br /><sub><b>Jake Goulding</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=shepmaster" title="Code">💻</a></td>
    <td align="center"><a href="https://github.com/hirochachacha"><img src="https://avatars0.githubusercontent.com/u/898442?v=4" width="100px;" alt=""/><br /><sub><b>hirochachacha</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=hirochachacha" title="Code">💻</a></td>
    <td align="center"><a href="http://www.ronenbarzel.org"><img src="https://avatars2.githubusercontent.com/u/125620?v=4" width="100px;" alt=""/><br /><sub><b>ronen barzel</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=ronen" title="Code">💻</a></td>
    <td align="center"><a href="https://jamison.dance"><img src="https://avatars1.githubusercontent.com/u/72027?v=4" width="100px;" alt=""/><br /><sub><b>Jamison Dance</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=jergason" title="Code">💻</a> <a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=jergason" title="Documentation">📖</a></td>
    <td align="center"><a href="http://josephholsten.com"><img src="https://avatars3.githubusercontent.com/u/7495?v=4" width="100px;" alt=""/><br /><sub><b>Joseph Anthony Pasquale Holsten</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=josephholsten" title="Code">💻</a></td>
    <td align="center"><a href="http://mfcabrera.com"><img src="https://avatars0.githubusercontent.com/u/12527?v=4" width="100px;" alt=""/><br /><sub><b>Miguel Cabrera</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=mfcabrera" title="Code">💻</a></td>
  </tr>
  <tr>
    <td align="center"><a href="http://www.miketheman.net"><img src="https://avatars0.githubusercontent.com/u/529516?v=4" width="100px;" alt=""/><br /><sub><b>Mike Fiedler</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=miketheman" title="Code">💻</a></td>
    <td align="center"><a href="http://nathanmlong.com"><img src="https://avatars2.githubusercontent.com/u/338814?v=4" width="100px;" alt=""/><br /><sub><b>Nathan Long</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=nathanl" title="Code">💻</a> <a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=nathanl" title="Documentation">📖</a></td>
    <td align="center"><a href="http://ollehost.dk/"><img src="https://avatars0.githubusercontent.com/u/211?v=4" width="100px;" alt=""/><br /><sub><b>Olle Jonsson</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=olleolleolle" title="Code">💻</a> <a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=olleolleolle" title="Documentation">📖</a></td>
    <td align="center"><a href="http://postmodern.github.com/"><img src="https://avatars2.githubusercontent.com/u/12671?v=4" width="100px;" alt=""/><br /><sub><b>Postmodern</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=postmodern" title="Code">💻</a></td>
    <td align="center"><a href="https://www.versioneye.com"><img src="https://avatars0.githubusercontent.com/u/652130?v=4" width="100px;" alt=""/><br /><sub><b>Robert Reiz</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=reiz" title="Code">💻</a></td>
    <td align="center"><a href="http://46halbe.de"><img src="https://avatars2.githubusercontent.com/u/1724196?v=4" width="100px;" alt=""/><br /><sub><b>Göran Bodenschatz</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=coding46" title="Code">💻</a></td>
    <td align="center"><a href="https://www.hsbt.org/"><img src="https://avatars1.githubusercontent.com/u/12301?v=4" width="100px;" alt=""/><br /><sub><b>SHIBATA Hiroshi</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=hsbt" title="Code">💻</a></td>
  </tr>
  <tr>
    <td align="center"><a href="https://github.com/moracca"><img src="https://avatars1.githubusercontent.com/u/7213746?v=4" width="100px;" alt=""/><br /><sub><b>moracca</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=moracca" title="Code">💻</a></td>
    <td align="center"><a href="https://github.com/TPei"><img src="https://avatars0.githubusercontent.com/u/4004343?v=4" width="100px;" alt=""/><br /><sub><b>TPei</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=TPei" title="Documentation">📖</a></td>
    <td align="center"><a href="https://github.com/villuorav"><img src="https://avatars3.githubusercontent.com/u/8543094?v=4" width="100px;" alt=""/><br /><sub><b>Villu Orav</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=villuorav" title="Documentation">📖</a></td>
    <td align="center"><a href="https://github.com/deivid-rodriguez"><img src="https://avatars2.githubusercontent.com/u/2887858?v=4" width="100px;" alt=""/><br /><sub><b>David Rodríguez</b></sub></a><br /><a href="https://github.com/glejeune/Ruby-Graphviz/commits?author=deivid-rodriguez" title="Code">💻</a> <a href="#platform-deivid-rodriguez" title="Packaging/porting to new platform">📦</a></td>
  </tr>
</table>

<!-- markdownlint-enable -->
<!-- prettier-ignore-end -->
<!-- ALL-CONTRIBUTORS-LIST:END -->

This project follows the [all-contributors](https://github.com/kentcdodds/all-contributors) specification. Contributions of any kind welcome!
