//roarmonhttp.c:

/*
 *      Copyright (C) Philipp 'ph3-der-loewe' Schafft - 2008-2012
 *
 *  This file is part of roarclients a part of RoarAudio,
 *  a cross-platform sound system for both, home and professional use.
 *  See README for details.
 *
 *  This file is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License version 3
 *  as published by the Free Software Foundation.
 *
 *  RoarAudio is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this software; see the file COPYING.  If not, write to
 *  the Free Software Foundation, 51 Franklin Street, Fifth Floor,
 *  Boston, MA 02110-1301, USA.
 *
 */

/* ckport options:
 * ckport: ignore-symbol: alarm of target win32
 *           checked by configure. Needed to clear timeout some small HTTPds implement
 *           to avoid problems with bad CGI scripts.
 */

#include <roaraudio.h>

#if defined(ROAR_HAVE_SETENV) || defined(ROAR_HAVE_PUTENV)
#define _CAN_SET_ENV
#endif

#define BUFSIZE 1024

void usage (void) {
 printf("roarmonhttp [OPTIONS]...\n");

 printf("\nOptions:\n\n");

 printf("  --server    SERVER    - Set server hostname\n"
        "  --rate  -R  RATE      - Set sample rate\n"
        "  --bits  -B  BITS      - Set bits per sample\n"
        "  --chans -C  CHANNELS  - Set number of channels\n"
        "  --codec     CODEC     - Set the codec\n"
        "  --rel-id ID           - Set ID of relative stream\n"
        "  --inetd               - Start in inetd mode (STDIN and STDOUT connected to socket)\n"
        "  --help                - Show this help\n"
       );

}

void print_header (int codec, int rate, int channels) {
 const char * mime;

 mime = roar_codec2mime(codec);

 if ( mime == NULL )
  mime = "application/octet-stream";

 printf("Content-type: %s\r\n", mime);
 printf("ice-audio-info: ice-samplerate=%i;ice-channels=%i\r\n", rate, channels);
 printf("icy-pub:0\r\n");
 printf("Server: RoarAudio (roarmonhttp $Revision: 1.35 $)\r\n");
 printf("\r\n");

 fflush(stdout);
}

int stream (struct roar_vio_calls * dest, struct roar_vio_calls * src) {
 struct roar_vio_select vios[2];
 struct roar_buffer *ring = NULL, *cur;
 ssize_t len;
 size_t  todo;
 int alive = 1;
 void * data;
 int ret;

 if ( roar_vio_nonblock(src,  ROAR_SOCKET_NONBLOCK) == -1 ) {
  ROAR_WARN("stream(dest=%p, src=%p): Can not set source stream non-blocking: %s",
            dest, src, roar_error2str(roar_error));
 }
 if ( roar_vio_nonblock(dest, ROAR_SOCKET_NONBLOCK) == -1 ) {
  ROAR_WARN("stream(dest=%p, src=%p): Can not set destination stream non-blocking: %s",
            dest, src, roar_error2str(roar_error));
 }

 ROAR_VIO_SELECT_SETVIO(&(vios[0]), src,  ROAR_VIO_SELECT_READ);
 ROAR_VIO_SELECT_SETVIO(&(vios[1]), dest, ROAR_VIO_SELECT_WRITE);

 while (alive) {
  ret = roar_vio_select(vios, ring != NULL ? 2 : 1, NULL, NULL);
  if ( ret == -1 ) {
   alive = 0;
  } else if ( ret == 0 ) {
   // nothing happend.
  } else {
   if ( vios[0].eventsa & ROAR_VIO_SELECT_READ ) { // we can read!
    if ( roar_buffer_new_data(&cur, BUFSIZE, &data) == -1 )
     return -1;

    len = roar_vio_read(src, data, BUFSIZE);

    switch (len) {
     case  0:
     case -1:
       roar_buffer_free(cur);
       cur = NULL;

       if ( ring != NULL )
        roar_buffer_free(ring);

       ring = NULL;

       return -1;
      break;
    }

    if ( cur != NULL ) {
     if ( roar_buffer_set_len(cur, len) == -1 )
      return -1;

     if ( ring == NULL ) {
      ring = cur;
     } else {
      if ( roar_buffer_moveinto(ring, &cur) == -1 ) {
       ROAR_ERR("stream(dest=%p, src=%p): Can not append buffer to ring: %s",
                dest, src, roar_error2str(roar_error));
       roar_buffer_free(ring);
       roar_buffer_free(cur);
       return -1;
      }
     }
    }
   } else if ( (vios[1].eventsa & ROAR_VIO_SELECT_WRITE) && ring != NULL ) { // we can write!
    if ( roar_buffer_get_data(ring, &data) == -1 )
     return -1;

    if ( roar_buffer_get_len(ring, &todo) == -1 )
     return -1;

    len = roar_vio_write(dest, data, todo);

    if ( len < 1 ) {
     if ( roar_error != ROAR_ERROR_AGAIN ) {
      roar_buffer_free(ring);
      return -1;
     }
    }

    if ( (ssize_t)todo == len ) { // we wrote all of the pkg
     if ( roar_buffer_next(&ring) == -1 )
      return -1;
    } else {
     if ( roar_buffer_set_offset(ring, len) == -1 )
      return -1;
    }

   }
  }
 }

 return 0;
}

#ifdef _CAN_SET_ENV
int parse_http (int * gopher) {
 struct roar_keyval kv;
 char buf[1024];
 char * qs = buf, *str;
 ssize_t len;
 int dir = ROAR_DIR_MONITOR;

 if ( (len = read(ROAR_STDIN, buf, 1023)) == -1 )
  return -1;

 buf[len] = 0;

 if ( strncmp(buf, "GET /", 5) ) {
  if ( strncmp(buf, "SOURCE /", 8) ) {
   if ( buf[0] != '/' ) {
    return -1;
   } else {
    *gopher = 1;
   }
  } else {
   dir = ROAR_DIR_PLAY;
   qs += 3; 
  }
 }

 if ( !*gopher ) {
  qs += 5;

  if ( (str = strstr(qs, " ")) == NULL )
   return -1;

  *str = 0;
 } else {
  if ( (str = strstr(qs, "\r")) != NULL )
   *str = 0;
  if ( (str = strstr(qs, "\n")) != NULL )
   *str = 0;
 }

 for (; *qs != '?'; qs++)
  if ( !*qs )
   break;

 if ( *qs == '?' )
  qs++;

 if ( !*gopher )
  printf("HTTP/1.0 200 OK\r\n");
// printf("QS: %s\r\n", qs);

 fflush(stdout);

 kv.key   = "QUERY_STRING";
 kv.value = qs;

 roar_env_set(&kv);

 return dir;
}
#endif

int main (int argc, char * argv[]) {
 int    rate     = 44100;
 int    bits     = 16;
 int    channels = 2;
 int    codec    = ROAR_CODEC_OGG_VORBIS;
 int    rel_id   = -1;
 int    sflags   = ROAR_FLAG_NONE;
// int    codec    = ROAR_CODEC_DEFAULT;
 char * server   = NULL;
 int    i;
 char * c, * k, * v;
 char * sp0 = NULL, * sp1 = NULL;
 int dir = ROAR_DIR_MONITOR;
 int gopher = 0;
 struct roar_connection    con;
 struct roar_stream        s;
 struct roar_vio_calls   * vio;

#ifdef ROAR_HAVE_ALARM
 alarm(0); // reset alarm timers from httpd 
#endif

 for (i = 1; i < argc; i++) {
  k = argv[i];
  if ( !strcmp(k, "--inetd") ) {
#ifdef _CAN_SET_ENV
   if ( (dir = parse_http(&gopher)) == -1 )
    return 1;
#else
   return 1;
#endif
  } else if ( !strcmp(k, "--server") ) {
   roar_libroar_set_server(argv[++i]);
  } else if ( !strcmp(k, "--codec") ) {
   codec = roar_str2codec(argv[++i]);
  } else if ( !strcmp(k, "--rate") || !strcmp(k, "-r") || !strcmp(k, "-R") ) {
   rate = roar_str2rate(argv[++i]);
  } else if ( !strcmp(k, "--bits") || !strcmp(k, "-B") ) {
   bits = roar_str2bits(argv[++i]);
  } else if ( !strcmp(k, "--channels") || !strcmp(k, "--chans") || !strcmp(k, "-C") ) {
   channels = roar_str2channels(argv[++i]);
  } else if ( !strcmp(k, "--rel-id") ) {
   rel_id = atoi(argv[++i]);
  } else if ( !strcmp(k, "--help") && !strcmp(k, "-h") ) {
   usage();
   return 0;
  } else {
   ROAR_ERR("Unknown parameter: %s", k);
   usage();
   return 1;
  }
 }

 c = getenv("QUERY_STRING");
 if ( c == NULL )
  c = "";

 c = roar_mm_strtok_r(c, "&", &sp0);

 while (c != NULL) {
  k = roar_mm_strtok_r(c,    "=", &sp1);
  v = roar_mm_strtok_r(NULL, "=", &sp1);

  if ( !strcmp(k, "codec") ) {
   if ( (codec = roar_str2codec(v)) == -1 )
    return 1;
  } else if ( !strcmp(k, "channels") ) {
   channels = roar_str2channels(v);
  } else if ( !strcmp(k, "rate") ) {
   rate = roar_str2rate(v);
  } else if ( !strcmp(k, "bits") ) {
   bits = roar_str2bits(v);
  } else if ( !strcmp(k, "rel-id") || !strcmp(k, "relid") ) {
   rel_id = atoi(v);
  } else if ( !strcmp(k, "set-flag") ) {
   if ( !strcmp(v, "meta") ) {
    sflags |= ROAR_FLAG_META;
   } else if ( !strcmp(v, "cleanmeta") ) {
    sflags |= ROAR_FLAG_CLEANMETA;
   } else if ( !strcmp(v, "prethru") ) {
    sflags |= ROAR_FLAG_PRETHRU;
   } else {
    return 1;
   }
  } else if ( !strcmp(k, "dir") ) {
   if ( (dir = roar_str2dir(v)) == -1 )
    return 1;
  } else {
   return 1;
  }

  c = roar_mm_strtok_r(NULL, "&", &sp0);
 }

 if ( roar_simple_connect(&con, server, "roarmonhttp") == -1 ) {
  return 10;
 }

 if ( roar_stream_new(&s, rate, channels, bits, codec) == -1 ) {
  roar_disconnect(&con);
  return 20;
 }

 if ( rel_id != -1 ) {
  if ( roar_stream_set_rel_id(&s, rel_id) ) {
   roar_disconnect(&con);
   return 21;
  }
 }

 if ( roar_stream_connect(&con, &s, dir, -1) == -1 ) {
  roar_disconnect(&con);
  return 11;
 }

 if ( sflags != ROAR_FLAG_NONE ) {
  if ( roar_stream_set_flags(&con, &s, sflags, ROAR_SET_FLAG) == -1 ) {
   roar_disconnect(&con);
   return 14;
  }
 }

 if ( roar_stream_exec(&con, &s) == -1 ) {
  roar_disconnect(&con);
  return 12;
 }

 if ( (vio = roar_get_connection_vio2(&con)) == NULL )
  return 1;

 if ( !gopher )
  print_header(codec, rate, channels);

/*
 while((i = read(fh, buf, BUFSIZE)))
  if (write(out, buf, i) != i)
   break;
*/

 switch (dir) {
  case ROAR_DIR_PLAY:
    stream(vio, roar_stdin);
   break;
  case ROAR_DIR_MONITOR:
  case ROAR_DIR_THRU:
    stream(roar_stdout, vio);
   break;
 }

 roar_vio_close(vio);

 return 0;
}

//ll
