// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "arrow.h"
#include GL_INCLUDE    // see platform.h
#include <cassert>

namespace visual {
      
int
arrow::get_fixedwidth() const
{
	return fixedwidth;
}

void 
arrow::set_fixedwidth( const int& width) 
{
	write_lock L( mtx);
	fixedwidth = width;
}


double 
arrow::get_shaftwidth()
{
    getScale();
    return effective_shaftwidth;
}

void 
arrow::set_shaftwidth( const double& width) 
{
	write_lock L( mtx);
	radius = width;
}


double 
arrow::get_headwidth()
{
	getScale();
	return effective_headwidth;
}

void 
arrow::set_headwidth( const double& width)
{
	write_lock L( mtx);
	headwidth = width;
}

double 
arrow::get_headlength()
{
	getScale();
	return effective_headlength;
}

void 
arrow::set_headlength( const double& length) 
{
	write_lock L( mtx);
	headlength = length;
}

arrow::arrow() 
	: fixedwidth(0), headwidth(0), headlength(0)
	, effective_headwidth(0), effective_shaftwidth(0), effective_headlength(0) 
{ 
	radius = 0.0;
}

arrow::arrow( const arrow& other)
	: axialSymmetry( other), fixedwidth( other.fixedwidth), 
	headwidth( other.headwidth), headlength( other.headlength),
	effective_headwidth( other.effective_headwidth),
	effective_shaftwidth( other.effective_shaftwidth),
	effective_headlength( other.effective_headlength)
{
}

vector 
arrow::getScale()
{
	double length = axis.mag();
	if (length == 0) {
		degenerate = true;
	}
	else {
		degenerate = false;
	}
	// Rules:
	//   min_sw     <= shaftwidth/length    <= max_sw
	//   min_hw     <= headwidth/shaftwidth <= max_hw
	//   min_aspect <= headlength/headwidth <= max_aspect
	//                 headlength/length    <= max_hl
	double min_sw = 0.02,  default_sw = 0.1, max_sw UNUSED = 0.25;
	double min_hw UNUSED = 1,  default_hw = 2, max_hw UNUSED = 10;
	double min_aspect UNUSED = 0.2, default_aspect = 1.5, max_aspect UNUSED = 10;
	double max_hl = 0.5;

	// shaftwidth is radius
	// Defalt width of the shaft is equal to 10% of the total length
	if (!radius)
		effective_shaftwidth = length * default_sw;
	else
		effective_shaftwidth = radius;

	// Default headwidth = 20% of shaft width (or 2x shaft width).
	if (!headwidth)
		effective_headwidth = effective_shaftwidth * default_hw;
	else
		effective_headwidth = headwidth;

	// Default headlength = 30% of the total length, (or 1.5x head width)
	if (!headlength)
		effective_headlength = effective_headwidth * default_aspect;
	else
		effective_headlength = headlength;

	if (fixedwidth) {
		if (effective_headlength > max_hl*length) {
			effective_headlength = max_hl*length;
		}
	}
	else {
		if (effective_shaftwidth < length * min_sw) {
			double r = length * min_sw / effective_shaftwidth;
			effective_shaftwidth *= r;
			effective_headwidth *= r;
			effective_headlength *= r;
		}
		if (effective_headlength > length * max_hl) {
			double r = length * max_hl / effective_headlength;
			effective_shaftwidth *= r;
			effective_headwidth *= r;
			effective_headlength *= r;
		}
	}
	return vector(length, effective_headwidth/2, effective_headwidth/2);
}

void 
arrow::glRender(rView &view)
{
	if (degenerate) 
		return;
	if (scale.x == 0)
		return;

	static double brect[] = { 1.0,  1.0,  1.0,
	                          0.0, -1.0, -1.0};
	view.ext_brect(mwt, brect);

	tmatrix mct( mwt, view.wct);

	float X = 1.0 - effective_headlength/scale.x;

	float nr = 1.0 / std::sqrt(effective_headwidth*effective_headwidth/4 + effective_headlength*effective_headlength);
	float nc = effective_headwidth/2 * nr;
	float ns = effective_headlength * nr;

	// Calculate the lighting on each face.
	float L[10];
	for(int i=0; i<10; i++) 
		L[i] = view.lights.ambient;
	for(int l=0; l<view.lights.n_lights; l++) {
		vector lt = wlt * view.lights.L[l];

		// For the axis-aligned normals, the dot products are
		//   e.g. [1 0 0]T * lt = lt.x
		if (lt.x < 0)
			L[2]-=lt.x;
		if (lt.y > 0)
			L[3]+=lt.y;
		else
			L[1]-=lt.y;
		if (lt.z > 0)
			L[4]+=lt.z;
		else
			L[0]-=lt.z;

		// For the faces of the head, the normals 
		//   are [nc +/-ns 0]T or [nc 0 +/-ns].  The dot products
		//   degenerate accordingly:
		lt.x *= nc;
		lt.y *= ns;
		lt.z *= ns;
		if (lt.x >  lt.z) L[6] += lt.x - lt.z;
		if (lt.x >  lt.y) L[7] += lt.x - lt.y;
		if (lt.x > -lt.y) L[8] += lt.x + lt.y;
		if (lt.x > -lt.z) L[9] += lt.x + lt.z;
	}
	L[5] = L[2];

	static int s[10][4] = { {1,3,0,2},         // shaft
	                        {0,4,1,5},
	                        {0,2,4,6},
	                        {6,2,7,3},
	                        {4,6,5,7},
	                        {10,11,8,9},       // back of head
	                        {12,8,9, -1},  // front of head (triangles)
	                        {12,10,8, -1},
	                        {12,9,11, -1},
	                        {12,11,10, -1} };

	float shf = effective_shaftwidth / effective_headwidth;
	float v[13][3] = { { 0.0, -shf, -shf },  // shaft
	                   {   X, -shf, -shf },
	                   { 0.0,  shf, -shf },
	                   {   X,  shf, -shf },
	                   { 0.0, -shf,  shf },
	                   {   X, -shf,  shf },
	                   { 0.0,  shf,  shf },
	                   {   X,  shf,  shf },
	                   
	                   {   X, -1.0,  -1.0 },   // head
	                   {   X,  1.0,  -1.0 },
	                   {   X, -1.0,   1.0 },
	                   {   X,  1.0,   1.0 },

	                   { 1.0,  0.0,   0.0 }    // tip
	                 };

	// Projection and rendering
	mct.gl_load();

	glEnableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_COLOR_ARRAY);
	glVertexPointer(3, GL_FLOAT, 3 * sizeof(float), v);
	glShadeModel(GL_FLAT);

	for(int side=0; side<10; side++) {
		glColor3f(L[side]*color.r, L[side]*color.g, L[side]*color.b);
		glDrawElements(GL_TRIANGLE_STRIP, s[side][3]<0 ? 3 : 4, GL_UNSIGNED_INT, s[side]);
	}
	glLoadIdentity();
}

} // !namespace visual
