/*
 * $Id: history.c,v 1.20 2001/10/27 18:56:41 nordstrom Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2001, Mark Ian Lillywhite and Michael Nordstrm
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "debug.h"
#include "document.h"
#include "documentdata.h"
#include "history.h"
#include "util.h"


const Boolean GetAddHistory( void ) PLKRDB_SECTION;
const Int16 GetHistoryPrev( void ) PLKRDB_SECTION;
const Int16 GetHistoryNext( void ) PLKRDB_SECTION;
const Int16 GetHistoryCurrent( void ) PLKRDB_SECTION;
void InitHistory( void ) PLKRDB_SECTION;
void ReadHistory( History* recordPtr ) PLKRDB_SECTION;
void SaveHistory( History* recordPtr ) PLKRDB_SECTION;
void SetHistoryFirstParagraph( const Int16 offset, const Int16 y ) PLKRDB_SECTION;
void SetHistoryLastParagraph( const Int16 offset, const Int16 y ) PLKRDB_SECTION;
void SetHistoryVerticalOffset( const Int16 offset ) PLKRDB_SECTION;
void AddToHistory( const Int16 recordId ) PLKRDB_SECTION;
void ResetHistory( void ) PLKRDB_SECTION;

static void RollHistoryIndex( Int16* index ) PLKRDB_SECTION;
static void RestoreData( void ) PLKRDB_SECTION;


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static History history;



/* Return next history index */
static void RollHistoryIndex
    (
    Int16* index    /* pointer to history index ( upon return, it will
                       contain the next history index ) */
    )
{
    ( *index )++;
    if ( HISTORYSIZE <= *index )
        *index = 0;
}



/* Restore data for current record from history */
static void RestoreData( void )
{
    MetaRecord* meta;
    MemHandle   handle;
    Int16       recordId;

    recordId    = history.records[ history.currentRecord ].recordId;
    handle      = GetMetaHandle( recordId, false );
    meta        = (MetaRecord*) MemHandleLock( handle );
    ErrFatalDisplayIf( meta == NULL, "RestoreData: MemHandleLock failed" );

    DmWrite( meta, OFFSETOF( MetaRecord, verticalOffset ), 
        (void*) &history.records[ history.currentRecord ], 
        sizeof( HistoryData ) - sizeof( Int16 ) );

    MemPtrUnlock( meta );
}



/* Initialize the history structure */
void InitHistory( void )
{
    MemSet( (void*) &history, sizeof( History ), 0 );

    history.addHistory              = false;
    history.records[ 0 ].recordId   = HOME_PAGE_ID;
}



/* Read history from session record */
void ReadHistory
    (
    History* recordPtr /* pointer to session record */
    )
{
    MemMove( (void*) &history, (void*) recordPtr, sizeof( History ) );
}



/* Save history in session record */
void SaveHistory
    (
    History* recordPtr /* pointer to session record */
    )
{
    DmWrite( (void*) recordPtr, 0, (void*) &history, sizeof( History ) );
}



/* Add record to history */
void AddToHistory
    (
    const Int16 recordId  /* record ID */
    )
{
    if ( ! history.addHistory ) {
        history.addHistory = true;
        return;
    }

    RollHistoryIndex( &history.currentRecord );
    history.lastRecord = history.currentRecord;

    if ( history.currentRecord == history.firstRecord )
        RollHistoryIndex( &history.firstRecord );

    history.records[ history.currentRecord ].recordId = recordId;
}



/* Reset vertical offset and data for first and last paragraph in the history structure */
void ResetHistory( void )
{
    UInt8 i;

    for ( i = 0; i < HISTORYSIZE; i++ ) {
        history.records[ i ].verticalOffset         = 0;
        history.records[ i ].firstVisibleParagraph  = 0;
        history.records[ i ].firstParagraphY        = 0;
        history.records[ i ].lastVisibleParagraph   = 0;
        history.records[ i ].lastParagraphY         = 0;
    }
}



/* Store data for first paragraph in current record from history */
void SetHistoryFirstParagraph
    (
    const Int16 offset, /* offset to paragraph in record */
    const Int16 y       /* paragraph's y coordinate within record */
    )
{
    history.records[ history.currentRecord ].firstVisibleParagraph  = offset;
    history.records[ history.currentRecord ].firstParagraphY        = y;
}



/* Store data for last paragraph in current record from history */
void SetHistoryLastParagraph
    (
    const Int16 offset, /* offset to paragraph in record */
    const Int16 y       /* paragraph's y coordinate within record */
    )
{
    history.records[ history.currentRecord ].lastVisibleParagraph   = offset;
    history.records[ history.currentRecord ].lastParagraphY         = y;
}



/* Store vertical offset for current record from history */
void SetHistoryVerticalOffset
    (
    const Int16 offset  /* vertical offset in record */
    )
{
    history.records[ history.currentRecord ].verticalOffset = offset;
}



/* Get the add history status */
const Boolean GetAddHistory( void )
{
    return history.addHistory;
}



/* Get the current record ID from the history */
const Int16 GetHistoryCurrent( void )
{
    return history.records[ history.currentRecord ].recordId;
}



/* Get the next record from the history */
const Int16 GetHistoryNext( void )
{
    if ( history.currentRecord == history.lastRecord )
        return NOT_FOUND;

    RollHistoryIndex( &history.currentRecord );

    RestoreData();

    return history.records[ history.currentRecord ].recordId;
}



/* Get the previous record from the history */
const Int16 GetHistoryPrev( void )
{
    if ( history.currentRecord == history.firstRecord )
        return NOT_FOUND;

    if ( --history.currentRecord < 0 )
        history.currentRecord = HISTORYSIZE - 1;

    RestoreData();

    return history.records[ history.currentRecord ].recordId;
}
