// Copyright (c) Lawrence Livermore National Security, LLC and other VisIt
// Project developers.  See the top-level LICENSE file for dates and other
// details.  No copyright assignment is required to contribute to VisIt.

#include <PixiePluginInfo.h>
#include <avtPixieFileFormat.h>
#include <avtMTSDFileFormatInterface.h>
#include <avtGenericDatabase.h>
#include <avtPixieOptions.h>

// ****************************************************************************
//  Method:  PixieCommonPluginInfo::GetDatabaseType
//
//  Purpose:
//    Returns the type of a Pixie database.
//
//  Programmer:  generated by xml2info
//  Creation:    omitted
//
// ****************************************************************************
DatabaseType
PixieCommonPluginInfo::GetDatabaseType()
{
    return DB_TYPE_MTSD;
}

// ****************************************************************************
//  Method: PixieCommonPluginInfo::SetupDatabase
//
//  Purpose:
//      Sets up a Pixie database.
//
//  Arguments:
//      list    A list of file names.
//      nList   The number of timesteps in list.
//      nBlocks The number of blocks in the list.
//
//  Returns:    A Pixie database from list.
//
//  Programmer: whitlocb -- generated by xml2info
//  Creation:   Fri Aug 13 14:31:41 PST 2004
//
//  Modifications:
//    Brad Whitlock, Fri Aug 13 18:47:12 PST 2004
//    Added some exception handline so the file formats are deleted if
//    if creating one of them causes an exception to be generated.
//
//    Jeremy Meredith, Thu Jan 28 12:28:07 EST 2010
//    MTSD now accepts grouping multiple files into longer sequences, so
//    its interface has changed to accept both a number of timestep groups
//    and a number of blocks.
// ****************************************************************************

avtDatabase *
PixieCommonPluginInfo::SetupDatabase(const char *const *list,
                                     int nList, int nBlock)
{
    int nTimestepGroups = nList / nBlock;
    avtMTSDFileFormat ***ffl = new avtMTSDFileFormat**[nTimestepGroups];
    for (int i = 0 ; i < nTimestepGroups ; i++)
    {
        ffl[i] = new avtMTSDFileFormat*[nBlock];
        for (int j = 0 ; j < nBlock ; j++)
        {
            ffl[i][j] = NULL;
        }
    }

    TRY
    {
        for (int i = 0 ; i < nTimestepGroups ; i++)
        {
            for (int j = 0 ; j < nBlock ; j++)
            {
                ffl[i][j] = new avtPixieFileFormat(list[i*nBlock+j], readOptions);
            }
        }
    }
    CATCH(VisItException)
    {
        for (int i = 0 ; i < nTimestepGroups ; i++)
        {
            for (int j = 0 ; j < nBlock ; j++)
            {
                if(ffl[i][j] != 0)
                    delete ffl[i][j];
            }
            delete [] ffl[i];
        }
        delete [] ffl;
        RETHROW;
    }
    ENDTRY

    avtMTSDFileFormatInterface *inter
        = new avtMTSDFileFormatInterface(ffl, nTimestepGroups, nBlock);
    return new avtGenericDatabase(inter);
}


// ****************************************************************************
//  Method: PixieCommonPluginInfo::GetReadOptions
//
//  Purpose:
//      Gets the read options.
//
//  Programmer: generated by xml2info
//  Creation:   omitted
//
// ****************************************************************************

DBOptionsAttributes *
PixieCommonPluginInfo::GetReadOptions() const
{
    return GetPixieReadOptions();
}

// ****************************************************************************
//  Method: PixieCommonPluginInfo::GetWriteOptions
//
//  Purpose:
//      Gets the write options.
//
//  Programmer: generated by xml2info
//  Creation:   omitted
//
// ****************************************************************************

DBOptionsAttributes *
PixieCommonPluginInfo::GetWriteOptions() const
{
    return GetPixieWriteOptions();
}

