;;;-*- Mode: Lisp; Package: CCL -*-
;;;
;;;   Copyright (C) 2001 Clozure Associates
;;;   This file is part of Opensourced MCL.
;;;
;;;   Opensourced MCL is free software; you can redistribute it and/or
;;;   modify it under the terms of the GNU Lesser General Public
;;;   License as published by the Free Software Foundation; either
;;;   version 2.1 of the License, or (at your option) any later version.
;;;
;;;   Opensourced MCL is distributed in the hope that it will be useful,
;;;   but WITHOUT ANY WARRANTY; without even the implied warranty of
;;;   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
;;;   Lesser General Public License for more details.
;;;
;;;   You should have received a copy of the GNU Lesser General Public
;;;   License along with this library; if not, write to the Free Software
;;;   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
;;;

(in-package "CCL")

;; basic socket API
(eval-when (:compile-toplevel :load-toplevel :execute)
  (export '(MAKE-SOCKET
	    ACCEPT-CONNECTION
	    DOTTED-TO-IPADDR
	    IPADDR-TO-DOTTED
	    IPADDR-TO-HOSTNAME
	    LOOKUP-HOSTNAME
	    LOOKUP-PORT
	    ;;with-pending-connect
	    RECEIVE-FROM
	    SEND-TO
	    SHUTDOWN
	    ;;socket-control
	    SOCKET-OS-FD
	    REMOTE-HOST
	    REMOTE-PORT
	    ;;remote-filename
	    LOCAL-HOST
	    LOCAL-PORT
	    ;;local-filename
	    SOCKET-ADDRESS-FAMILY
	    SOCKET-CONNECT
	    SOCKET-FORMAT
	    SOCKET-TYPE
	    SOCKET-ERROR
	    SOCKET-ERROR-CODE
	    SOCKET-ERROR-IDENTIFIER
	    SOCKET-ERROR-SITUATION
	    WITH-OPEN-SOCKET)))

(eval-when (:compile-toplevel :execute)
  #+linuxppc-target
  (require "LINUX-SYSCALLS")
  #+darwinppc-target
  (require "DARWIN-SYSCALLS"))

(define-condition socket-error (simple-error)
  ((code :initarg :code :reader socket-error-code)
   (identifier :initform :unknown :initarg :identifier :reader socket-error-identifier)
   (situation :initarg :situation :reader socket-error-situation)))

(defvar *socket-error-identifiers*
  (list #$EADDRINUSE :address-in-use
	#$ECONNABORTED :connection-aborted
	#$ENOBUFS :no-buffer-space
	#$ENOMEM :no-buffer-space
	#$ENFILE :no-buffer-space
	#$ETIMEDOUT :connection-timed-out
	#$ECONNREFUSED :connection-refused
	#$ENETUNREACH :host-unreachable
	#$EHOSTUNREACH :host-unreachable
	#$EHOSTDOWN :host-down
	#$ENETDOWN :network-down
	;; ?? :address-not-available
	;; ?? :network-reset
	;; ?? :connection-reset
	;; ?? :shutdown
	#$EACCES :access-denied
	#$EPERM :access-denied))


(declaim (inline socket-call))
(defun socket-call (where res)
  (if (< res 0)
    (socket-error where res)
    res))

(defun socket-error (where errno)
  (when (< errno 0)
    (setq errno (- errno)))
  (error (make-condition 'socket-error
			 :code errno
			 :identifier (getf *socket-error-identifiers* errno :unknown)
			 :situation where
			 ;; TODO: this is a constant arg, there is a way to put this
			 ;; in the class definition, just need to remember how...
			 :format-control "~a (error #~d) in ~a"
			 :format-arguments (list (%strerror errno) errno where))))

;; If true, this will try to allow other processes to run while
;; socket io is happening.
(defvar *multiprocessing-socket-io* t)

(defclass socket ()
  ())

(defmacro with-open-socket ((var . args) &body body
			    &aux (socket (make-symbol "socket"))
			         (done (make-symbol "done")))
  `(let (,socket ,done)
     (unwind-protect
	 (multiple-value-prog1
	   (let ((,var (setq ,socket (make-socket ,@args))))
	     ,@body)
	   (setq ,done t))
       (when ,socket (close ,socket :abort (not ,done))))))

(defclass ip-socket (socket)
  ())

(defmethod SOCKET-ADDRESS-FAMILY ((socket ip-socket)) :internet)

(defclass tcp-socket (ip-socket)
  ())

(defmethod SOCKET-TYPE ((socket tcp-socket)) :stream)

;; An active TCP socket is an honest-to-goodness stream.
(defclass tcp-stream (tcp-socket fd-stream
				 buffered-binary-io-stream-mixin
				 buffered-character-io-stream-mixin)
  ())

(defmethod SOCKET-CONNECT ((stream tcp-stream)) :active)

(defmethod SOCKET-FORMAT ((stream tcp-stream))
  (if (eq (stream-element-type stream) 'character)
    :text
    ;; Should distinguish between :binary and :bivalent, but hardly
    ;; seems worth carrying around an extra slot just for that.
    :bivalent))

(defmethod socket-device ((stream tcp-stream))
  (let ((ioblock (stream-ioblock stream)))
    (and ioblock (ioblock-device ioblock))))

(defmethod select-stream-class ((class tcp-stream) in-p out-p char-p)
  (declare (ignore char-p)) ; TODO: is there any real reason to care about this?
  (assert (and in-p out-p) () "Non-bidirectional tcp stream?")
  'tcp-stream)

(defclass unconnected-socket (socket)
  ((device :initarg :device :accessor socket-device)
   (keys :initarg :keys :reader socket-keys)))

(defmethod SOCKET-FORMAT ((socket unconnected-socket))
  (or (getf (socket-keys socket) :format) :text))

(defmethod CLOSE ((socket unconnected-socket) &key abort)
  (declare (ignore abort))
  (when (socket-device socket)
    (fd-close (socket-device socket))
    (setf (socket-device socket) nil)))

;; A passive tcp socket just generates connection streams
(defclass listener-socket (tcp-socket unconnected-socket) ())

(defmethod SOCKET-CONNECT ((stream listener-socket)) :passive)

;; A udp socket just sends and receives packets.
(defclass udp-socket (ip-socket unconnected-socket) ())

(defmethod SOCKET-TYPE ((stream udp-socket)) :datagram)
(defmethod SOCKET-CONNECT ((stream udp-socket)) nil)

;; Returns nil for closed stream...
(defmethod SOCKET-OS-FD ((socket socket))
  (socket-device socket))

;; Returns nil for closed stream
(defun local-socket-info (fd type)
  (and fd
       (rlet ((sockaddr :sockaddr_in)
	      (namelen :signed))
	     (setf (pref namelen :signed) (record-length :sockaddr_in))
	     (socket-call "getsockname" (c_getsockname fd sockaddr namelen))
	     (ecase type
	       (:host (#_ntohl (pref sockaddr :sockaddr_in.sin_addr.s_addr)))
	       (:port (#_ntohs (pref sockaddr :sockaddr_in.sin_port)))))))

(defun remote-socket-info (fd type)
  (and fd
       (rlet ((sockaddr :sockaddr_in)
	      (namelen :signed))
	     (setf (pref namelen :signed) (record-length :sockaddr_in))
	     (let ((err (c_getpeername fd sockaddr namelen)))
	       (cond ((eql err (- #$ENOTCONN)) nil)
		     ((< err 0) (socket-error "getpeername" err))
		     (t
		      (ecase type
			(:host (#_ntohl (pref sockaddr :sockaddr_in.sin_addr.s_addr)))
			(:port (#_ntohs (pref sockaddr :sockaddr_in.sin_port))))))))))

(defmethod LOCAL-PORT ((socket socket))
  (local-socket-info (socket-device socket) :port))

(defmethod LOCAL-HOST ((socket socket))
  (local-socket-info (socket-device socket) :host))

;; Returns NIL if socket is not connected
(defmethod REMOTE-HOST ((socket socket))
  (remote-socket-info (socket-device socket) :host))

(defmethod REMOTE-PORT ((socket socket))
  (remote-socket-info (socket-device socket) :port))
    

(defun set-socket-options (fd &key keepalive
			           reuse-address
				   nodelay
				   broadcast
				   linger
				   local-port
				   local-host
				   type
				   out-of-band-inline
				   &allow-other-keys)
  ;; see man socket(7) tcp(7) ip(7)
  (when keepalive
    (int-setsockopt fd #$SOL_SOCKET #$SO_KEEPALIVE 1))
  (when reuse-address
    (int-setsockopt fd #$SOL_SOCKET #$SO_REUSEADDR 1))
  (when nodelay
    (int-setsockopt fd
		    #+linuxppc-target #$SOL_TCP
		    #+darwinppc-target #$IPPROTO_TCP
		    #$TCP_NODELAY 1))
  (when broadcast
    (int-setsockopt fd #$SOL_SOCKET #$SO_BROADCAST 1))
  (when out-of-band-inline
    (int-setsockopt fd #$SOL_SOCKET #$SO_OOBINLINE 1))
  (when linger
    (%stack-block ((plinger 8))
      (setf (%get-long plinger 0) 1
	    (%get-long plinger 4) linger)
      (socket-call "setsockopt"
		   (c_setsockopt fd #$SOL_SOCKET #$SO_LINGER plinger 8))))
  (when (or local-port local-host)
    (let* ((proto (if (eq type :stream) "tcp" "udp"))
	   (port-n (if local-port (port-as-inet-port local-port proto) 0))
	   (host-n (if local-host (host-as-inet-host local-host) #$INADDR_ANY)))
      (rlet ((sockaddr :sockaddr_in))
	(setf (pref sockaddr :sockaddr_in.sin_family) #$AF_INET
	      (pref sockaddr :sockaddr_in.sin_port) port-n
	      (pref sockaddr :sockaddr_in.sin_addr.s_addr) host-n)
	(socket-call "bind" (c_bind fd sockaddr (record-length :sockaddr_in))))))
  (when *multiprocessing-socket-io*
    (socket-call "fcntl" (fd-set-flag fd #$O_NONBLOCK))))

(defun MAKE-SOCKET (&rest keys
		    &key address-family
		    ;; List all keys here just for error checking...
		    ;; &allow-other-keys
		    type connect remote-host remote-port eol format
		    keepalive reuse-address nodelay broadcast linger
		    local-port local-host backlog class out-of-band-inline)
  (declare (dynamic-extent keys))
  (declare (ignore type connect remote-host remote-port eol format
		   keepalive reuse-address nodelay broadcast linger
		   local-port local-host backlog class out-of-band-inline))
  (ecase address-family
    ;; ((:file) (apply #'make-file-socket keys))
    ((nil :internet) (apply #'make-ip-socket keys))))


;; I hope the inline declaration makes the &rest/apply's go away...
(declaim (inline make-ip-socket))
(defun make-ip-socket (&rest keys &key type &allow-other-keys)
  (declare (dynamic-extent keys))
  (ecase type
    ((nil :stream) (apply #'make-tcp-socket keys))
    ((:datagram) (apply #'make-udp-socket keys))))

(defun make-udp-socket (&rest keys &aux (fd -1))
  (unwind-protect
    (let (socket)
      (setq fd (socket-call "socket"
			    (c_socket #$AF_INET #$SOCK_DGRAM #$IPPROTO_UDP)))
      (apply #'set-socket-options fd keys)
      (setq socket (make-instance 'udp-socket
				  :device fd
				  :keys keys))
      (setq fd -1)
      socket)
    (unless (< fd 0)
      (fd-close fd))))

(defun make-tcp-socket (&rest keys &key connect &allow-other-keys &aux (fd -1))
  (unwind-protect
    (let (socket)
      (setq fd (socket-call "socket"
			    (c_socket #$AF_INET #$SOCK_STREAM #$IPPROTO_TCP)))
      (apply #'set-socket-options fd keys)
      (setq socket
	    (ecase connect
	      ((nil :active) (apply #'make-tcp-stream-socket fd keys))
	      ((:passive) (apply #'make-tcp-listener-socket fd keys))))
      (setq fd -1)
      socket)
    (unless (< fd 0)
      (fd-close fd))))

(defun inet-connect (fd host-n port-n)
  (let ((err (rlet ((sockaddr :sockaddr_in))
	       (setf (pref sockaddr :sockaddr_in.sin_family) #$AF_INET
		     (pref sockaddr :sockaddr_in.sin_port) port-n
		     (pref sockaddr :sockaddr_in.sin_addr.s_addr) host-n)
	       (c_connect fd sockaddr (record-length :sockaddr_in)))))
    (declare (fixnum err))
    (when (eql err (- #$EINPROGRESS))
      (process-output-wait fd)
      (setq err (- (int-getsockopt fd #$SOL_SOCKET #$SO_ERROR))))
    (unless (eql err 0) (socket-error "connect" err))))

(defun make-tcp-stream-socket (fd &key remote-host
				  remote-port
				  eol
				  format
				  (class 'tcp-stream)
				  &allow-other-keys)
  (inet-connect fd
		(host-as-inet-host remote-host)
		(port-as-inet-port remote-port "tcp"))
  (make-tcp-stream fd :format format :eol eol :class class))

(defun make-tcp-stream (fd &key format eol (class 'tcp-stream)  &allow-other-keys)
  (declare (ignore eol))		;???
  (let ((element-type (ecase format
			((nil :text) 'character)
			((:binary :bivalent) '(unsigned-byte 8)))))
    ;; TODO: check out fd-stream-advance, -listen, -eofp, -force-output, -close
    ;; See if should specialize any of 'em.
    (make-fd-stream fd
		    :class class
		    :direction :io
		    :element-type element-type)))

(defun make-tcp-listener-socket (fd &rest keys &key backlog &allow-other-keys)
  (socket-call "listen" (c_listen fd (or backlog 5)))
  (make-instance 'listener-socket
		 :device fd
		 :keys keys))

(defun inet-accept (fd wait)
  (flet ((_accept (fd async)
	   (let ((res (c_accept fd (%null-ptr) (%null-ptr))))
	     (declare (fixnum res))
	     ;; See the inscrutable note under ERROR HANDLING in
	     ;; man accept(2). This is my best guess at what they mean...
	     (if (and async (< res 0)
		      (or (eql res (- #$ENETDOWN))
			  (eql res (- #+linuxppc-target #$EPROTO
				      #+darwinppc-target #$EPROTOTYPE))
			  (eql res (- #$ENOPROTOOPT))
			  (eql res (- #$EHOSTDOWN))
			  (eql res (- #+linuxppc-target #$ENONET
				      #+darwinppc-target #$ENETDOWN))
			  (eql res (- #$EHOSTUNREACH))
			  (eql res (- #$EOPNOTSUPP))
			  (eql res (- #$ENETUNREACH))))
	       (- #$EAGAIN)
	       res))))
    (cond (wait
	    (with-eagain fd :input
	      (_accept fd *multiprocessing-socket-io*)))
	  (*multiprocessing-socket-io*
	    (_accept fd t))
	  (t
	    (let ((old (socket-call "fcntl" (fd-get-flags fd))))
	      (unwind-protect
		  (progn
		    (socket-call "fcntl" (fd-set-flags fd (logior old #$O_NONBLOCK)))
		    (_accept fd t))
		(socket-call "fcntl" (fd-set-flags fd old))))))))

(defmethod ACCEPT-CONNECTION ((socket listener-socket) &key (wait t))
  (let ((listen-fd (socket-device socket))
	(fd -1))
    (unwind-protect
      (progn
	(setq fd (inet-accept listen-fd wait))
	(cond ((>= fd 0)
	       (prog1 (apply #'make-tcp-stream fd (socket-keys socket))
		 (setq fd -1)))
	      ((eql fd (- #$EAGAIN)) nil)
	      (t (socket-error "accept" fd))))
      (when (>= fd 0)
	(fd-close fd)))))

(defun verify-socket-buffer (buf offset size)
  (unless offset (setq offset 0))
  (unless (<= (+ offset size) (length buf))
    (report-bad-arg size `(integer 0 ,(- (length buf) offset))))
  (multiple-value-bind (arr start) (array-data-and-offset buf)
    (setq buf arr offset (+ offset start)))
  ;; TODO: maybe should allow any raw vector
  (let ((subtype (typecode buf)))
    (unless (and (<= arch::min-8-bit-ivector-subtag subtype)
		 (<= subtype arch::max-8-bit-ivector-subtag))
      (report-bad-arg buf `(or (array character)
			       (array (unsigned-byte 8))
			       (array (signed-byte 8))))))
  (values buf offset))

(defmethod SEND-TO ((socket udp-socket) msg size
		    &key remote-host remote-port offset)
  (let ((fd (socket-device socket)))
    (multiple-value-setq (msg offset) (verify-socket-buffer msg offset size))
    (unless remote-host
      (setq remote-host (or (getf (socket-keys socket) :remote-host)
			    (remote-socket-info fd :host))))
    (unless remote-port
      (setq remote-port (or (getf (socket-keys socket) :remote-port)
			    (remote-socket-info fd :port))))
    (rlet ((sockaddr :sockaddr_in))
      (setf (pref sockaddr :sockaddr_in.sin_family) #$AF_INET)
      (setf (pref sockaddr :sockaddr_in.sin_addr.s_addr)
	    (if remote-host (host-as-inet-host remote-host) #$INADDR_ANY))
      (setf (pref sockaddr :sockaddr_in.sin_port)
	    (if remote-port (port-as-inet-port remote-port "udp") 0))
      (%stack-block ((bufptr size))
        (%copy-ivector-to-ptr msg offset bufptr 0 size)
	(socket-call "sendto"
	  (with-eagain fd :output
	    (c_sendto fd bufptr size 0 sockaddr (record-length :sockaddr_in))))))))

(defmethod RECEIVE-FROM ((socket udp-socket) size &key buffer extract offset)
  (let ((fd (socket-device socket))
	(vec-offset offset)
	(vec buffer)
	(ret-size -1))
    (when vec
      (multiple-value-setq (vec vec-offset)
	(verify-socket-buffer vec vec-offset size)))
    (rlet ((sockaddr :sockaddr_in)
	   (namelen :signed))
      (setf (pref sockaddr :sockaddr_in.sin_family) #$AF_INET)
      (setf (pref sockaddr :sockaddr_in.sin_addr.s_addr) #$INADDR_ANY)
      (setf (pref sockaddr :sockaddr_in.sin_port) 0)
      (setf (pref namelen :signed) (record-length :sockaddr_in))
      (%stack-block ((bufptr size))
	(setq ret-size (socket-call "recvfrom"
			 (with-eagain fd :input
			   (c_recvfrom fd bufptr size 0 sockaddr namelen))))
	(unless vec
	  (setq vec (make-array ret-size
				:element-type
				(ecase (socket-format socket)
				  ((:text) 'base-character)
				  ((:binary :bivalent) '(unsigned-byte 8))))
		vec-offset 0))
	(%copy-ptr-to-ivector bufptr 0 vec vec-offset ret-size))
      (values (cond ((null buffer)
		     vec)
		    ((or (not extract)
			 (and (eql 0 (or offset 0))
			      (eql ret-size (length buffer))))
		     buffer)
		    (t 
		     (subseq vec vec-offset (+ vec-offset ret-size))))
	      ret-size
	      (#_ntohl (pref sockaddr :sockaddr_in.sin_addr.s_addr))
	      (#_ntohs (pref sockaddr :sockaddr_in.sin_port))))))

(defmethod SHUTDOWN (socket &key direction)
  ;; TODO: should we ignore ENOTCONN error?  (at least make sure it
  ;; is a distinct, catchable error type).
  (let ((fd (socket-device socket)))
    (socket-call "shutdown"
      (c_shutdown fd (ecase direction
		       (:input 0)
		       (:output 1))))))

;; Accepts port as specified by user, returns port number in network byte
;; order.  Protocol should be one of "tcp" or "udp".  Error if not known.
(defun port-as-inet-port (port proto)
  (or (etypecase port
	(fixnum (#_htons port))
	(string (_getservbyname port proto))
	(symbol (_getservbyname (string-downcase (symbol-name port)) proto)))
      (socket-error "getservbyname" (- #$ENOENT))))

(defun LOOKUP-PORT (port proto)
  (if (fixnump port)
    port
    (#_ntohs (port-as-inet-port port proto))))

;; Accepts host as specified by user, returns host number in network byte
;; order.
(defun host-as-inet-host (host)
  (etypecase host
    (integer (#_htonl host))
    (string (or (and (every #'(lambda (c) (position c ".0123456789")) host)
		     (_inet_aton host))
		(multiple-value-bind (addr err) (c_gethostbyname host)
		  (or addr
		      (socket-error "gethostbyname" err)))))))


(defun DOTTED-TO-IPADDR (name &key (errorp t))
  (let ((addr (_inet_aton name)))
    (if addr (#_ntohl addr)
      (and errorp (error "Invalid dotted address ~s" name)))))
    
(defun LOOKUP-HOSTNAME (host)
  (if (typep host 'integer)
    host
    (#_ntohl (host-as-inet-host host))))

(defun IPADDR-TO-DOTTED (addr &key values)
  (if values
      (values (ldb (byte 8 24) addr)
	      (ldb (byte 8 16) addr)
	      (ldb (byte 8  8) addr)
	      (ldb (byte 8  0) addr))
    (_inet_ntoa (#_htonl addr))))

(defun IPADDR-TO-HOSTNAME (ipaddr &key ignore-cache)
  (declare (ignore ignore-cache))
  (multiple-value-bind (name err) (c_gethostbyaddr (#_htonl ipaddr))
    (or name (socket-error "gethostbyaddr" err))))
  

(defun int-getsockopt (socket level optname)
  (rlet ((valptr :signed)
         (vallen :signed))
    (setf (pref vallen :signed) 4)
    (let* ((err (c_getsockopt socket level optname valptr vallen)))
      (if (and (eql 0 err)
               (eql 4 (pref vallen :signed)))
        (pref valptr :signed)
	(socket-error "getsockopt" err)))))

(defun int-setsockopt (socket level optname optval)
  (rlet ((valptr :signed))
    (setf (pref valptr :signed) optval)
    (socket-call "setsockopt"
      (c_setsockopt socket level optname valptr (record-length :signed)))))

#+darwinppc-target
(defun c_gethostbyaddr (addr)
  (rlet ((addrp :unsigned))
    (setf (pref addrp :unsigned) addr)
    (without-interrupts
     (let* ((hp (#_gethostbyaddr addrp (record-length :unsigned) #$AF_INET)))
       (declare (dynamic-extent hp))
       (if (not (%null-ptr-p hp))
	 (%get-cstring (pref hp :hostent.h_name))
	 (values nil -1))))))		; screw: find h_errno
      
#+linuxppc-target
(defun c_gethostbyaddr (addr)
  (rlet ((hostent :hostent)
	 (hp (* (struct :hostent)))
	 (herr :signed)
	 (addrp :unsigned))
    (setf (pref addrp :unsigned) addr)
    (do* ((buflen 1024 (+ buflen buflen))) ()
      (declare (fixnum buflen))
      (%stack-block ((buf buflen))
	(let* ((res (#_gethostbyaddr_r addrp (record-length :unsigned) #$AF_INET
				       hostent buf buflen hp herr)))
	  (declare (fixnum res))
	  (unless (eql res #$ERANGE)
	    (return
	     (if (and (eql res 0) (not (%null-ptr-p (%get-ptr hp))))
		 (%get-cstring (pref (%get-ptr hp) :hostent.h_name))
	       (values nil (- (pref herr :signed)))))))))))

#+darwinppc-target
(defun c_gethostbyname (name)
  (with-cstrs ((name (string name)))
    (without-interrupts
     (let* ((hp (#_gethostbyname  name)))
       (declare (dynamic-extent hp))
       (if (not (%null-ptr-p hp))
	 (%get-unsigned-long
	  (%get-ptr (pref hp :hostent.h_addr_list)))
	 (values nil -1))))))

#+linuxppc-target
(defun c_gethostbyname (name)
  (with-cstrs ((name (string name)))
    (rlet ((hostent :hostent)
           (hp (* (struct :hostent)))
           (herr :signed))
       (do* ((buflen 1024 (+ buflen buflen))) ()
         (declare (fixnum buflen))
         (%stack-block ((buf buflen))
           (let* ((res (#_gethostbyname_r name hostent buf buflen hp herr)))
             (declare (fixnum res))
             (unless (eql res #$ERANGE)
	       (return
		 (if (eql res 0)
		   (%get-unsigned-long
		    (%get-ptr (pref (%get-ptr hp) :hostent.h_addr_list)))
		   (values nil (- (pref herr :signed))))))))))))

(defun _getservbyname (name proto)
  (with-cstrs ((name (string name))
	       (proto (string proto)))
    (let* ((servent-ptr (%null-ptr)))
      (declare (dynamic-extent servent-ptr))
      (%setf-macptr servent-ptr (#_getservbyname name proto))
      (unless (%null-ptr-p servent-ptr)
	(pref servent-ptr :servent.s_port)))))

(defun _inet_ntoa (addr)
  (rlet ((addrp :unsigned))
    (setf (pref addrp :unsigned) addr)
    (with-macptrs ((p))
      (%setf-macptr p (#_inet_ntoa addrp))
      (unless (%null-ptr-p p) (%get-cstring p)))))

(defun _inet_aton (string)
  (with-cstrs ((name string))
    (rlet ((addr :in_addr))
      (let* ((result (#_inet_aton name addr)))
	(unless (eql result 0)
	  (pref addr :in_addr.s_addr))))))

(defun c_socket (domain type protocol)
  #+darwinppc-target
  (syscall os::socket domain type protocol)
  #+linuxppc-target
  (%stack-block ((params 12))
    (setf (%get-long params 0) domain
          (%get-long params 4) type
          (%get-long params 8) protocol)
    (syscall os::socketcall 1 params)))

(defun c_bind (sockfd sockaddr addrlen)
  #+darwinppc-target
  (syscall os::bind sockfd sockaddr addrlen)
  #+linuxppc-target
  (%stack-block ((params 12))
    (setf (%get-long params 0) sockfd
          (%get-ptr params 4) sockaddr
          (%get-long params 8) addrlen)
    (syscall os::socketcall 2 params)))

(defun c_connect (sockfd addr len)
  #+darwinppc-target
  (syscall os::connect sockfd addr len)
  #+linuxppc-target
  (%stack-block ((params 12))
     (setf (%get-long params 0) sockfd
           (%get-ptr params 4) addr
           (%get-long params 8) len)
     (syscall os::socketcall 3 params)))

(defun c_listen (sockfd backlog)
  #+darwinppc-target
  (syscall os::listen sockfd backlog)
  #+linuxppc-target
  (%stack-block ((params 8))
     (setf (%get-long params 0) sockfd
           (%get-long params 4) backlog)
     (syscall os::socketcall 4 params)))

(defun c_accept (sockfd addrp addrlenp)
  #+darwinppc-target
  (syscall os::accept sockfd addrp addrlenp)
  #+linuxppc-target
  (%stack-block ((params 12))
    (setf (%get-long params 0) sockfd
          (%get-ptr params 4) addrp
          (%get-ptr params 8) addrlenp)
    (syscall os::socketcall 5 params)))

(defun c_getsockname (sockfd addrp addrlenp)
  #+darwinppc-target
  (syscall os::getsockname sockfd addrp addrlenp)
  #+linuxppc-target
  (%stack-block ((params 12))
    (setf (%get-long params 0) sockfd
          (%get-ptr params 4) addrp
          (%get-ptr params 8) addrlenp)
    (syscall os::socketcall 6 params)))

(defun c_getpeername (sockfd addrp addrlenp)
  #+darwinppc-target
  (syscall os::getpeername sockfd addrp addrlenp)
  #+linuxppc-target
  (%stack-block ((params 12))
    (setf (%get-long params 0) sockfd
          (%get-ptr params 4) addrp
          (%get-ptr params 8) addrlenp)
    (syscall os::socketcall 7 params)))

(defun c_socketpair (domain type protocol socketsptr)
  #+darwinppc-target
  (syscall os::socketpair domain type protocol socketsptr)
  #+linuxppc-target
  (%stack-block ((params 16))
    (setf (%get-long params 0) domain
          (%get-long params 4) type
          (%get-long params 8) protocol
          (%get-ptr params 12) socketsptr)
    (syscall os::socketcall 8 params)))

(defun c_send (sockfd msgptr len flags)
  #+darwinppc-target
  (syscall os::sendto sockfd msgptr len flags (%null-ptr) 0)
  #+linuxppc-target
  (%stack-block ((params 16))
    (setf (%get-long params 0) sockfd
	  (%get-ptr params  4) msgptr
	  (%get-long params 8) len
	  (%get-long params 12) flags)
    (syscall os::socketcall 9 params)))

(defun c_recv (sockfd bufptr len flags)
  #+darwinppc-target
  (syscall os::recvfrom sockfd bufptr len flags (%null-ptr) (%null-ptr))
  #+linuxppc-target
  (%stack-block ((params 16))
    (setf (%get-long params 0) sockfd
	  (%get-ptr params  4) bufptr
	  (%get-long params 8) len
	  (%get-long params 12) flags)
    (syscall os::socketcall 10 params)))

(defun c_sendto (sockfd msgptr len flags addrp addrlen)
  #+darwinppc-target
  (syscall os::sendto sockfd msgptr len flags addrp addrlen)
  #+linuxppc-target
  (%stack-block ((params 24))
    (setf (%get-long params 0) sockfd
	  (%get-ptr params  4) msgptr
	  (%get-long params 8) len
	  (%get-long params 12) flags
	  (%get-ptr params  16) addrp
	  (%get-long params 20) addrlen)
    (syscall os::socketcall 11 params)))

(defun c_recvfrom (sockfd bufptr len flags addrp addrlenp)
  #+darwinppc-target
  (syscall os::recvfrom sockfd bufptr len flags addrp addrlenp)
  #+linuxppc-target
  (%stack-block ((params 24))
    (setf (%get-long params 0) sockfd
	  (%get-ptr params  4) bufptr
	  (%get-long params 8) len
	  (%get-long params 12) flags
	  (%get-ptr params  16) addrp
	  (%get-ptr params  20) addrlenp)
    (syscall os::socketcall 12 params)))

(defun c_shutdown (sockfd how)
  #+darwinppc-target
  (syscall os::shutdown sockfd how)
  #+linuxppc-target
  (%stack-block ((params 8))
    (setf (%get-long params 0) sockfd
	  (%get-long params 4) how)
    (syscall os::socketcall 13 params)))

(defun c_setsockopt (sockfd level optname optvalp optlen)
  #+darwinppc-target
  (syscall os::setsockopt sockfd level optname optvalp optlen)
  #+linuxppc-target
  (%stack-block ((params 20))
    (setf (%get-long params 0) sockfd
          (%get-long params 4) level
          (%get-long params 8) optname
          (%get-ptr params 12) optvalp
          (%get-long params 16) optlen)
    (syscall os::socketcall 14 params)))

(defun c_getsockopt (sockfd level optname optvalp optlenp)
  #+darwinppc-target
  (syscall os::getsockopt sockfd level optname optvalp optlenp)
  #+linuxppc-target
  (%stack-block ((params 20))
    (setf (%get-long params 0) sockfd
          (%get-long params 4) level
          (%get-long params 8) optname
          (%get-ptr params 12) optvalp
          (%get-ptr params 16) optlenp)
    (syscall os::socketcall 15 params)))

(defun c_sendmsg (sockfd msghdrp flags)
  #+darwinppc-target
  (syscall os::sendmsg sockfd msghdrp flags)
  #+linuxppc-target
  (%stack-block ((params 12))
    (setf (%get-long params 0) sockfd
	  (%get-ptr params 4) msghdrp
	  (%get-long params 8) flags)
    (syscall os::socketcall 16 params)))

(defun c_recvmsg (sockfd msghdrp flags)
  #+darwinppc-target
  (syscall os::recvmsg sockfd msghdrp flags)
  #+linuxppc-target
  (%stack-block ((params 12))
    (setf (%get-long params 0) sockfd
	  (%get-ptr params 4) msghdrp
	  (%get-long params 8) flags)
    (syscall os::socketcall 17 params)))
