<details>
  <summary><strong>What does this do?</strong></summary>

<br>

This libary generates the `source` string to be passed to `new RegExp()` for matching a range of numbers. 

**Example**

```js
const toRegexRange = require('{%= name %}');
const regex = new RegExp(toRegexRange('15', '95'));
```

A string is returned so that you can do whatever you need with it before passing it to `new RegExp()` (like adding `^` or `$` boundaries, defining flags, or combining it another string). 

<br>

</details>

<details>
  <summary><strong>Why use this library?</strong></summary>

<br>

### Convenience

Creating regular expressions for matching numbers gets deceptively complicated pretty fast. 

For example, let's say you need a validation regex for matching part of a user-id, postal code, social security number, tax id, etc:

- regex for matching `1` => `/1/` (easy enough)
- regex for matching `1` through `5` => `/[1-5]/` (not bad...)
- regex for matching `1` or `5` => `/(1|5)/` (still easy...)
- regex for matching `1` through `50` => `/([1-9]|[1-4][0-9]|50)/` (uh-oh...)
- regex for matching `1` through `55` => `/([1-9]|[1-4][0-9]|5[0-5])/` (no prob, I can do this...)
- regex for matching `1` through `555` => `/([1-9]|[1-9][0-9]|[1-4][0-9]{2}|5[0-4][0-9]|55[0-5])/` (maybe not...)
- regex for matching `0001` through `5555` => `/(0{3}[1-9]|0{2}[1-9][0-9]|0[1-9][0-9]{2}|[1-4][0-9]{3}|5[0-4][0-9]{2}|55[0-4][0-9]|555[0-5])/` (okay, I get the point!)

The numbers are contrived, but they're also really basic. In the real world you might need to generate a regex on-the-fly for validation. 

**Learn more**

If you're interested in learning more about [character classes](http://www.regular-expressions.info/charclass.html) and other regex features, I personally have always found [regular-expressions.info](http://www.regular-expressions.info/charclass.html) to be pretty useful.


### Heavily tested

As of {%= date() %}, this library runs [>1m test assertions](./test/test.js) against generated regex-ranges to provide brute-force verification that results are correct. 

Tests run in ~280ms on my MacBook Pro, 2.5 GHz Intel Core i7.

### Optimized

Generated regular expressions are optimized:

- duplicate sequences and character classes are reduced using quantifiers
- smart enough to use `?` conditionals when number(s) or range(s) can be positive or negative
- uses fragment caching to avoid processing the same exact string more than once 

<br>

</details>

## Usage

Add this library to your javascript application with the following line of code

```js
const toRegexRange = require('{%= name %}');
```

The main export is a function that takes two integers: the `min` value and `max` value (formatted as strings or numbers).

```js
const source = toRegexRange('15', '95');
//=> 1[5-9]|[2-8][0-9]|9[0-5]

const regex = new RegExp(`^${source}$`);
console.log(regex.test('14')); //=> false
console.log(regex.test('50')); //=> true
console.log(regex.test('94')); //=> true
console.log(regex.test('96')); //=> false
```

## Options

### options.capture

**Type**: `boolean`

**Deafault**: `undefined`

Wrap the returned value in parentheses when there is more than one regex condition. Useful when you're dynamically generating ranges.

```js
console.log(toRegexRange('-10', '10'));
//=> -[1-9]|-?10|[0-9]

console.log(toRegexRange('-10', '10', { capture: true }));
//=> (-[1-9]|-?10|[0-9])
```

### options.shorthand

**Type**: `boolean`

**Deafault**: `undefined`

Use the regex shorthand for `[0-9]`:

```js
console.log(toRegexRange('0', '999999'));
//=> [0-9]|[1-9][0-9]{1,5}

console.log(toRegexRange('0', '999999', { shorthand: true }));
//=> \d|[1-9]\d{1,5}
```

### options.relaxZeros

**Type**: `boolean`

**Default**: `true`

This option relaxes matching for leading zeros when when ranges are zero-padded.  

```js
const source = toRegexRange('-0010', '0010');
const regex = new RegExp(`^${source}$`);
console.log(regex.test('-10')); //=> true
console.log(regex.test('-010')); //=> true
console.log(regex.test('-0010')); //=> true
console.log(regex.test('10')); //=> true
console.log(regex.test('010')); //=> true
console.log(regex.test('0010')); //=> true
```

When `relaxZeros` is false, matching is strict:

```js
const source = toRegexRange('-0010', '0010', { relaxZeros: false });
const regex = new RegExp(`^${source}$`);
console.log(regex.test('-10')); //=> false
console.log(regex.test('-010')); //=> false
console.log(regex.test('-0010')); //=> true
console.log(regex.test('10')); //=> false
console.log(regex.test('010')); //=> false
console.log(regex.test('0010')); //=> true
```

## Examples

{%= examples() %}

## Heads up!

**Order of arguments**

When the `min` is larger than the `max`, values will be flipped to create a valid range:

```js
toRegexRange('51', '29');
```

Is effectively flipped to:

```js
toRegexRange('29', '51');
//=> 29|[3-4][0-9]|5[0-1]
```

**Steps / increments**

This library does not support steps (increments). A pr to add support would be welcome.


## History

### v5.0.0 - 2019-04-07

Optimizations. Updated code to use newer ES features. 

### v2.0.0 - 2017-04-21

**New features**

Adds support for zero-padding!

### v1.0.0

**Optimizations**

Repeating ranges are now grouped using quantifiers. rocessing time is roughly the same, but the generated regex is much smaller, which should result in faster matching.

## Attribution

Inspired by the python library [range-regex](https://github.com/dimka665/range-regex).