/********************************************************************
 *                                                                  *
 * File: certutils.c                                                *
 *                                                                  *
 ********************************************************************
 *                                                                  *
 * Authors:                                                         *
 *   David Corcoran       <corcoran@linuxnet.com>                   *
 *   Eirik A. Herskedal   <ehersked@cs.purdue.edu>                  *
 *   Peter Huang          <peter_huang@hp.com>                      *
 ********************************************************************
 *                                                                  *
 * Utility functions for reading certificates and extracting data   *
 * from the certificates.                                           *
 * It also includes a random number function for                    * 
 * challenge/response.                                              *
 *                                                                  *
 ********************************************************************/

#include <syslog.h>
#include <pem.h>

#include "certutils.h"
#include "preferences.h"

int getRandom(char *buf, int len)
{
  int fh, i, rv;

  /* open random device */
  fh = open(RANDOM_DEVICE, O_RDONLY);
  if (fh == -1) return -1;

  /* read len bytes */
  for (i = 0; i < len; ) {
    rv = read(fh, &buf[i], len-i);
    if (rv == -1) return -1;
    i += rv;
  }

  /* the most significant bit must be zero */
  buf[0] &= 0x7F;

  /* close device */
  close(fh);
  return 0;
}

/* Method to read a certificate from a file */
int getFileCert(const char *file, X509 **cert) {
  FILE *fp;
  X509 *mycert = NULL;

  // Open file
  fp = fopen(file, "rb");
  if (fp == NULL)
    return -1;
  
  // Get the cert
  d2i_X509_fp(fp, &mycert);

  fclose(fp);
  
  if (mycert == NULL)
    return -1;

  *cert = mycert;

  return 0;
}

/* add the code to read PEM certificate */
int getFileCertPEM(const char *file, X509 **cert) {
  FILE *fp;
  X509 *mycert = NULL;
 
  // Open file
  fp = fopen(file, "rb");
  if (fp == NULL)
    return -1;
 
  // Get the cert
  PEM_read_X509(fp,&mycert,NULL,NULL);

  fclose(fp);

  if (mycert == NULL)
    return -1;
 
  *cert = mycert;
 
  return 0;
}

/* Method to read a certificate from a a smartcard */
int getCardCert(MSCTokenConnection pConnection, X509 **cert) {
  int rv;
  MSCULong32 objSize;
  MSCPUChar8 buf;
  BIO *mem = BIO_new(BIO_s_mem());
  X509 *mycert = NULL;
  char certID[3];

  snprintf(certID, 3, "C%d", pr.certnumber);
  certID[2] = '\0';

  // Read certificate object from card
  rv = MSCReadAllocateObject(&pConnection, certID, &buf, &objSize, NULL, NULL);
  
  if (rv != MSC_SUCCESS)
    return -1;
  
  BIO_write(mem, buf, objSize);
  
  // Get the cert
  d2i_X509_bio(mem, &mycert);

  if (mycert == NULL)
    return -1;

  *cert = mycert;
  
  BIO_vfree(mem);

  return 0;
}

/* Check validity of a cert */
int checkCert(X509* cert) {
  int before, after;

  // Check the expiration dates
  before = X509_cmp_current_time(cert->cert_info->validity->notBefore);
  after = X509_cmp_current_time(cert->cert_info->validity->notAfter);

  if (before > 0 || after < 0)
    return -1;

  return 0;
}

/* Extract certificate from smartcard */
int getPublicKey(X509 *cert, EVP_PKEY **pub_key) {
  // Get the public key from the certificate
  *pub_key = (EVP_PKEY*) X509_get_pubkey(cert);
  if (*pub_key == NULL)
    return -1;
  
  return 0;
}













