#define _LARGEFILE64_SOURCE     /* required for GLIBC to enable stat64 and friends */
#include <sys/types.h>
#include <regex.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>

#include "mt.h"
#include "error.h"
#include "mem.h"
#include "term.h"
#include "utils.h"
#include "globals.h"

void add_color_scheme(int **schemes, int *n_schemes, int cur_scheme)
{
	*schemes = (int *)myrealloc(*schemes, sizeof(int) * (*n_schemes + 1), "add_color_scheme: list of colorschemes");
	(*schemes)[*n_schemes] = cur_scheme;
	(*n_schemes)++;
}

int gen_color(char *start, char *end)
{
	char *loop;
	int chk = 0;

	for(loop=start; loop<end; loop++)
	{
		chk ^= *loop;
	}

	return abs(chk) % cp.n_def;
}

myattr_t gen_syslog_progname_color(char *string)
{
	myattr_t cdev = { -1, -1 };

	if (use_colors)
	{
		if (strlen(string) >= 16)
		{
			string = strchr(&string[16], ' ');
		}

		while(string && isspace(*string)) string++;

		if (string)
		{
			char *end1 = strchr(string, '[');
			char *end2 = strchr(string, ' ');
			char *end3 = strchr(string, ':');
			char *end = NULL;

			end = end1;
			if ((end2 && end2 < end) || (end == NULL))
				end = end2;
			if ((end3 && end3 < end) || (end == NULL))
				end = end3;

			if (end)
				cdev.colorpair_index = gen_color(string, end);
			else
				cdev.colorpair_index = gen_color(string, &string[strlen(string)]);
		}
	}

	return cdev;
}

myattr_t gen_color_from_field(char *string, char *field_del, int field_nr)
{
	myattr_t cdev = { -1, -1 };

	if (use_colors)
	{
		int nlen = strlen(field_del), loop;
		char *dummy = NULL;

		for(loop=0; loop<field_nr; loop++)
		{
			string = strstr(string, field_del);
			while(string)
			{
				if (strncmp(string, field_del, nlen) == 0)
				{
					string += nlen;
				}
				else
				{
					break;
				}
			}

			if (!string)
				break;
		}

		if (string)
			dummy = strstr(string, field_del);

		if (dummy != NULL && string != NULL)
			cdev.colorpair_index = gen_color(string, dummy);
		else if (string)
			cdev.colorpair_index = gen_color(string, &string[strlen(string)]);
		else
			cdev.colorpair_index = 0;
	}

	return cdev;
}

void get_colors_from_colorscheme(char *string, int *color_schemes, int n_color_schemes, color_offset_in_line **cmatches, int *n_cmatches)
{
	int cs_index, loop;
	regmatch_t colormatches[MAX_N_RE_MATCHES];
	int cur_n_cmatches = 0;
	int len = strlen(string);

	*n_cmatches = 0;
	*cmatches = NULL;

	for(loop=0; loop<n_color_schemes; loop++)
	{
		for(cs_index=0; cs_index<cschemes[color_schemes[loop]].n; cs_index++)
		{
			csreflag_t flags = cschemes[color_schemes[loop]].flags[cs_index];
			int offset = 0;
			int re_start, re_end;

			if (flags)
			{
				re_start = 1;
				re_end = MAX_N_RE_MATCHES;
			}
			else
			{
				re_start = 0;
				re_end = 1;
			}

			do
			{
				/* FIXME: what to do with regexp errors? */
				if (regexec(&cschemes[color_schemes[loop]].regex[cs_index], &string[offset], MAX_N_RE_MATCHES, colormatches, offset?REG_NOTBOL:0) == 0)
				{
					int match_index;
					int cur_offset = offset;

					for(match_index=re_start; match_index<re_end; match_index++)
					{
						int this_start_offset, this_end_offset;

						if (colormatches[match_index].rm_so == -1)
							break;

						this_start_offset = colormatches[match_index].rm_so + cur_offset;
						this_end_offset   = colormatches[match_index].rm_eo + cur_offset;

						offset = max(offset + 1, this_end_offset);

						if (flags == CSREFLAG_CMP_VAL_LESS || flags == CSREFLAG_CMP_VAL_BIGGER || flags == CSREFLAG_CMP_VAL_EQUAL)
						{
							int val_size = this_end_offset - this_start_offset;
							char *valstr = (char *)mymalloc(val_size + 1, "get_colors_from_colorscheme: compare value");
							double value = 0.0;
							char value_match = 0;

							if (val_size > 0)
								memcpy(valstr, &string[this_start_offset], val_size);
							valstr[val_size] = 0x00;
							value = atof(valstr);
							myfree(valstr);

							if (flags == CSREFLAG_CMP_VAL_LESS && value < cschemes[color_schemes[loop]].cmp_value[cs_index])
								value_match = 1;
							if (flags == CSREFLAG_CMP_VAL_BIGGER && value > cschemes[color_schemes[loop]].cmp_value[cs_index])
								value_match = 1;
							if (flags == CSREFLAG_CMP_VAL_EQUAL && value == cschemes[color_schemes[loop]].cmp_value[cs_index])
								value_match = 1;

							if (!value_match)
								continue;
						}

						if (cur_n_cmatches == *n_cmatches)
						{
							cur_n_cmatches = (cur_n_cmatches + 8) * 2;

							*cmatches = (color_offset_in_line *)myrealloc(*cmatches, cur_n_cmatches * sizeof(color_offset_in_line), "get_colors_from_colorscheme: cmatches");
						}

						(*cmatches)[*n_cmatches].start = this_start_offset;
						(*cmatches)[*n_cmatches].end   = this_end_offset;
						(*cmatches)[*n_cmatches].attrs = cschemes[color_schemes[loop]].attrs[cs_index];

						(*n_cmatches)++;
					} /* iterate all substringmatches or just the first which is the globl one */
				} /* did the reg exp match? */
				else
				{
					break;
				}
			}
			while(offset < len); /* do the whole line */
		} /* go through all lines of the current colorscheme */
	} /* go through all colorschemes for this (sub-)window */
}

myattr_t choose_color(char *string, proginfo *cur, color_offset_in_line **cmatches, int *n_cmatches, char **new_string)
{
	myattr_t cdev = { -1, -1 };

	*new_string = NULL;

	if (cur -> term_emul != TERM_IGNORE)
	{
		LOG("terminal: %d\n", cur -> term_emul);
		*new_string = emulate_terminal(string, cmatches, n_cmatches);
		return find_attr(COLOR_WHITE, -1, -1);
	}

	/* find color */
	switch(cur -> colorize)
	{
		case 'i':
			cdev = cur -> attributes;
			break;

		case 'a':
			/* alternating colors */
			if (string[0] != 0x00)
			{
				if (cur -> alt_col)
					cdev = cur -> alt_col_cdev1;
				else
					cdev = cur -> alt_col_cdev2;

				cur -> alt_col = !cur -> alt_col;
			}
			break;

		case 's':
			/* use the program name for the color */
			cdev = gen_syslog_progname_color(string);
			break;

		case 'f':
			/* user selected field for coloring */
			cdev = gen_color_from_field(string, cur -> field_del, cur -> field_nr);
			break;

		case 'm':
			/* complete string */
			if (use_colors)
				cdev.colorpair_index = gen_color(string, &string[strlen(string)]);
			break;

		case 'S':
			/* color scheme */
			get_colors_from_colorscheme(string, cur -> color_schemes, cur -> n_color_schemes, cmatches, n_cmatches);
			break;
	}

	if (cdev.colorpair_index == -1)
		cdev.colorpair_index = 0;
	if (cdev.attrs == -1)
		cdev.attrs = A_NORMAL;

	if (cur -> syslog_noreverse)
		cdev.colorpair_index %= DEFAULT_COLORPAIRS;

	return cdev;
}

int find_colorscheme(char *name)
{
	int loop;

	for(loop=0; loop<n_cschemes; loop++)
	{
		if (strcasecmp(cschemes[loop].name, name) == 0)
			return loop;
	}

	return -1;
}

void init_colors(void)
{
	/* add standard colors */
	cp.size = min(256, COLOR_PAIRS);
	if (cp.size)
	{
		cp.fg_color = (int *)mymalloc(cp.size * sizeof(int), "main: fg colors");
		cp.bg_color = (int *)mymalloc(cp.size * sizeof(int), "main: bg colors");
	}

	(void)find_or_init_colorpair(-1, -1, 1);		/* colorpair 0 is always default, cannot be changed */
	(void)find_or_init_colorpair(COLOR_RED, -1, 1);
	(void)find_or_init_colorpair(COLOR_GREEN, -1, 1);
	(void)find_or_init_colorpair(COLOR_YELLOW, -1, 1);
	(void)find_or_init_colorpair(COLOR_BLUE, -1, 1);
	(void)find_or_init_colorpair(COLOR_MAGENTA, -1, 1);
	(void)find_or_init_colorpair(COLOR_CYAN, -1, 1);
	(void)find_or_init_colorpair(COLOR_WHITE, -1, 1);
	/*	(void)find_or_init_colorpair(COLOR_BLACK, -1, 1); */

	if (use_colors && cp.n_def != DEFAULT_COLORPAIRS)
		error_exit("unexpected number of colors: %d (%d)\n", cp.n_def, DEFAULT_COLORPAIRS);
}
