/****************************************************************************

  module      : SAPDB_ToStringClass.cpp

  -------------------------------------------------------------------------

  responsible : FrankH

  special area: Common
  description:  Value to String conversion i.e. for argument passing


  last changed: 2000-06-15  17:30
  see also    : example.html ...

  -------------------------------------------------------------------------

  copyright:    (c) 1999-2004 SAP AG




    ========== licence begin  GPL
    Copyright (c) 1999-2005 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end




*****************************************************************************/



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/

#include <stdio.h>
#include <string.h>
#include "SAPDBCommon/SAPDB_ToString.hpp"
#include "SAPDBCommon/MemoryManagement/SAPDBMem_IRawAllocator.hpp"
#include "RunTime/RTE_IInterface.hpp"
#include "hsp77.h" /* sp77sprintf */

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/



/*===========================================================================*
 *  LOCAL CLASSES, STRUCTURES, TYPES, UNIONS ...                             *
 *===========================================================================*/



/*===========================================================================*
 *  STATIC/INLINE FUNCTIONS (PROTOTYPES)                                     *
 *===========================================================================*/



/*===========================================================================*
 *  METHODS                                                                  *
 *===========================================================================*/

SAPDB_ToStringClass::~SAPDB_ToStringClass()
{
    if ( 0 != m_Buf )
    {
        RTE_IInterface::Instance().Allocator().Deallocate(m_Buf);
    }
}

// ----------------------- STRING -----------------------

void SAPDB_ToStringClass::FillStringFormatBuffer( const char       *String ,
                                             const SAPDB_UInt2 Width,
                                             const SAPDB_UInt2 Precision,
                                             const int         Flags)
{
  int StringLen;
  
  if (Precision > 0) 
  {
    const char * nullPos = (const char *) memchr (String, '\0', Precision);
    if (nullPos == NULL) 
    {
      StringLen = Precision;
    }
    else 
    {
      StringLen = (int) (nullPos - String) ;
    }
  }
  else 
  {
      StringLen = (int)strlen ( String ) ;
  }

  if ( Precision == 0 
    && Width <= StringLen 
    && StringLen <= MAX_CHARACTER_ALLOWED_IN_PRESENTATION )
  {
      strcpy(&m_PresentationBuffer[0], String);
      m_OutputString = &m_PresentationBuffer[0];
      m_Buf          = 0;
  }
  else
  {
    int FormatBufferLen = 0 ;

    m_FormatBuffer[FormatBufferLen] = '%' ;
    FormatBufferLen++ ;

    if (  Flags & left )
    {
      m_FormatBuffer[FormatBufferLen] = '-' ;
      FormatBufferLen++ ;
    }

    if ( Precision > 0 )
      FormatBufferLen += sp77sprintf( m_FormatBuffer+FormatBufferLen, 
                                      MAX_CHARACTER_ALLOWED_IN_FORMAT-FormatBufferLen,
                                      "%d.%ds", 
                                      Width, 
                                      Precision ) ;
    else
      FormatBufferLen += sp77sprintf( m_FormatBuffer+FormatBufferLen, 
                                      MAX_CHARACTER_ALLOWED_IN_FORMAT-FormatBufferLen, 
                                      "%ds", 
                                      Width ) ;

    int OutputLen = Precision > Width ? Precision : Width ;

    if ( OutputLen < StringLen )
      OutputLen = StringLen ;

    if ( OutputLen > MAX_CHARACTER_ALLOWED_IN_PRESENTATION )
    {
        m_Buf = (char *)RTE_IInterface::Instance().Allocator().Allocate(OutputLen + 1);

        if ( 0 != m_Buf )
        {
            sp77sprintf(m_Buf, OutputLen + 1, m_FormatBuffer, String ) ;
            m_OutputString = m_Buf ;
        }
        else
        {
            m_OutputString = "NO MORE MEMORY" ;
            m_Buf = 0;
        }
    }
    else
    {
        m_PresentationBuffer[MAX_CHARACTER_ALLOWED_IN_PRESENTATION] = 0;
        sp77sprintf(m_PresentationBuffer, 
            MAX_CHARACTER_ALLOWED_IN_PRESENTATION, 
            m_FormatBuffer, 
            String );
        m_OutputString = m_PresentationBuffer ;
        m_Buf = 0;
    }
  }
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass(const char *String,
                               const SAPDB_UInt2 Width,
                               const fmtflags Flag1,
                               const fmtflags Flag2)
{
  FillStringFormatBuffer( String, Width, 0, Flag1 | Flag2 ) ;
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass(const char *String,
                               const SAPDB_UInt2 Width,
                               const SAPDB_UInt2 Precision,
                               const fmtflags Flag1,
                               const fmtflags Flag2)
{
  FillStringFormatBuffer( String, Width, Precision, Flag1 | Flag2 ) ;
}

// ----------------------- BOOL / INT / UINT  -----------------------

void SAPDB_ToStringClass::FillBoolAlphaFormatBuffer(const SAPDB_Bool isTrue, const SAPDB_UInt2 width, const int combinedFlags)
{
    if ( combinedFlags & uppercase )
    {
        FillStringFormatBuffer(isTrue ? "TRUE" : "FALSE", width, width, combinedFlags);
    }
    else
    {
        FillStringFormatBuffer(isTrue ? "true" : "false", width, width, combinedFlags);
    }
}

//-------------------

void SAPDB_ToStringClass::FillNumberFormatBuffer( const SAPDB_UInt2 Width,
                                             const int         Flags,
                                             const bool        SignedValue,
                                             const bool        QuadWord)
{
    SAPDB_UInt4 usedWidth;

    m_Buf          = NULL ;

    int FormatBufferLen = 0 ;

    // silently, but reasonable.... PTS 1110611
    usedWidth = ( Width > MAX_CHARACTER_ALLOWED_IN_PRESENTATION ? 
                    MAX_CHARACTER_ALLOWED_IN_PRESENTATION : Width );

    m_FormatBuffer[FormatBufferLen] = '%' ;
    FormatBufferLen++ ;

    if ( Flags & showbase ) 
    {
        m_FormatBuffer[FormatBufferLen] = '#' ;
        FormatBufferLen++ ;
    }

    if ( Flags & showpos )
    {
        m_FormatBuffer[FormatBufferLen] = '+' ;
        FormatBufferLen++ ;
    }

    if ( Flags & showpoint )
	{
      m_FormatBuffer[FormatBufferLen] = '0';
      FormatBufferLen++ ;
	}

    if ( usedWidth )
      if ( Flags & left )
        FormatBufferLen += sp77sprintf( m_FormatBuffer+FormatBufferLen, 
                                        MAX_CHARACTER_ALLOWED_IN_FORMAT-FormatBufferLen,
                                        "-%d",
                                        usedWidth );
      else
        FormatBufferLen += sp77sprintf( m_FormatBuffer+FormatBufferLen,
                                        MAX_CHARACTER_ALLOWED_IN_FORMAT-FormatBufferLen,
                                        "%d",
                                        usedWidth );

    if ( Flags & ptr )
    {
        m_FormatBuffer[FormatBufferLen] = 'p';
    }
    else 
    {
        if (QuadWord) {
            m_FormatBuffer[FormatBufferLen] = 'q';
            FormatBufferLen++ ;
        }
        if ( Flags & oct )
          m_FormatBuffer[FormatBufferLen] = 'o' ;
        else if ( Flags & hex )
          m_FormatBuffer[FormatBufferLen] = Flags & uppercase ? 'X' : 'x';
        else if ( SignedValue )
          m_FormatBuffer[FormatBufferLen] = 'd' ;
        else
          m_FormatBuffer[FormatBufferLen] = 'u' ;
    }

    FormatBufferLen++ ;
    m_FormatBuffer[FormatBufferLen] = '\0' ;
}

//---------------------

void SAPDB_ToStringClass::FillSignedNumberFormatBuffer( const SAPDB_Int8  Number,
                                                   const SAPDB_UInt2 Width,
                                                   const int         Flags,
                                                   const bool        QuadWord )
{
    if ( Flags & boolalpha )
    {
        FillBoolAlphaFormatBuffer(Number != 0, Width, Flags);
    }
    else
    {
        FillNumberFormatBuffer(Width, Flags, true, QuadWord);

        m_PresentationBuffer[MAX_CHARACTER_ALLOWED_IN_PRESENTATION] = 0;
        if ( QuadWord )
        {
            sp77sprintf(m_PresentationBuffer, 
                MAX_CHARACTER_ALLOWED_IN_PRESENTATION, 
                m_FormatBuffer, 
                Number );
        }
        else
        {
            sp77sprintf(m_PresentationBuffer, 
                MAX_CHARACTER_ALLOWED_IN_PRESENTATION, 
                m_FormatBuffer, 
                (SAPDB_Int4)Number ) ;
        }
        m_OutputString = m_PresentationBuffer ;
        m_Buf = 0;
    }
}

//---------------------

void SAPDB_ToStringClass::FillUnsignedNumberFormatBuffer( const SAPDB_UInt8 Number,
                                                     const SAPDB_UInt2 Width,
                                                     const int         Flags,
                                                     const bool        QuadWord )
{
    if ( Flags & boolalpha )
    {
        FillBoolAlphaFormatBuffer(Number != 0, Width, Flags);
    }
    else
    {
        FillNumberFormatBuffer(Width, Flags, false, QuadWord);

        m_PresentationBuffer[MAX_CHARACTER_ALLOWED_IN_PRESENTATION] = 0;
        if ( QuadWord )
        {
            sp77sprintf(m_PresentationBuffer, 
                MAX_CHARACTER_ALLOWED_IN_PRESENTATION, 
                m_FormatBuffer, 
                Number );
        }
        else
        {
            sp77sprintf(m_PresentationBuffer, 
                MAX_CHARACTER_ALLOWED_IN_PRESENTATION, 
                m_FormatBuffer, 
                (SAPDB_UInt4)Number ) ;
        }
        m_OutputString = m_PresentationBuffer ;
        m_Buf = 0;
    }
}

// ------------------------ BOOL ---------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Bool Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillSignedNumberFormatBuffer(Number ? (SAPDB_Int8)true : (SAPDB_Int8)false, 0, Flag1 | Flag2 | Flag3 | Flag4) ;
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Bool  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillSignedNumberFormatBuffer(Number ? (SAPDB_Int8)true : (SAPDB_Int8)false, Width, Flag1 | Flag2 | Flag3 | Flag4) ;
}

// ----------------------- INT1  -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Int1  Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    int        combinedFlags = Flag1 | Flag2 | Flag3 | Flag4;
    SAPDB_Int8 outputNumber;

    if ( (combinedFlags & (hex|oct)) != 0 )
    {
        outputNumber = (SAPDB_Int8)((SAPDB_UInt1)Number);
    }
    else
    {
        outputNumber = (SAPDB_Int8)Number;
    }

    FillSignedNumberFormatBuffer( outputNumber, 0, combinedFlags );
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Int1  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    int        combinedFlags = Flag1 | Flag2 | Flag3 | Flag4;
    SAPDB_Int8 outputNumber;

    if ( (combinedFlags & (hex|oct)) != 0 )
    {
        outputNumber = (SAPDB_Int8)((SAPDB_UInt1)Number);
    }
    else
    {
        outputNumber = (SAPDB_Int8)Number;
    }

    FillSignedNumberFormatBuffer( outputNumber, Width, combinedFlags );
}

// ----------------------- UINT1  -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_UInt1  Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt8)Number, 0, Flag1 | Flag2 | Flag3 | Flag4);
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_UInt1  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt8)Number, Width, Flag1 | Flag2 | Flag3 | Flag4);
}

// ----------------------- INT2  -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Int2  Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    int        combinedFlags = Flag1 | Flag2 | Flag3 | Flag4;
    SAPDB_Int8 outputNumber;

    if ( (combinedFlags & (hex|oct)) != 0 )
    {
        outputNumber = (SAPDB_Int8)((SAPDB_UInt2)Number);
    }
    else
    {
        outputNumber = (SAPDB_Int8)Number;
    }

    FillSignedNumberFormatBuffer( outputNumber, 0, combinedFlags );
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Int2  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    int        combinedFlags = Flag1 | Flag2 | Flag3 | Flag4;
    SAPDB_Int8 outputNumber;

    if ( (combinedFlags & (hex|oct)) != 0 )
    {
        outputNumber = (SAPDB_Int8)((SAPDB_UInt2)Number);
    }
    else
    {
        outputNumber = (SAPDB_Int8)Number;
    }

    FillSignedNumberFormatBuffer( outputNumber, Width, combinedFlags );
}

// ----------------------- UINT2  -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_UInt2  Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt8)Number, 0, Flag1 | Flag2 | Flag3 | Flag4);
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_UInt2  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt8)Number, Width, Flag1 | Flag2 | Flag3 | Flag4);
}

// ----------------------- INT4  -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Int4  Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    int        combinedFlags = Flag1 | Flag2 | Flag3 | Flag4;
    SAPDB_Int8 outputNumber;

    if ( (combinedFlags & (hex|oct)) != 0 )
    {
        outputNumber = (SAPDB_Int8)((SAPDB_UInt4)Number);
    }
    else
    {
        outputNumber = (SAPDB_Int8)Number;
    }

    FillSignedNumberFormatBuffer( outputNumber, 0, combinedFlags );
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Int4  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    int        combinedFlags = Flag1 | Flag2 | Flag3 | Flag4;
    SAPDB_Int8 outputNumber;

    if ( (combinedFlags & (hex|oct)) != 0 )
    {
        outputNumber = (SAPDB_Int8)((SAPDB_UInt4)Number);
    }
    else
    {
        outputNumber = (SAPDB_Int8)Number;
    }

    FillSignedNumberFormatBuffer( outputNumber, Width, combinedFlags );
}

// ----------------------- UINT4  -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_UInt4  Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt8)Number, 0, Flag1 | Flag2 | Flag3 | Flag4);
}

// -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_UInt4  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt8)Number, Width, Flag1 | Flag2 | Flag3 | Flag4);
}

#if !defined (BIT64) && !defined (_WIN64)
// ----------------------- INT8  -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Int8  Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillSignedNumberFormatBuffer( Number, 0, Flag1 | Flag2 | Flag3 | Flag4, true);
}

// -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Int8  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillSignedNumberFormatBuffer( Number, Width, Flag1 | Flag2 | Flag3 | Flag4, true);
}

// ----------------------- UINT8  -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_UInt8  Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillUnsignedNumberFormatBuffer( Number, 0, Flag1 | Flag2 | Flag3 | Flag4, true);
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_UInt8  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillUnsignedNumberFormatBuffer( Number, Width, Flag1 | Flag2 | Flag3 | Flag4, true);
}
#endif

#if defined (_WIN32) && defined (_WIN64)
// ----------------------- long  -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const long  Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillSignedNumberFormatBuffer( (SAPDB_Int4)Number, 0, Flag1 | Flag2 | Flag3 | Flag4, false);
}

// -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const long  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillSignedNumberFormatBuffer( (SAPDB_Int4)Number, Width, Flag1 | Flag2 | Flag3 | Flag4, false);
}

// ----------------------- unsigned long  ---------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const unsigned long  Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt4)Number, 0, Flag1 | Flag2 | Flag3 | Flag4, false);
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const unsigned long  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt4)Number, Width, Flag1 | Flag2 | Flag3 | Flag4, false);
}

#endif

// ----------------------- LONG  -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Long Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
#if defined (BIT64) || defined (_WIN64)
    FillSignedNumberFormatBuffer( (SAPDB_Int8)Number, 0, Flag1 | Flag2 | Flag3 | Flag4, true );
#else
    int        combinedFlags = Flag1 | Flag2 | Flag3 | Flag4;
    SAPDB_Int8 outputNumber;

    if ( (combinedFlags & (hex|oct)) != 0 )
    {
        outputNumber = (SAPDB_Int8)((SAPDB_UInt4)Number);
    }
    else
    {
        outputNumber = (SAPDB_Int8)Number;
    }

    FillSignedNumberFormatBuffer( outputNumber, 0, combinedFlags );
#endif
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Long  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
#if defined (BIT64) || defined (_WIN64)
    FillSignedNumberFormatBuffer( (SAPDB_Int8)Number, Width, Flag1 | Flag2 | Flag3 | Flag4, true );
#else
    int        combinedFlags = Flag1 | Flag2 | Flag3 | Flag4;
    SAPDB_Int8 outputNumber;

    if ( (combinedFlags & (hex|oct)) != 0 )
    {
        outputNumber = (SAPDB_Int8)((SAPDB_UInt4)Number);
    }
    else
    {
        outputNumber = (SAPDB_Int8)Number;
    }

    FillSignedNumberFormatBuffer( outputNumber, Width, combinedFlags );
#endif
}

// ----------------------- ULONG  -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_ULong  Number,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
#if defined (BIT64) || defined (_WIN64)
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt8)Number, 0, Flag1 | Flag2 | Flag3 | Flag4, true);
#else
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt8)Number, 0, Flag1 | Flag2 | Flag3 | Flag4);
#endif
}

// -----------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_ULong  Number,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
#if defined (BIT64) || defined (_WIN64)
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt8)Number, Width, Flag1 | Flag2 | Flag3 | Flag4, true);
#else
    FillUnsignedNumberFormatBuffer( (SAPDB_UInt8)Number, Width, Flag1 | Flag2 | Flag3 | Flag4);
#endif
}

// ----------------------- REAL4 / REAL8 -------------------

void SAPDB_ToStringClass::FillRealFormatBuffer( const SAPDB_UInt2 Width,
                                           const SAPDB_UInt2 Precision,
                                           const int         Flags,
                                           const SAPDB_Real8 Real )
{
    SAPDB_UInt4 usedWidth;
    SAPDB_UInt4 usedPrecision;

    int FormatBufferLen = 0 ;

    // silently, but reasonable.... PTS 1110611
    usedWidth = ( Width > MAX_CHARACTER_ALLOWED_IN_PRESENTATION ?
                        MAX_CHARACTER_ALLOWED_IN_PRESENTATION : Width );

    // silently, but reasonable.... PTS 1110611
    usedPrecision = ( Precision > MAX_CHARACTER_ALLOWED_IN_PRESENTATION ?
                        MAX_CHARACTER_ALLOWED_IN_PRESENTATION : Precision );

    m_FormatBuffer[FormatBufferLen] = '%' ;
    FormatBufferLen++ ;

    if ( Flags & showpos )
    {
        m_FormatBuffer[FormatBufferLen] = '+' ;
        FormatBufferLen++ ;
    }

    if ( Flags & showpoint )
	{
      m_FormatBuffer[FormatBufferLen] = '0';
      FormatBufferLen++ ;
	}

    if ( usedWidth )
      if ( Flags & left )
        FormatBufferLen += sp77sprintf( m_FormatBuffer+FormatBufferLen, 
                                        MAX_CHARACTER_ALLOWED_IN_FORMAT-FormatBufferLen,
                                        "-%d", 
                                        usedWidth );
      else
        FormatBufferLen += sp77sprintf( m_FormatBuffer+FormatBufferLen, 
                                        MAX_CHARACTER_ALLOWED_IN_FORMAT-FormatBufferLen,
                                        "%d", 
                                        usedWidth );

    if ( usedPrecision )
      FormatBufferLen += sp77sprintf( m_FormatBuffer+FormatBufferLen, 
                                      MAX_CHARACTER_ALLOWED_IN_FORMAT-FormatBufferLen,
                                      ".%d", 
                                      usedPrecision ) ;

    if ( Flags & scientific )
        m_FormatBuffer[FormatBufferLen] = Flags & uppercase ? 'E' : 'e';
    else
      m_FormatBuffer[FormatBufferLen] = 'f' ;

    FormatBufferLen++ ;
    m_FormatBuffer[FormatBufferLen] = '\0' ;

    m_PresentationBuffer[MAX_CHARACTER_ALLOWED_IN_PRESENTATION] = 0;
    sp77sprintf(m_PresentationBuffer, MAX_CHARACTER_ALLOWED_IN_PRESENTATION, m_FormatBuffer, Real ) ;

    m_OutputString = m_PresentationBuffer ;
    m_Buf = 0;
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Real8  Real,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillRealFormatBuffer(0, 0, Flag1 | Flag2 | Flag3 | Flag4, Real ) ;
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Real8  Real,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillRealFormatBuffer(Width, 0, Flag1 | Flag2 | Flag3 | Flag4, Real ) ;
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const SAPDB_Real8  Real,
                                const SAPDB_UInt2 Width ,
                                const SAPDB_UInt2 Precision ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillRealFormatBuffer(Width, Precision, Flag1 | Flag2 | Flag3 | Flag4, Real ) ;
}


// ----------------------- VOID * (pointer) ----------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const void *Addr,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillNumberFormatBuffer(0, ptr | Flag1 |Flag2 | Flag3 | Flag4) ;

    m_PresentationBuffer[MAX_CHARACTER_ALLOWED_IN_PRESENTATION] = 0;
    sp77sprintf(m_PresentationBuffer, MAX_CHARACTER_ALLOWED_IN_PRESENTATION, m_FormatBuffer, Addr ) ;

    m_OutputString = m_PresentationBuffer ;
    m_Buf = 0;
}

// --------------------------

SAPDB_ToStringClass::SAPDB_ToStringClass( const void *Addr,
                                const SAPDB_UInt2 Width ,
                                const fmtflags Flag1,
                                const fmtflags Flag2,
                                const fmtflags Flag3,
                                const fmtflags Flag4)
{
    FillNumberFormatBuffer(Width, ptr | Flag1 | Flag2 | Flag3 | Flag4) ;

    m_PresentationBuffer[MAX_CHARACTER_ALLOWED_IN_PRESENTATION] = 0;
    sp77sprintf(m_PresentationBuffer, MAX_CHARACTER_ALLOWED_IN_PRESENTATION, m_FormatBuffer, Addr ) ;

    m_OutputString = m_PresentationBuffer ;
    m_Buf = 0;
}

// --------------------------

SAPDB_Int4 const SAPDB_ToStringClass::Length() const
{
    return (SAPDB_Int4)strlen(const_cast<char *>(m_OutputString));
}
/*===========================================================================*
 *  PUBLIC FUNCTIONS (CODE)                                                  *
 *===========================================================================*/

/*===========================================================================*
 *  END OF CODE                                                              *
 *===========================================================================*/

