#include "TePDIPIManager.hpp"

#include <TeAgnostic.h>

/* Forcing the instantiation of global settings 
   to avoid thread racing condition */
namespace {
  static TePDIPIManagerGlobalSettings& globalsettings = 
    TePDIPIManagerGlobalSettings::instance();
}

TePDIPIManager::TePDIPIManager( const std::string& message, 
  unsigned long int tsteps, bool enabled )
{
  enabled_ = false;
  total_steps_ = tsteps;
  curr_step_ = 0;
  message_ = message;
  
  if( enabled ) {
    Toggle( true );
  }
}


TePDIPIManager::~TePDIPIManager()
{
  Toggle( false );
}


void TePDIPIManager::Toggle( bool enabled )
{
  TeProgressBase* prog_intef_ptr = TeProgress::instance();
  
  if( prog_intef_ptr ) {
    TePDIPIManagerGlobalSettings& global_settings = 
      TePDIPIManagerGlobalSettings::instance();
    
    if( ( ! enabled_ ) && enabled && ( total_steps_ > 0 ) ) {
      /* Need to enable this instance */
     
      global_settings.lock_.lock();

      if( global_settings.active_manager_instances_ == 0 ) {
        /* No instance has control */
        
        global_settings.total_steps_ = total_steps_;
        global_settings.curr_step_ = curr_step_;
        /* keeping global_settings.curr_prop_step_ */
        global_settings.active_manager_instances_ = 1;
        global_settings.curr_message_ = message_;
      } else {
        /* More instances are controling */
        
        global_settings.total_steps_ += total_steps_;
        global_settings.curr_step_ += curr_step_;
        /* Keeping global_settings.curr_prop_step_ */
        global_settings.active_manager_instances_ += 1;
        global_settings.curr_message_.clear();
      }
      
      enabled_ = true;
      
      updateProgressInterface( global_settings, prog_intef_ptr );
      
      global_settings.lock_.unLock();
    } if( enabled_ && ( ! enabled ) ) {
      /* Need to disable this instance */
      
      global_settings.lock_.lock();

      if( global_settings.active_manager_instances_ == 1 ) {
        /* Just this instance has control */
        
        global_settings.total_steps_ = 0;
        global_settings.curr_step_ = 0;
        /* keeping global_settings.curr_prop_step_ */
        global_settings.active_manager_instances_ = 0;
        global_settings.curr_message_.clear();
      } else {
        /* More instance are controling */
        
        /* keeping global_settings.total_steps_ */
        global_settings.curr_step_ += ( total_steps_ - curr_step_ );
        /* keeping global_settings.curr_prop_step_ */
        global_settings.active_manager_instances_ -= 1;
      }
      
      enabled_ = false;
      
      updateProgressInterface( global_settings, prog_intef_ptr );
      
      global_settings.lock_.unLock();      
    }
  }
}


bool TePDIPIManager::updateProgressInterface( 
  TePDIPIManagerGlobalSettings& settings,
  TeProgressBase* prog_intef_ptr ) const
{
  if( prog_intef_ptr->wasCancelled() ) {
    if( settings.active_manager_instances_ == 0 ) {
      prog_intef_ptr->reset();
      return false;
    } else {
      return true;
    }
  } else {
    if( settings.active_manager_instances_ == 0 ) {
      settings.curr_prop_step_ = 0;
      prog_intef_ptr->reset();    
    } else {
      unsigned long int new_prop_step = (unsigned long int)
        ( 100.0 * 
        ( ((double)settings.curr_step_) / 
        ( (double)settings.total_steps_ ) ) );
          
      if( new_prop_step > settings.curr_prop_step_ ) {
        prog_intef_ptr->setMessage( settings.curr_message_ );
        prog_intef_ptr->setTotalSteps( 100 );
        prog_intef_ptr->setProgress( new_prop_step );
        
        settings.curr_prop_step_ = new_prop_step;
      }
    } 
    
    return false; 
  }
}


bool TePDIPIManager::Update( unsigned long int step )
{
  if( step > total_steps_ ) {
    step = total_steps_;
  }
  
  TeProgressBase* prog_intef_ptr = TeProgress::instance();
  
  if( enabled_ && prog_intef_ptr && ( step > curr_step_ ) ) {
    TePDIPIManagerGlobalSettings& global_settings = 
      TePDIPIManagerGlobalSettings::instance();
    
    global_settings.lock_.lock();

    /* keeping global_settings.total_steps_ */
    global_settings.curr_step_ += ( step - curr_step_ );
    /* keeping global_settings.curr_prop_step_ */
    /* keeping global_settings.active_manager_instances_ */
    
    curr_step_ = step;
    
    bool return_value = updateProgressInterface( global_settings, 
      prog_intef_ptr );
    
    global_settings.lock_.unLock();      
  
    return return_value;
  } else {
    return false;
  }
}


bool TePDIPIManager::Increment()
{
  return Update( (int)( curr_step_ + 1 ) );
}


void TePDIPIManager::Reset( const std::string& message, 
  unsigned long int tsteps )
{
  bool old_enabled_state = enabled_;
  
  Toggle( false );
  
  total_steps_ = tsteps; 
  curr_step_ = 0;
  message_ = message;

  if( old_enabled_state ) {
    Toggle( true );
  }  
}

