#include "TePDIBlending.hpp"

#include <TePDIBlendStratFactory.hpp>
#include <TePDIStatistic.hpp>
#include <TePDIContrast.hpp>
#include <TePDIUtils.hpp>
#include <TeAgnostic.h>

#include <TeOverlay.h>
#include <TeGeometryAlgorithms.h>


TePDIBlending::TePDIBlending()
{
}


TePDIBlending::~TePDIBlending()
{
}


void TePDIBlending::ResetState( const TePDIParameters& )
{
}


bool TePDIBlending::CheckParameters( const TePDIParameters& parameters ) const
{
  /* Checking input_raster1 */
  
  TePDITypes::TePDIRasterPtrType input_raster1;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "input_raster1", 
    input_raster1 ),
    "Missing parameter: input_raster1" );
  TEAGN_TRUE_OR_RETURN( input_raster1.isActive(),
    "Invalid parameter: input_raster1 inactive" );
  TEAGN_TRUE_OR_RETURN( input_raster1->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: input_raster1 not ready" );
  TEAGN_TRUE_OR_RETURN( ( input_raster1->params().projection() != 0 ),
    "Invalid parameter: input_raster1 do not have a projection" );
    
  /* Checking input_raster2 */
  
  TePDITypes::TePDIRasterPtrType input_raster2;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "input_raster2", 
    input_raster2 ),
    "Missing parameter: input_raster2" );
  TEAGN_TRUE_OR_RETURN( input_raster2.isActive(),
    "Invalid parameter: input_raster2 inactive" );
  TEAGN_TRUE_OR_RETURN( input_raster2->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: input_raster2 not ready" );    
  TEAGN_TRUE_OR_RETURN( ( input_raster2->params().projection() != 0 ),
    "Invalid parameter: input_raster2 do not have a projection" );
    
  /* Checking output_raster */
  
  TePDITypes::TePDIRasterPtrType output_raster;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "output_raster", 
    output_raster ),
    "Missing parameter: output_raster" );
  TEAGN_TRUE_OR_RETURN( output_raster.isActive(),
    "Invalid parameter: output_raster inactive" );
  TEAGN_TRUE_OR_RETURN( output_raster->params().status_ != 
    TeRasterParams::TeNotReady, "Invalid parameter: output_raster not ready" );    
    
  /* channels1 parameter checking */

  std::vector< int > channels1;
  TEAGN_TRUE_OR_RETURN( 
    parameters.GetParameter( "channels1", channels1 ), 
    "Missing parameter: channels1" );
  for( unsigned int channels1_index = 0 ; 
    channels1_index < channels1.size() ; 
    ++channels1_index ) {
    
    TEAGN_TRUE_OR_RETURN( ( channels1[ channels1_index ] >= 0 ) &&
      ( channels1[ channels1_index ] < input_raster1->nBands() ),
      "Invalid parameter: channels1" );
  }
  
  /* channels2 parameter checking */

  std::vector< int > channels2;
  TEAGN_TRUE_OR_RETURN( 
    parameters.GetParameter( "channels2", channels2 ), 
    "Missing parameter: channels2" );
  TEAGN_TRUE_OR_RETURN( ( channels2.size() == channels1.size() ),
    "Size mismatch between channels1 and channels2" );
  for( unsigned int channels2_index = 0 ; 
    channels2_index < channels2.size() ; 
    ++channels2_index ) {
    
    TEAGN_TRUE_OR_RETURN( ( channels2[ channels2_index ] >= 0 ) &&
      ( channels2[ channels2_index ] < input_raster2->nBands() ),
      "Invalid parameter: channels2" );
  } 
    
  /* Checking raster polygons */
  
  TePDITypes::TePDIPolygonPtrType raster1_pol_ptr;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster1_pol_ptr", 
    raster1_pol_ptr ), "Missing parameter : raster1_pol_ptr" );
  TEAGN_TRUE_OR_RETURN( raster1_pol_ptr.isActive(),
    "Invalid parameter : raster1_pol_ptr" )
  TEAGN_TRUE_OR_RETURN(
    ( ! input_raster1->begin( *raster1_pol_ptr, TeBoxPixelIn, 
    0 ).end() ), "Invalid parameter : raster1_pol_ptr" )
  TEAGN_TRUE_OR_RETURN( ( raster1_pol_ptr->size() == 1 ),
    "Invalid parameter : raster1_pol_ptr" )
    
  TePDITypes::TePDIPolygonPtrType raster2_pol_ptr;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster2_pol_ptr", 
    raster2_pol_ptr ), "Missing parameter : raster2_pol_ptr" );
  TEAGN_TRUE_OR_RETURN( raster2_pol_ptr.isActive(),
    "Invalid parameter : raster2_pol_ptr" )
  TEAGN_TRUE_OR_RETURN(
    ( ! input_raster2->begin( *raster2_pol_ptr, TeBoxPixelIn, 
    0 ).end() ), "Invalid parameter : raster2_pol_ptr" )
  TEAGN_TRUE_OR_RETURN( ( raster2_pol_ptr->size() == 1 ),
    "Invalid parameter : raster2_pol_ptr" )
    
  /* Checking raster2 polygon offsets */
  
  double raster2_pol_offset_x = 0;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster2_pol_offset_x", 
    raster2_pol_offset_x ), "Missing parameter : raster2_pol_offset_x" );  

  double raster2_pol_offset_y = 0;
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "raster2_pol_offset_y", 
    raster2_pol_offset_y ), "Missing parameter : raster2_pol_offset_y" );
    
  /* Checking the blending_type parameter */ 
  
  std::string blending_type;
  
  TEAGN_TRUE_OR_RETURN( parameters.GetParameter( "blending_type", 
    blending_type ), "Missing parameter: blending_type" );
    
  TePDIBlendStratFactory::TeFactoryMap::iterator fac_map_it = 
    TePDIBlendStratFactory::instance().find( blending_type );
    
  TEAGN_TRUE_OR_RETURN( 
    ( fac_map_it != TePDIBlendStratFactory::instance().end() ),
    "Invalid parameter: blending_type" );
       
  return true;
}


bool TePDIBlending::RunImplementation()
{
  bool auto_equalize = false;
  if( params_.CheckParameter< int >( "auto_equalize" ) ) {
    
    auto_equalize = true;
  }
  
  /* Converting input rasters to multiband */
  {
    TePDITypes::TePDIRasterPtrType input_raster1;
    params_.GetParameter( "input_raster1", input_raster1 );  
    
    TePDITypes::TePDIRasterPtrType input_raster2;
    params_.GetParameter( "input_raster2", input_raster2 );  
  
    TePDITypes::TePDIRasterPtrType input_raster1_multiband;
    
    TEAGN_TRUE_OR_RETURN( TePDIUtils::convert2MultiBand( input_raster1,
      progress_enabled_, input_raster1_multiband ), 
      "Multi-band conversion error" );
      
    params_.SetParameter( "input_raster1", input_raster1_multiband );    
  
    TePDITypes::TePDIRasterPtrType input_raster2_multiband;
    
    TEAGN_TRUE_OR_RETURN( TePDIUtils::convert2MultiBand( input_raster2,
      progress_enabled_, input_raster2_multiband ), 
      "Multi-band conversion error" );
      
    params_.SetParameter( "input_raster2", input_raster2_multiband );   
  }
  
  /* Reseting output raster */
  
  TEAGN_TRUE_OR_RETURN( resetOuputRaster(), "Output raster reset error" );
  
  /* Auto equalizing ( if required ) */
  
  if( auto_equalize ) {
    TEAGN_TRUE_OR_RETURN( equalizeInputRaster2(),
      "input_raster2 equalizing error" );
  }
  
  short pols_relation = 0;   
    
  /* Rendering the non-intersected areas */
  
  TEAGN_TRUE_OR_RETURN( renderNonIntersectedAreas( pols_relation ), 
    "Non-intersected area redering error" );     
    
  /* Calling the blending strategy */  
  
  if( ( pols_relation != TeDISJOINT ) && ( pols_relation != TeTOUCHES ) ) {
    std::string blending_type;
    params_.GetParameter( "blending_type", blending_type );  
    
    TePDIBlendingStrategy::pointer blending_strategy( 
      TePDIBlendStratFactory::make( blending_type, params_ ) );
        
    TEAGN_TRUE_OR_RETURN( blending_strategy.isActive(), 
      "Unable to buil blend strategy" );
      
    TEAGN_TRUE_OR_RETURN( blending_strategy->Apply( params_ ),
      "Error applying blending strategy" );    
  }
  
  return true;
}


bool TePDIBlending::renderNonIntersectedAreas( short& pols_relation )
{
  TePDITypes::TePDIRasterPtrType input_raster1;
  params_.GetParameter( "input_raster1", input_raster1 );
  TeRaster& input_raster1_ref = *( input_raster1.nakedPointer() );

  TePDITypes::TePDIRasterPtrType input_raster2;
  params_.GetParameter( "input_raster2", input_raster2 );  
  TeRaster& input_raster2_ref = *( input_raster2.nakedPointer() );

  TePDITypes::TePDIRasterPtrType output_raster;
  params_.GetParameter( "output_raster", output_raster );
  TeRaster& output_raster_ref = *( output_raster.nakedPointer() );

  std::vector< int > channels1;
  params_.GetParameter( "channels1", channels1 );
  
  std::vector< int > channels2;
  params_.GetParameter( "channels2", channels2 );  
  
  /* Guessing dummy use */
  
  bool not_using_dummy = true;
  double dummy_value = 0;
  
  if( params_.CheckParameter< double >( "dummy_value" ) ) {
    
    params_.GetParameter( "dummy_value", dummy_value );
    
    not_using_dummy = false;
  }  
  
  /* Calculating intersecion polygons data */
  
  TePolygon new_raster1_polygon;
  TePolygon new_raster2_polygon_ref_out;
  TePolygon dummy_inter_pol;
  TePolygon dummy_inter_pol_ref2;
  double raster1_rel_index_offset_x = 0;
  double raster1_rel_index_offset_y = 0;
  double raster2_rel_index_offset_x = 0;
  double raster2_rel_index_offset_y = 0;

  TEAGN_TRUE_OR_RETURN( extractPolygons( params_, new_raster1_polygon,
    new_raster2_polygon_ref_out, dummy_inter_pol, dummy_inter_pol_ref2,
    pols_relation,
    raster1_rel_index_offset_x, raster1_rel_index_offset_y,
    raster2_rel_index_offset_x, raster2_rel_index_offset_y ),
    "Error extracting intersection polygons" );
  
  /* Global vars */
  
  TeRaster::iteratorPoly input_raster1_it;
  if( new_raster1_polygon.size() != 0 ) {
    input_raster1_it =
      output_raster->begin( new_raster1_polygon, TeBoxPixelIn, 0 );
  }
      
  TeRaster::iteratorPoly input_raster2_it;
  if( new_raster2_polygon_ref_out.size() != 0 ) {
    input_raster2_it =
    output_raster->begin( new_raster2_polygon_ref_out, TeBoxPixelIn, 0 );
  }
      
  TePDIPIManager progress( "Rendering non-intersected area...", 
    channels1.size() * ( input_raster1_it.nLinesInPoly() + 
    input_raster2_it.nLinesInPoly() ), progress_enabled_ );    
  
  /* Transfering non-intersected areas */
  
  for( unsigned int channels_index = 0 ; channels_index < channels1.size() ;
    ++channels_index ) {
    
    const unsigned int channel1 = channels1[ channels_index ];
    const unsigned int channel2 = channels2[ channels_index ];

    /* Transfering raster 1 - non-intersected area */
    
    if( new_raster1_polygon.size() != 0 ) {
      input_raster1_it =
        output_raster->begin( new_raster1_polygon, TeBoxPixelIn, 0 );
    
      unsigned int last_line = 0;  
      unsigned int curr_line = 0;
      unsigned int curr_col = 0;
      double curr_value = 0;  
      int offset_x = TeRound( raster1_rel_index_offset_x );
      int offset_y = TeRound( raster1_rel_index_offset_y );
      
      while( ! input_raster1_it.end() ) {
        curr_line = input_raster1_it.currentLine();
        curr_col = input_raster1_it.currentColumn();
          
        if( input_raster1_ref.getElement( curr_col + offset_x, 
            curr_line + offset_y, curr_value, channel1 ) ) {
            
          if( not_using_dummy || 
            ( curr_value != dummy_value ) ) {
            
            TEAGN_TRUE_OR_RETURN( 
              output_raster_ref.setElement( curr_col, curr_line, curr_value, 
              channels_index ),
              "Unable to write to output raster at line=" + 
              Te2String( curr_line ) + 
              " col=" + Te2String( curr_col ) );                    
          }
        }
    
        if( curr_line != last_line ) {
          last_line = curr_line;
          TEAGN_FALSE_OR_RETURN( progress.Increment(), 
            "Canceled by the user" );
        }
        
        ++input_raster1_it;
      } 
    }
  
    /* Transfering raster 2 - non-intersected area */
    
    if( new_raster2_polygon_ref_out.size() != 0 ) {
      input_raster2_it =
        output_raster->begin( new_raster2_polygon_ref_out, TeBoxPixelIn, 0 );
    
      unsigned int last_line = 0;  
      unsigned int curr_line = 0;
      unsigned int curr_col = 0;
      double curr_value = 0;  
      int offset_x = TeRound( raster2_rel_index_offset_x );
      int offset_y = TeRound( raster2_rel_index_offset_y );
        
      while( ! input_raster2_it.end() ) {
        curr_line = input_raster2_it.currentLine();
        curr_col = input_raster2_it.currentColumn();
          
        if( input_raster2_ref.getElement( curr_col + offset_x, 
            curr_line + offset_y , curr_value, channel2 ) ) {
            
          if( not_using_dummy || 
            ( curr_value != dummy_value ) ) {
          
            TEAGN_TRUE_OR_RETURN( 
              output_raster_ref.setElement( curr_col, curr_line, curr_value, 
              channels_index ), "Unable to write to output raster at line=" + 
              Te2String( curr_line ) + " col=" + Te2String( curr_col ) );
          }
        }
    
        if( curr_line != last_line ) {
          last_line = curr_line;
          TEAGN_FALSE_OR_RETURN( progress.Increment(), 
            "Canceled by the user" );
        }
        
        ++input_raster2_it;
      }
    }
  }
  
  return true;
}


bool TePDIBlending::equalizeInputRaster2()
{
  TePDITypes::TePDIRasterPtrType input_raster1;
  params_.GetParameter( "input_raster1", input_raster1 );

  TePDITypes::TePDIRasterPtrType input_raster2;
  params_.GetParameter( "input_raster2", input_raster2 );
  
  std::vector< int > channels1;
  params_.GetParameter( "channels1", channels1 );  
  
  std::vector< int > channels2;
  params_.GetParameter( "channels2", channels2 );  
  
  bool enable_multi_thread = params_.CheckParameter< int >( 
    "enable_multi_thread" );
  
  /* Guessing dummy use */
  
  bool forced_dummy = false;
  double dummy_value = 0;
  
  if( params_.CheckParameter< double >( "dummy_value" ) ) {
    
    params_.GetParameter( "dummy_value", dummy_value );
    
    forced_dummy = true;
  }    
  
  /* Calculating intersecion polygons data */
  
  TePolygon new_raster1_polygon;
  TePolygon new_raster2_polygon;
  TePolygon inter_pol_ref1;
  TePolygon int_pol_ref2;
  double raster1_rel_index_offset_x = 0;
  double raster1_rel_index_offset_y = 0;
  double raster2_rel_index_offset_x = 0;
  double raster2_rel_index_offset_y = 0;
  short pols_relation = 0;

  TEAGN_TRUE_OR_RETURN( TePDIBlending::extractPolygons( params_, 
    new_raster1_polygon, new_raster2_polygon, inter_pol_ref1, int_pol_ref2,
    pols_relation,
    raster1_rel_index_offset_x, raster1_rel_index_offset_y,
    raster2_rel_index_offset_x, raster2_rel_index_offset_y ),
    "Error extracting intersection polygons" );  
  
  /* Defining the overlap polygon over raster1 ( raster1 world index )*/
    
  if( pols_relation != TeDISJOINT ) {
    /* Creating the thread parameters */
    
    std::vector< double > mean_vector_raster1;
    std::vector< double > mean_vector_raster2;
    
    std::vector< double > variance_vector_raster1;
    std::vector< double > variance_vector_raster2;
    
    TeThreadParameters t1_params;
    t1_params.store( "input_raster", input_raster1 );
    t1_params.store( "input_pol", inter_pol_ref1 );
    t1_params.store( "progress_enabled_flag", 
      TePDIAlgorithm::progress_enabled_ );
    t1_params.store( "input_channels", channels1 );
    t1_params.store( "mean_vector_ptr", &mean_vector_raster1 );
    t1_params.store( "variance_vector_ptr", &variance_vector_raster1 );

    TeThreadParameters t2_params;
    t2_params.store( "input_raster", input_raster2 );
    t2_params.store( "input_pol", int_pol_ref2 );
    t2_params.store( "progress_enabled_flag", 
      TePDIAlgorithm::progress_enabled_ );
    t2_params.store( "input_channels", channels2 );
    t2_params.store( "mean_vector_ptr", &mean_vector_raster2 );
    t2_params.store( "variance_vector_ptr", &variance_vector_raster2 );
    
    /* Starting both threads to generate the mean and variance vectors
       from both rasters */

    if( enable_multi_thread ) {
      TeThreadFunctor thread1;
      thread1.setStartFunctPtr( getMeanAndVarianceThreadEntry );
      
      TeThreadFunctor thread2;
      thread2.setStartFunctPtr( getMeanAndVarianceThreadEntry );
        
      thread1.setParameters( t1_params );
      TEAGN_TRUE_OR_RETURN( thread1.start(),
        "Error starting thread1" );  
      thread2.setParameters( t2_params );
      TEAGN_TRUE_OR_RETURN( thread2.start(),
        "Error starting thread2" );         
    
      TEAGN_TRUE_OR_RETURN( thread1.waitToFinish(),
        "Error waiting for thread to finish" )  
      TEAGN_TRUE_OR_RETURN( thread2.waitToFinish(),
        "Error waiting for thread to finish" )          
    } else {
      TEAGN_TRUE_OR_RETURN( getMeanAndVarianceThreadEntry( t1_params ),
        "Error calculatin mean and variance from raster 1" );      
      TEAGN_TRUE_OR_RETURN( getMeanAndVarianceThreadEntry( t2_params ),
        "Error calculatin mean and variance from raster 2" );         
    }
    
    TEAGN_DEBUG_CONDITION( ( mean_vector_raster1.size() == channels1.size() ),
      "Invalid mean vector size" )
    TEAGN_DEBUG_CONDITION( ( mean_vector_raster2.size() == channels2.size() ),
      "Invalid mean vector size" )

    TEAGN_DEBUG_CONDITION( ( variance_vector_raster1.size() == channels1.size() ),
      "Invalid variance vector size" )
    TEAGN_DEBUG_CONDITION( ( variance_vector_raster2.size() == channels1.size() ),
      "Invalid variance vector size" )
  
    /* Creating a new raster2 */
      
    TePDITypes::TePDIRasterPtrType new_input_raster2;
    
    TeRasterParams new_input_raster2_params = input_raster2->params();
    new_input_raster2_params.nBands( channels2.size() );
    if( forced_dummy ) {
      new_input_raster2_params.setDummy( dummy_value, -1 );      
    } else {
      if( input_raster2->params().useDummy_ ) {
        new_input_raster2_params.setDummy( 
          input_raster2->params().dummy_[ 0 ], -1 );
      }
    }
    
    TEAGN_TRUE_OR_RETURN( TePDIUtils::TeAllocRAMRaster( new_input_raster2,
      input_raster2->params(), TePDIUtils::TePDIUtilsAutoMemPol ), 
      "Unable to allocate the new equalized raster2" );
      
    /* Equalizing data */
        
    std::vector< int > new_channels2;
        
    for( unsigned int channels_index = 0 ; 
      channels_index < channels2.size() ; ++channels_index ) {
      
      TePDIParameters contra_params;
      contra_params.SetParameter( "contrast_type",
        TePDIContrast::TePDIContrastStat );
      contra_params.SetParameter( "input_image", input_raster2 );
      contra_params.SetParameter( "input_band", 
        channels2[ channels_index] );
      contra_params.SetParameter( "target_mean", 
        mean_vector_raster1[ channels_index ] );
      contra_params.SetParameter( "target_variance", 
        variance_vector_raster1[ channels_index ] );
      contra_params.SetParameter( "input_mean", 
        mean_vector_raster2[ channels_index ] );
      contra_params.SetParameter( "input_variance", 
        variance_vector_raster2[ channels_index ] );        
      contra_params.SetParameter( "output_image", new_input_raster2 );
      contra_params.SetParameter( "output_band", (int)channels_index );
      contra_params.SetParameter( "restrict_out_reset", (int)1 );
      if( forced_dummy ) {
        contra_params.SetParameter( "dummy_value", dummy_value );
      }
      if( forced_dummy ) {
        contra_params.SetParameter( "dummy_value", dummy_value );
      } else {
        if( input_raster2->params().useDummy_ ) {
          contra_params.SetParameter( "dummy_value", 
            input_raster2->params().dummy_[ 0 ] );
        }
      }      
        
      TePDIContrast contra_instance;
      TEAGN_TRUE_OR_RETURN( contra_instance.Reset( contra_params ),
        "TePDIContrast reset error" );
      TEAGN_TRUE_OR_RETURN( contra_instance.Apply(),
        "TePDIContrast apply error" );            
        
      new_channels2.push_back( channels_index );
    }        
     
    params_.SetParameter( "input_raster2", new_input_raster2 );
    params_.SetParameter( "channels2", new_channels2 );  
  }
  
  return true;
}


bool TePDIBlending::resetOuputRaster()
{
  TePDITypes::TePDIRasterPtrType input_raster1;
  params_.GetParameter( "input_raster1", input_raster1 );
  
  TePDITypes::TePDIRasterPtrType input_raster2;
  params_.GetParameter( "input_raster2", input_raster2 );  
  
  TePDITypes::TePDIRasterPtrType output_raster;
  params_.GetParameter( "output_raster", output_raster );  
  
  TePDITypes::TePDIPolygonPtrType raster1_pol_ptr;
  params_.GetParameter( "raster1_pol_ptr", raster1_pol_ptr );
  TePolygon& raster1_polygon = ( *raster1_pol_ptr );

  TePDITypes::TePDIPolygonPtrType raster2_pol_ptr;
  params_.GetParameter( "raster2_pol_ptr", raster2_pol_ptr );
  TePolygon& raster2_polygon = ( *raster2_pol_ptr );
  
  std::vector< int > channels1;
  params_.GetParameter( "channels1", channels1 );  
  
  double raster2_pol_offset_x = 0;
  params_.GetParameter( "raster2_pol_offset_x", raster2_pol_offset_x );

  double raster2_pol_offset_y = 0;
  params_.GetParameter( "raster2_pol_offset_y", raster2_pol_offset_y );  
  
  /* Dumyy value definition */
 
  bool output_raster_uses_dummy = output_raster->params().useDummy_;
  double output_raster_dummy = 0;
  
  if( output_raster_uses_dummy ) {
    output_raster_dummy = output_raster->params().dummy_[ 0 ];
  }
  
  if( params_.CheckParameter< double >( "dummy_value" ) ) {
    
    params_.GetParameter( "dummy_value", output_raster_dummy );
    
    output_raster_uses_dummy = true;
  }  
  
  /* Building raster 2 polygon - reference raster1 */
  
  TePolygon raster2_polygon_ref1;
  {
    TePolygon raster2_polygon_indexed_ref2;
    TePDIUtils::MapCoords2RasterIndexes( raster2_polygon,
      input_raster2, raster2_polygon_indexed_ref2 );
        
    double r2_min_x = 
      MIN( raster2_polygon_indexed_ref2.box().x1(),
      raster2_polygon_indexed_ref2.box().x2() );
    double r2_min_y = 
      MIN( raster2_polygon_indexed_ref2.box().y1(),
      raster2_polygon_indexed_ref2.box().y2() ); 
     
    for( unsigned int lr_index = 0 ; 
      lr_index < raster2_polygon_indexed_ref2.size() ; ++lr_index ) {
      for( unsigned int c2d_index = 0 ; 
        c2d_index < raster2_polygon_indexed_ref2[ lr_index ].size() ; 
        ++c2d_index ) {
          
        raster2_polygon_indexed_ref2[ lr_index ][ c2d_index ].x( 
          raster2_polygon_indexed_ref2[ lr_index ][ c2d_index ].x() +
          raster2_pol_offset_x - r2_min_x );
        raster2_polygon_indexed_ref2[ lr_index ][ c2d_index ].y( 
          raster2_polygon_indexed_ref2[ lr_index ][ c2d_index ].y() +
          raster2_pol_offset_y - r2_min_y );
      }
    }
  
    TePDIUtils::MapRasterIndexes2Coords( raster2_polygon_indexed_ref2,
      input_raster1, raster2_polygon_ref1 );
  }  

  /* Reseting output raster */
  
  TeBox global_box_ref1 = raster1_polygon.box();
  updateBox( global_box_ref1, raster2_polygon_ref1.box() );
  
  TeRasterParams output_raster_params = output_raster->params();
  
  output_raster_params.nBands( channels1.size() );
  if( input_raster1->projection() != 0 ) {
    TeSharedPtr< TeProjection > proj( TeProjectionFactory::make( 
      input_raster1->projection()->params() ) );  
    output_raster_params.projection( proj.nakedPointer() );
  }
  output_raster_params.boxResolution( global_box_ref1.x1(), 
    global_box_ref1.y1(), global_box_ref1.x2(), global_box_ref1.y2(), 
    input_raster1->params().resx_, input_raster1->params().resy_ );
  if( output_raster_uses_dummy ) {
    output_raster_params.setDummy( output_raster_dummy, -1 );
  } else {
    output_raster_params.useDummy_ = false;
  }
  output_raster_params.setPhotometric( TeRasterParams::TeMultiBand );
     
  TEAGN_TRUE_OR_RETURN( output_raster->init( output_raster_params ),
    "Output raster reset error" );  
    
  return true;
}


bool TePDIBlending::extractPolygons( const TePDIParameters& params, 
  TePolygon& new_raster1_pol_ref1, TePolygon& new_raster2_pol_ref1, 
  TePolygon& inter_pol_ref1, TePolygon& inter_pol_ref2,
  short& relation, double& raster1_rel_index_offset_x,
  double& raster1_rel_index_offset_y, double& raster2_rel_index_offset_x,
  double& raster2_rel_index_offset_y )
{
  new_raster1_pol_ref1.clear();
  new_raster2_pol_ref1.clear();
  inter_pol_ref1.clear();
  inter_pol_ref2.clear();
  
  /* Extracting the used parameter */
  
  TePDITypes::TePDIRasterPtrType input_raster1;
  TEAGN_TRUE_OR_THROW( params.GetParameter( "input_raster1", input_raster1 ),
    "Missing input_raster1" );
  
  TePDITypes::TePDIRasterPtrType input_raster2;
  TEAGN_TRUE_OR_THROW(params.GetParameter( "input_raster2", input_raster2 ),
    "Missing input_raster2" );  

  TePDITypes::TePDIRasterPtrType output_raster;
  TEAGN_TRUE_OR_THROW(params.GetParameter( "output_raster", output_raster ),
    "Missing output_raster" ); 
     
  TePDITypes::TePDIPolygonPtrType raster1_pol_ptr;
  TEAGN_TRUE_OR_THROW(params.GetParameter( "raster1_pol_ptr", 
    raster1_pol_ptr ), "Missing raster1_pol_ptr" ); 
  TePolygon& raster1_polygon = ( *raster1_pol_ptr ); 
  
  TePDITypes::TePDIPolygonPtrType raster2_pol_ptr;
  TEAGN_TRUE_OR_THROW(params.GetParameter( "raster2_pol_ptr", 
    raster2_pol_ptr ), "Missing raster2_pol_ptr" );  
  TePolygon& raster2_polygon = ( *raster2_pol_ptr );
  
  double raster2_pol_offset_x = 0;
  TEAGN_TRUE_OR_THROW(params.GetParameter( "raster2_pol_offset_x", 
    raster2_pol_offset_x ), "Missing raster2_pol_offset_x" );  

  double raster2_pol_offset_y = 0;
  TEAGN_TRUE_OR_THROW(params.GetParameter( "raster2_pol_offset_y", 
    raster2_pol_offset_y ), "Missing raster2_pol_offset_y" );

  /* Raster1 positioning */
  
  double raster1_abs_index_offset_x = 0;/* related to output raster indexes */
  double raster1_abs_index_offset_y = 0;/* related to output raster indexes */
  {
    TeBox pol1_box_indexed;
    TePDIUtils::MapCoords2RasterIndexes( raster1_polygon.box(),
      input_raster1, pol1_box_indexed );
          
    double r1_min_x_index_ref1 = MIN( pol1_box_indexed.x1(),
      pol1_box_indexed.x2() );
    double r1_min_y_index_ref1 = MIN( pol1_box_indexed.y1(),
      pol1_box_indexed.y2() );      

    TeCoord2D r1_UL_ref1_indexed( r1_min_x_index_ref1, r1_min_y_index_ref1 );  
    TeCoord2D r1_UL_ref1( input_raster1->index2Coord( r1_UL_ref1_indexed ) );  
    TeCoord2D r1_UL_refout_indexed( output_raster->coord2Index( r1_UL_ref1 ) );
    
    /* Calculating the absolute offset betweeen input raster1 and output raster
       following output raster indexed units */    
        
    raster1_abs_index_offset_x = r1_UL_refout_indexed.x();  
    raster1_abs_index_offset_y = r1_UL_refout_indexed.y();
    
    /* Calculating the relative offset betweeen input raster1 and output raster
       following output raster indexed units */    
      
    raster1_rel_index_offset_x = r1_UL_ref1_indexed.x() -
      raster1_abs_index_offset_x;
    raster1_rel_index_offset_y = r1_UL_ref1_indexed.y() -
      raster1_abs_index_offset_y;
  }
  
  /* Raster2 positioning */
  
  TePolygon r2_pol_refout;
  {
    TePolygon r2_pol_ref2_indexed;
    TePDIUtils::MapCoords2RasterIndexes( raster2_polygon,
      input_raster2, r2_pol_ref2_indexed );
      
    double r2_min_x = MIN( r2_pol_ref2_indexed.box().x1(),
      r2_pol_ref2_indexed.box().x2() );
    double r2_min_y = MIN( r2_pol_ref2_indexed.box().y1(),
      r2_pol_ref2_indexed.box().y2() );
  
    raster2_rel_index_offset_x = r2_min_x - raster1_abs_index_offset_x -
      raster2_pol_offset_x;
    raster2_rel_index_offset_y = r2_min_y - raster1_abs_index_offset_y -
      raster2_pol_offset_y;
      
    TePolygon r2_pol_refout_indexed = r2_pol_ref2_indexed;
        
    for( unsigned int lr_index = 0 ; 
      lr_index < r2_pol_refout_indexed.size() ; ++lr_index ) {
      for( unsigned int c2d_index = 0 ; 
        c2d_index < r2_pol_refout_indexed[ lr_index ].size() ; 
        ++c2d_index ) {
            
        r2_pol_refout_indexed[ lr_index ][ c2d_index ].x( 
          r2_pol_refout_indexed[ lr_index ][ c2d_index ].x() -
          r2_min_x + raster1_abs_index_offset_x + raster2_pol_offset_x );
        r2_pol_refout_indexed[ lr_index ][ c2d_index ].y( 
          r2_pol_refout_indexed[ lr_index ][ c2d_index ].y() -
          r2_min_y + raster1_abs_index_offset_y + raster2_pol_offset_y );          
      }
    }
    
    TePDIUtils::MapRasterIndexes2Coords( r2_pol_refout_indexed,
      output_raster, r2_pol_refout );
  }
  
  /* Defining the new rasters polygons and the intersection polygons */
  
  relation = TeRelation( raster1_polygon, 
    r2_pol_refout );
  
  switch( relation ) {
    case TeEQUALS :
    {
      inter_pol_ref1 = raster1_polygon;
      
      return true;
      break;
    }
    case TeDISJOINT :
    case TeTOUCHES :
    {
      new_raster1_pol_ref1 = raster1_polygon;
      new_raster2_pol_ref1 = r2_pol_refout;
    
      break;
    }
    case TeOVERLAPS:
    {
      TePolygonSet r1_ps;
      r1_ps.add( raster1_polygon );

      TePolygonSet r2_ps;
      r2_ps.add( r2_pol_refout );
      
      /* Calculating the new raster1 polygon  - following raster1 projected 
         coords */
      
      {
        TePolygonSet ps_new;
        
        TEAGN_TRUE_OR_THROW( TeOVERLAY::TeDifference( r1_ps, 
          r2_ps, ps_new ),
            "Unable to find the new input_raster1 polygon" );
        
        TEAGN_TRUE_OR_THROW( ( ps_new.size() == 1 ),
          "Invalid polygon set size" );
          
        new_raster1_pol_ref1 = ps_new[ 0 ];
      }      
      
      /* Calculating the new raster2 polygon  - following raster1 projected 
         coords */
      
      {
        TePolygonSet ps_new;
        
        TEAGN_TRUE_OR_THROW( TeOVERLAY::TeDifference( r2_ps, 
          r1_ps, ps_new ),
            "Unable to find the new input_raster1 polygon" );
          
        TEAGN_TRUE_OR_THROW( ( ps_new.size() == 1 ),
          "Invalid polygon set size" );
          
        new_raster2_pol_ref1 = ps_new[ 0 ];
      }    
      
      /* Calculating the intersection polygon  - following raster1 projected 
         coords */
      
      {
        TePolygonSet ps_new;
        
        TEAGN_TRUE_OR_THROW( TeOVERLAY::TeIntersection( r1_ps, r2_ps, 
          ps_new ), "Unable to find polygons intersection" );
        
        TEAGN_TRUE_OR_THROW( ( ps_new.size() == 1 ),
          "Invalid polygon set size" );
          
        inter_pol_ref1 = ps_new[ 0 ];
      }

      /* Calculating the intersection polygon  - following raster2 projected 
         coords */
            
      {
        TePolygon inter_pol_indexed_refout;
        TePDIUtils::MapCoords2RasterIndexes( inter_pol_ref1,
          output_raster, inter_pol_indexed_refout );
          
        for( unsigned int lr_index = 0 ; 
          lr_index < inter_pol_indexed_refout.size() ; ++lr_index ) {
          for( unsigned int c2d_index = 0 ; 
            c2d_index < inter_pol_indexed_refout[ lr_index ].size() ; 
            ++c2d_index ) {
                
            inter_pol_indexed_refout[ lr_index ][ c2d_index ].x( 
              inter_pol_indexed_refout[ lr_index ][ c2d_index ].x() +
              raster2_rel_index_offset_x );
            inter_pol_indexed_refout[ lr_index ][ c2d_index ].y( 
              inter_pol_indexed_refout[ lr_index ][ c2d_index ].y() +
              raster2_rel_index_offset_y );          
          }
        }
        
        TePDIUtils::MapRasterIndexes2Coords( inter_pol_indexed_refout,
          input_raster2, inter_pol_ref2 );        
      }
    
      break;
    }
    case TeCONTAINS :
    case TeCOVERS :    
    {
      TePolygonSet r1_ps;
      r1_ps.add( raster1_polygon );

      TePolygonSet r2_ps;
      r2_ps.add( r2_pol_refout );
      
      /* Calculating the new raster1 polygon  - following raster1 projected 
         coords */      
          
      {
        TePolygonSet ps_new;
        
        TEAGN_TRUE_OR_THROW( TeOVERLAY::TeDifference( r1_ps, 
          r2_ps, ps_new ),
            "Unable to find the new input_raster1 polygon" );
        
        TEAGN_TRUE_OR_THROW( ( ps_new.size() == 1 ),
          "Invalid polygon set size" );
          
        new_raster1_pol_ref1 = ps_new[ 0 ];
      }
      
      inter_pol_ref1 = r2_pol_refout;
      
      /* Calculating the intersection polygon  - following raster2 projected 
         coords */
            
      {
        TePolygon inter_pol_indexed_refout;
        TePDIUtils::MapCoords2RasterIndexes( inter_pol_ref1,
          output_raster, inter_pol_indexed_refout );
          
        for( unsigned int lr_index = 0 ; 
          lr_index < inter_pol_indexed_refout.size() ; ++lr_index ) {
          for( unsigned int c2d_index = 0 ; 
            c2d_index < inter_pol_indexed_refout[ lr_index ].size() ; 
            ++c2d_index ) {
                
            inter_pol_indexed_refout[ lr_index ][ c2d_index ].x( 
              inter_pol_indexed_refout[ lr_index ][ c2d_index ].x() +
              raster2_rel_index_offset_x );
            inter_pol_indexed_refout[ lr_index ][ c2d_index ].y( 
              inter_pol_indexed_refout[ lr_index ][ c2d_index ].y() +
              raster2_rel_index_offset_y );          
          }
        }
        
        TePDIUtils::MapRasterIndexes2Coords( inter_pol_indexed_refout,
          input_raster2, inter_pol_ref2 );        
      }   
      
      break;
    }
    case TeWITHIN :
    case TeCOVEREDBY :
    {
      TePolygonSet r1_ps;
      r1_ps.add( raster1_polygon );

      TePolygonSet r2_ps;
      r2_ps.add( r2_pol_refout );     
      
      /* Calculating the new raster2 polygon  - following raster1 projected 
         coords */      

      {
        TePolygonSet ps_new;
        
        TEAGN_TRUE_OR_THROW( TeOVERLAY::TeDifference( r2_ps, 
          r1_ps, ps_new ),
            "Unable to find the new input_raster1 polygon" );
          
        TEAGN_TRUE_OR_THROW( ( ps_new.size() == 1 ),
          "Invalid polygon set size" );
          
        new_raster2_pol_ref1 = ps_new[ 0 ];
      }
      
      inter_pol_ref1 = raster1_polygon;
      
      /* Calculating the intersection polygon  - following raster2 projected 
         coords */
            
      {
        TePolygon inter_pol_indexed_refout;
        TePDIUtils::MapCoords2RasterIndexes( inter_pol_ref1,
          output_raster, inter_pol_indexed_refout );
          
        for( unsigned int lr_index = 0 ; 
          lr_index < inter_pol_indexed_refout.size() ; ++lr_index ) {
          for( unsigned int c2d_index = 0 ; 
            c2d_index < inter_pol_indexed_refout[ lr_index ].size() ; 
            ++c2d_index ) {
                
            inter_pol_indexed_refout[ lr_index ][ c2d_index ].x( 
              inter_pol_indexed_refout[ lr_index ][ c2d_index ].x() +
              raster2_rel_index_offset_x );
            inter_pol_indexed_refout[ lr_index ][ c2d_index ].y( 
              inter_pol_indexed_refout[ lr_index ][ c2d_index ].y() +
              raster2_rel_index_offset_y );          
          }
        }
        
        TePDIUtils::MapRasterIndexes2Coords( inter_pol_indexed_refout,
          input_raster2, inter_pol_ref2 );        
      }       
      
      break;
    }
    default :
    {
      TEAGN_LOG_AND_THROW( "Invalid polygon relation" );
      break;
    }
  }
  
  return true;
}

bool TePDIBlending::getMeanAndVarianceThreadEntry( const TeThreadParameters& pars )
{
  /* Retriving input parameters */
  
  TePDITypes::TePDIRasterPtrType input_raster;
  TEAGN_TRUE_OR_RETURN( pars.retrive( "input_raster", input_raster ),
    "Missing input_raster parameter" );  
  
  TePolygon input_pol;
  TEAGN_TRUE_OR_RETURN( pars.retrive( "input_pol", input_pol ),
    "Missing input_pol parameter" );
    
  bool progress_enabled_flag = false;
  TEAGN_TRUE_OR_RETURN( pars.retrive( "progress_enabled_flag", 
    progress_enabled_flag ),
    "Missing progress_enabled_flag parameter" ); 
    
  std::vector< int > input_channels;   
  TEAGN_TRUE_OR_RETURN( pars.retrive( "input_channels", 
    input_channels ),
    "Missing input_channels parameter" ); 
        
  /* Retriving output parameters */
    
  std::vector< double >* mean_vector_ptr = 0;
  TEAGN_TRUE_OR_RETURN( pars.retrive( "mean_vector_ptr", mean_vector_ptr ),
    "Missing mean_vector_ptr parameter" );
  mean_vector_ptr->clear();
  
  std::vector< double >* variance_vector_ptr = 0;
  TEAGN_TRUE_OR_RETURN( pars.retrive( "variance_vector_ptr", 
    variance_vector_ptr ),
    "Missing variance_vector_ptr parameter" );
  variance_vector_ptr->clear();
    
  /* Generating algorithm parameters */
    
  TePDIParameters stat_params;
    
  TePDITypes::TePDIRasterVectorType rasters;
  for( unsigned int channels_index = 0 ; 
    channels_index < input_channels.size() ; ++channels_index ) {
      
    rasters.push_back( input_raster );
  }
  stat_params.SetParameter( "rasters", rasters );
  
  stat_params.SetParameter( "bands", input_channels );
  
  TePDITypes::TePDIPolygonSetPtrType polygonset_ptr( new TePolygonSet );
  polygonset_ptr->add( input_pol );
  stat_params.SetParameter( "polygonset", polygonset_ptr );
  
  /* Generating mean and variance vectors */
  
  TePDIStatistic stat_instance;
  stat_instance.ToggleProgInt( progress_enabled_flag );

  TEAGN_TRUE_OR_RETURN( stat_instance.Reset( stat_params ),
    "TePDIStatistic reset error" );
    
  for( unsigned int channels_index = 0 ; 
    channels_index < input_channels.size() ; ++channels_index ) {
      
    mean_vector_ptr->push_back( stat_instance.getMean( channels_index, 0 ) );
    variance_vector_ptr->push_back( stat_instance.getVariance( channels_index, 
      0 ) );
  }
  
  return true;
}

