/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#ifndef  __TERRALIB_INTERNAL_STOBJECTSET_H
#define  __TERRALIB_INTERNAL_STOBJECTSET_H

#include "TeSTElement.h"

class TeQuerier; 

//! A TeSTElementSet is a set of spatial temporal elements
/*
	A spatial temporal set of elements or objects is obtained from a theme. 
*/
class TeSTElementSet  
{

protected:

	//! valid time interval of all ST elements in the set
	TeTimeInterval			totalTime_;		
	
	//! querier used to fill this structure
	TeQuerier*				querier_;       
	
	//! map of object identification to its instances in time
	map<string, TeSTElement >	elements_;	
	
	typedef map<string, TeSTElement >::iterator  STElementSetIterator;
	typedef TeSTElement::iterator				 STElemIterator;	

	//! Internal function to build the STElement set
	bool buildImpl(int slide=-1);

public:

	
	//! Default constructor
	TeSTElementSet()
	{	querier_ = 0; }

	//! Constructor
	TeSTElementSet(TeTheme* theme); 
	
	//! Constructor
	TeSTElementSet(TeLayer* layer); 

	//! Constructor
	TeSTElementSet(TeQuerier* querier); 
	
	//! Copy constructor
	TeSTElementSet (const TeSTElementSet& other); 

	//! Destructor 
	~TeSTElementSet();  
	
	//! Assignment operator 
	TeSTElementSet& operator= (const TeSTElementSet& other);

	//! Operator ==
	bool operator== (const TeSTElementSet& other) const
	{ return ((totalTime_==other.totalTime_)); } 
		
	//! Returns the time inteval associated with the STOSet
	TeTimeInterval totalTime() { return totalTime_;} 

	//! Sets the time inteval associated with the STOSet
	void totalTime(TeTimeInterval t) { totalTime_ = t;}

	//! Returns the querier
	TeQuerier* querier() { return querier_; }

	// Returns the theme
	TeTheme* theme();  

	//! Returns a reference to the map of elements
	map<string, TeSTElement >& elements() { return elements_; }

	//! Return if the element (objId) exists in the STElemSet 
	bool hasElement(const string& objId) { return (elements_.find (objId) != elements_.end()); }
	
	//! Insert an ST element instance 
	bool insertSTInstance (TeSTInstance& object);  

	//! Sets the geometries of a STObjetct
	bool setGeometry(const string& object_id, const TePolygonSet& geomSet, TeTimeInterval time = TeTimeInterval());
	bool setGeometry(const string& object_id, const TeLineSet& geomSet, TeTimeInterval time = TeTimeInterval());
	bool setGeometry(const string& object_id, const TePointSet& geomSet, TeTimeInterval time = TeTimeInterval());
	bool setGeometry(const string& object_id, const TeCellSet& geomSet, TeTimeInterval time = TeTimeInterval());
	bool setGeometry(const string& object_id, vector<TeGeometry*>& geomSet, TeTimeInterval time = TeTimeInterval());
	bool setGeometry(const string& object_id, const TeMultiGeometry& geomSet, TeTimeInterval time = TeTimeInterval());

	//! Gets the geometries of an object with a time stamp
	bool getGeometry(const string& object_id, TePolygonSet& geomSet, TeTimeInterval time = TeTimeInterval());
	bool getGeometry(const string& object_id, TeLineSet& geomSet, TeTimeInterval time = TeTimeInterval());
	bool getGeometry(const string& object_id, TePointSet& geomSet, TeTimeInterval time = TeTimeInterval());
	bool getGeometry(const string& object_id, TeCellSet& geomSet, TeTimeInterval time = TeTimeInterval());
	bool getGeometry(const string& object_id, vector<TeGeometry*>& geomSet, TeTimeInterval time = TeTimeInterval());
	bool getGeometry(const string& object_id, TeMultiGeometry& geomSet, TeTimeInterval time = TeTimeInterval());

 
	//! Gets the value of an attribute (as string) of a object with a time stamp
	bool getAttributeValue (const string&  object_id, const string& attr_name, 
		                    string& val, TeTimeInterval time = TeTimeInterval()); 
	

	//! Gets the value of the i-th attribute (as string) of a object with a time stamp
	bool getAttributeValue (const string& object_id, int attr_index, string& val,
							TeTimeInterval time = TeTimeInterval()); 

	//! Gets the value of the i-th attribute (as double) of a object with a time stamp
	bool getAttributeValue (const string& object_id, int attr_index, double& val,
							TeTimeInterval time = TeTimeInterval()); 

	//! Returns the index of an attribute
	int getAttributeIndex(const string& attrName);

	//! Gets a vector of attributes or properties of a object with a time stamp
	bool getPropertyVector (const string& object_id, TePropertyVector& propVec,
							TeTimeInterval time = TeTimeInterval()); 

	//! Sets the value of the property or adds if there is no such property in the object
	bool addProperty(const string& object_id, TeProperty& prop);

	//! Returns true if exists in the set, an object with more than one instance
	bool existMultiTemporalElem(const string& objId);

	//! Adds the instances of a element
	void insertSTElement(TeSTElement& stElem)
	{ elements_[stElem.objectId()] = stElem;	}
	
	//! Retuns the number of elements in the set
	int numElements() const  { return elements_.size();}

	//! Returns the number of instances of an object
	int numSTInstance(const string& objectId = ""); 

	//! Fills the STElement set
	bool build(bool loadGeometries=false, bool loadAllAttributes=true, vector<string> attrNames=vector<string>(), int slide=-1); 

	//! Fills the STElement set
	bool build(TeGroupingAttr& groupAttr, bool loadGeometries=false, int slide=-1); 

	//! Clears the set 
	void clear() { elements_.clear(); }
		
	//---------------------
	//! An iterator to a set of spatial temportal elements
	/*! 
		An strucuture that allows the traversal STElement set 
		(ST elements) in a similar way as the STL iterators. 
	*/
	class iterator 
	{
		public:

			iterator(TeSTElementSet* stoSet, STElementSetIterator stoSetP, STElemIterator stoVecP) : 
				curObj_(stoSetP),
				curSTO_(stoVecP),
				stoSet_(stoSet) 
		{}

		//! Prefix move forward operator
		iterator& operator++(); 
		
		TeSTInstance& operator*()
		{	return (*curSTO_); 	}

		STElemIterator getCurSTO()
		{	return curSTO_; 	}

		TeSTElementSet* elemSet()
		{	return stoSet_; 	}

		bool operator==(const iterator& rhs) const
		{ return (this->curSTO_ == rhs.curSTO_ && this->curObj_== rhs.curObj_); }

		bool operator!=(const iterator& rhs) const
		{ return (this->curSTO_ != rhs.curSTO_ && this->curObj_!= rhs.curObj_);	}

		protected:
			STElementSetIterator	curObj_;
			STElemIterator			curSTO_;
			TeSTElementSet*			stoSet_;
	};
	//---------------
	
	//---------------------
	//! An iterator to a given numeric property in a set of spatial temportal obejcts 
	/*! 
		An strucuture that allows the traversal STO set 
		(ST objects) in a similar way as the STL iterators. 
	*/
	class propertyIterator : public iterator
	{
		public:

			propertyIterator (TeSTElementSet* stoSet, 
				STElementSetIterator stoSetP, 
				STElemIterator stoVecP, 
				string attrName):
				iterator(stoSet, stoSetP, stoVecP),
				attrName_ (attrName)
				{}
		

			double operator*();  //if too slow, get property index. The problem is 
				// that nothing garantees it is going to be the same for all objects.
		
			bool getProperty(TeProperty& prop); 
				// same implemented for raster: to be used when attribute is categorical
		protected:
			string	attrName_;

	};
	//---------------

	//! Returns an iterator to the first element of the set
	iterator begin();     
	
	//! Returns an iterator to the first element of the set
	propertyIterator begin(const string& attrName);     
	
	//! Returns an iterator to the one past last element of the set
	iterator end();       
	
	//! Returns an iterator to the one past last element of the set
	propertyIterator end(const string& attrName);       
	
	//---------------
};

#endif

