/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeIntersector.h
    This file contains structures and definitions for line intersection algorithms.
 */

/**
  *@author Gilberto Ribeiro de Queiroz
  */

#ifndef  __TERRALIB_INTERNAL_INTERSECTOR2_H
#define  __TERRALIB_INTERNAL_INTERSECTOR2_H

//TerraLib's include
#include "TeCounted.h"
#include "TeCoord2D.h"
#include "TeGeometryAlgorithms.h"

//STL's include
#include <list>
#include <vector>
#include <map>
#include <algorithm>
#include <string>

using namespace std;


#define PRINT

namespace TeINTERSECTOR2
{

#define EPSILON_COMPARE 0.00000000005

/*! \fn inline short TeCCW(const TeCoord2D& c1, const TeCoord2D& c2, const TeCoord2D& c3, bool& between)
    \brief Tells if three points makes a right turn, a left turn or are collinear.
    \param c1      The first coordinate.
	\param c2      The second coordinate.
	\param c3      The coordinate to test the relative position.
	\param between Tells if c3 is between c1 and c2.
*/
inline short TeCCW(const TeCoord2D& c1, const TeCoord2D& c2, const TeCoord2D& c3, bool& between)
{
	double tol = TePrecision::instance().precision();
	TePrecision::instance().setPrecision(EPSILON_COMPARE);


	double dx1 = c2.x() - c1.x();
	double dx2 = c3.x() - c1.x();
	double dy1 = c2.y() - c1.y();
	double dy2 = c3.y() - c1.y();

	double dx1y2 = dx1 * dy2;
	double dy1x2 = dy1 * dx2;

	// slope of the second line is greater than the first, so counterclockwise.
	if(TeGeometryAlgorithmsPrecision::IsGreater(dx1y2, dy1x2))
	{
		TePrecision::instance().setPrecision(tol);

		return TeCOUNTERCLOCKWISE;
	}

	// slope of the first line is greater than the second, so clockwise.
	if(TeGeometryAlgorithmsPrecision::IsSmaller(dx1y2, dy1x2))
	{
		TePrecision::instance().setPrecision(tol);

		return TeCLOCKWISE;
	}
	

	TePrecision::instance().setPrecision(tol);
	// if we are here, so the three points are collinear.

	// check if c3 is between c1 and c2
	if(TeGeometryAlgorithmsPrecision::IsDifferent(c1.x(), c2.x()))	// c1c2 is not vertical
		between = (TeGeometryAlgorithmsPrecision::IsSmallerEqual(c1.x(), c3.x()) && TeGeometryAlgorithmsPrecision::IsSmallerEqual(c3.x(), c2.x())) ||
				  (TeGeometryAlgorithmsPrecision::IsGreaterEqual(c1.x(), c3.x()) && TeGeometryAlgorithmsPrecision::IsGreaterEqual(c3.x(), c2.x()));
	else	// c1c2 is vertical
		between = (TeGeometryAlgorithmsPrecision::IsSmallerEqual(c1.y(), c3.y()) && TeGeometryAlgorithmsPrecision::IsSmallerEqual(c3.y(), c2.y())) ||
		          (TeGeometryAlgorithmsPrecision::IsGreaterEqual(c1.y(), c3.y()) && TeGeometryAlgorithmsPrecision::IsGreaterEqual(c3.y(), c2.y()));



	//string linha = "";
	//linha  = Te2String(c1.x(), 15);
	//linha += " | ";
	//linha += Te2String(c1.y(), 15);
	//linha += " | ";
	//linha += Te2String(c2.x(), 15);
	//linha += " | ";
	//linha += Te2String(c2.y(), 15);
	//linha += " | ";
	//linha += Te2String(c3.x(), 15);
	//linha += " | ";
	//linha += Te2String(c3.y(), 15);
	//linha += " | ";
	//linha += Te2String(dx1y2 - dy1x2, 15);
	//linha += "\n";

	//TeWriteToFile("c:\\log_ccw.txt", linha, "aw");


	return TeNOTURN;
}

/*! \fn inline bool TeBoxIntersects(const TeCoord2D& a, const TeCoord2D& b, const TeCoord2D& c, const TeCoord2D& d)
    \brief Tells if the box of segments intersects.
    \param a  The first coordinate of the first segment.
	\param b  The second coordinate of the first segment.
	\param c  The first coordinate of the second segment.
	\param d  The second coordinate of the second segment.
*/
inline bool TeBoxIntersects(const TeCoord2D& a, const TeCoord2D& b, const TeCoord2D& c, const TeCoord2D& d)
{	
	/* X bound box test */
	if(d.x() > c.x()) 
	{
		if(b.x() < a.x())
		{
			if(TeGeometryAlgorithmsPrecision::IsGreater(c.x(), a.x()))
				return false;	// don't intersect => S2 is to the right of S1

			if(TeGeometryAlgorithmsPrecision::IsGreater(b.x(), d.x())) 
				return false;	// don't intersect => S2 is to the left of S1
		}
		else
		{
			if(TeGeometryAlgorithmsPrecision::IsGreater(c.x(), b.x()))
				return false;	// don't intersect => S2 is to the right of S1

			if(TeGeometryAlgorithmsPrecision::IsGreater(a.x(), d.x())) 
				return false;	// don't intersect => S2 is to the left of S1
		}			
	} 
	else 
	{
		if(b.x() < a.x())
		{
			if(TeGeometryAlgorithmsPrecision::IsGreater(d.x(), a.x()))
				return false;	// don't intersect => S2 is to the right of S1

			if(TeGeometryAlgorithmsPrecision::IsGreater(b.x(), c.x()))
				return false;	// don't intersect => S2 is to the left of S1
		}
		else
		{
			if(TeGeometryAlgorithmsPrecision::IsGreater(d.x(), b.x()))
				return false;	// don't intersect => S2 is to the right of S1

			if(TeGeometryAlgorithmsPrecision::IsGreater(a.x(), c.x()))
				return false;	// don't intersect => S2 is to the left of S1
		}
	}
	
	/* Y bound box test */
	if(d.y() > c.y()) 
	{
		if(b.y() < a.y())
		{
			if(TeGeometryAlgorithmsPrecision::IsGreater(c.y(), a.y()))
				return false; // don't intersect => S2 is above of S1

			if(TeGeometryAlgorithmsPrecision::IsGreater(b.y(), d.y())) 
				return false; // don't intersect => S1 is above of S2
		}
		else
		{
			if(TeGeometryAlgorithmsPrecision::IsGreater(c.y(), b.y()))
				return false; // don't intersect => S2 is above of S1

			if(TeGeometryAlgorithmsPrecision::IsGreater(a.y(), d.y())) 
				return false; // don't intersect => S1 is above of S2
		}
	} 
	else 
	{
		if(b.y() < a.y())
		{
			if(TeGeometryAlgorithmsPrecision::IsGreater(d.y(), a.y()))
				return false; // don't intersect => S2 is above of S1

			if(TeGeometryAlgorithmsPrecision::IsGreater(b.y(), c.y())) 
				return false; // don't intersect => S1 is above of S2
		}
		else
		{
			if(TeGeometryAlgorithmsPrecision::IsGreater(d.y(), b.y()))
				return false; // don't intersect => S2 is above of S1

			if(TeGeometryAlgorithmsPrecision::IsGreater(a.y(), c.y())) 
				return false; // don't intersect => S1 is above of S2
		}
	}

	return true;
}

struct TeBoundaryIP
{

	vector<TeCoord2D> coords_;

	unsigned int redSegNum_;
	unsigned int redPartNum_;

	unsigned int blueSegNum_;
	unsigned int bluePartNum_;	
};

typedef vector<TeBoundaryIP> TeVectorBoundaryIP;

/*! \fn inline bool TeIntersection(const TeCoord2D& a, const TeCoord2D& b, const TeCoord2D& c, const TeCoord2D& d, TeIntersCoordsVec& coords, TeSegmentIntersectionType& intersectionType)
    \brief Returns the intersection point of the segments.
    \param a                 The first coordinate of the first segment.
	\param b                 The second coordinate of the first segment.
	\param c                 The first coordinate of the second segment.
	\param d                 The second coordinate of the second segment.
	\param coords            The intersection coordinates (0, 1 or 2).
	\param intersectionType  An intersection may be proper or improper.
*/
inline bool TeIntersection(const TeCoord2D& a, const TeCoord2D& b, const TeCoord2D& c, const TeCoord2D& d, TeBoundaryIP& ips, TeSegmentIntersectionType& intersectionType)
{
	if(TeEquals(a, b))
		return false;

	if(TeEquals(c, d))
		return false;

	if(TeBoxIntersects(a, b, c, d))
	{
		if((TeEquals(a, c) || TeEquals(a, d)) && (TeEquals(b, c) || TeEquals(b, d)))
		{
			intersectionType = TeImproperIntersection;

			ips.coords_.push_back(a);
			ips.coords_.push_back(b);

			return true;
		}

		bool between1 = false;
		short sign1 = TeCCW(a, b, c, between1);

		bool between2 = false;
		short sign2 = TeCCW(a, b, d, between2);

		bool between3 = false;
		short sign3 = TeCCW(c, d, a, between3);

		bool between4 = false;
		short sign4 = TeCCW(c, d, b, between4);

		if((sign1 * sign2) <= 0 && (sign3 * sign4 <= 0))	// if there is an intersection
		{
			intersectionType = TeProperIntersection;

			if(between1)
			{			
				intersectionType = TeImproperIntersection;
				ips.coords_.push_back(c);
			}
			
			if(between2)
			{
				intersectionType = TeImproperIntersection;
				ips.coords_.push_back(d);
			}

			if(between3 && !TeEquals(a, c) && !TeEquals(a, d))
			{
				intersectionType = TeImproperIntersection;
				ips.coords_.push_back(a);

				return true;
			}
			
			if(between4 && !TeEquals(b, c) && !TeEquals(b, d))
			{
				intersectionType = TeImproperIntersection;
				ips.coords_.push_back(b);

				return true;
			}

			if(intersectionType == TeImproperIntersection)
				return true;

			double denominator = (d.y() - c.y()) * (b.x() - a.x()) - (d.x() - c.x()) * (b.y() - a.y());
			
			if(denominator == 0.0)	// parallel can not occur here any more! I expect this is true!
				return false;		// ERRO!!!!!!! Lanar Exceo: Montar um esquema de excees

			// parameters
			double Ua = ((d.x() - c.x()) * (a.y() - c.y()) - (d.y() - c.y()) * (a.x() - c.x())) / denominator; 
			double Ub = ((b.x() - a.x()) * (a.y() - c.y()) - (b.y() - a.y()) * (a.x() - c.x())) / denominator; 

			if(Ua > 0.0 && Ua < 1.0  && Ub > 0.0 && Ub < 1.0)
			{
				ips.coords_.push_back(TeCoord2D(a.x() + Ua * (b.x() - a.x()), a.y() + Ua * (b.y() - a.y())));
				return true;
			}
		}
	}

	intersectionType = TeImproperIntersection;

	return false;
}

/*! \fn inline bool TeIntersects(const TeCoord2D& a, const TeCoord2D& b, const TeCoord2D& c, const TeCoord2D& d, TeSegmentIntersectionType& intersectionType)
    \brief Tells if two segments intersects.
    \param a                 The first coordinate of the first segment.
	\param b                 The second coordinate of the first segment.
	\param c                 The first coordinate of the second segment.
	\param d                 The second coordinate of the second segment.
	\param intersectionType  An intersection may be proper or improper.
*/
inline bool TeIntersects(const TeCoord2D& a, const TeCoord2D& b, const TeCoord2D& c, const TeCoord2D& d, TeSegmentIntersectionType& intersectionType)
{	
	if(TeBoxIntersects(a, b, c, d))
	{
		if((TeEquals(a, c) || TeEquals(a, d)) && (TeEquals(b, c) || TeEquals(b, d)))
		{
			intersectionType = TeImproperIntersection;

			return true;
		}

		bool between1 = false;
		short sign1 = TeCCW(a, b, c, between1);

		bool between2 = false;
		short sign2 = TeCCW(a, b, d, between2);

		bool between3 = false;
		short sign3 = TeCCW(c, d, a, between3);

		bool between4 = false;
		short sign4 = TeCCW(c, d, b, between4);

		if((sign1 * sign2) <= 0 && (sign3 * sign4 <= 0))	// if there is an intersection
		{
			intersectionType = TeProperIntersection;

			if(between1)
			{			
				intersectionType = TeImproperIntersection;

				return true;
			}
			
			if(between2)
			{
				intersectionType = TeImproperIntersection;

				return true;
			}

			if(between3 && !TeEquals(a, c) && !TeEquals(a, d))
			{
				intersectionType = TeImproperIntersection;

				return true;
			}
			
			if(between4 && !TeEquals(b, c) && !TeEquals(b, d))
			{
				intersectionType = TeImproperIntersection;

				return true;
			}

			return true;
		}
	}
		
	return false;
}

/*! \fn bool TeSafeIntersections(const TePolygonSet& redPols, const TePolygonSet& bluePols, TeReportVector& report)
    \brief Verifies if there is an intersection between two given polygonsets.
    \param redPols  The first polygonset to test.
	\param bluePols The second polygonset to test.
	\param report   A list with the intersection points sorted according to redPols segments.
	
    This is a lazy algorithm. It can be used, for example, in intersections between a box and a line.
	It is O(n*m) - where n and m are the numbers of segments in the first and second polygonsets.
 */
bool TeSafeIntersections(const TePolygonSet& redPols, const TePolygonSet& bluePols, TeVectorBoundaryIP& report);

/*! \fn bool TeSafeIntersections(const TeLine2D& redLine, const TeLine2D& blueLine, TeReportVector& report, const unsigned int& redObjId = 0, const unsigned int& blueObjId = 0, const bool& sortReport = true)
    \brief Verifies if there is an intersection between two given lines.
    \param redLine  The first line to test.
	\param blueLine The second line to test.
	\param report   A list with the intersection points sorted according to lred segments.
	\param redObjId Red line object id
	\param blueObjId Blue line object id
	\param sortReport Flag to indicate that report should be sorted
	
    This is a lazy algorithm. It can be used, for example, in intersections between a box and a line.
	It is O(n*m) - where n and m are the numbers of segments in the first and second lines.
 */
bool TeSafeIntersections(const TeLine2D& redLine, const TeLine2D& blueLine, TeVectorBoundaryIP& report, const unsigned int& redObjId = 0, const unsigned int& blueObjId = 0);

/*! \fn bool TeIntersects(const TeLine2D& redLine, const TeLine2D& blueLine)
    \brief Returns true if the lines intersects.
	\param redLine			The line to test.
	\param blueLine			The line to test.

 */
bool TeIntersects(const TeLine2D& redLine, const TeLine2D& blueLine);



}	// end namespace TeINTERSECTOR2

#endif //__TERRALIB_INTERNAL_INTERSECTOR2_H

