/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeRasterRemap.h"
#include "TeDecoderDatabase.h"
#include "TeRaster.h"
#include "TeDecoderMemory.h" 
#include "TeRasterParams.h"
#include "TeProgress.h"
#include "TeVectorRemap.h"
#include "TeImportRaster.h"


TeLayer* 
TeImportRaster (const string& layerName, TeRaster* rasterIn, TeDatabase* database)
{
	if (!database || layerName.empty() || !rasterIn || 
		(rasterIn->status() != TeREADYTOREAD && rasterIn->status() != TeREADYTOWRITE))
		return 0;

	// find a valid layer name
	string newLayerName = layerName;
	TeLayerMap& layerMap = database->layerMap();
	TeLayerMap::iterator it;
	bool flag = true;
	int n = 1;
	while (flag)
	{
		for (it = layerMap.begin(); it != layerMap.end(); ++it)
		{
			if (TeStringCompare(it->second->name(),newLayerName))
				break;
		}
		if (it == layerMap.end())
			flag = 0;
		else
			newLayerName = layerName + "_" +Te2String(n);
		n++;	
	}

	TeLayer* newLayer = new TeLayer(newLayerName,database,rasterIn->projection());
	if (newLayer->id() <= 0 )
		return 0;				// couldnt create new layer

	unsigned int bw = rasterIn->params().ncols_;
	unsigned int bh = 1;
	if (rasterIn->params().nlines_ > 128 || rasterIn->params().ncols_ > 128)
	{
		bw = 128;
		bh = 128;
	}

	bool res = TeImportRaster(newLayer, rasterIn, bw, bh, TeNoCompression, 
		       "", rasterIn->params().dummy_[0], false, TeNoExpansible);
	if (res)
		return newLayer;

	database->deleteLayer(newLayer->id());
	delete newLayer;
	return 0;
}


// This function mosaics an input raster to one previsously existing in a TerraLib layer
bool TeMosaicRaster(TeRaster* rasterIn, TeLayer* layer,  const string& objectId)
{
	// layer must exist and input raster should be read to be read
	if (!layer || !rasterIn  || rasterIn->status() != TeREADYTOREAD)
		return false;
	
	TeRaster* rasterOut;
	if (!objectId.empty())	// if object id informed try to get raster associated to it
		rasterOut = layer->raster(objectId,'w');
	else					// else try to get first one found
		rasterOut = layer->raster();

	if (!rasterOut)			// no raster asked or no raster at all
		return false;				// can not do mosaic

	// allows mosaic of same photometric types only
	if (rasterOut->params().photometric_[0] != rasterIn->params().photometric_[0])
		return false;

	// increases output box to: input raster box + output raster box 
	TeBox boxIn = rasterIn->params().box();

	// if necessary remap input box to output projection 
	if (!(*(rasterIn->projection()) == *(rasterOut->projection())))
		boxIn = TeRemapBox(boxIn,rasterIn->projection(),rasterOut->projection());

	// adjust it to expansible values
	boxIn = adjustToCut(boxIn, rasterOut->params().blockWidth_*rasterOut->params().resx_, 
							   rasterOut->params().blockHeight_*rasterOut->params().resy_);

	// sum stored and input boxes
	TeBox newBox = rasterOut->params().boundingBox();
	updateBox(newBox,boxIn);

	// update box in parameters of the raster and it's decoder
	rasterOut->params().boundingBoxResolution(newBox.x1_,newBox.y1_,newBox.x2_,newBox.y2_,
											  rasterOut->params().resx_, rasterOut->params().resy_);

	rasterOut->decoder()->params().boundingBoxResolution(newBox.x1_,newBox.y1_,newBox.x2_,newBox.y2_,
											  rasterOut->params().resx_, rasterOut->params().resy_);
	rasterOut->params().nTilesInMemory_ = 0;
	
	// creates a remapping from input raster to output raster
	TeRasterRemap remap(rasterIn, rasterOut);
	if (remap.apply(true))			// o.k.
	{
		TeRasterParams rp = rasterOut->params();

		// atualizes the representation box in memory
		TeRepresentation* repp = layer->getRepresentation(TeRASTER);
		TeBox boxNew = rp.box();
		updateBox(repp->box_,rp.box());
		repp->nCols_ = rp.ncols_;
		repp->nLins_ = rp.nlines_;
		repp->resX_ = rp.resx_;
		repp->resY_ = rp.resy_;

		// atualizes representation in the database
		layer->database()->updateRasterRepresentation(layer->id(),rp,objectId);
		layer->updateLayerBox(rp.boundingBox());
		rasterOut->clear();
		return true;
	}
	return false;
}


bool TeImportRaster (TeLayer* layer, TeRaster* rasterIn, 
					 unsigned int bWidth, unsigned int bHeight,TeCompressionMode compress,
					 const string& objectId, double dummy, bool useDummy,
					 TeRasterTilingType indext)
{	
	if (!layer || !rasterIn)
		return false;

	string objId;
	if (objectId.empty())
		objId = "O1";
	else
		objId = objectId;

	// check if there  is a raster geometry to be modified
	string tableGeo;
	TeRepresentation* repp = 0;
	repp = layer->getRepresentation(TeRASTER);
	if (!repp || !layer->raster(objectId))	// layer doesnt have any 
	{										// or the desired raster geometry
											// a new one should be created

		TeRasterParams parOut = rasterIn->params();		// parameters of the raster being created	
														// start with the same parameters as input raster
		
		parOut.setCompressionMode(compress);		// overwrites some parameters 
		parOut.nTilesInMemory_ = 0;					// according to the specified in the interface
		parOut.blockHeight_ = bHeight;
		parOut.blockWidth_ = bWidth;
		parOut.setDummy(dummy);
		parOut.useDummy_ = useDummy;
		parOut.tiling_type_ = indext;
		
		parOut.dxI_= 0.0;							// raster stored is corrected  
		parOut.dyJ_= 0.0;							// to translation/rotation 
		parOut.dxJ_ = 0.0;
		parOut.dyI_ = 0.0;		
		
		TeBox newBox = rasterIn->params().boundingBox();
		TeProjection *projIn = rasterIn->projection();	
		TeProjection *projOut = layer->projection();
		parOut.projection(projOut);
		if (projIn && projOut && !(*projIn == *projOut))
		{
			TeBox boxIn = rasterIn->params().boundingBox();
			newBox = TeRemapBox (boxIn,projIn,projOut);
			parOut.resx_ = newBox.width()/parOut.ncols_;	// recalculates resolutions 
			parOut.resy_ = newBox.height()/parOut.nlines_;  // for the new projection
			parOut.boundingBoxResolution(newBox.x1_,newBox.y1_,newBox.x2_,newBox.y2_,parOut.resx_,parOut.resy_);
			newBox = parOut.box();
		}
	
		if (indext == TeExpansible)							// if mosaicable adjust box 
			parOut.resizeToTiling(newBox,bWidth,bHeight);	// to be cut in blocks of bWidth X bHeight (in projection units)

		parOut.decoderIdentifier_ = "DB";					// parameters of the decoder
		parOut.fileName_ = "RasterLayer" + Te2String(layer->id()) + "_R_" + objId;

		if ((parOut.photometric_[0] == TeRASTERPALETTE 
           || parOut.photometric_[0] == TeRASTERKERNEL))
			parOut.lutName_ = parOut.fileName_ + "_LUT";

		if (!layer->addRasterGeometry(parOut,objId))		// creates the empty raster geometry
			return false;
		
		TeRaster* rasterOut = layer->raster(objId,'w');
		TeRasterRemap remap(rasterIn, rasterOut);
		if (remap.apply(true))			// o.k.
		{
			TeRasterParams rp = rasterOut->params();

			// atualizes the representation box in memory
			TeRepresentation* repp = layer->getRepresentation(TeRASTER);
			TeBox boxNew = rp.box();
			updateBox(repp->box_,rp.box());
			repp->nCols_ = rp.ncols_;
			repp->nLins_ = rp.nlines_;
			repp->resX_ = rp.resx_;
			repp->resY_ = rp.resy_;

			// atualizes representation in the database
			layer->database()->updateRasterRepresentation(layer->id(),rp,objectId);
			layer->updateLayerBox(rp.boundingBox());
			rasterOut->clear();
			return true;
		}
		return false;
	}
	else
	{
		// layer contains already a raster representation associated to the
		// object id: calls mosaic operation
		return TeMosaicRaster(rasterIn,layer,objId);
	}
}


bool 
TeBuildLowerResolution(TeLayer* layer, TeRaster* rasterIn, int resFac, const string& objectId)
{
	if (!layer || !rasterIn)	// layer and input raster pointers shouldn't be null
		return false;

	string objId;				// if empty try to get the default object identification
	if (objectId.empty())
		objId = "O1";
	else
		objId = objectId;	
	
	string tableName;
	TeRepresentation* repp = layer->getRepresentation(TeRASTER);
	if (repp)
	{
		tableName = layer->database()->getRasterTable(layer->id(), objId);
		if (tableName.empty())	
			return false;		// layer doesnt have a raster geometry to this object id
	}
	else					
		return false;			// layer doesnt have any raster geometry

	// retrieves the original raster
	TeRaster* originalRaster = layer->raster(objId);
	TeBox bbOriginalRes = originalRaster->params().boundingBox();

	// builds some parameters relative to the new resolution
	TeRasterParams parNewRes = originalRaster->params();
	parNewRes.resolution_ = resFac;
	parNewRes.mode_ = 'w';
	parNewRes.boundingBoxResolution(bbOriginalRes.x1_, bbOriginalRes.y1_,bbOriginalRes.x2_, bbOriginalRes.y2_,
									parNewRes.resx_*resFac, parNewRes.resy_*resFac);

	// adjust box to tiling in geographical coordinates
	if (originalRaster->params().tiling_type_ == TeExpansible)
	{
		TeBox b = parNewRes.boundingBox();
		int w = parNewRes.blockWidth_;
		int h = parNewRes.blockHeight_;
		parNewRes.resizeToTiling(b, w, h);
	} 


	// create a decoder to raster in a TerraLib database 
	TeDecoderDatabase *dbDecoder = new TeDecoderDatabase(parNewRes);
	dbDecoder->setDB (layer->database());

	TeRaster* rasterOut = new TeRaster();
	rasterOut->setDecoder(dbDecoder);
	rasterOut->init();

	// remap;
	TeRasterRemap remap;
	remap.setInput(rasterIn);
	remap.setOutput(rasterOut);
	bool status = remap.apply(true);
	rasterOut->clear();
	delete dbDecoder;
	return status;
}
