/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeBufferRegion.h"
#include "TeOverlay.h"
#include "TeGeometry.h" 
#include "TeGeometryAlgorithms.h"

#include <map>
#include <algorithm>
#include <cmath>

using namespace std;

namespace TeBUFFERREGION
{

//---------------- Auxiliary Constants ----------------//
const double pi      = 3.14159265358979323846;

//---------------- Auxiliary Functions ----------------//
// A partir de dois pontos, gera um polgono no sentido anti-horrio
void TeGenerateBufferPolygon(TeCoord2D first, TeCoord2D last, const double& bufferDistance, const short& numPoints, TePolygon& pol)
{
	double xLength  = 0.0;
	double yLength  = 0.0;
	double x        = 0.0;
	double y        = 0.0;
	double val_ang  = 0.0;
	double ang		= 0.0;

	TeLine2D contourLine;

	if(TeGeometryAlgorithmsPrecision::IsGreater(first.x(), last.x()))
	    swap(first, last);
	else
		if(TeGeometryAlgorithmsPrecision::IsEqual(first.x(), last.x()))
		{
			if(TeGeometryAlgorithmsPrecision::IsSmaller(first.y(), last.y()))
				swap(first, last);
		}


	xLength = last.x() - first.x();

	if(first.y() < last.y())
		yLength = last.y() - first.y();
	else
		yLength = first.y() - last.y();

	double segmentLength = TeDistance(first, last);

	val_ang = xLength / segmentLength;

	if(val_ang > 1.0)
		val_ang = 1.0;
	else
		if(val_ang < -1.0)
			val_ang = -1.0;

	if(TeGeometryAlgorithmsPrecision::IsEqual(val_ang, 0.0))
		ang = 0.0;
	else
	{
		if(TeGeometryAlgorithmsPrecision::IsEqual(first.y(), last.y()))
			ang = (pi/2.0);
		else
			if(TeGeometryAlgorithmsPrecision::IsGreater(first.y(), last.y()))
				ang = asin(val_ang);
			else
				ang = acos(val_ang) + (pi/2.0);
	}

	// Para cada extremo, calcula uma semi-circunferncia
	for(int i = 1; i < numPoints; ++i)
	{
		double s = sin(ang);
		double c = cos(ang);
		
		x = first.x() + bufferDistance * c ; 
		y = first.y() + bufferDistance * s ;

		contourLine.add(TeCoord2D(x, y));

		ang += (pi / (numPoints - 1));
	}

	x = first.x() + bufferDistance * cos(ang) ; 
	y = first.y() + bufferDistance * sin(ang) ;

	contourLine.add(TeCoord2D(x, y));

	for(int j = 0; j < numPoints; ++j)
	{		
		double s = sin(ang);
		double c = cos(ang);
		
		x = last.x() + bufferDistance * c ; 
		y = last.y() + bufferDistance * s ;

		contourLine.add(TeCoord2D(x, y));

		ang += (pi / (numPoints - 1));
	}

	contourLine.add(contourLine[0]);

	pol.add(TeLinearRing(contourLine));

	return;
}

}	// end namespace TeBUFFERREGION

//---------------- Buffer Algorithm ----------------//
bool TeBUFFERREGION::TeBufferRegion(const TeCoord2D& coord, const double& bufferDistance, const short& numPoints, TePolygon& pol)
{
	TeLine2D contourLine;

	contourLine.add(TeCoord2D((coord.x() + bufferDistance), (coord.y())));

	double initialAng = 0.0;

	const double two_pi = 2 * pi;

	double x = 0.0;
	double y = 0.0;

	for(int i = 1; i < numPoints; ++i)
	{
		initialAng += (two_pi / numPoints);
		
		double s = sin(initialAng);
		double c = cos(initialAng);
		
		x = coord.x() + bufferDistance * c ; 
		y = coord.y() + bufferDistance * s ;

		contourLine.add(TeCoord2D(x, y));
	}

	contourLine.add(TeCoord2D((coord.x() + bufferDistance), (coord.y())));

	pol.add(TeLinearRing(contourLine));

	return true;
}


bool TeBUFFERREGION::TeBufferRegion(const TeLine2D& line, const double& bufferDistance, const short& numPoints, TePolygonSet& ps)
{
	ps.clear();

	// Generate first polygon
	TePolygon pAux;
	TeGenerateBufferPolygon(line[0], line[1], bufferDistance, numPoints, pAux);
	ps.add(pAux);

	unsigned int nStep = line.size() - 1;
	for(unsigned int i = 1; i < nStep; ++i)
	{
		// Generate polygon from segment
		TePolygon p;
		TeGenerateBufferPolygon(line[i], line[i+1], bufferDistance, numPoints, p);

		//ps.add(p);
		// Gets union
		TePolygonSet psAux;
		psAux.add(p);

		TePolygonSet psAux2;

		if(TeOVERLAY::TeUnion(psAux, ps, psAux2) == false)
			return false;

		ps = psAux2;		
	}

	return true;
}

bool TeBUFFERREGION::TeBufferRegion(const TePolygon& pol, const double& bufferDistance, const short& numPoints, TePolygonSet& ps)
{
	return TeBufferRegion(pol[0], bufferDistance, numPoints, ps);
}

