/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

/*! \file TeBox.h
    This file contains structures and definitions to deal with a box
*/
#ifndef  __TERRALIB_INTERNAL_BOX_H
#define  __TERRALIB_INTERNAL_BOX_H

#include "TeCoord2D.h"
#include "TeDefines.h"

class TeGeometry;
class TePolygon;

//! Provides support for dealing with a rectangular a box. Used by all geometrical representations
/*!
 \sa
	TeCoord2D
 */
struct TeBox {

	enum TeBoxCorner { TeLOWERLEFT=0, TeUPPERLEFT=1, TeLOWERRIGHT=2, TeUPPERRIGHT=3 };

//! Default constructor 
/*!
	\param x1 X value of the lower left box coordinate 
	\param y1 Y value of the lower left box coordinate 
	\param x2 X value of the upper right box coordinate 
	\param y2 Y value of the upper right box coordinate 
*/
	TeBox ( const double& x1 = TeMAXFLOAT, const double y1 = TeMAXFLOAT,
		  const double& x2 = -TeMAXFLOAT, const double y2 = -TeMAXFLOAT ):
		  x1_ ( x1 ), y1_ ( y1 ), x2_ ( x2 ), y2_ ( y2 ) {}

//! Constructor from corners as TeCoord2D objects
/*!
	\param p1 lower left box coordinate
	\param p2 upper right box coordinate
*/
	TeBox( const TeCoord2D& p1, const TeCoord2D& p2 ):
		x1_ ( p1.x() ), y1_ ( p1.y() ),
		x2_ ( p2.x() ), y2_ ( p2.y() )
		{}

//! Copy constructor
	TeBox ( const TeBox& rhs )
	{
		x1_ = rhs.x1_;
		x2_ = rhs.x2_;
		y1_ = rhs.y1_;
		y2_ = rhs.y2_;
	}

//! Operator =
	TeBox& operator=(const TeBox& rhs)
	{
		if ( this == &rhs )
			return *this;
		else
		{
		x1_ = rhs.x1_;
		x2_ = rhs.x2_;
		y1_ = rhs.y1_;
		y2_ = rhs.y2_;
		}
		return *this;
	}

//!	Returns TRUE if current TeBox is equal to box; otherwise returns FALSE 
	bool operator== (const TeBox& box) const 
	{return (  ( fabs (y1_ - box.y1_) < TePrecision::instance().precision() ) 
			&& ( fabs (x1_ - box.x1_) < TePrecision::instance().precision() ) 
			&& ( fabs (y2_ - box.y2_) < TePrecision::instance().precision() ) 
			&& ( fabs (x2_ - box.x2_) < TePrecision::instance().precision() ) 
			); }

//! Returns the coordinate corresponding to the geometric center of the  box
	TeCoord2D center() const
	{	
		TeCoord2D p ( (  ( x1_ + x2_ ) / 2.), ( ( y1_ + y2_ ) / 2.) );
		return p;
	}	

//!	Returns TRUE if current box is valid; otherwise returns FALSE 
	bool isValid() const
	{
		if (x1_ == TeMAXFLOAT || y1_ == TeMAXFLOAT || 
		    x2_ == -TeMAXFLOAT || y2_ == -TeMAXFLOAT  ||
			x1_ > x2_ || y1_ > y2_)
			return false;
		return true;
	}

//! Returns the x component of the lower left corner
	double x1 () const 
	{ return x1_;}

//! Returns the y component of the lower left corner
	double y1 () const 
	{ return y1_;}

//! Returns the x component of the upper right corner
	double x2 () const 
	{ return x2_;}

//! Returns the y component of the upper right corner
	double y2 () const 
	{ return y2_;}

//! Returns the box width 
	double width () const 
	{ return x2_ - x1_; }
	
//! Returns the box height 
	double height () const 
	{ return y2_ - y1_; }
	
//! Returns the lower left coordinate of the box 
	TeCoord2D lowerLeft() const 
	{ return TeCoord2D ( x1_, y1_ );}

//! Returns the upper right coordinate of the box 
	TeCoord2D upperRight() const
	{ return TeCoord2D (x2_,y2_);}

//! Destructor
	~TeBox() {}  
	
	double x1_, //!< x coordinate of the lower left corner
		   y1_,	//!< y coordinate of the lower left corner
		   x2_,	//!< x coordinate of the upper right corner
		   y2_;	//!< y coordinate of the upper right corner	

};

// ALGORITHMS THAT USE A BOX

/*! \fn void updateBox void updateBox ( TeBox& box, const TeCoord2D& pt ) 
   \brief update a box to include a coordinate
 */
void updateBox ( TeBox& box, const TeCoord2D& pt );  

/*! \fn void updateBox void updateBox ( TeBox& box, const TeBox& other ) 
   \brief update a box to include another box
 */
void updateBox ( TeBox& box, const TeBox& other );

/*! \fn void updateBox ( TeBox& box, const TeGeometry& geo ) 
   \brief update a box to include a geometry
 */
void updateBox ( TeBox& box, const TeGeometry& geo );


// ZOOMING FUNCTIONS 

/*! \fn void zoomIn void zoomIn ( TeBox& box, double t = .8) 
   \brief increases the box by a factor t
 */
void zoomIn ( TeBox& box, double t = .8); 

/*! \fn  void zoomOut void zoomOut ( TeBox& box, double t = .8) 
   \brief decreases box by a factor of t
 */
void zoomOut ( TeBox& box, double t = .8);
 
/*! \fn  TeBox makeBox(double x1, double y1, double x2, double y2, const double& tol = 0.0) 
   \brief builds a box 
 */
TeBox makeBox(double x1, double y1, double x2, double y2, const double& tol = 0.0);

/*! \fn  TeBox adjustToCut(TeBox& box, double bWidth, double bHeight) 
   \brief finds the correspondent smallest box that allows a box to be cut in blocks of a given size 
 */
TeBox adjustToCut(TeBox& box, double bWidth, double bHeight);

/*! \fn TePolygon polygonFromBox(TeBox& bb) 
   \brief builds a TePolygon geometry from a box
 */
TePolygon polygonFromBox( TeBox& bb );

#endif


