/************************************************************************************
TerraView - visualization and exploration of geographical databases
using TerraLib.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.
This file is part of TerraView. TerraView is free software; you can redistribute it 
and/or modify it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

You should have received a copy of the GNU General Public License
along with TerraView.
The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The software provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use of
this program and its documentation.
*************************************************************************************/

#include "TeGeoProcessingFunctions.h"
#include "TeQuerier.h"
#include "TeQuerierParams.h"
#include "TeOverlay.h"
#include "TeProgress.h"
#include "TeVectorRemap.h"
#include "TeApplicationUtils.h"
#include "TeCellAlgorithms.h"
#include "TeAsciiFile.h"


bool changeAttrList(TeAttributeList& attrList)
{
	bool change = false;
	
	for(int i=0; i<(int)attrList.size(); ++i) 
	{
		string& nameOr = attrList[i].rep_.name_;
	
		attrList[i].rep_.isAutoNumber_ = false;
		attrList[i].rep_.isPrimaryKey_ = false;
		
		//replace "." for "_"
		string::size_type f = nameOr.find(".");
		if(f != string::npos)
		{
			nameOr.replace(f, 1, "_");
			change = true;
		}

		string nameTemp = TeConvertToUpperCase(nameOr);

		int count = 1;
		int j = i+1;
		while(j<(int)attrList.size())
		{
			if((nameTemp == TeConvertToUpperCase(attrList[j].rep_.name_)) && (i!=j))
			{
				nameTemp = nameTemp +"_"+ Te2String(count);
				nameOr = nameOr +"_"+ Te2String(count);

				j = 0;
				change = true;
				++count;
			}
			++j;
		}
	}

	return change;
}


bool multiGeometryToLayer(TeMultiGeometry& mGeom, const string& newId, TeLayer* newLayer, TeProjection* proj=0)
{
	bool flag = false;
	TeProjection* projLayer = newLayer->projection();
	
	if(mGeom.hasPolygons())  
	{
		TePolygonSet polSet, polSetTemp;
		mGeom.getGeometry (polSetTemp);

		if((proj) && (!((*proj) == (*projLayer))))
		{
			for(int i=0; i<(int)polSetTemp.size(); ++i)
			{
				TePolygon poly = polSetTemp[i];
				TePolygon pout;
				TeVectorRemap (poly, proj, pout, projLayer);
				pout.objectId(poly.objectId());
				polSet.add (pout);
			}
		}
		else
			polSet = polSetTemp;


		polSet.objectId(newId);
		flag = newLayer->addPolygons(polSet);
	} 
	else if (mGeom.hasLines())
	{
		TeLineSet lineSet, lineSetTemp;
		mGeom.getGeometry (lineSetTemp);

		if((proj) && (!((*proj) == (*projLayer))))
		{
			for(int i=0; i<(int)lineSetTemp.size(); ++i)
			{
				TeLine2D line = lineSetTemp[i];
				TeLine2D lout;
				TeVectorRemap (line, proj, lout, projLayer);
				lout.objectId(line.objectId());
				lineSet.add (lout);
			}
		}
		else
			lineSet = lineSetTemp;

		lineSet.objectId (newId);
		flag = newLayer->addLines(lineSet);
	}
	else if (mGeom.hasPoints())
	{
		TePointSet pointSet, pointSetTemp;
		mGeom.getGeometry (pointSetTemp);

		if((proj) && (!((*proj) == (*projLayer))))
		{
			for(int i=0; i<(int)pointSetTemp.size(); ++i)
			{
				TePoint point = pointSetTemp[i];
				TePoint pout;
				TeVectorRemap (point, proj, pout, projLayer);
				pout.objectId(point.objectId());
				pointSet.add (pout);
			}
		}
		else
			pointSet = pointSetTemp;

		pointSet.objectId (newId);
		flag = newLayer->addPoints(pointSet);
	}
	else if (mGeom.hasCells())
	{
		TeCellSet cellSet, cellSetTemp;
		mGeom.getGeometry (cellSetTemp);
	
		if((proj) && (!((*proj) == (*projLayer))))
		{
			for(int i=0; i<(int)cellSetTemp.size(); ++i)
			{
				TeCell cell = cellSetTemp[i];
				TeCell cellout;
				TeVectorRemap (cell, proj, cellout, projLayer);
				cellout.objectId(cell.objectId());
				cellSet.add (cellout);
			}
		}
		else
			cellSet = cellSetTemp;

		cellSet.objectId (newId);
		flag = newLayer->addCells(cellSet);
	}
	else 
		return false;

	return flag;
}



bool TeGeoOpAggregation(TeLayer* newLayer, TeTheme* theme, vector<string> agregAttrVec, 
						TeGroupingAttr& attrMM, TeSelectedObjects selOb, TeAsciiFile* logFile)		
{
	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;
	int	i, j, offset, steps = 0;
	TeStatisticValMap stat;
	TeStatisticStringValMap sstat;
	vector<string> valVec;
	map<int, vector<string> > stringMap;
	map<int, vector<double> > doubleMap;
	string geoName, query;
	string layerName = newLayer->name();
	TeLayer* layer = theme->layer();
	TeDatabase* db = layer->database();
	string CT = theme->collectionTable();
	string OID = CT + ".c_object_id";

	TeAttributeList aList = theme->sqlAttList();
	TeAttribute agregTeAttribute;
	agregTeAttribute.rep_.type_ = TeSTRING;
	agregTeAttribute.rep_.numChar_ = 200;
	agregTeAttribute.rep_.isPrimaryKey_ = false;
	agregTeAttribute.rep_.isAutoNumber_ = false;

	map<int, string> legendLabelMap;
	int legIndex = -1;

	for(j=0; j<(int)agregAttrVec.size(); ++j)
	{
		string name;
		if(agregAttrVec[j] == "CURRENT LEGEND")
		{
			agregAttrVec[j] = CT + ".c_legend_id";
			name = "agreg_leg_";
			legIndex = j;
		}
		else
		{	
			for(i=0; i<(int)aList.size(); ++i)
			{
				name = aList[i].rep_.name_;
				if(name == agregAttrVec[j])
				{
					int f = name.find(".");
					if(f >= 0)
						name.replace(f, 1, "_");
					break;
				}
			}
		}
		if(j==0)
			agregTeAttribute.rep_.name_ += name;
		else
			agregTeAttribute.rep_.name_ += "_" + name;
	}
	if(agregTeAttribute.rep_.name_.size() > 30)
		agregTeAttribute.rep_.name_.erase(30, agregTeAttribute.rep_.name_.size()-30);

	if(legIndex > -1)
	{
		TeLegendEntryVector& legVec = theme->legend();
		for(i=0; i<(int)legVec.size(); ++i)
			legendLabelMap[legVec[i].id()] = legVec[i].label();
	}

	TeAttributeList attList;

	TeAttribute at;
	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "object_id_";
	at.rep_.numChar_ = 255;
	at.rep_.isPrimaryKey_ = true;
	at.rep_.isAutoNumber_ = false;
	attList.push_back(at);

	attList.push_back(agregTeAttribute);
	
	at.rep_.type_ = TeINT;
	at.rep_.name_ = "agreg_count_";
	at.rep_.numChar_ = 0;
	at.rep_.isPrimaryKey_ = false;
	attList.push_back(at);	

	TeGroupingAttr::iterator it;
	string ss;
	for(it=attrMM.begin(); it!=attrMM.end(); ++it)
	{
		string s = it->first.name_;
		TeStatisticType sType = it->second;

		ss = getStringFromStatistic(sType);
		string sss = s + "_" + ss;

		if(((db->dbmsName()=="OracleAdo") || 
			(db->dbmsName()=="OracleSpatial") ) &&
			(sss.size() > 30))
		{
			int s = sss.size(); 
			at.rep_.name_ = sss.substr(s-30, s-1);
		}
		else
			at.rep_.name_ = sss;

		at.rep_ = it->first;
		at.rep_.isPrimaryKey_ = false;
		if(at.rep_.type_ == TeINT)
			at.rep_.type_ = TeREAL;
		attList.push_back(at);	
	}

	changeAttrList(attList);
	TeTable attTable(newLayer->name(), attList, "object_id_", "object_id_");

	if(!newLayer->createAttributeTable(attTable))
        return false;

	string queryAtr  = "SELECT " + OID;

	for(i=0; i<(int)agregAttrVec.size(); ++i)
		queryAtr += ", " + agregAttrVec[i];

	for(it=attrMM.begin(); it!=attrMM.end(); ++it)
	{
		string s = it->first.name_;
		bool existedName = false;
		vector<string>::iterator it2 = agregAttrVec.begin();
		while(it2!=agregAttrVec.end())
		{
			if((*it2)==s)
			{
				existedName=true;
				break;
			}
			++it2;
		}

		if((s != ss) && (!existedName))
			queryAtr += ", " + s;
		ss = s;
	}
	queryAtr += theme->sqlFrom();
	if(selOb == TeSelectedByPointing)
		queryAtr += " WHERE " + CT + ".c_object_status = 1 OR " + CT + ".c_object_status = 3";
	else if(selOb == TeSelectedByQuery)
		queryAtr += " WHERE " + CT + ".c_object_status = 2 OR " + CT + ".c_object_status = 3";
	queryAtr += " ORDER BY ";
	for(i=0; i<(int)agregAttrVec.size(); ++i)
	{
		queryAtr += agregAttrVec[i];
		if(i<(int)agregAttrVec.size()-1)
			queryAtr += ", ";
	}

	if(layer->hasGeometry(TePOLYGONS))
		geoName = layer->tableName(TePOLYGONS);
	else if(layer->hasGeometry(TeLINES))
		geoName = layer->tableName(TeLINES);
	else if(layer->hasGeometry(TePOINTS))
		geoName = layer->tableName(TePOINTS);
	else
		geoName = layer->tableName(TeCELLS);

	TeAttributeList geoAtList;
	db->getAttributeList(geoName, geoAtList);
	offset = geoAtList.size();

	if((db->dbmsName()=="OracleAdo") || (db->dbmsName()=="OracleSpatial"))
		query = "SELECT " + geoName + ".*, ATR.* FROM " + geoName + ", (" + queryAtr + ") ATR ";
	else
		query = "SELECT " + geoName + ".*, ATR.* FROM " + geoName + ", (" + queryAtr + ") AS ATR ";

	string newOID = TeGetExtension(OID.c_str());

	if(newOID.empty())
		newOID = OID;

	query += "WHERE ATR." + newOID + " = " + geoName + ".object_id";
	query += " ORDER BY ";
	for(i=0; i<(int)agregAttrVec.size(); ++i)
	{
		string newagre = TeGetExtension(agregAttrVec[i].c_str());

		if(newagre.empty())
			newagre = agregAttrVec[i];

		query += "ATR." + newagre;
		if(i<(int)agregAttrVec.size()-1)
			query += ", ";
	}
	query += ", " + geoName + ".object_id";


	if(layer->hasGeometry(TePOLYGONS))
	{
		if(db->dbmsName() != "OracleSpatial" && db->dbmsName() != "PostGIS")
			query += ", " + geoName + ".parent_id, " + geoName + ".num_holes DESC";
	}

	TeDatabasePortal* portal = db->getPortal();

	int total;
	string qc = "SELECT COUNT(*) FROM " + CT;
	if(selOb == TeSelectedByPointing)
		qc += " WHERE " + CT + ".c_object_status = 1 OR " + CT + ".c_object_status = 3";
	else if(selOb == TeSelectedByQuery)
		qc += " WHERE " + CT + ".c_object_status = 2 OR " + CT + ".c_object_status = 3";
	if(portal->query(qc) && portal->fetchRow())
		total = atoi(portal->getData(0));

	portal->freeResult();
	if(portal->query(query) && portal->fetchRow())
	{
		TeAttributeList portalAttList = portal->AttributeList();

		if(layer->hasGeometry(TePOLYGONS))
			newLayer->addGeometry(TePOLYGONS);
		else if(layer->hasGeometry(TeLINES))
			newLayer->addGeometry(TeLINES);
		else if(layer->hasGeometry(TePOINTS))
			newLayer->addGeometry(TePOINTS);
		else
			newLayer->addGeometry(TeCELLS);

		bool flag = false;
		int count = 0;			
		int objectId = 1;
		string lastOid = portal->getData(offset);
		lastOid += "aaa";
		string lastVal;

		for(i=0; i<(int)agregAttrVec.size(); ++i)
		{
			string s = portal->getData(agregAttrVec[i]);
			if(legIndex == i)
				s = legendLabelMap[atoi(s.c_str())];
			lastVal += s;
			if(i<(int)agregAttrVec.size() - 1)
				lastVal += "_";
		}
		
		TePolygonSet ps;
		TeMultiGeometry multGeom; 

		if(TeProgress::instance())
		{
			string caption = "Aggregation";
			TeProgress::instance()->setCaption(caption.c_str());
			string msg = "Executing aggregation. Please, wait!";
			TeProgress::instance()->setMessage(msg);
			TeProgress::instance()->setTotalSteps(total);
			t2 = clock();
			t0 = t1 = t2;
		}


		TePrecision::instance().setPrecision(TeGetPrecision(layer->projection()));

		do
		{	
			string oid = portal->getData(offset);
			string currVal;
			for(i=0; i<(int)agregAttrVec.size(); ++i)
			{
				string s = portal->getData(agregAttrVec[i]);
				if(legIndex == i)
					s = legendLabelMap[atoi(s.c_str())];
				currVal += s;
				if(i<(int)agregAttrVec.size() - 1)
					currVal += "_";
			}

			TePolygon p;
			TeLine2D  l;
			TePoint	  pt;
			TeCell    c;
			double	  dval;

			if(lastVal == currVal)
			{				
				if(oid != lastOid)
				{
					count++;
					for(j=3, i=offset+(int)agregAttrVec.size()+1; i<(int)portalAttList.size(); ++i, ++j)
					{
						string s = portal->getData(i);
						TeAttrDataType aType = portalAttList[i].rep_.type_;
						if(aType != TeREAL && aType != TeINT)
							stringMap[j].push_back(s);
						else
						{
							if(s.empty())
								dval = TeMAXFLOAT;
							else
								dval = atof(s.c_str());
							doubleMap[j].push_back(dval);
						}
					}
				}

				if(layer->hasGeometry(TePOLYGONS))
				{
					flag = portal->fetchGeometry(p);
					TePolygonSet psaux;
					psaux.add(p);

					TePolygonSet psauxOut;
					if(!TeOVERLAY::TeUnion(ps, psaux, psauxOut))
					{
						// when the operation returns false, write in the log a possible
						// inconsistency in the data
						if (logFile)
						{
							string mess = "Possivel inconsistencia na uniao do objeto ";
							mess += ps.objectId();
							mess += " e ";
							mess += p.objectId();
							logFile->writeString(mess);
							logFile->writeNewLine();
						}
					}
					ps = psauxOut;
					multGeom.setGeometry(ps);
				}
				else if(layer->hasGeometry(TeLINES))
				{
					flag = portal->fetchGeometry(l);
					multGeom.addGeometry(l);
				}
				else if(layer->hasGeometry(TePOINTS))
				{
					flag = portal->fetchGeometry(pt);
					multGeom.addGeometry(pt);
				}
				else
				{
					flag = portal->fetchGeometry(c);
					multGeom.addGeometry(c);
				}
			}
			else  
			{
				if(!multiGeometryToLayer(multGeom, Te2String(objectId), newLayer))
				{
					delete portal;
					return false; 
				}
				
				valVec.push_back(Te2String(objectId));
				valVec.push_back(lastVal);
				valVec.push_back(Te2String(count));

				i = 3;
				it=attrMM.begin();
				while(it != attrMM.end())
				{
					TeAttributeRep atRep = it->first;
					TeAttrDataType aType = atRep.type_;

					stat.clear();
					sstat.clear();
					if(aType == TeREAL || aType == TeINT)
						TeCalculateStatistics(doubleMap[i].begin(), doubleMap[i].end(), stat);
					else
						TeCalculateStatistics(stringMap[i].begin(), stringMap[i].end(), sstat);

					int ssize = attrMM.count(atRep);
					for(j=0; j<ssize; ++j)
					{
						TeStatisticType sType = it->second;

						if(aType == TeREAL || aType == TeINT)
						{
							double d = stat[sType];
							valVec.push_back(Te2String(d));
						}
						else
						{
							string s = sstat[sType];
							valVec.push_back(s);
						}
						it++;
					}
					i++;
				}

				TeTableRow row;
				for(i=0; i<(int)valVec.size(); ++i)
					row.push_back(valVec[i]);
				attTable.add(row);

				valVec.clear();
				stringMap.clear();
				doubleMap.clear();
				count = 0;
				ps.clear();
				multGeom.clear();
				
				if(oid != lastOid)
				{
					count++;
					for(j=3, i=offset+(int)agregAttrVec.size()+1; i<(int)portalAttList.size(); ++i, ++j)
					{
						string s = portal->getData(i);
						TeAttrDataType aType = portalAttList[i].rep_.type_;
						if(aType != TeREAL && aType != TeINT)
							stringMap[j].push_back(s);
						else
						{
							if(s.empty())
								dval = TeMAXFLOAT;
							else
								dval = atof(s.c_str());
							doubleMap[j].push_back(dval);
						}
					}
				}

				if(layer->hasGeometry(TePOLYGONS))
				{
					flag = portal->fetchGeometry(p);
					TePolygonSet psaux;
					psaux.add(p);

					TePolygonSet psauxOut;

					if(!TeOVERLAY::TeUnion(ps, psaux, psauxOut))
					{
						// when the operation returns false, write in the log a possible
						// inconsistency in the data
						if (logFile)
						{
							string mess = "Possivel inconsistencia na uniao do objeto ";
							mess += ps.objectId();
							mess += " e ";
							mess += p.objectId();
							logFile->writeString(mess);
							logFile->writeNewLine();
						}
					}
					ps = psauxOut;

					multGeom.setGeometry(ps);
				}
				else if(layer->hasGeometry(TeLINES))
				{
					flag = portal->fetchGeometry(l);
					multGeom.addGeometry(l);
				}
				else if(layer->hasGeometry(TePOINTS))
				{
					flag = portal->fetchGeometry(pt);
					multGeom.addGeometry(pt);
				}
				else
				{
					flag = portal->fetchGeometry(c);
					multGeom.addGeometry(c);
				}

				++objectId;
			}

			if((oid != lastOid) && TeProgress::instance())
			{
				steps++;
				t2 = clock();
				if (int(t2-t1) > dt)
				{
					t1 = t2;
					if(TeProgress::instance()->wasCancelled())
					{
						TeProgress::instance()->reset();
						delete portal;
						return false;
					}
					
					if((int)(t2-t0) > dt2)
						TeProgress::instance()->setProgress(steps);
				}
			}
			lastOid = oid;
			lastVal = currVal;

		}while(flag);
		if(TeProgress::instance())
			TeProgress::instance()->reset();

		if(!multGeom.empty())  
		{
			if(!multiGeometryToLayer(multGeom, Te2String(objectId), newLayer))
			{
				delete portal;
				return false; 
			}

			valVec.push_back(Te2String(objectId));
			valVec.push_back(lastVal);
			valVec.push_back(Te2String(count));

			i = 3;
			it=attrMM.begin();
			while(it != attrMM.end())
			{
				TeAttributeRep atRep = it->first;
				TeAttrDataType aType = atRep.type_;

				stat.clear();
				sstat.clear();
				if(aType == TeREAL || aType == TeINT)
					TeCalculateStatistics(doubleMap[i].begin(), doubleMap[i].end(), stat);
				else
					TeCalculateStatistics(stringMap[i].begin(), stringMap[i].end(), sstat);

				int ssize = attrMM.count(atRep);
				for(j=0; j<ssize; ++j)
				{
					TeStatisticType sType = it->second;

					if(aType == TeREAL || aType == TeINT)
					{
						double d = stat[sType];
						valVec.push_back(Te2String(d));
					}
					else
					{
						string s = sstat[sType];
						valVec.push_back(s);
					}
					it++;
				}
				i++;
			}

			TeTableRow row;
			for(i=0; i<(int)valVec.size(); ++i)
				row.push_back(valVec[i]);
			attTable.add(row);

			valVec.clear();
			stringMap.clear();
			doubleMap.clear();
			count = 0;
			ps.clear();
			multGeom.clear();
		}			

		newLayer->saveAttributeTable(attTable);
	}
	else
	{
		delete portal;
		return false;
	}

	delete portal;
	return true;
}


bool TeGeoOpAdd(TeLayer* newLayer, TeTheme* theme, TeThemeVector themeVec, 
				TeSelectedObjects selOb,TeAsciiFile* logFile)
{
	
	bool insertAttr = false;
	
	// create attribute table from main theme
	TeAttributeList attrList;
	TeAttribute at;
	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "object_id_"+ Te2String(newLayer->id());
	at.rep_.numChar_ = 255;
	at.rep_.isPrimaryKey_ = true;
	at.rep_.isAutoNumber_ = false;
	attrList.push_back(at);

	//main theme
	TeQuerierParams mainParams(true, true);
	mainParams.setParams(theme);
	mainParams.setSelecetObjs(selOb);
	TeQuerier* mainQuerier = new TeQuerier(mainParams);

	if(!mainQuerier->loadInstances())
		return false;

	//get attribute list from querier
	TeAttributeList attribs = mainQuerier->getAttrList();
	TeAttributeList::iterator it = attribs.begin();
	while(it != attribs.end())
	{
		attrList.push_back (*it);
		++it;
	}

	//verify if there are duplicate names
	changeAttrList(attrList);
	attrList[0].rep_.isPrimaryKey_ = true;  //link attribute is primary key
		
	// --------- create table 
	TeTable attrTable (newLayer->name(), attrList, attrList[0].rep_.name_, attrList[0].rep_.name_);

	if(!newLayer->createAttributeTable(attrTable))
        return false;

	int newId = 0;

	// ---------------------------------- Begin main theme
	TeSTInstance st;
	while(mainQuerier->fetchInstance(st))
	{
		TeTableRow row;
		TePropertyVector vect = st.getPropertyVector();
		row.push_back(Te2String(++newId));

		//insert os valores do tema principal
		for(int p=0; p<(int)vect.size();++p)
			row.push_back(vect[p].value_);

		attrTable.add(row);
				
		if(!multiGeometryToLayer(st.geometries(), Te2String(newId), newLayer))
			return false; 
		
		if((newId%100)==0)
		{
			//insert attributes
			newLayer->saveAttributeTable(attrTable); //save 100 records!!!
			if(attrTable.size()>0)
				insertAttr = true;
			attrTable.clear();
		}
	} 	//while fetchInstance no querier

	
	// save the last records 
	newLayer->saveAttributeTable(attrTable); 
	if(attrTable.size()>0)
		insertAttr = true;
	attrTable.clear();
	st.clear();
	delete mainQuerier;

	// ---------------------------------- End main theme

	for(int i=0; i<(int)themeVec.size(); ++i)
	{

		TeTheme* temp = themeVec[i];
		TeProjection* projTemp = temp->layer()->projection();
		
		TeQuerierParams params(true, true);

		params.setParams(temp);
		params.setSelecetObjs (selOb);
		TeQuerier* querier = new TeQuerier(params);

		if(!querier->loadInstances())
			return false;

		attribs.clear();
		attribs = querier->getAttrList();
		vector<int> indexVec;

		for(int k=0; k<(int)attrList.size(); ++k)
		{
			string name = attrList[k].rep_.name_;
			int ind = -1;
			
			for(int p=0; p<(int)attribs.size();++p)
			{
				if(TeConvertToUpperCase(name) == TeConvertToUpperCase(attribs[p].rep_.name_))
				{
					ind = p;
					break;
				}
			}

			indexVec.push_back(ind);
		}
					
		while(querier->fetchInstance(st))
		{
			TeTableRow row;
			TePropertyVector vect = st.getPropertyVector();
			row.push_back(Te2String(++newId));

			//insert the values when the names are equals
			for(int p=1; p<(int)attrList.size();++p)
			{
				int s = indexVec[p];
				if(s<0)
					row.push_back("");
				else
					row.push_back(vect[s].value_);
			}

			attrTable.add(row);
					
			if(!multiGeometryToLayer(st.geometries(), Te2String(newId), newLayer, projTemp))
				return false; 
										
			if((newId%100)==0)
			{
				//insert attributes
				newLayer->saveAttributeTable(attrTable); //save 100 records!!!
				if(attrTable.size()>0)
					insertAttr = true;
				attrTable.clear();
			}
		}  //while fetchInstance no querier
	
		newLayer->saveAttributeTable(attrTable); //save the last records!!!
		if(attrTable.size()>0)
			insertAttr = true;
		attrTable.clear();
		st.clear();
	}

	if(!newLayer->box().isValid())
		return false;

	if(!insertAttr)
		return false; 

	return true;
}

bool TeGeoOpOverlayIntersection(TeLayer* newLayer, TeTheme* theme, TeTheme* themeTrim, 
								TeSelectedObjects selOb, TeSelectedObjects selObTrim, 
								bool attrTrim, TeAsciiFile* logFile)
{
	if((!newLayer) || (!theme) || (!themeTrim))
		return false;

	bool insertAttr = false;
	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;
		
	TeLayer* layer = theme->layer();
	TeLayer* layerTrim = themeTrim->layer();

	TeProjection* proj = layer->projection();
	TeProjection* projTrim = layerTrim->projection();

	//Querier Trim
	TeQuerierParams paramsTrim(true, true);
	paramsTrim.setParams(themeTrim);
	paramsTrim.setSelecetObjs(selObTrim);
	TeQuerier  querierTrim(paramsTrim); 

	if(!querierTrim.loadInstances())
		return  false;

	// ----------------- begin create attribute table

	TeAttributeList attrList;
	TeAttribute at;
	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "object_id_"+ Te2String(newLayer->id());
	at.rep_.numChar_ = 255;
	at.rep_.isPrimaryKey_ = true;
	at.rep_.isAutoNumber_ = false;
	attrList.push_back(at);

	//theme
	TeQuerierParams paramsTemp(false, true);
	paramsTemp.setParams(theme);
	TeQuerier* querierTemp = new TeQuerier(paramsTemp);

	if(!querierTemp->loadInstances())
		return false;

	//get attribute list from querier
	TeAttributeList attribs = querierTemp->getAttrList();
	TeAttributeList::iterator it = attribs.begin();
	while(it != attribs.end())
	{
		attrList.push_back (*it);
		++it;
	}
						
	delete querierTemp;

	//trim
	attribs.clear();
	attribs = querierTrim.getAttrList(); 

	//fill attributes from Trim theme 
	if(attrTrim)
	{
		it = attribs.begin();
		while(it != attribs.end())
		{
			attrList.push_back (*it);
			++it;
		}
	}

	changeAttrList(attrList);
	attrList[0].rep_.isPrimaryKey_ = true;  //link attribute is primary key
	
	TeTable attrTable (newLayer->name(), attrList, attrList[0].rep_.name_, attrList[0].rep_.name_);

	if(!newLayer->createAttributeTable(attrTable))
        return false;
	
	// ----------------- End create attribute table
	int numPolygonsTrim = querierTrim.numElemInstances();
	// set progress bar
	if(TeProgress::instance())
	{
		string caption = "Overlay Intersection";
		string msg = "Executing Overlay Intersection. Please, wait!";
		TeProgress::instance()->setCaption(caption.c_str());
		TeProgress::instance()->setMessage(msg);
		TeProgress::instance()->setTotalSteps(numPolygonsTrim);
		t2 = clock();
		t0 = t1 = t2;
	}

	double tol = TeGetPrecision(proj);
	TePrecision::instance().setPrecision(tol);

	int steps = 0;
	TeSTInstance trim;
	int newId = 0;

	while(querierTrim.fetchInstance(trim))
	{
		TePolygonSet polSet;
		if(!trim.getGeometry(polSet))
			continue;

		TeTableRow rowTrim;
		if(attrTrim)
		{
			//fill attribute values from trim
			TePropertyVector vecTrim = trim.getPropertyVector();

			for(int p=0; p<(int)vecTrim.size();++p)
				rowTrim.push_back(vecTrim[p].value_);
		}

		TePolygonSet ps;
		TeLineSet	ls;
		TePointSet pts;
		TeCellSet cs;

		for(int i=0; i<(int)polSet.size(); ++i)
		{
			TePolygon polyTrim = polSet[i];
						
			if(!((*proj) == (*projTrim)))
			{
				TePolygon pout;
				TeVectorRemap (polyTrim, projTrim, pout, proj);
				polyTrim = pout;
			}

			TePolygonSet polyTrimSet;
			polyTrimSet.add(polyTrim);

			TeBox boxTrim = polyTrim.box();

			//Querier - theme that will be clip
			TeQuerierParams params(true, true);
			params.setParams(theme);
			params.setSelecetObjs(selOb);
			params.setSpatialRest(boxTrim, TeINTERSECTS, TeGEOMETRYNONE);

			TeQuerier querier(params);

			if(!querier.loadInstances())
				continue;
			
			TeSTInstance sti;
			while(querier.fetchInstance(sti))
			{
				TeTableRow row;
				TePropertyVector vect = sti.getPropertyVector();
				row.push_back("");

				//insert values of the theme that is been clip 
				for(int p=0; p<(int)vect.size();++p)
					row.push_back(vect[p].value_);

				if(attrTrim)  
				{
					for(int i=0; i<(int)rowTrim.size(); ++i)
						row.push_back(rowTrim[i]);
				}
				
				if(sti.hasPolygons())
				{
					TePolygonSet polSet;
					sti.getGeometry (polSet);

					for(int j=0; j<(int)polSet.size(); ++j)
					{
						TePolygonSet psAux, resultPS;
						psAux.add(polSet[j]);
						if(!TeOVERLAY::TeIntersection(polyTrimSet, psAux, resultPS))
						{
							// when the operation returns false, write in the log a possible
							// inconsistency in the data
							if (logFile)
							{
								string mess = "Possivel inconsistencia na interseccao do objeto ";
								mess += polyTrim.objectId();
								mess += " do tema " + themeTrim->name() + " e ";
								mess += polSet[j].objectId();
								mess += theme->name();
								logFile->writeString(mess);
								logFile->writeNewLine();
							}
						}
						
						for(int c=0; c<(int)resultPS.size(); ++c)
						{
							newId++;
							TePolygon paux = resultPS[c];
							paux.objectId(Te2String(newId));
							ps.add(paux);
							row[0] = Te2String(newId);
							attrTable.add(row);
						}
					}
				}
				else if (sti.hasLines())
				{
					TeLineSet lineSet;
					sti.getGeometry (lineSet);

					for(int j=0; j<(int)lineSet.size(); ++j)
					{
						TeLineSet lsAux, resultLS;
						TeMultiGeometry resultMGEOM;
						lsAux.add(lineSet[j]);
						resultMGEOM = TeOVERLAY::TeIntersection(lsAux, polyTrimSet);

						resultMGEOM.getGeometry(resultLS);
						
						for(int c=0; c<(int)resultLS.size(); ++c)
						{
							newId++;
							TeLine2D laux = resultLS[c];
							laux.objectId(Te2String(newId));

							ls.add(laux);
							row[0] = Te2String(newId);
							attrTable.add(row);
						}
					}
				}
				else if (sti.hasPoints())
				{
					TePointSet pointSet;
					sti.getGeometry (pointSet);

					for(int j=0; j<(int)pointSet.size(); ++j)
					{
						if(TeWithin(pointSet[j], polyTrim))
						{
							newId++;
							pointSet[j].objectId(Te2String(newId));
							pts.add(pointSet[j]);
							row[0] = Te2String(newId);
							attrTable.add(row);
						}
					}
				}
				else if (sti.hasCells())
				{
					TeCellSet cellSet;
					sti.getGeometry (cellSet);

					for(int j=0; j<(int)cellSet.size(); ++j)
					{
						if(TeIntersects(cellSet[j], polyTrim))
						{
							newId++;
							cellSet[j].objectId(Te2String(newId));
							cs.add(cellSet[j]);
							row[0] = Te2String(newId);
							attrTable.add(row);
						}
					}

				}
			} //while fetchInstance 
		}//for each polygon of the Trim theme

		//inserir a geometria no banco
		if(layer->hasGeometry(TePOLYGONS) && ps.size())
			newLayer->addPolygons(ps);
		else if(layer->hasGeometry(TeLINES) && ls.size())
			newLayer->addLines(ls);
		else if(layer->hasGeometry(TePOINTS) && pts.size())
			newLayer->addPoints(pts);
		else if(cs.size())
			newLayer->addCells(cs);

		trim.clear();

		steps++;
		t2 = clock();
		if ((TeProgress::instance()) && (int(t2-t1) > dt))
		{
			t1 = t2;
			if(TeProgress::instance()->wasCancelled())
			{
				TeProgress::instance()->reset();
				return false;
			}
			if((int)(t2-t0) > dt2)
				TeProgress::instance()->setProgress(steps);
		}

		if((steps%100)==0)
		{
			newLayer->saveAttributeTable(attrTable); //save 100 records!!!
			if(attrTable.size()>0)
				insertAttr = true;
			attrTable.clear();
		}
	} //while fetch instance querier Trim

	if(TeProgress::instance())
			TeProgress::instance()->reset();

	newLayer->saveAttributeTable(attrTable); //save the last records!!!
	if(attrTable.size()>0)
		insertAttr = true;

	if(!newLayer->box().isValid())
		return false;

	if(!insertAttr)
		return false; 

	return true;
}

bool TeGeoOpAssignDataLocationDistribute(TeTheme* themeMod, TeTheme* theme, const string& tableName, 
										 const int& spatialRelation)
{
	if((!themeMod) || (!themeMod->layer()) ||  (!theme))
		return false;

	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;

	TeLayer* layerMod = themeMod->layer();	
	TeProjection* projMod = layerMod->projection();
	TeProjection* proj = theme->layer()->projection();
	
	//Querier from theme 
	TeQuerierParams params(true, true);
	params.setParams(theme);
	TeQuerier  querier(params); 

	if(!querier.loadInstances())
		return  false;

	// ----------------- begin create attribute table

	TeAttributeList attrList;
	TeAttribute at;
	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "object_id_";
	at.rep_.numChar_ = 255;
	at.rep_.isPrimaryKey_ = true;
	at.rep_.isAutoNumber_ = false;
	attrList.push_back(at);

	//get attribute list from querier
	TeAttributeList attribs = querier.getAttrList();
	TeAttributeList::iterator it = attribs.begin();
	while(it != attribs.end())
	{
		attrList.push_back (*it);
		++it;
	}
						
	changeAttrList(attrList);
	attrList[0].rep_.isPrimaryKey_ = true;  //link attribute is primary key
	
	TeTable attrTable (tableName, attrList, attrList[0].rep_.name_, attrList[0].rep_.name_);

	if(!layerMod->createAttributeTable(attrTable))
        return false;

	//empty rows
	TeTable updateAttrTable (tableName, attrList, attrList[0].rep_.name_, attrList[0].rep_.name_);
	
	// ----------------- End create attribute table
	vector<string> objIds;
	if( (theme->layer()->hasGeometry(TePOLYGONS)) || 
		(theme->layer()->hasGeometry(TeCELLS))    ||
		((themeMod->layer()->hasGeometry(TeLINES)) && (theme->layer()->hasGeometry(TeLINES)))) 
	{
		int numElem = querier.numElemInstances();
		
		// set progress bar
		if(TeProgress::instance())
		{
			string caption = "Spatial Join";
			string msg = "Executing Spatial Join. Please, wait!";
			TeProgress::instance()->setCaption(caption.c_str());
			TeProgress::instance()->setMessage(msg);
			TeProgress::instance()->setTotalSteps(numElem);
			t2 = clock();
			t0 = t1 = t2;
		}
	
		int steps = 0;
		TeSTInstance sti;

		double tol = TeGetPrecision(projMod);
		TePrecision::instance().setPrecision(tol);	

		while(querier.fetchInstance(sti))
		{
			// Get attributes
			TeTableRow row;
			TeTableRow emptyRow;

			row.push_back("");
			emptyRow.push_back("");

			TePropertyVector vec = sti.getPropertyVector();
			for(int p=0; p<(int)vec.size();++p)
			{
				row.push_back(vec[p].value_);
				emptyRow.push_back ("");
			}
			
			//Get geometry
			TeMultiGeometry mGeom;
			sti.getGeometry(mGeom);

			vector<TeGeometry*> geoms;
			mGeom.getGeometry(geoms);

			for(int i=0; i<(int)geoms.size(); ++i)
			{
				if(!((*proj) == (*projMod)))
				{
					TeGeometry* gout; 
					if(geoms[i]->elemType() == TePOLYGONS)
					{
						gout = new TePolygon;
						TeVectorRemap (*((TePolygon*)geoms[i]), proj, *((TePolygon*)gout), projMod);
					}
					else if (geoms[i]->elemType() == TeCELLS)
					{
						gout = new TeCell;
						TeVectorRemap (*((TeCell*)geoms[i]), proj, *((TeCell*)gout), projMod);
					}
					else
					{
						gout = new TeLine2D;
						TeVectorRemap (*((TeLine2D*)geoms[i]), proj, *((TeLine2D*)gout), projMod);
					}
					gout->objectId(geoms[i]->objectId());
					delete geoms[i];
					geoms[i] = gout;
				}
				
				//querier that will be modified
				TeQuerierParams paramsMod(false, false);
				paramsMod.setParams(themeMod);
				paramsMod.setSpatialRest(geoms[i], spatialRelation, TeGEOMETRYNONE);
				TeQuerier querierMod(paramsMod);

				if(!querierMod.loadInstances())
					continue;
				
				TeSTInstance stMod;
				while(querierMod.fetchInstance(stMod))
				{
					string obj = stMod.objectId();
					if( find(objIds.begin(), objIds.end(), obj) == objIds.end())
					{
						objIds.push_back (obj);
						row[0] = obj;
						attrTable.add(row);
					}
					else 
					{
						emptyRow[0] = obj;
						updateAttrTable.add(emptyRow);
					}
				}
				
				if(attrTable.size()>100)
				{
					layerMod->saveAttributeTable(attrTable); //save 100 records!!!
					layerMod->database()->updateTable(updateAttrTable);
					attrTable.clear();
					updateAttrTable.clear();
				}
			} // for
			
			for(int j=0; j<(int)geoms.size(); ++j)
				delete (geoms[j]);  //delete allocated geometries 

			steps++;
			t2 = clock();
			if ((TeProgress::instance()) && (int(t2-t1) > dt))
			{
				t1 = t2;
				if(TeProgress::instance()->wasCancelled())
				{
					TeProgress::instance()->reset();
					return false;
				}
				if((int)(t2-t0) > dt2)
					TeProgress::instance()->setProgress(steps);
			}

		} //while 
	} //if geometry type
	else 
	{
		return false;  //use nearest neighbor
	}
				
	if(TeProgress::instance())
		TeProgress::instance()->reset();

	layerMod->saveAttributeTable(attrTable); //save the last records!!!
	layerMod->database()->updateTable(updateAttrTable);
	
	return true;
}


bool TeGeoOpOverlayUnion(TeLayer* newLayer, TeTheme* theme, TeTheme* themeOverlay, 
						 TeSelectedObjects selOb, TeSelectedObjects selObOverlay, TeAsciiFile* logFile)
{
///////////////////////////////////////////////////////////////////
return false;
// verifivar se vai trabalhar com celulas
// se vai, modificar a interface.....
///////////////////////////////////////////////////////////////////
	if((!newLayer) || (!theme) || (!themeOverlay))
		return false;

	bool insertAttr = false;
	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;
		
	TeLayer* layer = theme->layer();
	TeLayer* layerOverlay = themeOverlay->layer();

	TeProjection* proj = layer->projection();
	TeProjection* projOverlay = layerOverlay->projection();

	//Querier Overlay
	TeQuerierParams paramsOverlay(true, true);
	paramsOverlay.setParams(themeOverlay);
	paramsOverlay.setSelecetObjs(selObOverlay);
	TeQuerier  querierOverlay(paramsOverlay); 

	if(!querierOverlay.loadInstances())
		return  false;

	// ----------------- begin create attribute table

	TeAttributeList attrList;
	TeAttribute at;
	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "object_id_"+ Te2String(newLayer->id());
	at.rep_.numChar_ = 255;
	at.rep_.isPrimaryKey_ = true;
	at.rep_.isAutoNumber_ = false;
	attrList.push_back(at);

	//theme
	TeQuerierParams paramsTemp(false, true);
	paramsTemp.setParams(theme);
	TeQuerier* querierTemp = new TeQuerier(paramsTemp);

	if(!querierTemp->loadInstances())
		return false;

	//get attribute list from querier
	TeAttributeList attribs = querierTemp->getAttrList();
	TeAttributeList::iterator it = attribs.begin();
	while(it != attribs.end())
	{
		attrList.push_back (*it);
		++it;
	}
						
	delete querierTemp;

	//overlay
	attribs.clear();
	attribs = querierOverlay.getAttrList(); 

	//fill attributes from Overlay theme 
	it = attribs.begin();
	while(it != attribs.end())
	{
		attrList.push_back (*it);
		++it;
	}

	changeAttrList(attrList);
	attrList[0].rep_.isPrimaryKey_ = true;  //link attribute is primary key
	
	TeTable attrTable (newLayer->name(), attrList, attrList[0].rep_.name_, attrList[0].rep_.name_);

	if(!newLayer->createAttributeTable(attrTable))
        return false;
	
	// ----------------- End create attribute table
	int numPolygonsOverlay = querierOverlay.numElemInstances();
	// set progress bar
	if(TeProgress::instance())
	{
		string caption = "Overlay Union";
		string msg = "Executing Overlay Union. Please, wait!";
		TeProgress::instance()->setCaption(caption.c_str());
		TeProgress::instance()->setMessage(msg);
		TeProgress::instance()->setTotalSteps(numPolygonsOverlay);
		t2 = clock();
		t0 = t1 = t2;
	}

	double tol = TeGetPrecision(layer->projection());
	TePrecision::instance().setPrecision(tol);	

	int steps = 0;
	TeSTInstance Overlay;
	int newId = 0;

	while(querierOverlay.fetchInstance(Overlay))
	{
		TePolygonSet polSet;
		if(!Overlay.getGeometry(polSet))
			continue;

		TeTableRow rowOverlay;
		//fill attribute values from Overlay
		TePropertyVector vecOverlay = Overlay.getPropertyVector();

		for(int p=0; p<(int)vecOverlay.size();++p)
			rowOverlay.push_back(vecOverlay[p].value_);

		TePolygonSet ps;
//		TeCellSet cs;

		for(int i=0; i<(int)polSet.size(); ++i)
		{
			TePolygon polyOverlay = polSet[i];
						
			if(!((*proj) == (*projOverlay)))
			{
				TePolygon pout;
				TeVectorRemap (polyOverlay, projOverlay, pout, proj);
				pout.objectId(polyOverlay.objectId());
				polyOverlay = pout;
			}

			TePolygonSet polyOverlaySet;
			polyOverlaySet.add(polyOverlay);

			TeBox boxOverlay = polyOverlay.box();

			//Querier - theme that will be clip
			TeQuerierParams params(true, true);
			params.setParams(theme);
			params.setSelecetObjs(selOb);
			params.setSpatialRest(boxOverlay, TeINTERSECTS, TeGEOMETRYNONE);

			TeQuerier querier(params);

			if(!querier.loadInstances())
				continue;
			
			TeSTInstance sti;
			while(querier.fetchInstance(sti))
			{
				TeTableRow row;
				TePropertyVector vect = sti.getPropertyVector();
				row.push_back("");

				//insert values of the theme that is been clip 
				for(int p=0; p<(int)vect.size();++p)
					row.push_back(vect[p].value_);

				for(int i=0; i<(int)rowOverlay.size(); ++i)
					row.push_back(rowOverlay[i]);
				
				if(sti.hasPolygons())
				{
					TePolygonSet polSet;
					sti.getGeometry (polSet);

					for(int j=0; j<(int)polSet.size(); ++j)
					{
						TePolygonSet psAux, resultPS;
						psAux.add(polSet[j]);
						if(!TeOVERLAY::TeIntersection(polyOverlaySet, psAux, resultPS))
						{
							// when the operation returns false, write in the log a possible
							// inconsistency in the data
							if (logFile)
							{
								string mess = "Possivel inconsistencia na interseccao do objeto ";
								mess += polyOverlay.objectId();
								mess += " e ";
								mess += polSet[j].objectId();
								logFile->writeString(mess);
								logFile->writeNewLine();
							}
						}
						
						for(int c=0; c<(int)resultPS.size(); ++c)
						{
							newId++;
							TePolygon paux = resultPS[c];
							paux.objectId(Te2String(newId));
							ps.add(paux);
							row[0] = Te2String(newId);
							attrTable.add(row);
						}
					}
				}
//				else if (sti.hasCells())
//				{
//					TeCellSet cellSet;
//					sti.getGeometry (cellSet);
//
//					for(int j=0; j<(int)cellSet.size(); ++j)
//					{
//						if(TeIntersects(cellSet[j], polyOverlay))
//						{
//							newId++;
//							cellSet[j].objectId(Te2String(newId));
//							cs.add(cellSet[j]);
//							row[0] = Te2String(newId);
//							attrTable.add(row);
//						}
//					}
//				}
			} //while fetchInstance 
		}//for each polygon of the Overlay theme

		//inserir a geometria no banco
		if(layer->hasGeometry(TePOLYGONS) && ps.size())
			newLayer->addPolygons(ps);
//		else if(cs.size())
//			newLayer->addCells(cs);

		Overlay.clear();

		steps++;
		t2 = clock();
		if ((TeProgress::instance()) && (int(t2-t1) > dt))
		{
			t1 = t2;
			if(TeProgress::instance()->wasCancelled())
			{
				TeProgress::instance()->reset();
				return false;
			}
			if((int)(t2-t0) > dt2)
				TeProgress::instance()->setProgress(steps);
		}

		if((steps%100)==0)
		{
			newLayer->saveAttributeTable(attrTable); //save 100 records!!!
			if(attrTable.size()>0)
				insertAttr = true;
			attrTable.clear();
		}
	} //while fetch instance querier Overlay

	if(TeProgress::instance())
			TeProgress::instance()->reset();

	newLayer->saveAttributeTable(attrTable); //save the last records!!!
	if(attrTable.size()>0)
		insertAttr = true;

	if(!newLayer->box().isValid())
		return false;

	if(!insertAttr)
		return false; 

	return true;
}


bool TeGeoOpAssignByLocationCollect(TeTheme* restrTheme, TeTheme* srcTheme, 
									const string& newTableName, TeGroupingAttr& measuresColl, 
									const int& spatialRelation)
{
	if(!restrTheme || !restrTheme->layer() ||		// no spatial restriction theme
	   !srcTheme ||									// no source theme
	   newTableName.empty() ||						// no output table name 
	   measuresColl.empty())							// no set of measures to collect
		return false;

	// verifies that spatial restriction is given by polygons or cells
	TeGeomRep restrRep = (TeGeomRep)(restrTheme->layer()->geomRep() & ~TeTEXT);
	if (restrRep != TePOLYGONS && restrRep != TeCELLS)
		return false;

	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;

	TeProjection* restrProj = restrTheme->layer()->projection();
	TeProjection* srcProj	= srcTheme->layer()->projection();

	bool doRemap = !(*restrProj == *srcProj);

	// try to retrieve the geometries of the spatial restriction theme
	TeQuerierParams params(true, false);
	params.setParams(restrTheme);
	TeQuerier spatialRestrictions(params); 

	if(!spatialRestrictions.loadInstances())
		return false;

	// there is at least one restriction geometry, so create the output table
	// defines its attribute list
	TeAttributeList attrList;
	TeAttribute at;
	at.rep_.type_ = TeSTRING;
	at.rep_.name_ = "object_id_";
	at.rep_.numChar_ = 255;
	attrList.push_back(at);
	vector<int>  countAttr;

	int attrNum = 1;
	TeGroupingAttr::iterator it = measuresColl.begin();		// for each different measure
	while (it != measuresColl.end())								// create a new attribute
	{
		at.rep_ = it->first;
		at.rep_.isPrimaryKey_ = false;
		string nameTemp = it->first.name_ + "_" + getStringFromStatistic(it->second);
		if(((restrTheme->layer()->database()->dbmsName()=="OracleAdo") || 
			(restrTheme->layer()->database()->dbmsName()=="OracleSpatial") ) &&
			(nameTemp.size() > 30))
		{
			int s = nameTemp.size(); 
			at.rep_.name_ = nameTemp.substr(s-30, s-1);
		}
		else
			at.rep_.name_ = nameTemp;
				
		if (it->second == TeCOUNT || it->second == TeVALIDCOUNT)
		{
			at.rep_.type_ = TeINT;
			countAttr.push_back (attrNum);
		}
		else if (at.rep_.type_ == TeINT)
			at.rep_.type_ = TeREAL;
		attrList.push_back(at);
		++it;
		++attrNum;
	}	
	changeAttrList(attrList);	// validates attribute list						
	attrList[0].rep_.isPrimaryKey_ = true;
	attrList[0].rep_.isAutoNumber_ = false;

	TeTable measuresTable (newTableName, attrList, attrList[0].rep_.name_, attrList[0].rep_.name_);
	if(!restrTheme->layer()->createAttributeTable(measuresTable)) // creates the new attribute table
        return false;

	//use function to point and cell
	TeGeomRep srcRep = (TeGeomRep)(srcTheme->layer()->geomRep() & ~TeTEXT);
	TeDatabase* db = restrTheme->layer()->database();
	int spatialRes = TeOVERLAPS | TeWITHIN | TeCOVEREDBY | TeCROSSES;
	bool sqlGroup = false;
	
	if( (*restrProj==*srcProj) && 
		(((restrRep==TeCELLS) && (srcRep==TePOINTS)) ||
		((restrRep==TeCELLS) && (srcRep==TeCELLS) && ((spatialRelation==TeINTERSECTS) || (spatialRelation==spatialRes)))))
	{
		if(TeCellStatistics(restrTheme, measuresTable, measuresColl, srcTheme, restrTheme->layer()->database()))
			sqlGroup = db->insertTableInfo(restrTheme->layer()->id(),measuresTable);
	}

	if(sqlGroup)
		return true;
	
	if(TeProgress::instance())								// sets the progress bar
	{
		string caption = "Assign Data By Location";
		string msg = "Executing collecting. Please, wait!";
		TeProgress::instance()->setCaption(caption.c_str());
		TeProgress::instance()->setMessage(msg);
		TeProgress::instance()->setTotalSteps(spatialRestrictions.numElemInstances());
		t2 = clock();
		t0 = t1 = t2;
	}

	double tol = TeGetPrecision(srcProj);
	TePrecision::instance().setPrecision(tol);	

	unsigned int nregions = 0;
	TePolygonSet restrPols;
	TeCellSet restrCells;
	TeSTInstance restrInstance;

	while(spatialRestrictions.fetchInstance(restrInstance)) // collect data for each geometry
	{														// of the spatial restriction set
		// defines a querier to select measures from objects that
		// are delimited by the a geometry and a spatial restriction
		TeQuerierParams par2(false,measuresColl);
		par2.setParams(srcTheme);
		restrCells.clear();
		restrPols.clear();
		TeMultiGeometry mGeom;
		restrInstance.getGeometry(mGeom);
		// get the geometry of the instance
		if (restrRep == TeCELLS)						// restriction is given by a cell set geometry
		{
			mGeom.getGeometry(restrCells);
			if (doRemap)						// bring it to the source projection
			{									// if necessary
				TeCellSet restrCellsAux;
				TeVectorRemap(restrCells,restrProj,restrCellsAux,srcProj);
				restrCells.clear();
				restrCells = restrCellsAux;
			}
			par2.setSpatialRest(&restrCells,spatialRelation,TeGEOMETRYNONE);
		}
		else									// restriction is given by a cell set geometry
		{
			mGeom.getGeometry(restrPols);
			if (doRemap)						// bring it to the source projection
			{									// if necessary
				TePolygonSet restrPolsAux;
				TeVectorRemap(restrPols,restrProj,restrPolsAux,srcProj);
				restrPols.clear();
				restrPols = restrPolsAux;
			}
			par2.setSpatialRest(&restrPols,spatialRelation,TeGEOMETRYNONE);
		}
		// creates a querier to retrieve the desired measures of the objects inside the 
		// spatial restriction
		TeQuerier sourceMeasures(par2);

		// process each source within the spatial restriction geometry
		TeTableRow row;
		int nrows = 0;
		row.push_back(restrInstance.objectId());	// identify the instance
		
		if (sourceMeasures.loadInstances())
		{
			TeSTInstance sourceInstance;
			unsigned int p;								// get the calculated attributes
			sourceMeasures.fetchInstance(sourceInstance);
			TePropertyVector vec = sourceInstance.getPropertyVector();
			for(p=0; p<vec.size();++p)
				row.push_back(vec[p].value_);
		}
		else 
		{
			//insert null or zero attribute value
			for(int p=1; p<(int)attrList.size();++p)
			{
				if(find(countAttr.begin(), countAttr.end(), p) != countAttr.end())
					row.push_back(Te2String(0));
				else
					row.push_back("");
			}
		}
	
		measuresTable.add(row);
		row.clear();
		nrows++;
			
		if(measuresTable.size()>100)	// saves chunks of 100 records
		{
			restrTheme->layer()->saveAttributeTable(measuresTable); 
			measuresTable.clear();
		}

		++nregions;
		t2 = clock();
		if ( int(t2-t1) > dt && TeProgress::instance())
		{
			t1 = t2;
			if(TeProgress::instance()->wasCancelled())
			{
				TeProgress::instance()->reset();
				return false;
			}
			if((int)(t2-t0) > dt2)
				TeProgress::instance()->setProgress(nregions);
		}
	}
	if (measuresTable.size() > 0)	// saves remaining records
	{
		restrTheme->layer()->saveAttributeTable(measuresTable); 
		measuresTable.clear();
	}

	if (TeProgress::instance())
		TeProgress::instance()->reset();
	return true;
}


bool TeGeoOpOverlayDifference(TeLayer* newLayer, TeTheme* theme1, 
							  TeTheme* theme2, TeSelectedObjects selOb1, 
							  TeSelectedObjects selOb2, TeAsciiFile* logFile)
{
	if((!newLayer) || (!theme1) || (!theme2))
		return false;

	int dt = CLOCKS_PER_SEC/4;
	int dt2 = CLOCKS_PER_SEC * 5;
	clock_t	t0, t1, t2;
		
	TeLayer* layer1 = theme1->layer();
	TeLayer* layer2 = theme2->layer();

	unsigned int t;
	vector<string> linkNames;
	TeAttrTableVector vetTables;
	if (layer1->getAttrTables(vetTables,TeAttrStatic))
	{
		for (t=0; t<vetTables.size(); ++t)
			linkNames.push_back(vetTables[t].linkName());
	}
	vetTables.clear();
	if (layer1->getAttrTables(vetTables,TeAttrEvent))
	{
		for (t=0; t<vetTables.size(); ++t)
			linkNames.push_back(vetTables[t].linkName());
	}

	TeProjection* proj1 = layer1->projection();
	TeProjection* proj2 = layer2->projection();

	// querier to get the objects from theme 1
	TeQuerierParams params1(true, true);
	params1.setParams(theme1);
	params1.setSelecetObjs(selOb1);
	TeQuerier  querier1(params1); 

	if(!querier1.loadInstances())
		return false;				// there aren't objects on theme 1

	// create the new attribute table with the attributes from theme 1 
	// preserving the same link attribute
	TeAttributeList attrList;
	int npk=-1, aux1=0;
	TeAttributeList theme1AttrList = querier1.getAttrList();
	TeAttributeList::iterator it = theme1AttrList.begin();
	while (it != theme1AttrList.end())
	{
		attrList.push_back(*it);
		vector<string>::iterator it2 = find(linkNames.begin(), linkNames.end(), (*it).rep_.name_);
		if (it2 != linkNames.end())
			npk = aux1;
		++it;
		++aux1;
	}

	// change attribute names if necessary and create a new attribute table
	changeAttrList(attrList);
	attrList[0].rep_.isPrimaryKey_ = true;  
	TeTable attrTable (newLayer->name(), attrList, attrList[npk].rep_.name_, attrList[npk].rep_.name_);
	if(!newLayer->createAttributeTable(attrTable))
       return false;

	// ----- 

	int numPolygons1 = querier1.numElemInstances();
	if(TeProgress::instance())
	{
		string caption = "Overlay Difference";
		string msg = "Executing Overlay Difference. Please, wait!";
		TeProgress::instance()->setCaption(caption.c_str());
		TeProgress::instance()->setMessage(msg);
		TeProgress::instance()->setTotalSteps(numPolygons1);
		t2 = clock();
		t0 = t1 = t2;
	}

	double tol = TeGetPrecision(proj1);
	TePrecision::instance().setPrecision(tol);

	int steps = 0;
	TeSTInstance elem1;
	while(querier1.fetchInstance(elem1))	// for each object of theme 1
	{
		TePolygonSet polSet1;
		if(!elem1.getGeometry(polSet1))
			continue;

		TePolygonSet resultAux;

		// querier to the theme 2
		TeQuerierParams params2(true,false);
		params2.setParams(theme2);
		params2.setSelecetObjs(selOb2);
		params2.setSpatialRest(polSet1.box(), TeINTERSECTS, TeGEOMETRYNONE);

		TeQuerier querier2(params2);
		if(!querier2.loadInstances())
		{
			// there is no intersection between this object and the objects from theme 2
			// so include the entire geometry of the object in the result
			for (unsigned int i=0; i<polSet1.size(); ++i)
				resultAux.add(polSet1[i]);
		}
		else
		{
			unsigned int m;
			TePolygonSet polSet2;
			TeSTInstance sti2;
			vector<string> oids;
			// build a polygon set with the representation of all objetcs from theme 2
			// that intercepts current object of theme 1
			while(querier2.fetchInstance(sti2))  
			{
				if(sti2.hasPolygons())
				{
					TePolygonSet polSetAux2;
					sti2.getGeometry (polSetAux2);
					if (!((*proj1) == (*proj2)))
					{
						TePolygonSet polySet2Out;
						TeVectorRemap (polSetAux2, proj2, polySet2Out, proj1);
						for (m=0; m<polySet2Out.size(); m++)
							polSet2.add(polySet2Out[m]);
					}
					else
					{
						for (m=0; m<polSetAux2.size(); m++)
							polSet2.add(polSetAux2[m]);
					}
					// keep track of the object ids for logging
					oids.push_back(sti2.objectId());
				}
			}// for each intersection object of theme 2 

			// try to recover the difference between the geometry of the current object
			// from theme 1 and the geometries of the objects from theme 2 that intercept
			// current object from theme 1
			if(!TeOVERLAY::TeDifference(polSet1, polSet2, resultAux))
			{
				// when the operation returns false, writes in the error log file a possible
				// inconsistency in the data
				if (logFile)
				{
					string mess = "Possivel inconsistencia na diferenca do objeto ";
					mess += polSet1.objectId();
					mess += " do tema " + theme1->name() + " e ";
					if (!oids.empty())
						mess += oids[0];
					for (m=1;m<oids.size();++m)
						mess += ", " + oids[m];
					mess += " do tema " + theme2->name();
					logFile->writeString(mess);
					logFile->writeNewLine();
				}
			} 
		}
		// save partial results
		if (!resultAux.empty())
		{
			TeTableRow row;
			TePropertyVector vecOverlay = elem1.getPropertyVector();

			for (unsigned int p=0; p<vecOverlay.size();++p)
				row.push_back(vecOverlay[p].value_);
			attrTable.add(row);
			row.clear();
			for (unsigned int l=0; l <resultAux.size(); l++)
				resultAux[l].objectId(vecOverlay[npk].value_);
			newLayer->addPolygons(resultAux);
		}
		steps++;
		t2 = clock();
		if ((TeProgress::instance()) && (int(t2-t1) > dt))
		{
			t1 = t2;
			if(TeProgress::instance()->wasCancelled())
			{
				TeProgress::instance()->reset();
				return false;
			}
			if((int)(t2-t0) > dt2)
				TeProgress::instance()->setProgress(steps);
		}
		if((steps%100)==0)
		{
			newLayer->saveAttributeTable(attrTable); 
			attrTable.clear();
		}
	} // for each object of theme 1

	if(TeProgress::instance())
		TeProgress::instance()->reset();

	if(attrTable.size()>0)
		newLayer->saveAttributeTable(attrTable); 

	if(!newLayer->box().isValid())
		return false;
	return true;
}
