/************************************************************************************
TerraLib - a library for developing GIS applications.
Copyright  2001-2004 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular purpose.
The library provided hereunder is on an "as is" basis, and the authors have no
obligation to provide maintenance, support, updates, enhancements, or modifications.
In no event shall INPE and Tecgraf / PUC-Rio be held liable to any party for direct,
indirect, special, incidental, or consequential damages arising out of the use
of this library and its documentation.
*************************************************************************************/

#include "TeCellAlgorithms.h"
#include "TeGeometryAlgorithms.h"
#include "TeDatabase.h"
#include "TeProgress.h"
#include "TeRepresentation.h"
#include "TeGeneralizedProxMatrix.h"

TeLayer* 
TeCreateCells( const string& layerName, TeLayer* layerBase, 
			   double resX, double resY, TeBox& box, bool mask)
{
	if (!layerBase || layerName.empty())
		return 0;

	TeDatabase* db = layerBase->database();
	if (!db)
		return 0;

	TeDatabasePortal* portal = db->getPortal();
	if (!portal)
		return 0;

	TeProjection* proj = layerBase->projection();

	string newLayerName = layerName;
	TeLayerMap& layerMap = db->layerMap();
	TeLayerMap::iterator it;
	bool flag = true;
	int n = 1;
	while (flag)
	{
		for (it = layerMap.begin(); it != layerMap.end(); ++it)
		{
			if (TeStringCompare(it->second->name(),newLayerName))
				break;
		}
		if (it == layerMap.end())
			flag = 0;
		else
			newLayerName = layerName + "_" +Te2String(n);
		n++;	
	}

	string polTableName;
	if (mask)
	{
		TeRepresentation* repp = layerBase->getRepresentation(TePOLYGONS);
		if (repp)
		{
			if (!box.isValid())
				box = adjustToCut(repp->box_,resX,resY);
			polTableName = repp->tableName_;
		}
		else
			mask = false;
	}
	else 
	{
		if (!box.isValid())
			box = adjustToCut(layerBase->box(),resX,resY);
	}

	TeBox newBox = adjustToCut(box,resX,resY);
	double x,y,x1,x2,y1,y2;
	x1 = newBox.x1_;
	y1 = newBox.y1_;
	x2 = newBox.x2_;
	y2 = newBox.y2_;

	int maxcols, maxlines;
	maxlines = TeRound((y2-y1)/resY);
	maxcols = TeRound((x2-x1)/resX);

	TeAttribute attribute;
	TeAttributeList attList;
	attribute.rep_.name_ = "object_id_";
	attribute.rep_.type_ = TeSTRING;
	attribute.rep_.numChar_ = 48;
	attribute.rep_.isPrimaryKey_ = true;
	attList.push_back ( attribute );
	attribute.rep_.name_ = "Col";
	attribute.rep_.type_ = TeINT;
	attList.push_back ( attribute );
	attribute.rep_.name_ = "Lin";
	attribute.rep_.type_ = TeINT;
	attribute.rep_.isPrimaryKey_ = false;
	attList.push_back ( attribute );	

	TeLayer* newLayer = new TeLayer(layerName,db,newBox,proj);
	if (!newLayer || newLayer->id() <= 0)
	{
		delete portal;
		return 0;
	}
	newLayer->addGeometry(TeCELLS);
	TeRepresentation* repp = newLayer->getRepresentation(TeCELLS);
	if (!repp)
	{
		db->deleteLayer(newLayer->id());
		delete portal;
		return 0;
	}
	repp->box_ = newBox;
	repp->resX_ = resX;
	repp->resY_ = resY;
	repp->nCols_ = maxcols;
	repp->nLins_ = maxlines;
	db->updateRepresentation(newLayer->id(),*repp);
	
	TeTable attTable (layerName,attList,"object_id_","object_id_",TeAttrStatic);
	newLayer->createAttributeTable(attTable);

	TeCellSet cells;
	cells.resX(resX);
	cells.resY(resY);

	int col,lin,ncels=0;
	bool status;
	if(TeProgress::instance())
		TeProgress::instance()->setTotalSteps(maxlines);

	TePolygon curPol;
	bool found = false;

	TePrecision::instance().setPrecision(TeGetPrecision(proj));
	
	y=y2;
	for (lin=0; lin<maxlines; ++lin) 
	{	
		double yu = y;
		y=y-resY;
		x=x1;

		for (col=0; col<maxcols; ++col)
		{
			TeBox box(x,y,x+resX,yu);
			found = false;
			// check if there should be used the polygon representation as a mask
			if (mask) 
			{
				TePolygon polBox = polygonFromBox(box);
				// check if the cell intersects the current polygon
				if (TeIntersects(polBox,curPol))
					found = true;
				else
				{
					if (db->spatialRelation(polTableName, TePOLYGONS, 
						(TePolygon*)&polBox, portal, TeINTERSECTS))
					{
						portal->fetchGeometry(curPol);
						found = true;
					}
					portal->freeResult();
				}
			}

			if (!mask || found)
			{
				// build geometry
				TeCell cell(box,col,lin);
				char celId[32];
				sprintf(celId,"C%02dL%02d",col,lin);
				cell.objectId(string(celId));
				cells.add(cell);
				
				// build default attributes
				TeTableRow row;
				row.push_back(cell.objectId()); 
				row.push_back(Te2String(col)); 
				row.push_back(Te2String(lin)); 
				attTable.add(row);
				ncels++;
			}
			x=x+resX;
		}
		if (attTable.size() > 0)	// if there is some attributes in this line
		{
			status = newLayer->saveAttributeTable(attTable);
			attTable.clear();
			status = newLayer->addCells(cells);
			cells.clear();
			if(TeProgress::instance())
			{
				if (TeProgress::instance()->wasCancelled())
					break;
				else
					TeProgress::instance()->setProgress(lin);
			}			
		}
	}
	delete portal;
	if (TeProgress::instance())
		TeProgress::instance()->reset();

	if (ncels > 0)
		return newLayer;
	else
	{
		db->deleteLayer(newLayer->id());
		return 0;
	}
}

	
TeLayer*
TeCreateCells(const string& layerName,TeTheme* theme, double resX, double resY, TeBox& box)
{
	TeLayer* inputLayer = theme->layer();
	TeRepresentation* pp = inputLayer->getRepresentation(TePOLYGONS);
	if (!pp)
		return 0;

	TeDatabase* db = inputLayer->database();
	TeDatabasePortal* portal = db->getPortal();
	if (!portal)
		return 0;

	string newLayerName = layerName;
	TeLayerMap& layerMap = db->layerMap();
	TeLayerMap::iterator it;
	bool flag = true;
	int n = 1;
	while (flag)
	{
		for (it = layerMap.begin(); it != layerMap.end(); ++it)
		{
			if (TeStringCompare(it->second->name(),newLayerName))
				break;
		}
		if (it == layerMap.end())
			flag = 0;
		else
			newLayerName = layerName + "_" +Te2String(n);
		n++;	
	}

	if (!box.isValid())
		box = pp->box_;

	TeBox newBox = adjustToCut(box,resX,resY);
	double x,y,x1,x2,y1,y2;
	x1 = newBox.x1_;
	y1 = newBox.y1_;
	x2 = newBox.x2_;
	y2 = newBox.y2_;

	int maxcols, maxlines;
	maxlines = TeRound((y2-y1)/resY);
	maxcols = TeRound((x2-x1)/resX);

	TeAttribute attribute;
	TeAttributeList attList;
	attribute.rep_.name_ = "object_id_";
	attribute.rep_.type_ = TeSTRING;
	attribute.rep_.isPrimaryKey_ = true;
	attribute.rep_.numChar_ = 48;
	attList.push_back ( attribute );

	attribute.rep_.name_ = "Col";
	attribute.rep_.type_ = TeINT;
	attribute.rep_.isPrimaryKey_ = false;
	attList.push_back ( attribute );

	attribute.rep_.name_ = "Lin";
	attribute.rep_.type_ = TeINT;
	attribute.rep_.isPrimaryKey_ = false;
	attList.push_back ( attribute );	

	TeLayer* newLayer = new TeLayer(layerName,db,newBox,inputLayer->projection());
	if (!newLayer || newLayer->id() <= 0)
	{
		return 0;
		delete portal;
	}
	newLayer->addGeometry(TeCELLS);
	TeRepresentation* repp = newLayer->getRepresentation(TeCELLS);
	if (!repp)
	{
		db->deleteLayer(newLayer->id());
		delete portal;
		return 0;
	}
	repp->box_ = newBox;
	repp->resX_ = resX;
	repp->resY_ = resY;
	repp->nCols_ = maxcols;
	repp->nLins_ = maxlines;
	db->updateRepresentation(newLayer->id(),*repp);

	TeTable attTable (layerName,attList,"object_id_","object_id_",TeAttrStatic);
	newLayer->createAttributeTable(attTable);

	string polTableName = inputLayer->tableName(TePOLYGONS);

	TeCellSet cells;
	cells.resX(resX);
	cells.resY(resY);

	TePolygon curPol;
	int col,lin;
	bool status;
	int ncels = 0;

	TePrecision::instance().setPrecision(TeGetPrecision(newLayer->projection()));

	if(TeProgress::instance())
		TeProgress::instance()->setTotalSteps(maxlines);

	bool found;
	y=y2;
	for (lin=0; lin<maxlines; ++lin) 
	{	
		double yu = y;
		y=y-resY;
		x=x1;
		for (col=0; col<maxcols; ++col)
		{
			found = false;
			TeBox box(x,y,x+resX,yu);

			TePolygon polBox = polygonFromBox(box);
			// check if the cell intersects the current polygon
			if (TeIntersects(polBox,curPol))
				found = true;
			else
			{
				if (db->spatialRelation(polTableName, TePOLYGONS, 
					(TePolygon*)&polBox, portal, TeINTERSECTS, theme->collectionTable()))
				{
					portal->fetchGeometry(curPol);
					found = true;
				}
				portal->freeResult();
			}
			if (found)
			{
				TeCell cell(box,col,lin);
				char celId[32];
				sprintf(celId,"C%02dL%02d",col,lin);
				cell.objectId(string(celId));
				cells.add(cell);

				TeTableRow row;
				row.push_back(cell.objectId()); 
				row.push_back(Te2String(col)); 
				row.push_back(Te2String(lin)); 
				attTable.add(row);
				++ncels;		
			}
			x=x+resX;
		}
		if (attTable.size() > 0)	// if there is some attributes in this line
		{
			status = newLayer->saveAttributeTable(attTable);
			attTable.clear();
			status = newLayer->addCells(cells);
			cells.clear();
			if(TeProgress::instance())
			{
				if (TeProgress::instance()->wasCancelled())
					break;
				else
					TeProgress::instance()->setProgress(lin);
			}	
		}
	}
	delete portal;
	if (TeProgress::instance())
		TeProgress::instance()->reset();

	if (ncels > 0)
		return newLayer;
	else
	{
		db->deleteLayer(newLayer->id());
		return 0;
	}
}


bool 
TeCellStatistics(TeTheme* themeOut, TeTable& newAttrTable, TeGroupingAttr& stat, 
				 TeTheme* themeIn, TeDatabase* db)
{

	TeGroupingAttr tempStat = stat;
	string sqlResut = db->getSQLStatistics (tempStat);

	bool dbStat = true;
	TeGroupingAttr::iterator it = tempStat.begin();
	while(it!= tempStat.end())
	{
		if(it->second != TeNOSTATISTIC)
		{
			dbStat = false;
			break;
		}
		++it;
	}
		
	if((!dbStat) || (sqlResut.empty()))
		return false; 

	TeDatabasePortal* portal = db->getPortal();

	// mount sql 
	//geom and collection tables from cell layer (cell)
	string tableGeomOut = themeOut->layer()->tableName (TeCELLS);
	string tableCollOut; 
	if(themeOut->hasRestriction())
		tableCollOut = themeOut->collectionTable();

	//geom and collection tables from in layer (point or cell)
	string tableGeomIn = themeIn->layer()->tableName (TePOINTS);
	TeGeomRep geomRepIn = TePOINTS;
	if(tableGeomIn.empty())
	{
		geomRepIn = TeCELLS;
		tableGeomIn = themeIn->layer()->tableName (TeCELLS);
	}

	string tableCollIn; 
	if(themeIn->hasRestriction())
		tableCollIn = themeIn->collectionTable();

	//sql from database
	string sql = " SELECT " + tableGeomOut + ".object_id,  ";
	sql += 	sqlResut;
	sql += " FROM " + tableGeomOut +","+ tableGeomIn;

	//for each point attr table 
	string sqlWhere ="";
	TeAttrTableVector vec = themeIn->attrTables();
	for(unsigned int index=0; index<vec.size(); ++index)
	{
		if( (vec[index].tableType() != TeAttrStatic) && 
			(vec[index].tableType() != TeAttrEvent)  &&
			(vec[index].tableType() != TeFixedGeomDynAttr)) //only to static attribute table
			continue;

		//from clause
		sql += ","+ vec[index].name();

		//where clause
		if(!sqlWhere.empty())
			sqlWhere += " AND ";
		sqlWhere += vec[index].name() +"."+ vec[index].linkName() +" = "+ tableGeomIn +".object_id ";
	}

	if(!tableCollIn.empty())
	{
		sql += ", "+ tableCollIn;
		if(!sqlWhere.empty())
			sqlWhere += " AND ";
	
		sqlWhere = tableGeomIn +".object_id = "+ tableCollIn +".c_object_id ";
	}
	if(!tableCollOut.empty())
	{
		sql += ", "+ tableCollOut;
		if(!sqlWhere.empty())
			sqlWhere += " AND ";
	
		sqlWhere += tableGeomOut +".object_id = "+ tableCollOut +".c_object_id ";
	}

	//box where
	if(!sqlWhere.empty())
		sqlWhere += " AND ";
	
	sqlWhere += db->getSQLBoxWhere (tableGeomOut, tableGeomIn, geomRepIn);

	sql += " WHERE "+ sqlWhere; 
	sql += " GROUP BY "+ tableGeomOut +".object_id";

	if(!portal->query(sql))
	{
		delete portal;
		return false;
	}
	
	//Keep statistics in the tableAttrCell table   
	if(!TeKeepStatistics(newAttrTable, portal))
	{
		delete portal;
		return false;
	}
	
	delete portal;
	return true;
}


bool	
TeKeepStatistics(TeTable& tableAttrCell, TeDatabasePortal* portal)
{
	
	TeAttributeList attr = portal->AttributeList();

	//Mount the update sql from portal
	string insert = "INSERT INTO "+ tableAttrCell.name() +" VALUES ( ";
	while(portal->fetchRow())
	{
		string insert2 = "";
		for(unsigned int i=0; i<attr.size(); ++i)
		{
			if(i>0)
				insert2 += ",";
			
			string val = portal->getData(i);
			if (attr[i].rep_.type_ == TeSTRING) 
				insert2 += "'"+ val +"'";
			else
				insert2 += val;
		}

		string result = insert + insert2 +" )";
		
		if(!portal->getDatabase()->execute(result))
			return false;
	}
		
	return true;
}	





//////////////////////////////////////////////////////////////////////
//
//				Fill Cell Auxiliary Funcions
//
/////////////////////////////////////////////////////////////////////


bool TeFillCellInitLoad (TeLayer* cell_layer, const string& cell_tablename, TeCellSet& cells)
{
	if (!cell_layer) return false;

	// get cells
	if (!cell_layer->getCells (cells)) return false;

	// if dynamic table was not created, create it
	TeTable table;
	if (!cell_layer->getAttrTablesByName(cell_tablename, table, TeFixedGeomDynAttr))
		if (!TeCreateBasicDymanicCellTable (cell_layer, cell_tablename))
			return false;

	return true;
}



void TeFillCellInitSTO (const TeCell& cell, TeTimeInterval& t, TePropertyVector& result, TeSTElementSet& cellObjSet)
{

		TeSTInstance cellObj;
		cellObj.objectId (cell.objectId());
		string uniqueId = cell.objectId();
		cellObj.addUniqueId (uniqueId); // ANAP
		uniqueId += t.getInitialDate() + t.getInitialTime();
		uniqueId += t.getFinalDate() + t.getFinalTime() ;
		cellObj.addUniqueId (uniqueId);
		cellObj.timeInterval (t);
		TePropertyVector::iterator itProp = result.begin();
		while (itProp != result.end())
		{
			cellObj.addProperty (*itProp);
			itProp++;
		}
		cellObjSet.insertSTInstance (cellObj);
		return;
}



bool
TeCreateBasicDymanicCellTable (TeLayer* cell_layer, const string cell_tablename)
{
	if (!cell_layer) return false;

		TeAttribute		attribute;
		TeAttributeList attList;
		TeAttributeList keyList;

		// attr_id
		attribute.rep_.name_ = "attr_id";
		attribute.rep_.type_ = TeSTRING;
		attribute.rep_.numChar_ = 48;
		attribute.rep_.isPrimaryKey_ = true;
		attList.push_back (attribute);
		keyList.push_back(attribute);

		// object_id
		attribute.rep_.name_ = "object_id";
		attribute.rep_.type_ = TeSTRING;
		attribute.rep_.numChar_ = 48;
		attList.push_back ( attribute );
	

		//  initial_time
		attribute.rep_.name_ = "initial_time";
	 	attribute.rep_.type_ = TeDATETIME;   
		attribute.dateTimeFormat_ = "YYYYsMMsDDsHHsMMsSS";
		attribute.dateChronon_ = TeSECOND;
		attribute.rep_.numChar_ = 48;
		attList.push_back ( attribute );


		//  final_time
		attribute.rep_.name_ = "final_time";
	 	attribute.rep_.type_ = TeDATETIME;  
		attribute.dateTimeFormat_ = "YYYYsMMsDDsHHsMMsSS";
		attribute.dateChronon_ = TeSECOND;
		attribute.rep_.numChar_ = 48;
		attList.push_back ( attribute );
		
	// Create table and initialize attributes

		TeTable cells_attTable (cell_tablename);       
		cells_attTable.setAttributeList(attList);
		cells_attTable.setTableType(TeFixedGeomDynAttr);
		cells_attTable.setLinkName("object_id");
		cells_attTable.setUniqueName("attr_id");
		cells_attTable.attInitialTime ("initial_time"); 
		cells_attTable.attFinalTime ("final_time"); 
		cells_attTable.attTimeUnit (TeSECOND); 
			
		if (!cell_layer->createAttributeTable(cells_attTable))
			return false;

		TeAttributeList attr; 
		cell_layer->database ()->getAttributeList (cell_tablename, attr);

		return true;
}



//////////////////////////////////////////////////////////////////////
//
//					Fill Cell Operations
//
/////////////////////////////////////////////////////////////////////

bool TeFillCellSpatialOperation (TeDatabase* db,
										const string& input_layername, 
										TeGeomRep rep,
										const string& input_tablename, 
										const string& input_attrname,
										TeTimeInterval t,
										const string& cell_layername, 
										const string& cell_tablename, 
										const string& output_columnName,
										TeComputeAttrSpatialStrategy* operation)

{
	if (!operation) return false;
	if (!db) return false;

// Load input layers
	TeLayer* input_layer = new TeLayer (input_layername);
	if (!db->loadLayer (input_layer))
	{
		 cout << "\tLayer de entrada inexistente: " << input_layername << endl; 
		 db->close();
		 return false;
	}

// Load output cells layer with geometry previously created
	TeLayer* cell_layer = new TeLayer (cell_layername);
	if (!db->loadLayer (cell_layer))
	{
		 cout << "\tLayer de entrada inexistente: " << cell_layername << endl; 
		 db->close();
		 return false;
	}

// Initialize cell set
	TeCellSet cells;
	if (!TeFillCellInitLoad (cell_layer, cell_tablename, cells)) return false;


// Initialize object set to store cell properties
	TeSTElementSet cellObjSet (cell_layer);

// Initialize theme

	TeTheme* theme = new TeTheme ("", input_layer);
	vector<string> attrTableNames;
	attrTableNames.push_back (input_tablename);
	TeAttrTableVector atts;
	if (!input_layer->getAttrTablesByName(attrTableNames, atts)) return false;
	theme->setAttTables (atts);

// Process
	TePropertyVector result;
	TeCellSet::iterator cell_it = cells.begin();
	while (cell_it != cells.end())
	{
		// Set restrictions on a theme and create stoset
		theme->setSpatialRest((*cell_it).box(), rep);	
		result = operation->compute (theme, input_attrname, (*cell_it).box(), output_columnName);
		TeFillCellInitSTO ((*cell_it), t, result, cellObjSet);
		cell_it++;

	} 

// Update DB
	if (!TeUpdateDBFromSet(&cellObjSet, cell_tablename))
			return  false;
	return true;
}



bool TeFillCellNonSpatialOperation (TeDatabase* db,
										const string& input_layername, 
										TeGeomRep rep,
										const string& input_tablename, 
										const string& input_attrname,
										TeTimeInterval t,
										const string& cell_layername, 
										const string& cell_tablename, 
										const string& output_columnName,
										TeComputeAttrStrategy<TeSTElementSet::propertyIterator>* operation)
{
	if (!db) return false;

// Load input layers
	TeLayer* input_layer = new TeLayer (input_layername);
	if (!db->loadLayer (input_layer))
	{
		 cout << "\tLayer de entrada inexistente: " << input_layername << endl; 
		 db->close();
		 return false;
	}

// Load output cells layer with geometry previously created
	TeLayer* cell_layer = new TeLayer (cell_layername);
	if (!db->loadLayer (cell_layer))
	{
		 cout << "\tLayer de entrada inexistente: " << cell_layername << endl; 
		 db->close();
		 return false;
	}

// Initialize cell set
	TeCellSet cells;
	if (!TeFillCellInitLoad (cell_layer, cell_tablename, cells)) return false;

// Initialize object set to store cell properties
	TeSTElementSet cellObjSet (cell_layer);

// Initialize theme
	TeTheme* theme = new TeTheme ("", input_layer);
	vector<string> attrTableNames;
	attrTableNames.push_back (input_tablename);
	TeAttrTableVector atts;
	if (!input_layer->getAttrTablesByName(attrTableNames, atts)) return false;
	theme->setAttTables (atts);

// Process
	TePropertyVector result;
	TeCellSet::iterator cell_it = cells.begin();
	while (cell_it != cells.end())
	{
		// Set restrictions on a theme and create stoset
		theme->setSpatialRest((*cell_it).box(), rep);	
		TeSTElementSet stos (theme);
		vector<string> attrNames;
		attrNames.push_back (input_attrname);	
		if (TeSTOSetBuildDB (&stos, false, false, attrNames))
				result = operation->compute (stos.begin(input_attrname), stos.end(input_attrname), output_columnName); 	// if property not found, stos iterator will return zero to operation. To check this, have to be less general and use getProperty inside the compute method.		

		TeFillCellInitSTO ((*cell_it), t, result, cellObjSet);	
		cell_it++;

	} 

// Update DB
	if (!TeUpdateDBFromSet (&cellObjSet, cell_tablename))
			return  false;
	return  true;
}





bool TeFillCellNonSpatialRasterOperation (TeDatabase* db,
										const string& input_raster_layername, 
										TeTimeInterval t,
										const string& cell_layername, 
										const string& cell_tablename, 
										const string& output_columnName,
										TeComputeAttrStrategy<TeRaster::iteratorPoly>* operation)
{
	if (!db) return false;

// Load input layers
	TeLayer* input_layer = new TeLayer (input_raster_layername);
	if (!db->loadLayer (input_layer))
	{
		 cout << "\tLayer de entrada inexistente: " << input_raster_layername << endl; 
		 db->close();
		 return false;
	}

	TeRaster* raster = input_layer->raster();
	if (!raster) return false;


// Load output cells layer with geometry previously created
	TeLayer* cell_layer = new TeLayer (cell_layername);
	if (!db->loadLayer (cell_layer))
	{
		 cout << "\tLayer de entrada inexistente: " << cell_layername << endl; 
		 db->close();
		 return false;
	}


// Initialize cell set
	TeCellSet cells;
	if (!TeFillCellInitLoad (cell_layer, cell_tablename, cells)) return false;


// Initialize object set to store cell properties
	TeSTElementSet cellObjSet (cell_layer);


// Process
	TePropertyVector result;
	TeCellSet::iterator cell_it = cells.begin();
	while (cell_it != cells.end())
	{
		TePolygon p = TeBox2Polygon((*cell_it).box());
		TeRaster::iteratorPoly rasterItBegin(raster);
		rasterItBegin = raster->begin(p, TeBBoxPixelInters, 0);
		TeRaster::iteratorPoly rasterItEnd(raster);
		rasterItEnd = raster->end(p, TeBBoxPixelInters, 0);

		result = operation->compute (rasterItBegin, rasterItEnd, output_columnName);										
		TeFillCellInitSTO ((*cell_it), t, result, cellObjSet);	
		cell_it++;
	} 

// Update DB
	if (!TeUpdateDBFromSet (&cellObjSet, cell_tablename))
			return  false;
	return  true;
}



bool TeFillCellAggregateOperation (  TeDatabase* db,
										const string& input_layername, 
										const string& input_tablename, 
										TeTimeInterval t,
										const string& cell_layername, 
										const string& cell_tablename, 
										vector<string>& attrNames,
										TeComputeAttrStrategy<TeSTElementSet::propertyIterator>* operation)
{
	if (!db) return false;


// Load input layers
	TeLayer* input_layer = new TeLayer (input_layername);
	if (!db->loadLayer (input_layer))
	{
		 cout << "\tLayer de entrada inexistente: " << input_layername << endl; 
		 db->close();
		 return false;
	}


// Load output cells layer with geometry previously created
	TeLayer* cell_layer = new TeLayer (cell_layername);

	if (!db->loadLayer (cell_layer))
	{
		 cout << "\tLayer de entrada inexistente: " << cell_layername << endl; 
		 db->close();
		 return false;
	}

// Initialize cell set
	TeCellSet cells;
	if (!TeFillCellInitLoad (cell_layer, cell_tablename, cells)) return false;


// Initialize object set to store cell properties
	TeSTElementSet cellObjSet (cell_layer);

// Initialize theme

	TeTheme* theme = new TeTheme ("", input_layer);
	vector<string> attrTableNames;
	attrTableNames.push_back (input_tablename);
	TeAttrTableVector atts;
	if (!input_layer->getAttrTablesByName(attrTableNames, atts)) return false;
	theme->setAttTables (atts);


// Process
	TeCellSet::iterator cell_it = cells.begin();
	while (cell_it != cells.end())
	{
		TeSTInstance cellObj;
		cellObj.objectId ((*cell_it).objectId());
		string uniqueId = (*cell_it).objectId();
		cellObj.addUniqueId (uniqueId); 
		uniqueId += t.getInitialDate() + t.getInitialTime();
		uniqueId += t.getFinalDate() + t.getFinalTime() ;
		cellObj.addUniqueId (uniqueId);
		cellObj.timeInterval (t);
		
		// Set restrictions on a theme and create stoset
		theme->setSpatialRest((*cell_it).box(), TeCELLS); //????? TeWITHIN ok? ANAP	
		TeSTElementSet stos (theme);
		TeSTOSetBuildDB (&stos, true, false, attrNames);
			
		// Create stoset based on input layer
		TePropertyVector result;
		vector<string>::iterator attIt = attrNames.begin();
		while (attIt != attrNames.end())
		{
			result = operation->compute (stos.begin(*attIt), stos.end(*attIt), (*attIt)); 	// if property not found, stos iterator will return zero to operation. To check this, have to be less general and use getProperty inside the compute method.		
			TePropertyVector::iterator itProp = result.begin();
			while (itProp != result.end())
			{
				cellObj.addProperty (*itProp);
				itProp++;
			}
			attIt++;
		}
		cellObjSet.insertSTInstance (cellObj);

		cell_it++;
	}

// Update DB
	if (!TeUpdateDBFromSet (&cellObjSet, cell_tablename))
			return  false;

	return  true;
}






bool TeFillCellConnectionOperation (	TeDatabase* db,
										TeSTElementSet objects,
										TeGeomRep rep,
										TeGraphNetwork* net,
										const string& input_attrName,

										TeTimeInterval t,
										const string& cell_layername, 
										const string& cell_tablename, 
										const string& output_columnName,
										
								
										TeComputeAttrStrategy<TeSTElementSet::propertyIterator>* operation,
										double local_distance_factor,
										double net_distance_factor,
										double net_conn_factor,
										double mult_factor)
{

	if (!operation) return false;
	if (!net) return false;
	if (!db) return false;

	// Load output cells layer with geometry previously created

	TeLayer* cell_layer = new TeLayer (cell_layername);

	if (!db->loadLayer (cell_layer))
	{
		 cout << "\tLayer de entrada inexistente: " << cell_layername << endl; 
		 db->close();
		 return false;
	}

// Initialize cell set
	TeCellSet cells;
	if (!TeFillCellInitLoad (cell_layer, cell_tablename, cells)) return false;


// Initialize object set to store cell properties
	TeSTElementSet cellObjSet (cell_layer);


// Process
	TePropertyVector result;
	TeCellSet::iterator cell_it = cells.begin();
	while (cell_it != cells.end())
	{
		TeSTInstance cellObj;
		cellObj.objectId ((*cell_it).objectId());
		string uniqueId = (*cell_it).objectId();
		cellObj.addUniqueId (uniqueId);
		uniqueId += t.getInitialDate() + t.getInitialTime();
		uniqueId += t.getFinalDate() + t.getFinalTime() ;
		cellObj.addUniqueId (uniqueId);
		cellObj.timeInterval (t);	
		cellObjSet.insertSTInstance (cellObj);
		cell_it++;
	} 

	TeProxMatrixOpenNetworkStrategy2    sc_net (&cellObjSet, TeCELLS, &objects, rep, 0, TeMAXFLOAT, TeMAXFLOAT, net);
 //	TeProxMatrixConnectionStrenghtStrategy sw(4, 0.0, false);
	TeProxMatrixInverseDistanceStrategy sw (local_distance_factor, net_distance_factor, net_conn_factor, mult_factor, false);
	TeProxMatrixNoSlicingStrategy ss_no;
	TeGeneralizedProxMatrix mat  (&sc_net, &sw, &ss_no);

	// initialize object set to store cell properties
	TeSTElementSet cellObjSet2 (cell_layer);
	TeCellSet::iterator cell_it2 = cells.begin();
	while (cell_it2 != cells.end())
	{
		TeSTElementSet neigh = mat.getSTENeighbours((*cell_it2).objectId());	

		result = operation->compute (neigh.begin(input_attrName), neigh.end(input_attrName), output_columnName); 	// if property not found, stos iterator will return zero to operation. To check this, have to be less general and use getProperty inside the compute method.		
		TeFillCellInitSTO ((*cell_it2), t, result, cellObjSet2);	
		cell_it2++;
	} 


// Update DB
	if (!TeUpdateDBFromSet (&cellObjSet2, cell_tablename))
			return  false;

	return  true;
}


